/* ============================================================
 * File  : streambrowser.h
 * Author: Eric Giesselbach <ericgies@kabelfoon.nl>
 * Description : stream handling,
 *               inspired by mythtv http://www.mythtv.org
 *
 * Copyright 2003 by Eric Giesselbach

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published bythe Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// development snapshot, many todo's

#ifndef STREAMBROWSER_H
#define STREAMBROWSER_H

#include <qwidget.h>
#include <qdatetime.h>
#include <qprocess.h>
#include <qvbox.h>
#include <qlayout.h>
#include <qtextedit.h>
#include <qlineedit.h>
#include <qpushbutton.h>

#include "streamstatus.h"
#include "harvester.h"
#include "streamstorage.h"
#include "itemtree.h"
#include "fft.h"
#include "configelements.h"
#include "recorder.h"

// Main app. GUI API

// holding audio spectrum
struct Spectrum
{
  int values[100];
};

class ViewerWindow : public QWidget
{
  Q_OBJECT

  public:
     ViewerWindow(QWidget *parent, const char *name, WFlags f);

     void setContent(QString& content);
     void keyPressEvent( QKeyEvent *k );
  private:
     QTextEdit *textEdit;
     QWidget *parentWidget;
     
};


// window displaying player stdout on request
class DumpWindow : public QWidget
{
  Q_OBJECT

  public:
     DumpWindow(QWidget *parent, const char *name, WFlags f);
     void setText(QString line);
     void clear();
     
  private slots:
     void slotCommitClicked();
     
  signals:
     void manualUrlEntered(QString url);
     
  private:
     QTextEdit *textEdit;
     QLineEdit *newUrl;
     QPushButton * commit;
};


class StreamBrowser : public QObject
{
    Q_OBJECT

  public:
    enum Parameter       // stream parameter, get it with getStreamParameter(Parameter)
    {                    // after eventValuesUpdated(EventUpdatedType) signal
      // stream status
      streamName = 0,
      streamDescr,
      streamUrl,
      streamStatusStr,
      streamAudioFormat,
      streamAudioChannels,
      streamAudioSps,
      streamAudioBps,
      streamVideoFormat,
      streamVideoCodec,
      streamVideoFps,
      streamVideoBps,
      streamVideoLength,
      // stream play
      streamTime,
      streamCache,
      streamStability,
      streamVolume
    };

    enum EventUpdatedType
    {
      streamstatus = 0, // "stream status"-Parameter update (fixed stream properties)
      streamplay,       // "stream play"-Parameter update (stream variables)
      folderlist,       // folderlist change
      itemlist         // itemlist change
      //, information       // information about StreamObject under cursor requested
    };

    enum Action         // Action related to StreamObject
    {
      none = 0,
      stream,
      folder,
      goback,
      quitharvest,
      quitstorage,
      quitmessage,
      quitinfo,
      storestation,
      storesite,
      storeitem,
      filterlist,
      appendmarked,    // append marked streams to selected storage
      htmlviewer,      // view with html renderer
      download         // download before play
    };
    
    enum ItemTreeLoaded
    {
      streams = 0, // itemTree loaded
      harvester,   // harvestTree loaded
      storages,    // storageTree loaded  
      messages,    // some messages
      info         // item details
    };

    StreamBrowser(QWidget *parent, QWidget *focusProxy, StreamStorage *storage);
    ~StreamBrowser();

    void setVideoRect(QRect& area);          // set video std screen bounds
    void setVideoMaxRect(QRect& area);       // set video fullscreen bounds
    void hideVideo();
    void showVideo();

    void hideViewer();
    void showViewer(QString& content);
    void setViewerRect(QRect& rect);
    void setViewerMaxRect(QRect& rect);

    void setItemDisplayFolded(int offset, bool folded);   // offset>=0: fixed cursorposition, -1: stepping cursor
    void setFolderDisplayFolded(int offset, bool folded); // folded: loop items in list

    void enableFFT(int displayPoints);        // enables FFT signals

    bool handlePressedKey(int key);   // see source for action values. todo: enumeration
    void initReady();                        // (gui) init ready.


    QString getStreamParameter(Parameter parameter);      // fixed stream parameters (player.xml)
    QString getStreamCustomParameter(unsigned int index, QString& area, QString& title); // user defined stream parameters (player.xml)
    unsigned int getStreamCustomParameterCount();         // number of user defined parameters

    QString  getHarvestedLink();             // get harvested link information
    Action   getCurrentFolderAction();       // get current folder action
    QString  getCurrentFolderCaption();      // get current folder caption (title of display)
    bool     getCurrentObjectDetails( QString& title, QString& field1, QString& field2, 
               bool preferFolder = false); // get current folder or item name, url, descr 
               
    // new, remove above? Deny folders focus?
    // ... what is this comment about?
    // Thank the lords of Kobol - that this isn't embedded software for a medical device
    bool getCurrentStreamObjectDetails(QString& folderName, QString& itemName,
                    QString& url, QString& descr, QString& handler, QStringList& meta);

    int getDisplayFolderList(int rows, QStringList& list, bool& onTop, bool& atBottom); // returns cursorposition in &list of
    int getDisplayItemList(int rows, QStringList& list, bool& onTop, bool& atBottom);   //         strings to display
    QString& getUserMessage();                             // see eventUserMessage() signal.

  signals:
    void eventItemTreeSwitchedTo(int itemTreeLoaded);   // reports change of itemTree
    void eventHarvesterBusy(bool busy, QString message); // fetching or processing http url.
    void eventVideoActive(bool active);           // video window shown / hidden
    void eventUserMessage();                      // message to fetch with getUserMessage
    void eventValuesUpdated(int eventUpdatedType);// see eventUpdatedType and Parameter
    void eventFFTReady(Spectrum* spectrum);       // *spectrum[0..fftNumPoints-1] FFT points
    void eventRecorderActive(bool active);             // one or more active recorders
  
  private slots:
    void streamStatusChanged();
    void streamPollEvent();
    void harvesterReady(int, unsigned int count);
    void slotStorageEvent(int ident, int eventType, bool error);
    void slotFFTReady(SpectrumValues*);

    void slotListLoaded();
    void slotRecordInserted(ChangedRecord* rec);
    void slotRecordUpdated(ChangedRecord* rec);
    void slotRecordRemoved(ChangedRecord* rec);
    void slotMarkedRecordInserted(ChangedRecord* rec);
    void markedStreamsInsertReady();  //sent by timer after last insert
    
    void slotRecordingStarted(QString recName);
    void slotRecordingStopped(QString recName, RecordStopReason reason);
    void slotScheduleEvent(QString recName, QString errorMessage, bool isScheduled);

    void destroyedDumpWindow();
    void handleExternalURI( QString uri );
    void slotDownloadFinished(QString url, bool error, QString errorMessage);

  protected:
    bool eventFilter( QObject *obj, QEvent *ev);

  private:
    typedef QPtrList<StreamItem> StreamCopyList;
    StreamCopyList streamCopyList;
    QStringList streamCopyResult;
    int streamCopyIndex;
    
    void folderPrev();
    void folderNext();
    void itemPrev();
    void itemNext();

    void setFolderList();
    void setStationList();
    void setVideoFields();

    void initStream();
    void startStream();
    void stopStream();
    void playerCommand(QString command);
    void streamUndetected();
    void checkFolderCommand();
    void checkRecordCommand();
    void showHtmlViewer();
    void startDownload();
    
    void storeStationInFolder(QString folderName);
    void storeSiteInFolder(QString folderName);
    void storeLinkInFolder(QString folderName);
    void storeItemInFolder(QString folderName, QString name,
                           QString url, QString descr, QString handler);
    void markStreamItem();
    void storeMarkedStreamsPrepare(QString storageName);
    bool storeMarkedStreamsInsertNext(bool first);

    void initRecording(QString url, QString name, QString descr, QString handler);
    void stopRecording();

    void checkAndSetIcon(StreamItem *item);

    void loadDumpWindow();
    void toggleDumpWindow();

    StreamStatus    *streamStatus;
    StreamHarvester *streamHarvester;
    Downloader      *downloader;
    StreamStorage   *streamStorage;
    RecorderManager *recorderManager;

    void setStorageMode(bool goStorageMode);
    void setMessageMode(bool goMessageMode, QStringList* list);    
    void setInfoMode(bool goInfoMode);
    
    void setHarvestMode(bool goAndHarvest);
    void fillHarvestList(bool filterStreams);
    ItemTree databaseTree;  // stream folders and items
    ItemTree harvestTree;   // harvested links and harvester commands
    ItemTree storageTree;   // storage names (copying marked stream urls)
    ItemTree messageTree;   // messages (e.g. result of copying marked streams)
    ItemTree infoTree;      // item detail screen
    ItemTree *itemTree;     // pointer to one of the trees above
    ItemTreeLoaded itemTreeLoaded; // the xxxxTree that *itemTree refers to
    ItemTreeLoaded lastItemTreeLoaded; // former itemTreeLoaded (to return to when leaving info mode)
    
    ItemTree *prevItemTree;     // pointer to previous tree (use with message, storage and info)
    ItemTreeLoaded prevItemTreeLoaded; // previous tree (use with message, storage and info)
    
    StreamStorage *markedStorage; // holds storage to append marked stream items to
        
    void reportEvent(const QString& eventInfo);

    bool insertStationInStorage(QString& folder, QString& name, QString& url, QString& descr, QString& handler);

    unsigned int streamCustomParameterCount; // StreamCustomEvent% with 0 <= % < count in player.xml

    QString userMessage;
    bool video;

    QWidget *focusProxy;

    bool m_itemdisplayfolded;
    int  m_itemdisplayoffset;
    bool m_folderdisplayfolded;
    int  m_folderdisplayoffset;

    // FFT
    SampleObject *sampler;
    FFTConverter *converter;
    void samplerActive(bool active);

    VideoContainer  *videoContainer;
    ViewerWindow    *viewerWindow;
    DumpWindow      *m_dumpwindow;
    QRect m_viewerrect;
    QRect m_viewermaxrect;

};


#endif /* STREAMBROWSER_H */
