/* ============================================================
 * File        : itemtree.cpp
 * Author      : Eric Giesselbach <ericgies@kabelfoon.nl>
 * Date        : 2003-12-22
 * Description : tree of stream folder and items
 *
 * Copyright 2003 by Eric Giesselbach

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published bythe Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// development snapshot, many todo's

#include <iostream>

#include <qdir.h>
#include <qdict.h>

#include "itemtree.h"


using namespace std;

static QString emptyStr = "";
static QString nullStr  = QString::null;

StreamObject::StreamObject(const QString& sName) : QObject()
{
    name = sName;
    marked = false;
    prefix = " "; // for folder/item list display
    objectList.setAutoDelete(true);
    objectIndex = 0;
    objectList.clear();
    
    displaySize   = 0;
    displayOffset = 0;
    displayFolded = true;

    // display cursor on childlist
    cursor = 0;

    // action on select
    action = 0;
}

StreamObject::~StreamObject()
{
    objectList.clear();
    objectIndex = 0;
}

void StreamObject::setName(QString nme)
{
   name = nme;
   prefix = " ";
}

QString& StreamObject::getName()
{
   return name;
}

void StreamObject::addObjectToList(StreamObject* streamObject)
{
    objectList.append(streamObject);
    if (  objectList.count() == 1 )
      objectIndex = 0;
}

void StreamObject::removeObjectFromList(StreamObject* streamObject)
{
    objectList.remove(streamObject);
    if (  objectList.count() == 0 )
      objectIndex = 0;
}

void StreamObject::setAction(unsigned int act)
{
  action = act;
}

void StreamObject::setPrefix(QString prefix)
{
  this->prefix = prefix;
}

QString& StreamObject::getPrefixedName()
{  
  if (!marked)
    prefixedName = prefix + name;
  else
    prefixedName = "!" + name;
    
  return prefixedName;
}

void StreamObject::toggleMarked()
{
    marked = !marked;
}

unsigned int StreamObject::getAction()
{
  return action;
}

StreamObject* StreamObject::getObject()
{
  if ( objectIndex < int(objectList.count()) && objectIndex >= 0 )
    return objectList.at(objectIndex);
  else
    return 0;
}

StreamObject* StreamObject::findObject(const QString& searchName)
{
    StreamObject* obj = 0;
    for ( unsigned int i = 0; i < objectList.count(); i++)
      if ( objectList.at(i)->name == searchName)
        obj = objectList.at(i);
    return obj;
}

void StreamObject::nextObject(bool folded)
{
    int cnt = objectList.count();
    if ( cnt > 0 && !( !folded && objectIndex>=(cnt-1) ) )
    {
      objectIndex++;
      if ( objectIndex >= cnt)
        objectIndex = 0;
    }
}

bool StreamObject::nextMarkedObject()
{
   int cnt = objectList.count();
   bool fnd = false;
   
   if ( cnt > 0 && !( objectIndex >= (cnt-1) ) )
   {
     for (unsigned int i = objectIndex + 1; i < objectList.count(); i++)
     if ( objectList.at(i)->marked )
     {
        objectIndex = i;
        i = objectList.count();
        fnd = true;
     }
   }

   return fnd;
}

void StreamObject::prevObject(bool folded)
{
  if ( objectList.count() > 0 && !( !folded && objectIndex<=0 ) )
  {
    objectIndex--;
    if ( objectIndex < 0 )
      objectIndex = objectList.count() - 1;
  }
}


void StreamObject::setIndex(int index)
{
    int cnt = objectList.count();
    if ( index >= 0 && index < cnt )
      objectIndex = index;
}

int StreamObject::getIndex()
{
    return objectIndex;
}

void StreamObject::resetDisplaySize(bool foldDisplay, int windowSize, int offset, bool& onTop, bool& atBottom)
{
   displayFolded = foldDisplay;
   displaySize   = windowSize;
   displayOffset = offset;
   displayIndex  = objectIndex - offset;
   
   onTop    = ( displayIndex <= 0 );
   atBottom = ( uint(displayIndex + windowSize) >= objectList.count() );
}

QString& StreamObject::getDisplayStr(int index)
{
  unsigned int cnt = objectList.count();
  int count = QMAX(cnt, displaySize);

  if ( displayFolded )
  {
      if ( index < 0 )
        index += count;
      if ( index >= count )
        index -= count;
  }

  if ( index >= 0 && index < int(cnt))
  {
     StreamObject* streamObject = objectList.at(index);
     if ( streamObject )
       return streamObject->getPrefixedName();
     else
       return emptyStr;
  }
    else return nullStr; 
}

QString& StreamObject::getNextDisplayStr()
{
   if ( displayIndex < int (objectIndex - displayOffset + displaySize) )
     return getDisplayStr(displayIndex++);
   else
     return nullStr;
}

void StreamObject::clearObjectList()
{
   objectList.clear();
   objectIndex = 0;
   cursor = 0;
}


// ----------------------------------------------------------------------

StreamItem::StreamItem(StreamFolder *sParent, const QString& sName,
                         const QString& sUrl, const QString& sDescr, const QString& sHandler)
           : StreamObject(sName)
{
    url     = sUrl;
    descr   = sDescr;
    handler = sHandler;
    parent  = sParent;
    parent->addObjectToList(this);
    setAction(1); // oops #1
}

QString& StreamItem::getFolderName()
{  
   if (parent)
     return parent->getName();
   else
     return nullStr;
}


// ----------------------------------------------------------------------

StreamFolder::StreamFolder( const QString& sName)
            : StreamObject( sName )
{
    setAction(2); //oops #2
    caption = "";
    descr = "";
}

StreamItem* StreamFolder::getStreamItem()
{
    return dynamic_cast<StreamItem*>( getObject() );
}

// ----------------------------------------------------------------------

ItemTree::ItemTree() : StreamObject( "" )
{
}

ItemTree::~ItemTree()
{
}

StreamFolder* ItemTree::getStreamFolder()
{
    return dynamic_cast<StreamFolder*>( getObject() );
}

