#define TRACE_INTERNALS
#include "trace.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

/* MS compiler has _snprintf rather than snprintf */

#ifdef _MSC_VER
#define snprintf _snprintf
#endif

/* Indentation and output files */

int   indent = 0;
int   indentStep = 2;
char  indentBuffer[1024];
int   base64 = 1;
FILE *out = NULL;
FILE *err = NULL;

#define MAX_GET_PROC_ADDRESS 32

static GetProcAddressFunc getProcAddress[MAX_GET_PROC_ADDRESS];
static int numGetProcAddress = 0;

static const char *getEnvironment(const char *var)
{
  const char *ret = NULL;

  if (var)
  {
    ret = getenv(var);

    /* Treat empty environment variable the same as non-existant */
    if (!ret || *ret=='\0')
      return NULL;
  }

  return ret;
}

void traceInit(void)
{
  static int initialized = 0;
  const char *outFileName = NULL;
  const char *errFileName = NULL;

  if (!initialized)
  {
    outFileName = getEnvironment("CG_TRACE_FILE");
    errFileName = getEnvironment("CG_TRACE_ERROR");
    if (outFileName)
    {
      out = fopen(outFileName, "w");
      if (!out)
        fprintf(stderr, "Failed to open trace file: %s\n", outFileName);
    }
    if (errFileName)
    {
      if (outFileName && !strcmp(errFileName, outFileName))
        err = out;
      else
      {
        err = fopen(errFileName, "w");
        if (!err)
          fprintf(stderr, "Failed to open error file: %s\n", errFileName);
      }
    }
    out = out ? out : stdout;
    err = err ? err : stderr;
    initialized = 1;
  }
}

const char *traceLibraryLocation(const char *lib)
{
  /* Force MS compiler to support floating point */

#if defined(_WIN32)
  float dummy = 0.0f;
#endif

  /* Where to look for Cg runtime libraries */

  const char *ret = NULL;

  #define bufferSize 1024
  static char buffer[bufferSize];

  traceInit();

  if (!strcmp(lib, "Cg"))
  {
    /* For OSX, use libCg.dylib symlink */

#if defined(__APPLE__)
    return "libCg.dylib";
#endif

    /* First try CG_TRACE_CG_LIBRARY variable */

    ret = getEnvironment("CG_TRACE_CG_LIBRARY");

    /* Second, try CG_BIN_PATH or CG_LIB_PATH variable */

    if (!ret)
    {
#if defined(_WIN32)
      ret = getEnvironment("CG_BIN_PATH");
      if (ret)
      {
        snprintf(buffer, bufferSize, "%s\\cg.dll", ret);
        ret = buffer;
      }
#else
      ret = getEnvironment("CG_LIB_PATH");
      if (ret)
      {
        snprintf(buffer, bufferSize, "%s/libCg.so", ret);
        ret = buffer;
      }
#endif
    }

    /* Third, try default installation location */

    if (!ret)
    {
#if defined(_WIN32)
      ret = getEnvironment("PROGRAMFILES");
      if (ret)
      {
        snprintf(buffer, bufferSize, "%s\\NVIDIA Corporation\\Cg\\bin\\cg.dll", ret);
        ret = buffer;
      }
#else
      ret = "/usr/lib/libCg.so";
#endif
    }
  }

  if (!strcmp(lib, "CgGL"))
  {
    /* For OSX, use libCg.dylib symlink */

#if defined(__APPLE__)
    return "libCg.dylib";
#endif

    /* First try CG_TRACE_CGGL_LIBRARY variable */

    ret = getEnvironment("CG_TRACE_CGGL_LIBRARY");

    /* Second, try CG_BIN_PATH or CG_LIB_PATH variable */

    if (!ret)
    {
#if defined(_WIN32)
      ret = getEnvironment("CG_BIN_PATH");
      if (ret)
      {
        snprintf(buffer, bufferSize, "%s\\cgGL.dll", ret);
        ret = buffer;
      }
#else
      ret = getEnvironment("CG_LIB_PATH");
      if (ret)
      {
        snprintf(buffer, bufferSize, "%s/libCgGL.so", ret);
        ret = buffer;
      }
#endif
    }

    /* Third, try default installation location */

    if (!ret)
    {
#if defined(_WIN32)
      ret = getEnvironment("PROGRAMFILES");
      if (ret)
      {
        snprintf(buffer, bufferSize, "%s\\NVIDIA Corporation\\Cg\\bin\\cgGL.dll", ret);
        ret = buffer;
      }
#else
      ret = "/usr/lib/libCgGL.so";
#endif
    }
  }

  if (!strcmp(lib, "GL") || !strcmp(lib, "WGL") || !strcmp(lib, "GLX"))
  {
    /* For OSX, use libGL.dylib symlink */

#if defined(__APPLE__)
    return "libGL.dylib";
#endif

    /* First, try CG_TRACE_GL_LIBRARY variable */

    ret = getEnvironment("CG_TRACE_GL_LIBRARY");

    /* Second, try default installation location */

    if (!ret)
    {
#if defined(_WIN32)
      ret = getEnvironment("windir");
      if (ret)
      {
        /* TODO: How to choose System32 vs SysWOW64 */
        snprintf(buffer, bufferSize, "%s\\system32\\opengl32.dll", ret);
        ret = buffer;
      }
#else
      ret = "/usr/lib/libGL.so";
#endif
    }
  }

  if (!strcmp(lib, "GLUT"))
  {
    /* For OSX, use libGLUT.dylib symlink */

#if defined(__APPLE__)
    return "libGLUT.dylib";
#endif

    /* First try CG_TRACE_GLUT_LIBRARY variable */

    ret = getEnvironment("CG_TRACE_GLUT_LIBRARY");

    /* Second, try CG_BIN_PATH or CG_LIB_PATH variable */

    if (!ret)
    {
#if defined(_WIN32)
      ret = getEnvironment("CG_BIN_PATH");
      if (ret)
      {
        snprintf(buffer, bufferSize, "%s\\glut32.dll", ret);
        ret = buffer;
      }
#endif
    }

    /* Third, try default installation location */

    if (!ret)
    {
#if defined(_WIN32)
      ret = getEnvironment("PROGRAMFILES");
      if (ret)
      {
        snprintf(buffer, bufferSize, "%s\\NVIDIA Corporation\\Cg\\bin\\glut32.dll", ret);
        ret = buffer;
      }
#else
      ret = "/usr/lib/libglut.so";
#endif
    }
  }

  return ret;
}

void
traceRegister(GetProcAddressFunc f)
{
  traceInit();

  if (f && numGetProcAddress < MAX_GET_PROC_ADDRESS)
    getProcAddress[numGetProcAddress++] = f;
}

void *
traceGetProcAddress(const char *name)
{
  int i;
  void *proc = NULL;

  if (!name)
    return NULL;

  for (i = 0; i < numGetProcAddress; ++i)
  {
    proc = getProcAddress[i](name);
    if (proc)
      return proc;
  }

  return NULL;
}

void updateIndent(int step)
{
  int i;

  if (step>0)
  {
    for (i=0; i<step; ++i)
      indentBuffer[indent++] = ' ';
  }
  else
    indent += step;

  indentBuffer[indent] = '\0';
}

void traceInfo(const char *format, ...)
{
  va_list argptr;

  traceInit();

  fprintf(err, "info: ");
  va_start(argptr, format);
  vfprintf(err, format, argptr);
  va_end(argptr);
  fprintf(err, "\n");
}

void traceWarning(const char *format, ...)
{
  va_list argptr;

  traceInit();

  fprintf(err, "warning: ");
  va_start(argptr, format);
  vfprintf(err, format, argptr);
  va_end(argptr);
  fprintf(err, "\n");
}

void traceError(const char *format, ...)
{
  va_list argptr;

  traceInit();

  fprintf(err, "error: ");
  va_start(argptr, format);
  vfprintf(err, format, argptr);
  va_end(argptr);
  fprintf(err, "\n");
  fflush(err);
}
