#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>

#include <sys/types.h>
#include <sys/wait.h>

#include "vobsub.h"
#include "spudec.h"
#include "mp_msg.h"

static char srtxname[FILENAME_MAX];
static char pgmname[FILENAME_MAX];
static spudec_handle_t *spudec;
static unsigned int sub_idx;
static FILE *fsrtx;

int forced;
int sub_pos;
int vobsub_id;
int verbosity = MSGL_STATUS;

static void
usage (const char *name)
{
  fprintf (stdout, "Usage:\n");
  fprintf (stdout, "  %s [OPTION...] <vobsub basename>\n", name);
  fprintf (stdout, "\n");
  fprintf (stdout, "Help Options:\n");
  fprintf (stdout, "  -h, --help                  Show help options\n");
  fprintf (stdout, "\n");
  fprintf (stdout, "Application Options:\n");
  fprintf (stdout, "  -o, --output=<filename>     Use filename for output (default: stdout)\n");
  fprintf (stderr, "  -s, --sid=<sid>             The subtitle id (default: 0)\n");
  fprintf (stderr, "  -f, --forced                Extract only forced subtitles\n");
  fprintf (stderr, "  -v, --verbose               Increase verbosity level\n");
  fprintf (stdout, "\n");
}

static void
output_srtx (const char *const fname, unsigned int start, unsigned int end)
{
  int temp, h, m, s, ms;

  fprintf (fsrtx, "%d\n", sub_idx);

  temp = start;
  temp /= 90;
  h = temp / 3600000;
  temp %= 3600000;
  m = temp / 60000;
  temp %= 60000;
  s = temp / 1000;
  temp %= 1000;
  ms = temp;

  fprintf (fsrtx, "%02d:%02d:%02d,%03d --> ", h, m, s, ms);

  temp = end;
  temp /= 90;
  h = temp / 3600000;
  temp %= 3600000;
  m = temp / 60000;
  temp %= 60000;
  s = temp / 1000;
  temp %= 1000;
  ms = temp;

  fprintf (fsrtx, "%02d:%02d:%02d,%03d\n", h, m, s, ms);
  fprintf (fsrtx, "%s.txt\n\n", fname);
  fflush (fsrtx);
}

static void
output_pgm (FILE * f, int w, int h, unsigned char *src, unsigned char *srca,
            int stride)
{
  int x, y;
  fprintf (f, "P5\n" "%d %d\n" "255\n", w, h);
  for (y = 0; y < h; ++y)
  {
    for (x = 0; x < w; ++x)
    {
      int res;
      if (srca[x])
        res = src[x] * (256 - srca[x]);
      else
        res = 0;
      res = (65535 - res) >> 8;
      putc (res & 0xff, f);
    }
    src += stride;
    srca += stride;
  }
  putc ('\n', f);
}

static void
draw_alpha (int x0, int y0, int w, int h, unsigned char *src,
            unsigned char *srca, int stride)
{
  char buf[FILENAME_MAX];
  FILE *f;

  sprintf (buf, "%s%04d.pgm", pgmname, ++sub_idx);

  f = fopen (buf, "w");
  output_pgm (f, w, h, src, srca, stride);
  fclose (f);

  output_srtx (buf, spudec->start_pts, spudec->end_pts);
}

static const char *shortopts = "hfvs:o:";
static const struct option longopts[] =
{ 
  { "help",    no_argument,       NULL, 'h' },
  { "forced",  no_argument,       NULL, 'f' },
  { "verbose", no_argument,       NULL, 'v' },
  { "sid",     required_argument, NULL, 's' },
  { "output",  required_argument, NULL, 'o' },
  { NULL,      0,                 NULL,  0  }
};

int
main (int argc, char **argv)
{
  const char *vobsubname;
  void *vobsub, *packet;
  unsigned int pts100;
  int packet_len, ch, optidx;

  if (argc < 2)
  {
    usage (argv[0]);
    return EXIT_FAILURE;
  }

  while ((ch = getopt_long (argc, argv, shortopts, longopts, &optidx)) != EOF)
  {
    switch (ch)
    {
      case 's':
        if (sscanf (optarg, "%d", &vobsub_id) != 1)
        {
          fprintf (stderr, "vobsub_id for -g option\n");
          return EXIT_FAILURE;
        }
        break;
      case 'o':
        if (sscanf (optarg, "%s", pgmname) != 1)
        {
          fprintf (stderr, "no filename specified to option -i\n");
          return EXIT_FAILURE;
        }
        break;
      case 'f':
        forced ++;
        break;
      case 'v':
        verbosity ++;
        break;
      case 'h':
        usage (argv[0]);
        return EXIT_SUCCESS;
        break;
      default:
        fprintf (stderr, "Unknown option. Use -h to list all valid options.\n");
        return EXIT_FAILURE;
        break;
    }
  }

  if (optind >= argc)
  {
    usage (argv[0]);
    return EXIT_FAILURE;
  }

  vobsubname = argv[optind];
  if (pgmname[0] == 0)
    strncpy (pgmname, vobsubname, FILENAME_MAX - 10);

  strncpy (srtxname, pgmname, FILENAME_MAX - 5);
  strncat (srtxname, ".srtx", FILENAME_MAX - 5);

  fsrtx = fopen (srtxname, "w");
  if (!fsrtx)
  {
    fprintf (stderr, "Error opening %s\n", srtxname);
    return EXIT_FAILURE;
  }

  vobsub = vobsub_open (vobsubname, NULL, 0, &spudec);
  if (!vobsub)
  {
    fprintf (stderr, "Error opening vobsubs\n");
    fclose (fsrtx);
    return EXIT_FAILURE;
  }
  if (!spudec)
  {
    fprintf (stderr, "Error creating spudec handle\n");
    vobsub_close (vobsub);
    fclose (fsrtx);
    return EXIT_FAILURE;
  }

  spudec_set_forced_subs_only (spudec, forced);

  while ((packet_len =
          vobsub_get_next_packet (vobsub, &packet, &pts100)) >= 0)
  {
    spudec_assemble (spudec, packet, packet_len, pts100);
    if (spudec->queue_head)
    {
      spudec_heartbeat (spudec, spudec->queue_head->start_pts);
      if (spudec_changed (spudec))
        spudec_draw (spudec, draw_alpha);
    }
  }

  if (vobsub)
    vobsub_close (vobsub);

  fprintf (stdout, "%u files generated\n", sub_idx);

  fclose (fsrtx);

  return EXIT_SUCCESS;
}
