/* OGMDvd - A wrapper library around libdvdread
 * Copyright (C) 2004-2008 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/**
 * SECTION:ogmdvd-disc
 * @title: OGMDvdDisc
 * @short_description: Structure describing a video DVD
 * @include: ogmdvd-disc.h
 */

#include "ogmdvd-disc.h"
#include "ogmdvd-priv.h"
#include "ogmdvd-enums.h"

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <fcntl.h>
#include <unistd.h>
/*
#include <stdio.h>
#include <string.h>
*/
#include <glib/gstdio.h>
#include <glib/gi18n-lib.h>

#if defined(__FreeBSD__) || defined(__NetBSD__)
#include <sys/cdio.h>
#define CDROMEJECT CDIOCEJECT
#if __FreeBSD_version < 500000
#undef MIN
#undef MAX
#include <sys/param.h>
#include <sys/mount.h>
#endif /* __FreeBSD_version */
#endif /* __FreeBSD__ */

#ifdef __linux__
#include <linux/cdrom.h>
#endif /* __linux__ */

/*
 * We keep a trace of all opened drives because, with libdvdread >= 0.9.5
 * a same drive cannot be opened multiple times at the same time.
 */
static GHashTable *opened_drives;

static gboolean
ogmdvd_device_tray_is_open (const gchar *device)
{
  gint fd;
#if defined(__linux__) || defined(__FreeBSD__)
  gint status = 0;
#endif
#ifdef __FreeBSD__
  struct ioc_toc_header h;
#endif

  g_return_val_if_fail (device && *device, FALSE);

  fd = g_open (device, O_RDONLY | O_NONBLOCK | O_EXCL, 0);

  if (fd < 0)
    return FALSE;

#if defined(__linux__)
  status = ioctl (fd, CDROM_DRIVE_STATUS, CDSL_CURRENT);
#elif defined(__FreeBSD__)
  status = ioctl (fd, CDIOREADTOCHEADER, &h);
#endif

  close (fd);

#if defined(__linux__)
  return status < 0 ? FALSE : status == CDS_TRAY_OPEN;
#elif defined(__FreeBSD__)
  return status < 0;
#else
  return FALSE;
#endif
}

/**
 * ogmdvd_error_quark:
 *
 * The function description goes here.
 *
 * Returns: the #GQuark
 */
GQuark
ogmdvd_error_quark (void)
{
  static GQuark quark = 0;

  if (quark == 0)
    quark = g_quark_from_static_string ("ogmdvd-error-quark");

  return quark;
}

/**
 * ogmdvd_disc_open:
 * @device: A DVD device.
 * @error: Location to store the error occuring, or NULL to ignore errors.
 *
 * Creates a new #OGMDvdDisc given a DVD device.
 *
 * Returns: The new #OGMDvdDisc, or NULL
 */
OGMDvdDisc *
ogmdvd_disc_open (const gchar *device, GError **error)
{
  OGMDvdDisc *disc;
  dvd_reader_t *reader;
  ifo_handle_t *vmg_file;

  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  if (!device || !*device)
    device = "/dev/dvd";

  if (!opened_drives)
    opened_drives = g_hash_table_new (g_str_hash, g_str_equal);

  disc = g_hash_table_lookup (opened_drives, device);
  if (disc)
  {
    ogmdvd_disc_ref (disc);
    return disc;
  }

  reader = DVDOpen (device);
  if (!reader)
  {
    struct stat buf;

    if (g_stat (device, &buf))
      g_set_error (error, OGMDVD_DISC_ERROR, OGMDVD_DISC_ERROR_EXIST, _("No such file or directory"));
    else
    {
      if (access (device, R_OK) < 0)
        g_set_error (error, OGMDVD_DISC_ERROR, OGMDVD_DISC_ERROR_PERM, _("Permission denied to access device"));
      else
      {
        if (S_ISBLK(buf.st_mode))
        {
          if (ogmdvd_device_tray_is_open (device))
            g_set_error (error, OGMDVD_DISC_ERROR, OGMDVD_DISC_ERROR_TRAY, _("Tray seems to be opened"));
          /*
          else if (ogmdvd_drive_has_valid_media (device))
            g_set_error (error, OGMDVD_ERROR, OGMDVD_DISC_ERROR_DEV, _("Device does not contain a valid DVD video"));
          else
            g_set_error (error, OGMDVD_ERROR, OGMDVD_DISC_ERROR_UNKNOWN, _("Unknown error"));
          */
            g_set_error (error, OGMDVD_DISC_ERROR, OGMDVD_DISC_ERROR_DEV, _("Device does not contain a valid DVD video"));
        }
        else if (S_ISDIR(buf.st_mode) || S_ISREG(buf.st_mode))
          g_set_error (error, OGMDVD_DISC_ERROR, OGMDVD_DISC_ERROR_PATH, _("Path does not contain a valid DVD structure"));
        else
          g_set_error (error, OGMDVD_DISC_ERROR, OGMDVD_DISC_ERROR_ACCESS, _("No such directory, block device or iso file"));
      }
    }

    return NULL;
  }

  vmg_file = ifoOpen (reader, 0);
/*
  if (!vmg_file)
  {
    g_set_error (error, OGMDVD_DISC_ERROR, OGMDVD_DISC_ERROR_VMG, _("Cannot open video manager"));
    DVDClose (reader);
    return NULL;
  }
*/
  disc = g_new0 (OGMDvdDisc, 1);
  disc->device = g_strdup (device);
  disc->vmg_file = vmg_file;
  disc->reader = reader;
  disc->ref = 1;

  g_hash_table_insert (opened_drives, disc->device, disc);

  return disc;
}

/**
 * ogmdvd_disc_ref:
 * @disc: An #OGMDvdDisc
 *
 * Increments the reference count of an #OGMDvdDisc.
 */
void
ogmdvd_disc_ref (OGMDvdDisc *disc)
{
  g_return_if_fail (disc != NULL);

  disc->ref ++;
}

/**
 * ogmdvd_disc_unref:
 * @disc: An #OGMDvdDisc
 *
 * Decrements the reference count of an #OGMDvdDisc.
 */
void
ogmdvd_disc_unref (OGMDvdDisc *disc)
{
  g_return_if_fail (disc != NULL);

  if (disc->ref > 0)
  {
    disc->ref --;
    if (disc->ref == 0)
    {
      g_hash_table_remove (opened_drives, disc->device);

      g_free (disc->device);
      disc->device = NULL;

      if (disc->vmg_file)
        ifoClose (disc->vmg_file);
      disc->vmg_file = NULL;

      if (disc->reader)
        DVDClose (disc->reader);
      disc->reader = NULL;

      g_free (disc);
    }
  }
}

/**
 * ogmdvd_disc_get_label:
 * @disc: An #OGMDvdDisc
 *
 * Returns the label of the DVD.
 *
 * Returns: The label of the DVD, or NULL
 */
gchar *
ogmdvd_disc_get_label (OGMDvdDisc *disc)
{
  gchar label[32];

  g_return_val_if_fail (disc != NULL, NULL);

  if (DVDUDFVolumeInfo (disc->reader, label, 32, NULL, 0) < 0)
    return g_strdup ("Unknown");

  return g_convert (label, -1, "UTF8", "LATIN1", NULL, NULL, NULL);
}

/**
 * ogmdvd_disc_get_id:
 * @disc: An #OGMDvdDisc
 *
 * Returns a unique 128 bit disc identifier.
 *
 * Returns: The identifier, or NULL
 */
gchar *
ogmdvd_disc_get_id (OGMDvdDisc *disc)
{
  GString *str;
  gchar id[16];
  gint i;

  g_return_val_if_fail (disc != NULL, NULL);

  if (DVDDiscID (disc->reader, (guchar *) id) < 0)
    return NULL;

  str = g_string_new (NULL);

  for (i = 0; i < 16; i++)
    g_string_append_printf (str, "%x", id[i]);

  return g_string_free (str, FALSE);
}

/**
 * ogmdvd_disc_get_device:
 * @disc: An #OGMDvdDisc
 *
 * Returns the DVD device.
 *
 * Returns: The device of the DVD.
 */
G_CONST_RETURN gchar *
ogmdvd_disc_get_device (OGMDvdDisc *disc)
{
  g_return_val_if_fail (disc != NULL, NULL);

  return disc->device;
}

/**
 * ogmdvd_disc_get_vmg_size:
 * @disc: An #OGMDvdDisc
 *
 * Returns the size of the video manager in bytes.
 *
 * Returns: The size in bytes, or -1
 */
gint64
ogmdvd_disc_get_vmg_size (OGMDvdDisc *disc)
{
  gint64 fullsize;

  g_return_val_if_fail (disc != NULL, -1);

  fullsize  = _ogmdvd_get_ifo_size (disc, 0);
  fullsize += _ogmdvd_get_bup_size (disc, 0);
  fullsize += _ogmdvd_get_menu_size (disc, 0);

  return fullsize;
}

/**
 * ogmdvd_disc_get_n_titles:
 * @disc: An #OGMDvdDisc
 *
 * Returns the number of video titles of this DVD.
 *
 * Returns: The number of video titles, or -1
 */
gint
ogmdvd_disc_get_n_titles (OGMDvdDisc *disc)
{
  g_return_val_if_fail (disc != NULL, -1);

  return disc->vmg_file ? disc->vmg_file->tt_srpt->nr_of_srpts : 1;
}

static gint
ogmdvd_title_find_by_nr (OGMDvdTitle *title, guint nr)
{
  return title->nr - nr;
}

/**
 * ogmdvd_disc_get_nth_title:
 * @disc: An #OGMDvdDisc
 * @nr: The title number
 *
 * Returns the video title at position nr. The first nr is 0.
 *
 * Returns: The #OGMDvdTitle, or NULL
 */
OGMDvdTitle *
ogmdvd_disc_get_nth_title (OGMDvdDisc *disc, guint nr)
{
  OGMDvdTitle *title;
  GSList *link;

  g_return_val_if_fail (disc != NULL, NULL);

  g_return_val_if_fail (nr == 0 || (disc->vmg_file && nr < disc->vmg_file->tt_srpt->nr_of_srpts), NULL);

  link = g_slist_find_custom (disc->titles, GUINT_TO_POINTER (nr), (GCompareFunc) ogmdvd_title_find_by_nr);
  if (link)
  {
    title = link->data;
    title->ref ++;
  }
  else
  {
    ifo_handle_t *vts_file;

    vts_file = ifoOpen (disc->reader, disc->vmg_file ? disc->vmg_file->tt_srpt->title[nr].title_set_nr : 1);
    if (!vts_file)
      return NULL;

    ogmdvd_disc_ref (disc);

    title = g_new0 (OGMDvdTitle, 1);
    title->vts_file = vts_file;
    title->disc = disc;
    title->ref = 1;
    title->nr = nr;

    title->ttn = disc->vmg_file ? disc->vmg_file->tt_srpt->title[nr].vts_ttn : 1;
    title->pgcn = title->vts_file->vts_ptt_srpt->title[title->ttn - 1].ptt[0].pgcn;

    disc->titles = g_slist_append (disc->titles, title);
  }

  return title;
}

/**
 * ogmdvd_disc_get_titles:
 * @disc: An #OGMDvdDisc
 *
 * Returns a list of all the titles of @disc.
 *
 * Returns: A #GList, or NULL
 */
GList *
ogmdvd_disc_get_titles (OGMDvdDisc *disc)
{
  OGMDvdTitle *title;
  GList *list = NULL;
  guint nr_of_srpts, nr;

  g_return_val_if_fail (disc != NULL, NULL);

  nr_of_srpts = disc->vmg_file ? disc->vmg_file->tt_srpt->nr_of_srpts : 1;

  for (nr = 0; nr < nr_of_srpts; nr ++)
  {
    title = ogmdvd_disc_get_nth_title (disc, nr);
    if (title)
      list = g_list_append (list, title);
  }

  return list;
}

