/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2009 Olivier Rolland <billl@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "ogmrip-fs.h"
#include "ogmrip-mplayer.h"
#include "ogmrip-plugin.h"
#include "ogmrip-version.h"
#include "ogmrip-xvid.h"

#include "ogmjob-exec.h"
#include "ogmjob-queue.h"

#include <stdio.h>
#include <unistd.h>
#include <glib/gstdio.h>
#include <glib/gi18n-lib.h>

#define OGMRIP_XVID_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMRIP_TYPE_XVID, OGMRipXvidPriv))

static gint ogmrip_xvid_run         (OGMJobSpawn       *spawn);
static void ogmrip_xvid_set_quality (OGMRipVideoCodec       *video,
                                     OGMRipQualityType quality);

struct _OGMRipXvidPriv
{
  gboolean gmc;
  gboolean chroma_opt;
  guint quant_type;
  guint bvhq;
  guint vhq;
};

static gdouble
ogmrip_xvid_get_quantizer (OGMRipVideoCodec *video)
{
  gdouble quantizer;

  quantizer = ogmrip_video_codec_get_quantizer (video);
  if (quantizer < 0.0)
    return 2.0;

  return quantizer;
}

static gchar **
ogmrip_xvid_command (OGMRipVideoCodec *video, guint pass, guint passes, const gchar *log_file)
{
  OGMRipXvid *xvid;
  OGMDvdTitle *title;
  GPtrArray *argv;
  GString *options;

  const char *output;
  gint bitrate, vid, threads;

  g_return_val_if_fail (OGMRIP_IS_VIDEO_CODEC (video), NULL);

  output = ogmrip_codec_get_output (OGMRIP_CODEC (video));
  g_return_val_if_fail (output != NULL, NULL);

  title = ogmrip_codec_get_input (OGMRIP_CODEC (video));
  g_return_val_if_fail (title != NULL, NULL);

  g_return_val_if_fail (pass == 1 || log_file != NULL, NULL);

  xvid = OGMRIP_XVID (video);

  argv = ogmrip_mencoder_video_command (video, pass == passes ? output : "/dev/null", pass);

  g_ptr_array_add (argv, g_strdup ("-ovc"));
  g_ptr_array_add (argv, g_strdup ("xvid"));

  options = g_string_new ("autoaspect:par=vga11");
  g_string_append_printf (options, ":vhq=%u:bvhq=%u", xvid->priv->vhq, xvid->priv->bvhq);

  if (xvid->priv->chroma_opt)
    g_string_append (options, ":chroma_opt");
  else
    g_string_append (options, ":nochroma_opt");

  if (xvid->priv->quant_type)
    g_string_append (options, ":quant_type=mpeg");
  else
    g_string_append (options, ":quant_type=h263");

  if (MPLAYER_CHECK_VERSION (1,0,0,6))
  {
    if (ogmrip_video_codec_get_cartoon (video))
      g_string_append (options, ":cartoon");
    else
      g_string_append (options, ":nocartoon");
  }

  if (ogmrip_video_codec_get_qpel (video))
    g_string_append (options, ":qpel");
  else
    g_string_append (options, ":noqpel");

  if (pass != passes && ogmrip_video_codec_get_turbo (video))
    g_string_append (options, ":turbo");

  if (ogmrip_video_codec_get_trellis (video))
    g_string_append (options, ":trellis");
  else
    g_string_append (options, ":notrellis");

  if (ogmrip_video_codec_get_grayscale (video))
    g_string_append (options, ":greyscale");
  else
    g_string_append (options, ":nogreyscale");

  if (ogmrip_xvid_get_gmc (OGMRIP_XVID (video)))
    g_string_append (options, ":gmc");
  else
    g_string_append (options, ":nogmc");

  g_string_append_printf (options, ":max_bframes=%d", ogmrip_video_codec_get_max_b_frames (video));

  bitrate = ogmrip_video_codec_get_bitrate (video);
  if (bitrate > 0)
  {
    if (bitrate < 16001)
      g_string_append_printf (options, ":bitrate=%u", bitrate / 1000);
    else
      g_string_append_printf (options, ":bitrate=%u", bitrate);
  }
  else
    g_string_append_printf (options, ":fixed_quant=%.0lf", ogmrip_xvid_get_quantizer (video));

  if (passes > 1 && log_file)
  {
    g_string_append_printf (options, ":pass=%u", pass);
    g_ptr_array_add (argv, g_strdup ("-passlogfile"));
    g_ptr_array_add (argv, g_strdup (log_file));
  }

  threads = ogmrip_video_codec_get_threads (video);
  if (threads > 0)
  {
    guint height;

    ogmrip_video_codec_get_scale_size (video, NULL, &height);
    g_string_append_printf (options, ":threads=%u", CLAMP (threads, 1, height / 16));
  }

  g_ptr_array_add (argv, g_strdup ("-xvidencopts"));
  g_ptr_array_add (argv, g_string_free (options, FALSE));

  vid = ogmdvd_title_get_nr (title);

  if (MPLAYER_CHECK_VERSION (1,0,0,1))
    g_ptr_array_add (argv, g_strdup_printf ("dvd://%d", vid + 1));
  else
  {
    g_ptr_array_add (argv, g_strdup ("-dvd"));
    g_ptr_array_add (argv, g_strdup_printf ("%d", vid + 1));
  }

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

G_DEFINE_TYPE (OGMRipXvid, ogmrip_xvid, OGMRIP_TYPE_VIDEO_CODEC)

static void
ogmrip_xvid_class_init (OGMRipXvidClass *klass)
{
  OGMJobSpawnClass *spawn_class;
  OGMRipVideoCodecClass *video_class;

  spawn_class = OGMJOB_SPAWN_CLASS (klass);
  spawn_class->run = ogmrip_xvid_run;

  video_class = OGMRIP_VIDEO_CODEC_CLASS (klass);
  video_class->set_quality = ogmrip_xvid_set_quality;

  g_type_class_add_private (klass, sizeof (OGMRipXvidPriv));
}

static void
ogmrip_xvid_init (OGMRipXvid *xvid)
{
  xvid->priv = OGMRIP_XVID_GET_PRIVATE (xvid);
}

static gint
ogmrip_xvid_run (OGMJobSpawn *spawn)
{
  OGMJobSpawn *queue, *child;
  gchar **argv, *log_file, *cwd = NULL;
  gint pass, passes, result;

  queue = ogmjob_queue_new ();
  ogmjob_container_add (OGMJOB_CONTAINER (spawn), queue);
  g_object_unref (queue);

  passes = ogmrip_video_codec_get_passes (OGMRIP_VIDEO_CODEC (spawn));

  log_file = NULL;
  if (passes > 1)
  {
    if (MPLAYER_CHECK_VERSION (1,0,0,8))
      log_file = ogmrip_fs_mktemp ("log.XXXXXX", NULL);
    else
    {
      /*
       * Workaround against xvid pass log file
       * Should disappear someday
       */
        log_file = g_build_filename (g_get_tmp_dir (), "xvid-twopass.stats", NULL);
    }
  }

  for (pass = 0; pass < passes; pass ++)
  {
    argv = ogmrip_xvid_command (OGMRIP_VIDEO_CODEC (spawn), pass + 1, passes, log_file);
    if (!argv)
      return OGMJOB_RESULT_ERROR;

    child = ogmjob_exec_newv (argv);
    ogmjob_exec_add_watch_full (OGMJOB_EXEC (child), (OGMJobWatch) ogmrip_mencoder_codec_watch, spawn, TRUE, FALSE, FALSE);
    ogmjob_container_add (OGMJOB_CONTAINER (queue), child);
    g_object_unref (child);
  }

  if (!MPLAYER_CHECK_VERSION (1,0,0,8))
  {
    /*
     * Workaround against xvid pass log file
     */
    cwd = g_get_current_dir ();
    g_chdir (g_get_tmp_dir ());
  }

  result = OGMJOB_SPAWN_CLASS (ogmrip_xvid_parent_class)->run (spawn);

  if (cwd)
  {
    /*
     * Return in cwd
     */
    g_chdir (cwd);
    g_free (cwd);
  }

  ogmjob_container_remove (OGMJOB_CONTAINER (spawn), queue);

  g_unlink (log_file);
  g_free (log_file);

  return result;
}

static void
ogmrip_xvid_set_quality (OGMRipVideoCodec *video, OGMRipQualityType quality)
{
  OGMRipXvid *xvid;

  xvid = OGMRIP_XVID (video);

  switch (quality)
  {
    case OGMRIP_QUALITY_EXTREME:
      xvid->priv->chroma_opt = TRUE;
      xvid->priv->quant_type = 1;
      xvid->priv->bvhq = 1;
      xvid->priv->vhq = 4;
      break;
    case OGMRIP_QUALITY_HIGH:
      xvid->priv->chroma_opt = TRUE;
      xvid->priv->quant_type = 1;
      xvid->priv->bvhq = 1;
      xvid->priv->vhq = 2;
      break;
    case OGMRIP_QUALITY_NORMAL:
      xvid->priv->chroma_opt = FALSE;
      xvid->priv->quant_type = 0;
      xvid->priv->bvhq = 0;
      xvid->priv->vhq = 0;
      break;
  }
}

/**
 * ogmrip_xvid_new:
 * @title: An #OGMDvdTitle
 * @output: The output file
 *
 * Creates a new #OGMRipXvid.
 *
 * Returns: the new #OGMRipXvid
 */
OGMJobSpawn *
ogmrip_xvid_new (OGMDvdTitle *title, const gchar *output)
{
  g_return_val_if_fail (title != NULL, NULL);
  g_return_val_if_fail (output && *output, NULL);

  return g_object_new (OGMRIP_TYPE_XVID, "input", title, "output", output, NULL);
}

/**
 * ogmrip_xvid_set_gmc:
 * @xvid: An #OGMRipXvid
 * @gmc: %TRUE to set global motion compensation
 *
 * Enables global motion compensation.
 */
void
ogmrip_xvid_set_gmc (OGMRipXvid *xvid, gboolean gmc)
{
  g_return_if_fail (OGMRIP_IS_XVID (xvid));

  xvid->priv->gmc = gmc;
}

/**
 * ogmrip_xvid_get_gmc:
 * @xvid: An #OGMRipXvid
 *
 * Gets global motion compensation.
 *
 * Returns: %TRUE if global motion compensation has been set
 */
gboolean
ogmrip_xvid_get_gmc (OGMRipXvid *xvid)
{
  g_return_val_if_fail (OGMRIP_IS_XVID (xvid), FALSE);

  return xvid->priv->gmc;
}

static OGMRipVideoPlugin xvid_plugin =
{
  NULL,
  G_TYPE_NONE,
  "xvid",
  N_("XviD"),
  OGMRIP_FORMAT_MPEG4,
  2,
  G_MAXINT
};

OGMRipVideoPlugin *
ogmrip_init_plugin (void)
{
  gchar *output;
  gboolean match;

  if (!ogmrip_check_mencoder ())
    return NULL;

  if (!g_spawn_command_line_sync ("mencoder -ovc help", &output, NULL, NULL, NULL))
    return NULL;

  match = g_regex_match_simple ("^ *xvid *- .*$", output, G_REGEX_MULTILINE, 0);
  g_free (output);

  if (!match)
    return NULL;

  xvid_plugin.type = OGMRIP_TYPE_XVID;

  return &xvid_plugin;
}

