/*
 *  OpenDuke
 *  Copyright (C) 1999  Rusty Wagner
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */


#include <windows.h>
#pragma hdrstop
#include <condefs.h>

#include "D3DRender.h"
#include "console.h"

char errorStr[256];
int error;

static Poly p;

#define SPRITE 0
#define WINDOW 1

extern int software;

extern HMODULE ddrawLib;
extern HINSTANCE hInst;

struct ModeEnumStruct
{
    ModeList *modes16;
    ModeList *modes32;
};

HRESULT WINAPI EnumModesFunc(LPDDSURFACEDESC sd,LPVOID ptr)
{
    ModeList *modes16=((ModeEnumStruct*)ptr)->modes16;
    ModeList *modes32=((ModeEnumStruct*)ptr)->modes32;
    if (sd->ddpfPixelFormat.dwRGBBitCount==16)
    {
        modes16->mode[modes16->count].w=sd->dwWidth;
        modes16->mode[modes16->count++].h=sd->dwHeight;
    }
    else if (sd->ddpfPixelFormat.dwRGBBitCount==32)
    {
        modes32->mode[modes32->count].w=sd->dwWidth;
        modes32->mode[modes32->count++].h=sd->dwHeight;
    }
    return DDENUMRET_OK;
}

void D3DRender::EnumModes(ModeList *modes16,ModeList *modes32)
{
    modes16->count=0; modes32->count=0;
    cprintf("Attempting to query DirectDraw for mode list...\n");
    if (ddrawLib==NULL)
        ddrawLib=LoadLibrary("ddraw.dll");
    if (ddrawLib==NULL)
    {
        cprintf("DirectDraw not installed\n");
        return;
    }
    HRESULT WINAPI (*ddc)(GUID FAR*,LPVOID *,IUnknown FAR *)=
        (HRESULT WINAPI (*)(GUID FAR*,LPVOID *,IUnknown FAR *))
        GetProcAddress(ddrawLib,"DirectDrawCreate");
    if (!ddc)
    {
        cprintf("DirectDraw not installed\n");
        return;
    }
    LPDIRECTDRAW lpDD;
	if (ddc(NULL,(void**)&lpDD,NULL)!=DD_OK)
    {
        cprintf("Could not initialize DirectDraw\n");
        return;
    }
    DDSURFACEDESC sd;
    memset(&sd,0,sizeof(sd));
    sd.dwSize=sizeof(sd);
    sd.dwFlags=DDSD_CAPS;
    sd.ddsCaps.dwCaps=DDSCAPS_3DDEVICE|DDSCAPS_COMPLEX|DDSCAPS_FLIP;
    ModeEnumStruct mes;
    mes.modes16=modes16;
    mes.modes32=modes32;
    lpDD->EnumDisplayModes(0,NULL,&mes,EnumModesFunc);
    lpDD->Release();
    lpDD=NULL;
    cprintf("Mode list obtained successfully\n");
}

D3DRender::D3DRender(HINSTANCE _hInst,HWND _hWnd,int _scrnX,int _scrnY,int depth)
{
    init=0;
    stencilValue=1; dc=NULL;
    extraPolyList=NULL;
    firstFont=lastFont=curFont=NULL;
    renderEnabled=sceneEnabled=false;
    scrnX=_scrnX; scrnY=_scrnY;
    hInst=_hInst; hWnd=_hWnd;
    cprintf("Attempting to initialize DirectDraw...\n");
    dd=new CDirectDraw(hWnd,scrnX,scrnY,depth);
    if (error)
    {
        cprintf("DirectDraw failed\n");
        delete d3d; d3d=NULL;
        delete dd; dd=NULL;
        return;
    }
    cprintf("Initialization OK, attempting to initialize Direct3D...\n");
    error=0;
    d3d=new CDirect3D(dd,dd->GetBackBuffer());
    if (error)
    {
        cprintf("Direct3D Error: %s\n",errorStr);
        delete d3d; d3d=NULL;
        delete dd; dd=NULL;
        return;
    }
    cprintf("Initialization OK, using Direct3D\n");
	lpView=new CView(d3d);
    lpView->SetBackgroundColor(RGB_MAKE(0,0,0));
    lpView->Update();
	lpZeroView=new CView(d3d);
    lpZeroView->SetXPos(0);
    lpZeroView->SetYPos(0);
    lpZeroView->SetZPos(0);
    lpZeroView->Update();
    d3d->SetView(lpView);
	d3d->SetRenderState(D3DRENDERSTATE_ZENABLE,software?D3DZB_TRUE:D3DZB_TRUE);
	d3d->SetRenderState(D3DRENDERSTATE_ZWRITEENABLE,TRUE);
	d3d->SetRenderState(D3DRENDERSTATE_FILLMODE,D3DFILL_SOLID);
	d3d->SetRenderState(D3DRENDERSTATE_SHADEMODE,software?D3DSHADE_FLAT:D3DSHADE_GOURAUD);
	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_CCW);
	d3d->SetRenderState(D3DRENDERSTATE_DITHERENABLE,software?FALSE:TRUE);
	d3d->SetRenderState(D3DRENDERSTATE_ALPHABLENDENABLE,software?FALSE:TRUE);
	d3d->SetRenderState(D3DRENDERSTATE_SRCBLEND,D3DBLEND_SRCALPHA);
	d3d->SetRenderState(D3DRENDERSTATE_DESTBLEND,D3DBLEND_INVSRCALPHA);
    d3d->SetRenderState(D3DRENDERSTATE_LIGHTING,FALSE);
    d3d->SetTextureStageState(0,D3DTSS_MAGFILTER,software?D3DTFG_POINT:D3DTFG_LINEAR);
    d3d->SetTextureStageState(0,D3DTSS_MINFILTER,software?D3DTFN_POINT:D3DTFN_LINEAR);
    d3d->SetTextureStageState(0,D3DTSS_MIPFILTER,software?D3DTFP_POINT:D3DTFP_POINT);
    d3d->SetTextureStageState(1,D3DTSS_MAGFILTER,software?D3DTFG_POINT:D3DTFG_LINEAR);
    d3d->SetTextureStageState(1,D3DTSS_MINFILTER,software?D3DTFN_POINT:D3DTFN_LINEAR);
    d3d->SetTextureStageState(1,D3DTSS_MIPFILTER,software?D3DTFP_POINT:D3DTFP_POINT);
    lpView->SetXPos(1);
    lpView->SetYPos(1);
    lpView->SetZPos(-5);
    lpView->Update();
    farPoly.AddVert(-210000,-210000,100000,0,0,0,0,0);
    farPoly.AddVert(210000,-210000,100000,0,0,0,0,0);
    farPoly.AddVert(210000,210000,100000,0,0,0,0,0);
    farPoly.AddVert(-210000,210000,100000,0,0,0,0,0);
    cprintf("Direct3D setup complete\n");
    init=1;
}

D3DRender::~D3DRender()
{
    cprintf("Deinitializing Direct3D...\n");
    if (d3d) { delete d3d; d3d=NULL; }
    if (dd) { delete dd; dd=NULL; }
    cprintf("Direct3D deinitialized successfully\n");
    init=0;
}

int D3DRender::Initialized()
{
    return init;
}

void D3DRender::ChangeResolution(int width,int height,int depth)
{
    scrnX=width; scrnY=height;
    Vector p=Vector(GetXPos(),GetYPos(),GetZPos());
    Vector r=Vector(GetXRotate(),GetYRotate(),GetZRotate());
    if (d3d) { delete d3d; d3d=NULL; }
    error=0;
    cprintf("Attempting to change DirectDraw resolution...\n");
    dd->ChangeResolution(hWnd,width,height,depth);
    if (error)
        throw "Could not change video settings\n";
    error=0;
    cprintf("Change successful, reinitializing Direct3D\n");
    d3d=new CDirect3D(dd,dd->GetBackBuffer());
    if (error)
    {
        cprintf("Direct3D error: %s\n",errorStr);
        throw "Could not change video settings\n";
    }
    cprintf("Direct3D initialization OK\n");
	lpView=new CView(d3d);
    lpView->SetBackgroundColor(RGB_MAKE(0,0,0));
    lpView->Update();
	lpZeroView=new CView(d3d);
    lpZeroView->SetXPos(0);
    lpZeroView->SetYPos(0);
    lpZeroView->SetZPos(0);
    lpZeroView->Update();
    d3d->SetView(lpView);
	d3d->SetRenderState(D3DRENDERSTATE_ZENABLE,software?D3DZB_TRUE:D3DZB_TRUE);
	d3d->SetRenderState(D3DRENDERSTATE_ZWRITEENABLE,TRUE);
	d3d->SetRenderState(D3DRENDERSTATE_FILLMODE,D3DFILL_SOLID);
	d3d->SetRenderState(D3DRENDERSTATE_SHADEMODE,software?D3DSHADE_FLAT:D3DSHADE_GOURAUD);
	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_CCW);
	d3d->SetRenderState(D3DRENDERSTATE_DITHERENABLE,software?FALSE:TRUE);
	d3d->SetRenderState(D3DRENDERSTATE_ALPHABLENDENABLE,software?FALSE:TRUE);
	d3d->SetRenderState(D3DRENDERSTATE_SRCBLEND,D3DBLEND_SRCALPHA);
	d3d->SetRenderState(D3DRENDERSTATE_DESTBLEND,D3DBLEND_INVSRCALPHA);
    d3d->SetRenderState(D3DRENDERSTATE_LIGHTING,FALSE);
    d3d->SetTextureStageState(0,D3DTSS_MAGFILTER,software?D3DTFG_POINT:D3DTFG_LINEAR);
    d3d->SetTextureStageState(0,D3DTSS_MINFILTER,software?D3DTFN_POINT:D3DTFN_LINEAR);
    d3d->SetTextureStageState(0,D3DTSS_MIPFILTER,software?D3DTFP_POINT:D3DTFP_POINT);
    d3d->SetTextureStageState(1,D3DTSS_MAGFILTER,software?D3DTFG_POINT:D3DTFG_LINEAR);
    d3d->SetTextureStageState(1,D3DTSS_MINFILTER,software?D3DTFN_POINT:D3DTFN_LINEAR);
    d3d->SetTextureStageState(1,D3DTSS_MIPFILTER,software?D3DTFP_POINT:D3DTFP_POINT);
    SetXPos(p.x); SetYPos(p.y); SetZPos(p.z);
    SetXRotate(r.x); SetYRotate(r.y); SetZRotate(r.z);
    cprintf("Direct3D setup complete\n");
}

void D3DRender::SetRenderWnd(HWND _hWnd)
{
    hWnd=_hWnd;
}

void D3DRender::BeginRender()
{
    if (!d3d) return;
    lpView->Update();
    d3d->BeginRender();
    polyDrawn=polyTotal=polyReflect=0;
    renderEnabled=true;
}

void D3DRender::BeginScene()
{
    if (!d3d) return;
    d3d->BeginScene();
    viewVect=Vector(lpView->GetXPos(),lpView->GetYPos(),lpView->GetZPos());
    sceneEnabled=true;
}

void D3DRender::EndRender()
{
    if (!d3d) return;
    d3d->EndRender();
    renderEnabled=false;
}

void D3DRender::EndScene()
{
    if (!d3d) return;
    d3d->EndScene();
    sceneEnabled=false;
}

/*void _export RenderReflectPoly(Poly &poly,Plane &p,float n,float r,int noCheck=0)
{
    if (poly.nVert<3) return;
    if (poly.flags&PF_INVISIBLE)
        return;
    if (!noCheck)
    {
        Poly posPoly,negPoly;
        switch(ClassifyPoly(poly,p))
        {
            case CT_COINCIDENT:
                return;
            case CT_NEGATIVE:
                if (r>=0) return;
                break;
            case CT_POSITIVE:
                if (r<=0) return;
                break;
            case CT_SPLIT:
                posPoly.Clear(); negPoly.Clear();
                SplitPoly(poly,p,posPoly,negPoly);
//                if (r>0) RenderReflectPoly(posPoly,p,n,r);
//                else RenderReflectPoly(negPoly,p,n,r);
                return;
        }
    }
    Poly np=poly;
    for (int i=0;i<np.nVert;i++)
    {
        float nr=p.Evaluate(np.v[i].x,np.v[i].y,np.v[i].z);
        np.v[i].x+=p.A*nr*n;
        np.v[i].y+=p.B*nr*n;
        np.v[i].z+=p.C*nr*n;
    }
    if (!IsBackFace(np,viewVect))
        return;
    if ((poly.texture!=NULL)&&(!(poly.flags&PF_NONTEXTURED)))
    {
        d3d->SetTextureStageState(0,D3DTSS_COLOROP,D3DTOP_MODULATE2X);
        d3d->SetTextureStageState(0,D3DTSS_ALPHAOP,D3DTOP_MODULATE2X);
        d3d->SetTexture(0,(CTexture*)(poly.texture->data));
    }
    else
        d3d->SetTextureStageState(0,D3DTSS_COLOROP,D3DTOP_DISABLE);
    if (poly.flags&PF_TWOSIDED)
    	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_NONE);
    else if (poly.flags&PF_REVERSE)
    	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_CCW);
    else
    	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_CW);
    polyReflect++;
    d3d->DrawPrimitive(D3DPT_TRIANGLEFAN,np.v,np.nVert,D3DDP_WAIT);
}

void RenderReflectPolyList(PolyList &list,Plane &p,float n,float r)
{
    PolyListElem *ptr=list.first;
    while (ptr!=NULL)
    {
        RenderReflectPoly(ptr->poly,p,n,r);
        ptr=ptr->next;
    }
}

void RenderReflectPolyListNonBSP(PolyList &list,Plane &p,float n,float r)
{
    PolyListElem *ptr=list.first;
    while (ptr!=NULL)
    {
        if (!(ptr->poly.flags&PF_INTREE))
            RenderReflectPoly(ptr->poly,p,n,r);
        ptr=ptr->next;
    }
}

void RenderReflectBSPNode(BSPNode *node,Plane &p,float n,float r,Vector eye)
{
    if (node==NULL) return;
    float result=node->part.Evaluate(eye.x,eye.y,eye.z);
    if (result>0)
    {
        RenderReflectBSPNode(node->neg,p,n,r,eye);
        RenderReflectPolyList(node->list,p,n,r);
        RenderReflectBSPNode(node->pos,p,n,r,eye);
    }
    else if (result<0)
    {
        RenderReflectBSPNode(node->pos,p,n,r,eye);
        RenderReflectPolyList(node->list,p,n,r);
        RenderReflectBSPNode(node->neg,p,n,r,eye);
    }
    else
    {
        RenderReflectBSPNode(node->pos,p,n,r,eye);
        RenderReflectBSPNode(node->neg,p,n,r,eye);
    }
}*/

void D3DRender::RenderPoly(Poly &poly)
{
    if ((!renderEnabled)||(!sceneEnabled))
        return;
    if (poly.flags&PF_INVISIBLE)
        return;
    if (IsBackFace(poly,viewVect))
        return;
/*    if (poly.flags&PF_REFLECT)
    {
        Plane p=PlaneFromPoly(poly);
        float r=p.Evaluate(GetXPos(),GetYPos(),GetZPos());
        float n=-2/(p.A*p.A+p.B*p.B+p.C*p.C);
        Vector eye=Vector(GetXPos()+p.A*r*n,GetYPos()+
            p.B*r*n,GetZPos()+p.C*r*n);
        r=-p.Evaluate(eye.x,eye.y,eye.z);
        stencilValue++;
        if (stencilValue>255)
        {
            d3d->ClearStencilBuffer();
            stencilValue=1;
        }
        d3d->SetRenderState(D3DRENDERSTATE_STENCILENABLE,TRUE);
        d3d->SetRenderState(D3DRENDERSTATE_STENCILPASS,D3DSTENCILOP_REPLACE);
        d3d->SetRenderState(D3DRENDERSTATE_STENCILZFAIL,D3DSTENCILOP_KEEP);
        d3d->SetRenderState(D3DRENDERSTATE_STENCILREF,stencilValue);
        d3d->SetRenderState(D3DRENDERSTATE_ZWRITEENABLE,FALSE);
        d3d->SetRenderState(D3DRENDERSTATE_STENCILFUNC,D3DCMP_ALWAYS);
        d3d->SetTextureStageState(0,D3DTSS_COLOROP,D3DTOP_DISABLE);
        if (poly.flags&PF_TWOSIDED)
        	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_NONE);
        else if (poly.flags&PF_REVERSE)
        	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_CW);
        else
        	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_CCW);
        Poly np=poly;
        for (int i=0;i<np.nVert;i++)
            np.v[i].color=RGBA_MAKE(128,128,128,255);
        d3d->DrawPrimitive(D3DPT_TRIANGLEFAN,np.v,np.nVert,D3DDP_WAIT);
        d3d->SetRenderState(D3DRENDERSTATE_ZWRITEENABLE,TRUE);
        d3d->SetRenderState(D3DRENDERSTATE_STENCILFAIL,D3DSTENCILOP_KEEP);
        d3d->SetRenderState(D3DRENDERSTATE_STENCILZFAIL,D3DSTENCILOP_KEEP);
        d3d->SetRenderState(D3DRENDERSTATE_STENCILPASS,D3DSTENCILOP_KEEP);
        d3d->SetRenderState(D3DRENDERSTATE_STENCILFUNC,D3DCMP_EQUAL);
        d3d->SetRenderState(D3DRENDERSTATE_ZFUNC,D3DCMP_ALWAYS);
      	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_NONE);
        d3d->SetView(lpZeroView);
        d3d->DrawPrimitive(D3DPT_TRIANGLEFAN,farPoly.v,farPoly.nVert,D3DDP_WAIT);
        d3d->SetView(lpView);
        d3d->SetRenderState(D3DRENDERSTATE_ZFUNC,D3DCMP_LESSEQUAL);
        if (extraPolyList) RenderReflectPolyListNonBSP(*extraPolyList,p,n,r);
        RenderReflectBSPNode(rootNode,p,n,r,eye);
        if (np.texture!=NULL)
        {
            d3d->SetTextureStageState(0,D3DTSS_COLOROP,D3DTOP_MODULATE2X);
            d3d->SetTextureStageState(0,D3DTSS_ALPHAOP,D3DTOP_MODULATE2X);
            d3d->SetTexture(0,(CTexture*)(poly.texture->data));
        }
        else
            d3d->SetTextureStageState(0,D3DTSS_COLOROP,D3DTOP_DISABLE);
        if (poly.flags&PF_TWOSIDED)
        	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_NONE);
        else if (poly.flags&PF_REVERSE)
        	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_CW);
        else
        	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_CCW);
        d3d->SetRenderState(D3DRENDERSTATE_ZFUNC,D3DCMP_ALWAYS);
        d3d->DrawPrimitive(D3DPT_TRIANGLEFAN,poly.v,poly.nVert,D3DDP_WAIT);
        d3d->SetRenderState(D3DRENDERSTATE_ZFUNC,D3DCMP_LESSEQUAL);
        d3d->SetRenderState(D3DRENDERSTATE_STENCILENABLE,FALSE);
        return;
    }*/
    if (poly.flags&PF_PRIORITY)
        d3d->SetRenderState(D3DRENDERSTATE_ZENABLE,FALSE);
    else
        d3d->SetRenderState(D3DRENDERSTATE_ZENABLE,TRUE);
    if ((poly.texture!=NULL)&&(!(poly.flags&PF_NONTEXTURED)))
    {
        d3d->SetTextureStageState(0,D3DTSS_COLOROP,D3DTOP_MODULATE2X);
        d3d->SetTextureStageState(0,D3DTSS_ALPHAOP,D3DTOP_MODULATE);
        d3d->SetTextureStageState(0,D3DTSS_TEXCOORDINDEX,0);
        d3d->SetTexture(0,(CTexture*)(poly.texture->data));
        if (poly.lightMap!=NULL)
        {
            d3d->SetTextureStageState(1,D3DTSS_COLOROP,D3DTOP_MODULATE2X);
            d3d->SetTextureStageState(1,D3DTSS_ALPHAOP,D3DTOP_SELECTARG2);
            d3d->SetTextureStageState(1,D3DTSS_TEXCOORDINDEX,1);
            d3d->SetTextureStageState(1,D3DTSS_MAGFILTER,D3DTFG_LINEAR);
            d3d->SetTextureStageState(1,D3DTSS_MINFILTER,D3DTFN_LINEAR);
            d3d->SetTextureStageState(1,D3DTSS_MIPFILTER,D3DTFP_LINEAR);
            d3d->SetTextureStageState(1,D3DTSS_ADDRESS,D3DTADDRESS_CLAMP);
            d3d->SetTexture(1,(CTexture*)(poly.lightMap->data));
        }
        else
        {
            d3d->SetTextureStageState(1,D3DTSS_COLOROP,D3DTOP_DISABLE);
            d3d->SetTextureStageState(1,D3DTSS_ALPHAOP,D3DTOP_DISABLE);
        }
    }
    else
    {
        if (poly.lightMap!=NULL)
        {
            d3d->SetTextureStageState(0,D3DTSS_COLOROP,D3DTOP_MODULATE2X);
            d3d->SetTextureStageState(0,D3DTSS_ALPHAOP,D3DTOP_SELECTARG2);
            d3d->SetTextureStageState(0,D3DTSS_TEXCOORDINDEX,1);
            d3d->SetTextureStageState(0,D3DTSS_ADDRESS,D3DTADDRESS_CLAMP);
            d3d->SetTexture(0,(CTexture*)(poly.lightMap->data));
        }
        else
        {
            d3d->SetTextureStageState(0,D3DTSS_COLOROP,D3DTOP_DISABLE);
            d3d->SetTextureStageState(0,D3DTSS_ALPHAOP,D3DTOP_DISABLE);
        }
    }
    if (poly.flags&PF_TWOSIDED)
    	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_NONE);
    else if (poly.flags&PF_REVERSE)
    	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_CW);
    else
    	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_CCW);
    polyDrawn++;
    if (poly.flags&PF_ZBIAS)
    {
        p.Clear();
        for (int i=0;i<poly.nVert;i++)
        {
            Vertex v=poly.v[i];
            v.x=GetXPos()+(v.x-GetXPos())*0.99;
            v.y=GetYPos()+(v.y-GetYPos())*0.99;
            v.z=GetZPos()+(v.z-GetZPos())*0.99;
            p.AddVert(v);
        }
        d3d->DrawPrimitive(D3DPT_TRIANGLEFAN,p.v,p.nVert,0);
    }
    else
        d3d->DrawPrimitive(D3DPT_TRIANGLEFAN,poly.v,poly.nVert,0);
    d3d->SetRenderState(D3DRENDERSTATE_ZENABLE,TRUE);
    d3d->SetTextureStageState(0,D3DTSS_ADDRESS,D3DTADDRESS_WRAP);
}

void D3DRender::RenderPolyList(PolyList &list)
{
    PolyListElem *ptr=list.first;
    while (ptr!=NULL)
    {
        RenderPoly(ptr->poly);
        ptr=ptr->next;
        polyTotal++;
    }
}

void D3DRender::RenderPolyListNonBSP(PolyList &list)
{
    PolyListElem *ptr=list.first;
    while (ptr!=NULL)
    {
        if (!(ptr->poly.flags&PF_INTREE))
        {
            RenderPoly(ptr->poly);
            polyTotal++;
        }
        ptr=ptr->next;
    }
}

void D3DRender::SetXPos(float x)
{
    if (!d3d) return;
    lpView->SetXPos(x);
}

void D3DRender::SetYPos(float y)
{
    if (!d3d) return;
    lpView->SetYPos(y);
}

void D3DRender::SetZPos(float z)
{
    if (!d3d) return;
    lpView->SetZPos(z);
}

void D3DRender::AddXPos(float x)
{
    if (!d3d) return;
    lpView->AddXPos(x);
}

void D3DRender::AddYPos(float y)
{
    if (!d3d) return;
    lpView->AddYPos(y);
}

void D3DRender::AddZPos(float z)
{
    if (!d3d) return;
    lpView->AddZPos(z);
}

void D3DRender::SetXRotate(float x)
{
    if (!d3d) return;
    lpView->SetXAngle(x);
}

void D3DRender::SetYRotate(float y)
{
    if (!d3d) return;
    lpView->SetYAngle(y);
}

void D3DRender::SetZRotate(float z)
{
    if (!d3d) return;
    lpView->SetZAngle(z);
}

void D3DRender::AddXRotate(float x)
{
    if (!d3d) return;
    lpView->AddXAngle(x);
}

void D3DRender::AddYRotate(float y)
{
    if (!d3d) return;
    lpView->AddYAngle(y);
}

void D3DRender::AddZRotate(float z)
{
    if (!d3d) return;
    lpView->AddZAngle(z);
}

void D3DRender::GoForward(float d)
{
    if (!d3d) return;
    lpView->GoForward(d);
}

void D3DRender::GoSide(float d)
{
    if (!d3d) return;
    lpView->GoSide(d);
}

float D3DRender::GetXPos()
{
    if (!d3d) return 0;
    return lpView->GetXPos();
}

float D3DRender::GetYPos()
{
    if (!d3d) return 0;
    return lpView->GetYPos();
}

float D3DRender::GetZPos()
{
    if (!d3d) return 0;
    return lpView->GetZPos();
}

float D3DRender::GetXRotate()
{
    if (!d3d) return 0;
    return lpView->GetXAngle();
}

float D3DRender::GetYRotate()
{
    if (!d3d) return 0;
    return lpView->GetYAngle();
}

float D3DRender::GetZRotate()
{
    if (!d3d) return 0;
    return lpView->GetZAngle();
}

Texture* D3DRender::LoadTextureBMP(char *filename)
{
    if (!d3d)
    {
        Texture *t=new Texture;
        t->location=TLOC_BMP;
        t->file=new char[strlen(filename)+1];
        strcpy(t->file,filename);
        t->alphaFile=NULL;
        CTexture *tex=new CTexture(d3d);
        t->data=tex;
        return t;
    }
    else
    {
        Texture *t=new Texture;
        t->location=TLOC_BMP;
        t->file=new char[strlen(filename)+1];
        strcpy(t->file,filename);
        t->alphaFile=NULL;
        CTexture *tex=new CTexture(d3d);
        t->data=tex;
        tex->LoadFromFile(filename);
        return t;
    }
}

Texture* D3DRender::LoadAlphaTextureBMP(char *filename,char *alphaname)
{
    if (!d3d)
    {
        Texture *t=new Texture;
        t->location=TLOC_BMP;
        t->file=new char[strlen(filename)+1];
        strcpy(t->file,filename);
        t->alphaFile=new char[strlen(alphaname)+1];
        strcpy(t->alphaFile,alphaname);
        CTexture *tex=new CTexture(d3d);
        t->data=tex;
        return t;
    }
    else
    {
        Texture *t=new Texture;
        t->location=TLOC_BMP;
        t->file=new char[strlen(filename)+1];
        strcpy(t->file,filename);
        t->alphaFile=new char[strlen(alphaname)+1];
        strcpy(t->alphaFile,alphaname);
        CTexture *tex=new CTexture(d3d);
        t->data=tex;
        tex->LoadFromFileWithAlpha(filename,alphaname);
        return t;
    }
}

void D3DRender::CreateMemoryTexture(Texture *tex)
{
    CTexture *t=new CTexture(d3d);
    tex->data=t;
    t->LoadFromMemoryTexture(tex);
}

void D3DRender::UpdateMemoryTexture(Texture *tex)
{
    if (!tex->data) return;
    ((CTexture*)tex->data)->UpdateMemoryTexture(tex);
}

void D3DRender::ReloadTexture(Texture *tex)
{
    if (tex->location==TLOC_BMP)
    {
        if (tex->alphaFile)
            ((CTexture*)(tex->data))->LoadFromFileWithAlpha(tex->file,tex->alphaFile);
        else
            ((CTexture*)(tex->data))->LoadFromFile(tex->file);
    }
}

void D3DRender::FreeTexture(Texture *tex)
{
    delete ((CTexture*)tex->data);
}

void D3DRender::UseOriginView()
{
    d3d->SetView(lpZeroView);
}

void D3DRender::UseNormalView()
{
    d3d->SetView(lpView);
}

void D3DRender::StartText()
{
    d3d->GetDDraw()->GetBackBuffer()->GetDC(&dc);
    SetBkMode(dc,TRANSPARENT);
    if (curFont) SelectObject(dc,curFont->font);
}

void D3DRender::EndText()
{
    d3d->GetDDraw()->GetBackBuffer()->ReleaseDC(dc);
    dc=NULL;
}

void D3DRender::RenderText(int x,int y,char *text,unsigned long color)
{
    SetTextColor(dc,color&0xffffff);
    TextOut(dc,x,y,text,strlen(text));
}

void* D3DRender::AddFont(HFONT font)
{
    if (firstFont==NULL)
    {
        firstFont=lastFont=new FontListElem;
        lastFont->next=NULL;
        lastFont->font=font;
        return lastFont;
    }
    lastFont->next=new FontListElem;
    lastFont=lastFont->next;
    lastFont->next=NULL;
    lastFont->font=font;
    return lastFont;
}

void D3DRender::UseFont(void* font)
{
    curFont=(FontListElem*)font;
    if (dc) SelectObject(dc,((FontListElem*)font)->font);
}

void D3DRender::GetStats(RenderStats *stat)
{
    stat->polyDrawn=polyDrawn;
    stat->polyTotal=polyTotal;
    stat->polyReflect=polyReflect;
}

void D3DRender::BeginBitmaps()
{
    UseOriginView();
    BeginScene();
}

void D3DRender::RenderBitmap(int x,int y,Art *art,int texIndex,unsigned long color)
{
    RenderBitmap(x,y,art->tex[1][texIndex],art->origSizeX[texIndex],art->origSizeY[texIndex],color);
}

void D3DRender::RenderBitmap(int x,int y,Art *art,int texIndex,int palNum,unsigned long color)
{
    RenderBitmap(x,y,art->tex[palNum][texIndex],art->origSizeX[texIndex],art->origSizeY[texIndex],color);
}

void D3DRender::RenderBitmap(int x,int y,Texture *tex,int w,int h,unsigned long color)
{
    if ((!renderEnabled)||(!sceneEnabled))
        return;
    float ofs=0.3125;
    if (tex->type!=TEXTYPE_REPEAT) ofs=1.3125;
    // Generate a polygon for the bitmap
    p.Clear();
    float xs=w;
    float ys=h;
    p.AddVert(20.0*(x-160.0)/160.0,20.0*(100.0-y)/100.0*((float)scrnY/(float)scrnX),20,ofs/(float)tex->memTex.width,ofs/(float)tex->memTex.height,0,0,color);
    p.AddVert(20.0*(x+xs-160.0)/160.0,20.0*(100.0-y)/100.0*((float)scrnY/(float)scrnX),20,(xs+ofs)/(float)tex->memTex.width,ofs/(float)tex->memTex.height,0,0,color);
    p.AddVert(20.0*(x+xs-160.0)/160.0,20.0*(100.0-(y+ys))/100.0*((float)scrnY/(float)scrnX),20,(xs+ofs)/(float)tex->memTex.width,(ys+ofs)/(float)tex->memTex.height,0,0,color);
    p.AddVert(20.0*(x-160.0)/160.0,20.0*(100.0-(y+ys))/100.0*((float)scrnY/(float)scrnX),20,ofs/(float)tex->memTex.width,(ys+ofs)/(float)tex->memTex.height,0,0,color);
    p.flags=PF_TWOSIDED|PF_PRIORITY;
    p.texture=tex;
    // Draw the polygon
    RenderPoly(p);
}

void D3DRender::RenderBitmapYFlipped(int x,int y,Art *art,int texIndex,unsigned long color)
{
    if ((!renderEnabled)||(!sceneEnabled))
        return;
    float ofs=0.3125;
    if (art->tex[1][texIndex]->type!=TEXTYPE_REPEAT) ofs=1.3125;
    // Generate a polygon for the bitmap
    p.Clear();
    float xs=art->origSizeX[texIndex];
    float ys=art->origSizeY[texIndex];
    p.AddVert(20.0*(x-160.0)/160.0,20.0*(100.0-y)/100.0*((float)scrnY/(float)scrnX),20,ofs/(float)art->sizeX[texIndex],(ys+ofs)/(float)art->sizeY[texIndex],0,0,color);
    p.AddVert(20.0*(x+xs-160.0)/160.0,20.0*(100.0-y)/100.0*((float)scrnY/(float)scrnX),20,(xs+ofs)/(float)art->sizeX[texIndex],(ys+ofs)/(float)art->sizeY[texIndex],0,0,color);
    p.AddVert(20.0*(x+xs-160.0)/160.0,20.0*(100.0-(y+ys))/100.0*((float)scrnY/(float)scrnX),20,(xs+ofs)/(float)art->sizeX[texIndex],ofs/(float)art->sizeY[texIndex],0,0,color);
    p.AddVert(20.0*(x-160.0)/160.0,20.0*(100.0-(y+ys))/100.0*((float)scrnY/(float)scrnX),20,ofs/(float)art->sizeX[texIndex],ofs/(float)art->sizeY[texIndex],0,0,color);
    p.flags=PF_TWOSIDED|PF_PRIORITY;
    p.texture=art->tex[1][texIndex];
    // Draw the polygon
    RenderPoly(p);
}

void D3DRender::RenderScaledBitmap(int x,int y,int w,int h,Art *art,int texIndex,unsigned long color)
{
    RenderScaledBitmap(x,y,w,h,art,texIndex,1,color);
}

void D3DRender::RenderScaledBitmap(int x,int y,int w,int h,Art *art,int texIndex,int palNum,unsigned long color)
{
    if ((!renderEnabled)||(!sceneEnabled))
        return;
    float ofs=0.3125;
    if (art->tex[palNum][texIndex]->type!=TEXTYPE_REPEAT) ofs=1.3125;
    // Generate a polygon for the bitmap
    p.Clear();
    float xs=art->origSizeX[texIndex];
    float ys=art->origSizeY[texIndex];
    p.AddVert(20.0*(x-160.0)/160.0,20.0*(100.0-y)/100.0*((float)scrnY/(float)scrnX),20,ofs/(float)art->sizeX[texIndex],ofs/(float)art->sizeY[texIndex],0,0,color);
    p.AddVert(20.0*(x+w-160.0)/160.0,20.0*(100.0-y)/100.0*((float)scrnY/(float)scrnX),20,(xs+ofs)/(float)art->sizeX[texIndex],ofs/(float)art->sizeY[texIndex],0,0,color);
    p.AddVert(20.0*(x+w-160.0)/160.0,20.0*(100.0-(y+h))/100.0*((float)scrnY/(float)scrnX),20,(xs+ofs)/(float)art->sizeX[texIndex],(ys+ofs)/(float)art->sizeY[texIndex],0,0,color);
    p.AddVert(20.0*(x-160.0)/160.0,20.0*(100.0-(y+h))/100.0*((float)scrnY/(float)scrnX),20,ofs/(float)art->sizeX[texIndex],(ys+ofs)/(float)art->sizeY[texIndex],0,0,color);
    p.flags=PF_TWOSIDED|PF_PRIORITY;
    p.texture=art->tex[palNum][texIndex];
    // Draw the polygon
    RenderPoly(p);
}

void D3DRender::EndBitmaps()
{
    EndScene();
    UseNormalView();
}

