/*
 *  OpenDuke
 *  Copyright (C) 1999  Rusty Wagner
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */


#ifndef PLATFORM_UNIX
#include <vcl.h>
#pragma hdrstop
#include <condefs.h>
#else
#include "linux_inc.h"
#endif

#include <math.h>
#include "SDLRender.h"
#include "console.h"

extern char errorStr[256];
extern int error;

static Poly p;

static MATRIX initViewMatrix={1, 0, 0, 0,
                              0, 1, 0, 0,
			      0, 0, 1, 0,
			      0, 0, 0, 1};

#define SPRITE 0
#define WINDOW 1

#ifndef PLATFORM_UNIX
extern HMODULE ddrawLib;
extern HINSTANCE hInst;
#endif

#ifndef PLATFORM_UNIX
struct ModeEnumStruct
{
    ModeList *modes16;
    ModeList *modes32;
};

HRESULT WINAPI EnumModesFunc(LPDDSURFACEDESC sd,LPVOID ptr)
{
    ModeList *modes16=((ModeEnumStruct*)ptr)->modes16;
    ModeList *modes32=((ModeEnumStruct*)ptr)->modes32;
    if (sd->ddpfPixelFormat.dwRGBBitCount==16)
    {
        modes16->mode[modes16->count].w=sd->dwWidth;
        modes16->mode[modes16->count++].h=sd->dwHeight;
    }
    else if (sd->ddpfPixelFormat.dwRGBBitCount==32)
    {
        modes32->mode[modes32->count].w=sd->dwWidth;
        modes32->mode[modes32->count++].h=sd->dwHeight;
    }
    return DDENUMRET_OK;
}
#endif

void SDLRender::EnumModes(ModeList *modes16,ModeList *modes32)
{
#ifndef PLATFORM_UNIX
    modes16->count=0; modes32->count=0;
    if (ddrawLib==NULL)
        ddrawLib=LoadLibrary("ddraw.dll");
    if (ddrawLib==NULL) return;
    HRESULT WINAPI (*ddc)(GUID FAR*,LPVOID *,IUnknown FAR *)=
        (HRESULT WINAPI (*)(GUID FAR*,LPVOID *,IUnknown FAR *))
        GetProcAddress(ddrawLib,"DirectDrawCreate");
    LPDIRECTDRAW lpDD;
	if (ddc(NULL,(void**)&lpDD,NULL)!=DD_OK) return;
    DDSURFACEDESC sd;
    memset(&sd,0,sizeof(sd));
    sd.dwSize=sizeof(sd);
    sd.dwFlags=DDSD_CAPS;
    sd.ddsCaps.dwCaps=DDSCAPS_3DDEVICE|DDSCAPS_COMPLEX|DDSCAPS_FLIP;
    ModeEnumStruct mes;
    mes.modes16=modes16;
    mes.modes32=modes32;
    lpDD->EnumDisplayModes(0,NULL,&mes,EnumModesFunc);
    lpDD->Release();
    lpDD=NULL;
#endif
}

#ifdef PLATFORM_UNIX
SDLRender::SDLRender(int _scrnX, int _scrnY, int depth)
#else
SDLRender::SDLRender(HINSTANCE _hInst,HWND _hWnd,int _scrnX,int _scrnY,int depth)
#endif
{
    init=0;
    firstFont=lastFont=curFont=NULL;
    renderEnabled=sceneEnabled=false;
    scrnX=_scrnX; scrnY=_scrnY;
#ifdef PLATFORM_UNIX
    cprintf ("Attempting to create an SDL window...\n");
    sdl=new CSDL(scrnX,scrnY,depth);
    if (error)
    {
	    cprintf("SDL failed\n");
	    delete sdl; sdl=NULL;
    }
    else
	    cprintf("Initialization OK, using SDL\n");
#else
    dc=NULL;
    hInst=_hInst; hWnd=_hWnd;
    cprintf("Attempting to initialize DirectDraw...\n");
    dd=new CDirectDraw(hWnd,scrnX,scrnY,depth,true);
    if (error)
    {
        cprintf("DirectDraw failed\n");
        cprintf("Initializing GDI renderer\n");
        delete dd; dd=NULL;
        bmp=new Graphics::TBitmap;
        bmp->Width=scrnX;
        bmp->Height=scrnY;
        bmp->PixelFormat=pf16bit;
    }
    else
        cprintf("Initialization OK, using DirectDraw\n");
#endif
    screen=new void*[scrnY];
    wbuffer=new unsigned long*[scrnY];
    for (int i=0;i<scrnY;i++)
        wbuffer[i]=new unsigned long[scrnX];
    for (int i=0;i<MAX_PAL;i++)
        colorTable[i]=NULL;
    init=1;
    xp=yp=zp=xa=ya=za=0;
    projYMult=0.5*(float)scrnX/(float)scrnY;
    UpdateViewMatrix();
}

SDLRender::~SDLRender()
{
    cprintf("Deinitializing software renderer...\n");
#ifndef PLATFORM_UNIX
    if (dd) { delete dd; dd=NULL; }
#endif
    cprintf("Software renderer deinitialized successfully\n");
    init=0;
}

int SDLRender::Initialized()
{
    return init;
}

void SDLRender::ChangeResolution(int width,int height,int depth)
{
    sdl->ChangeResolution(width,height,depth);
    if (wbuffer)
    {
        for (int i=0;i<scrnY;i++)
            delete[] wbuffer[i];
        delete[] wbuffer;
        wbuffer=NULL;
    }
    scrnX=width; scrnY=height;
    Vector p=Vector(GetXPos(),GetYPos(),GetZPos());
    Vector r=Vector(GetXRotate(),GetYRotate(),GetZRotate());
#ifndef PLATFORM_UNIX
    if (dd)
    {
        error=0;
        dd->ChangeResolution(hWnd,width,height,depth);
        if (error)
            throw "Could not change video settings\n";
    }
    else
    {
        bmp->Width=scrnX;
        bmp->Height=scrnY;
    }
#endif
    projYMult=0.5*(float)scrnX/(float)scrnY;
    SetXPos(p.x); SetYPos(p.y); SetZPos(p.z);
    SetXRotate(r.x); SetYRotate(r.y); SetZRotate(r.z);
    delete[] screen;
    screen=new void*[scrnY];
    wbuffer=new unsigned long*[scrnY];
    for (int i=0;i<scrnY;i++)
        wbuffer[i]=new unsigned long[scrnX];
    cprintf("Resolution change successful\n");
}

void SDLRender::SetRenderWnd(HWND _hWnd)
{
#ifndef PLATFORM_UNIX
    hWnd=_hWnd;
#endif
}

void SDLRender::BeginRender()
{
    polyDrawn=polyTotal=polyReflect=0;
    renderEnabled=true;
    for(int i=0;i<scrnY;i++)
    {
	    // scrnX * 2 * i = width x bytes per pixel x something or other
	    screen[i]=&(((char*)sdl->GetBackBuffer())[scrnX*2*i]);
	    memset(screen[i],0,scrnX*2);
	    memset(wbuffer[i],0,scrnX<<2);
    }
#ifndef PLATFORM_UNIX
    if (dd)
    {
        DDSURFACEDESC sd;
        memset(&sd,0,sizeof(sd));
        sd.dwSize=sizeof(sd);
        sd.dwFlags=DDSD_LPSURFACE;
        dd->GetBackBuffer3()->Lock(NULL,&sd,DDLOCK_WAIT|
            DDLOCK_SURFACEMEMORYPTR,NULL);
        for (int i=0;i<scrnY;i++)
        {
            screen[i]=&(((char*)sd.lpSurface)[sd.lPitch*i]);
            memset(screen[i],0,sd.lPitch);
            memset(wbuffer[i],0,scrnX<<2);
        }
    }
    else
    {
        for (int i=0;i<scrnY;i++)
        {
            screen[i]=bmp->ScanLine[i];
            memset(screen[i],0,bmp->Width<<1);
            memset(wbuffer[i],0,scrnX<<2);
        }
    }
#endif
}

void SDLRender::BeginScene()
{
    sceneEnabled=true;
}

void SDLRender::EndRender()
{
    renderEnabled=false;
//    cprintf("Updating\n");
    sdl->Flip();
#ifndef PLATFORM_UNIX
    if (dd)
    {
        dd->GetBackBuffer3()->Unlock(NULL);
        if (dd) dd->Flip();
    }
    else
    {
        dc=GetDC(hWnd);
        BitBlt(dc,0,0,bmp->Width,bmp->Height,bmp->Canvas->Handle,0,0,SRCCOPY);
        ReleaseDC(hWnd,dc);
    }
#endif
}

void SDLRender::EndScene()
{
    sceneEnabled=false;
}

static ProjVertex *LeftArray[3];
static ProjVertex *RightArray[3];
float DeltaLeftX,DeltaRightX,DeltaLeftU,DeltaLeftV;
float DeltaRightU,DeltaRightV;
float LeftX,RightX,LeftU,LeftV,RightU,RightV;
float LeftW,RightW,DeltaLeftW,DeltaRightW;
int LeftHeight,RightHeight;

static int CalculateLeftDelta(int index1,int index2)
{
    ProjVertex *v1=LeftArray[index1];
    ProjVertex *v2=LeftArray[index2];
    int height=v2->y-v1->y;
    if (height==0)
        return 0;
    float recip=1.0/(float)height;
    DeltaLeftX=(float)(v2->x-v1->x)*recip;
    DeltaLeftU=(float)(v2->uDivZ-v1->uDivZ)*recip;
    DeltaLeftV=(float)(v2->vDivZ-v1->vDivZ)*recip;
    DeltaLeftW=(float)(v2->w-v1->w)*recip;
    LeftX=v1->x;
    LeftU=v1->uDivZ;
    LeftV=v1->vDivZ;
    LeftW=v1->w;
    LeftHeight=height;
    return height;
}

static int CalculateRightDelta(int index1,int index2)
{
    ProjVertex *v1=RightArray[index1];
    ProjVertex *v2=RightArray[index2];
    int height=v2->y-v1->y;
    if (height==0)
        return 0;
    float recip=1.0/(float)height;
    DeltaRightX=(float)(v2->x-v1->x)*recip;
    DeltaRightU=(float)(v2->uDivZ-v1->uDivZ)*recip;
    DeltaRightV=(float)(v2->vDivZ-v1->vDivZ)*recip;
    DeltaRightW=(float)(v2->w-v1->w)*recip;
    RightX=v1->x;
    RightU=v1->uDivZ;
    RightV=v1->vDivZ;
    RightW=v1->w;
    RightHeight=height;
    return height;
}

static int deltau1,deltav1,deltaw;
static int curu1,curv1,curw,texAnd;
static unsigned char *texPtr;

void SDLRender::TexturedTriangle(ProjVertex *v1,ProjVertex *v2,ProjVertex *v3,Texture *tex)
{
    if (!tex) return;
    int rightindex,leftindex;
    if (v1->y>v2->y)
    {
        ProjVertex *temp=v1;
        v1=v2;
        v2=temp;
    }
    if (v1->y>v3->y)
    {
        ProjVertex *temp=v1;
        v1=v3;
        v3=temp;
    }
    if (v2->y>v3->y)
    {
        ProjVertex *temp=v2;
        v2=v3;
        v3=temp;
    }
    if (v1->y==v3->y) return;
    if (v1->y>=scrnY) return;
    if (v3->y<0) return;
    if ((v1->x<0)&&(v2->x<0)&&(v3->x<0)) return;
    if ((v1->x>=scrnX)&&(v2->x>=scrnX)&&(v3->x>=scrnX)) return;
    float t=(float)(v2->y-v1->y)/(float)(v3->y-v1->y);
    int widest=(v1->x+(int)(t*(float)(v3->x-v1->x)))-v2->x;
    if (widest>0)
    {
        rightindex=2;
        leftindex=1;
        LeftArray[0]=v1;
        LeftArray[1]=v2;
        LeftArray[2]=v3;
        RightArray[1]=v1;
        RightArray[2]=v3;
        if (CalculateRightDelta(rightindex-1,rightindex)<=0)
            return;
        if (CalculateLeftDelta(leftindex-1,leftindex)<=0)
        {
            leftindex++;
            if (CalculateLeftDelta(leftindex-1,leftindex)<=0)
                return;
        }
    }
    else if (widest<0)
    {
        leftindex=2;
        rightindex=1;
        RightArray[0]=v1;
        RightArray[1]=v2;
        RightArray[2]=v3;
        LeftArray[1]=v1;
        LeftArray[2]=v3;
        if (CalculateLeftDelta(leftindex-1,leftindex)<=0)
            return;
        if (CalculateRightDelta(rightindex-1,rightindex)<=0)
        {
            rightindex++;
            if (CalculateRightDelta(rightindex-1,rightindex)<=0)
                return;
        }
    }
    else
        return;
    int cury=v1->y;
    unsigned short *table=(unsigned short*)colorTable[tex->pal][(v1->color>>3)&31];
    texAnd=((tex->memTex.height-1)<<8)|(tex->memTex.width-1);
    texPtr=((unsigned char *)tex->data);
    while (1)
    {
        if (cury>=scrnY)
            return;
        if (cury<0)
        {
            if (LeftHeight>RightHeight)
            {
                if ((-cury)>RightHeight)
                {
                    LeftX+=DeltaLeftX*RightHeight;
                    LeftU+=DeltaLeftU*RightHeight;
                    LeftV+=DeltaLeftV*RightHeight;
                    LeftW+=DeltaLeftW*RightHeight;
                    LeftHeight-=RightHeight;
                    cury+=RightHeight;
                    rightindex++;
                    if (rightindex>2)
                        return;
                    else
                    {
                        if (CalculateRightDelta(rightindex-1,rightindex)<=0)
                            return;
                    }
                }
                else
                {
                    LeftX+=DeltaLeftX*(-cury);
                    LeftU+=DeltaLeftU*(-cury);
                    LeftV+=DeltaLeftV*(-cury);
                    LeftW+=DeltaLeftW*(-cury);
                    RightX+=DeltaRightX*(-cury);
                    RightU+=DeltaRightU*(-cury);
                    RightV+=DeltaRightV*(-cury);
                    RightW+=DeltaRightW*(-cury);
                    LeftHeight+=cury;
                    RightHeight+=cury;
                    cury=0;
                }
            }
            else
            {
                if ((-cury)>LeftHeight)
                {
                    RightX+=DeltaRightX*LeftHeight;
                    RightU+=DeltaRightU*LeftHeight;
                    RightV+=DeltaRightV*LeftHeight;
                    RightW+=DeltaRightW*LeftHeight;
                    RightHeight-=LeftHeight;
                    cury+=LeftHeight;
                    leftindex++;
                    if (leftindex>2)
                        return;
                    else
                    {
                        if (CalculateLeftDelta(leftindex-1,leftindex)<=0)
                            return;
                    }
                }
                else
                {
                    LeftX+=DeltaLeftX*(-cury);
                    LeftU+=DeltaLeftU*(-cury);
                    LeftV+=DeltaLeftV*(-cury);
                    LeftW+=DeltaLeftW*(-cury);
                    RightX+=DeltaRightX*(-cury);
                    RightU+=DeltaRightU*(-cury);
                    RightV+=DeltaRightV*(-cury);
                    RightW+=DeltaRightW*(-cury);
                    LeftHeight+=cury;
                    RightHeight+=cury;
                    cury=0;
                }
            }
            continue;
        }
        int x1=LeftX;
        int x2=RightX;
        int width=x2-x1;
        if ((x2>x1)&&(x1<scrnX)&&(x2>=0))
        {
            if (x1<0) x1=0;
            if (x2>=scrnX) x2=scrnX-1;
            unsigned short *row=&((unsigned short*)screen[cury])[x1];
            unsigned long *wbuf=&wbuffer[cury][x1];
            curw=LeftW*2147483648.0;
            float curu=LeftU;
            float curv=LeftV;
            float recip=1.0/(float)width;
            float deltau=(RightU-LeftU)*recip;
            float deltav=(RightV-LeftV)*recip;
            int deltaw=(RightW-LeftW)*recip*2147483648.0;
            int startX=x1-LeftX;
            curu+=deltau*startX;
            curv+=deltav*startX;
            curw+=deltaw*startX;
            float deltau16=deltau*16.0;
            float deltav16=deltav*16.0;
            int deltaw16=deltaw<<4;
            float z=2147483648.0/curw;
            curu1=(int)(curu*z*65536.0);
            curv1=(int)(curv*z*65536.0);
            int curu2,curv2;
            int left=x2-x1+1;
            while (left)
            {
                if (left>=16)
                {
                    curu+=deltau16;
                    curv+=deltav16;
                    float z=2147483648.0/(float)(curw+deltaw16);
                    curu2=(int)(curu*z*65536.0);
                    curv2=(int)(curv*z*65536.0);
                    int deltau1=(curu2-curu1)>>4;
                    int deltav1=(curv2-curv1)>>4;
                    for (int i=0;i<16;i++,row++,wbuf++)
                    {
                        if (curw>=(*wbuf))
                        {
                            int texelU=(curu1>>16)&(tex->memTex.width-1);
                            int texelV=(curv1>>16)&(tex->memTex.height-1);
                            unsigned char texel=((unsigned char *)tex->data)[texelV*tex->memTex.width+texelU];
                            if (texel!=255)
                            {
                                *row=table[texel];
                                *wbuf=curw;
                            }
                        }
                        curu1+=deltau1;
                        curv1+=deltav1;
                        curw+=deltaw;
                    }
                    left-=16;
                }
                else
                {
                    curu+=deltau*left;
                    curv+=deltav*left;
                    float z=2147483648.0/(float)(curw+deltaw*left);
                    curu2=(int)(curu*z*65536.0);
                    curv2=(int)(curv*z*65536.0);
                    int deltau1=(curu2-curu1)/left;
                    int deltav1=(curv2-curv1)/left;
                    for (int i=0;i<left;i++,row++,wbuf++)
                    {
                        if (curw>=(*wbuf))
                        {
                            int texelU=(curu1>>16)&(tex->memTex.width-1);
                            int texelV=(curv1>>16)&(tex->memTex.height-1);
                            unsigned char texel=((unsigned char *)tex->data)[texelV*tex->memTex.width+texelU];
                            if (texel!=255)
                            {
                                *row=table[texel];
                                *wbuf=curw;
                            }
                        }
                        curu1+=deltau1;
                        curv1+=deltav1;
                        curw+=deltaw;
                    }
                    left=0;
                }
                curu1=curu2;
                curv1=curv2;
            }
        }
        cury++;
        LeftHeight--;
        if (LeftHeight<=0)
        {
            leftindex++;
            if (leftindex>2)
                return;
            else
            {
                if (CalculateLeftDelta(leftindex-1,leftindex)<=0)
                    return;
            }
        }
        else
        {
            LeftX+=DeltaLeftX;
            LeftU+=DeltaLeftU;
            LeftV+=DeltaLeftV;
            LeftW+=DeltaLeftW;
        }
        RightHeight--;
        if (RightHeight<=0)
        {
            rightindex++;
            if (rightindex>2)
                return;
            else
            {
                if (CalculateRightDelta(rightindex-1,rightindex)<=0)
                    return;
            }
        }
        else
        {
            RightX+=DeltaRightX;
            RightU+=DeltaRightU;
            RightV+=DeltaRightV;
            RightW+=DeltaRightW;
        }
    }
}

void SDLRender::RenderPoly(Poly &poly)
{
    int n=poly.nVert;
    Vertex *tv=new Vertex[n];
    ProjVertex pv1,pv2,pv3;
    int start=-1;
    for (int i=0;i<n;i++)
    {
        tv[i]=poly.v[i];
        TranslateVertex(tv[i]);
        if ((start==-1)&&(tv[i].z>=1))
            start=i;
    }
    if (start==-1)
    {
        delete[] tv;
        return;
    }
    pv1.w=1.0/tv[start].z;
    pv1.x=(tv[start].x*pv1.w*0.5+0.5)*scrnX;
    pv1.y=(-tv[start].y*pv1.w*projYMult+0.5)*scrnY;
    if (poly.texture)
    {
        pv1.uDivZ=tv[start].u*poly.texture->memTex.width*pv1.w;
        pv1.vDivZ=tv[start].v*poly.texture->memTex.height*pv1.w;
    }
    else
        pv1.uDivZ=pv1.vDivZ=0;
    pv1.color=tv[start].color;
    bool pt2valid=false;
    bool split=false;
    int i;
    for (i=(start+1)%n;i!=start;i=(i+1)%n)
    {
        if (tv[i].z<1)
        {
            if (split) continue;
            pv2=pv3;
            float t=(tv[i].z-1)/(tv[i].z-tv[(i+n-1)%n].z);
            float x=t*(tv[(i+n-1)%n].x-tv[i].x)+tv[i].x;
            float y=t*(tv[(i+n-1)%n].y-tv[i].y)+tv[i].y;
            float u=t*(tv[(i+n-1)%n].u-tv[i].u)+tv[i].u;
            float v=t*(tv[(i+n-1)%n].v-tv[i].v)+tv[i].v;
            pv3.w=1.0;
            pv3.x=(x*0.5+0.5)*scrnX;
            pv3.y=(-y*projYMult+0.5)*scrnY;
            if (poly.texture)
            {
                pv3.uDivZ=u*poly.texture->memTex.width;
                pv3.vDivZ=v*poly.texture->memTex.height;
            }
            else
                pv3.uDivZ=pv3.vDivZ=0;
            pv3.color=tv[i].color;
            if (pt2valid)
                TexturedTriangle(&pv1,&pv2,&pv3,poly.texture);
            else
                pt2valid=true;
            split=true;
            continue;
        }
        pv2=pv3;
        if (split)
        {
            float t;
            if (tv[i].z!=tv[(i+n-1)%n].z)
                t=(tv[i].z-1)/(tv[i].z-tv[(i+n-1)%n].z);
            else
                t=0;
            float x=t*(tv[(i+n-1)%n].x-tv[i].x)+tv[i].x;
            float y=t*(tv[(i+n-1)%n].y-tv[i].y)+tv[i].y;
            float u=t*(tv[(i+n-1)%n].u-tv[i].u)+tv[i].u;
            float v=t*(tv[(i+n-1)%n].v-tv[i].v)+tv[i].v;
            pv3.w=1.0;
            pv3.x=(x*0.5+0.5)*scrnX;
            pv3.y=(-y*projYMult+0.5)*scrnY;
            if (poly.texture)
            {
                pv3.uDivZ=u*poly.texture->memTex.width;
                pv3.vDivZ=v*poly.texture->memTex.height;
            }
            else
                pv3.uDivZ=pv3.vDivZ=0;
            pv3.color=tv[i].color;
            if (pt2valid)
                TexturedTriangle(&pv1,&pv2,&pv3,poly.texture);
            else
                pt2valid=true;
            split=false;
            i=(i+n-1)%n;
            continue;
        }
        pv3.w=1.0/tv[i].z;
        pv3.x=(tv[i].x*pv3.w*0.5+0.5)*scrnX;
        pv3.y=(-tv[i].y*pv3.w*projYMult+0.5)*scrnY;
        if (poly.texture)
        {
            pv3.uDivZ=tv[i].u*poly.texture->memTex.width*pv3.w;
            pv3.vDivZ=tv[i].v*poly.texture->memTex.height*pv3.w;
        }
        else
            pv3.uDivZ=pv3.vDivZ=0;
        pv3.color=tv[i].color;
        if (pt2valid)
            TexturedTriangle(&pv1,&pv2,&pv3,poly.texture);
        else
            pt2valid=true;
    }
    if (split)
    {
        pv2=pv3;
        float t;
        if (tv[i].z!=tv[(i+n-1)%n].z)
            t=(tv[i].z-1)/(tv[i].z-tv[(i+n-1)%n].z);
        else
            t=0;
        float x=t*(tv[(i+n-1)%n].x-tv[i].x)+tv[i].x;
        float y=t*(tv[(i+n-1)%n].y-tv[i].y)+tv[i].y;
        float u=t*(tv[(i+n-1)%n].u-tv[i].u)+tv[i].u;
        float v=t*(tv[(i+n-1)%n].v-tv[i].v)+tv[i].v;
        pv3.w=1.0;
        pv3.x=(x*0.5+0.5)*scrnX;
        pv3.y=(-y*projYMult+0.5)*scrnY;
        if (poly.texture)
        {
            pv3.uDivZ=u*poly.texture->memTex.width;
            pv3.vDivZ=v*poly.texture->memTex.height;
        }
        else
            pv3.uDivZ=pv3.vDivZ=0;
        pv3.color=tv[i].color;
        if (pt2valid)
            TexturedTriangle(&pv1,&pv2,&pv3,poly.texture);
    }
    delete[] tv;
}

void SDLRender::RenderPolyList(PolyList &list)
{
    PolyListElem *ptr=list.first;
    while (ptr!=NULL)
    {
        RenderPoly(ptr->poly);
        ptr=ptr->next;
        polyTotal++;
    }
}

void SDLRender::RenderPolyListNonBSP(PolyList &list)
{
    PolyListElem *ptr=list.first;
    while (ptr!=NULL)
    {
        if (!(ptr->poly.flags&PF_INTREE))
        {
            RenderPoly(ptr->poly);
            polyTotal++;
        }
        ptr=ptr->next;
    }
}

static void MatrixRotateX(MATRIX& mat,float a)
{
	MATRIX temp;
	float s=sin(a);
	float c=cos(a);

	temp._12=mat._12*c+mat._13*s;
	temp._22=mat._22*c+mat._23*s;
	temp._32=mat._32*c+mat._33*s;
	temp._42=mat._42*c+mat._43*s;
	temp._13=mat._12*-s+mat._13*c;
	temp._23=mat._22*-s+mat._23*c;
	temp._33=mat._32*-s+mat._33*c;
	temp._43=mat._42*-s+mat._43*c;

	mat._12=temp._12; mat._22=temp._22; mat._32=temp._32;
	mat._42=temp._42; mat._13=temp._13; mat._23=temp._23;
	mat._33=temp._33; mat._43=temp._43;
}

static void MatrixRotateY(MATRIX& mat,float a)
{
	MATRIX temp;
	float s=sin(a);
	float c=cos(a);

	temp._11=mat._11*c+mat._13*-s;
	temp._21=mat._21*c+mat._23*-s;
	temp._31=mat._31*c+mat._33*-s;
	temp._41=mat._41*c+mat._43*-s;
	temp._13=mat._11*s+mat._13*c;
	temp._23=mat._21*s+mat._23*c;
	temp._33=mat._31*s+mat._33*c;
	temp._43=mat._41*s+mat._43*c;

	mat._11=temp._11; mat._21=temp._21; mat._31=temp._31;
	mat._41=temp._41; mat._13=temp._13; mat._23=temp._23;
	mat._33=temp._33; mat._43=temp._43;
}

static void MatrixRotateZ(MATRIX& mat,float a)
{
	MATRIX temp;
	float s=sin(a);
	float c=cos(a);

	temp._11=mat._11*c+mat._12*s;
	temp._21=mat._21*c+mat._22*s;
	temp._31=mat._31*c+mat._32*s;
	temp._41=mat._41*c+mat._42*s;
	temp._12=mat._11*-s+mat._12*c;
	temp._22=mat._21*-s+mat._22*c;
	temp._32=mat._31*-s+mat._32*c;
	temp._42=mat._41*-s+mat._42*c;

	mat._11=temp._11; mat._21=temp._21; mat._31=temp._31;
	mat._41=temp._41; mat._12=temp._12; mat._22=temp._22;
	mat._32=temp._32; mat._42=temp._42;
}

static void TransformVector(Vector& v,MATRIX& mat)
{
//	v=Vector(v.x*mat._11+v.y*mat._12+v.z*mat._13+mat._41,
//			 v.x*mat._21+v.y*mat._22+v.z*mat._23+mat._42,
//			 v.x*mat._31+v.y*mat._32+v.z*mat._33+mat._43);
	v=Vector(v.x*mat._11+v.y*mat._21+v.z*mat._31+mat._41,
			 v.x*mat._12+v.y*mat._22+v.z*mat._32+mat._42,
			 v.x*mat._13+v.y*mat._23+v.z*mat._33+mat._43);
}

void SDLRender::SetXPos(float _x)
{
    xp=_x;
    UpdateViewMatrix();
}

void SDLRender::SetYPos(float _y)
{
    yp=_y;
    UpdateViewMatrix();
}

void SDLRender::SetZPos(float _z)
{
    zp=_z;
    UpdateViewMatrix();
}

void SDLRender::AddXPos(float _x)
{
    xp+=_x;
    UpdateViewMatrix();
}

void SDLRender::AddYPos(float _y)
{
    yp+=_y;
    UpdateViewMatrix();
}

void SDLRender::AddZPos(float _z)
{
    zp+=_z;
    UpdateViewMatrix();
}

void SDLRender::SetXRotate(float x)
{
    xa=x;
    UpdateViewMatrix();
}

void SDLRender::SetYRotate(float y)
{
    ya=y;
    UpdateViewMatrix();
}

void SDLRender::SetZRotate(float z)
{
    za=z;
    UpdateViewMatrix();
}

void SDLRender::AddXRotate(float x)
{
    xa+=x;
    UpdateViewMatrix();
}

void SDLRender::AddYRotate(float y)
{
    ya+=y;
    UpdateViewMatrix();
}

void SDLRender::AddZRotate(float z)
{
    za+=z;
    UpdateViewMatrix();
}

void SDLRender::GoForward(float d)
{
	Vector forward(0,0,d);
	MATRIX transform;

	memcpy(&transform,&initViewMatrix,sizeof(MATRIX));
	MatrixRotateZ(transform,za);
	MatrixRotateY(transform,ya);
//	MatrixRotateX(transform,xa);
	TransformVector(forward,transform);
	xp+=forward.x;
	yp+=forward.y;
	zp+=forward.z;
    UpdateViewMatrix();
}

void SDLRender::GoSide(float d)
{
	Vector forward(d,0,0);
	MATRIX transform;

	memcpy(&transform,&initViewMatrix,sizeof(MATRIX));
	MatrixRotateZ(transform,za);
	MatrixRotateY(transform,ya);
//	MatrixRotateX(transform,xa);
	TransformVector(forward,transform);
	xp+=forward.x;
	yp+=forward.y;
	zp+=forward.z;
    UpdateViewMatrix();
}

float SDLRender::GetXPos()
{
    return xp;
}

float SDLRender::GetYPos()
{
    return yp;
}

float SDLRender::GetZPos()
{
    return zp;
}

float SDLRender::GetXRotate()
{
    return xa;
}

float SDLRender::GetYRotate()
{
    return ya;
}

float SDLRender::GetZRotate()
{
    return za;
}

Texture* SDLRender::LoadTextureBMP(char *filename)
{
    return NULL;
}

Texture* SDLRender::LoadAlphaTextureBMP(char *filename,char *alphaname)
{
    return NULL;
}

void SDLRender::CreateMemoryTexture(Texture *tex)
{
}

void SDLRender::UpdateMemoryTexture(Texture *tex)
{
}

void SDLRender::ReloadTexture(Texture *tex)
{
}

void SDLRender::FreeTexture(Texture *tex)
{
}

void SDLRender::UseOriginView()
{
}

void SDLRender::UseNormalView()
{
}

void SDLRender::StartText()
{
	// A NOP for SDL - DDOI
#ifndef PLATFORM_UNIX
    if (dd)
    {
        dd->GetBackBuffer3()->Unlock(NULL);
        dd->GetBackBuffer3()->GetDC(&dc);
    }
    else
        dc=bmp->Canvas->Handle;
    SetBkMode(dc,TRANSPARENT);
    if (curFont) SelectObject(dc,curFont->font);
#endif
}

void SDLRender::EndText()
{
#ifndef PLATFORM_UNIX
    if (dd)
    {
        dd->GetBackBuffer3()->ReleaseDC(dc);
        dc=NULL;
        DDSURFACEDESC sd;
        memset(&sd,0,sizeof(sd));
        sd.dwSize=sizeof(sd);
        dd->GetBackBuffer3()->Lock(NULL,&sd,DDLOCK_WAIT,NULL);
        for (int i=0;i<scrnY;i++)
            screen[i]=&(((char*)sd.lpSurface)[sd.lPitch*i]);
    }
#endif
}

void SDLRender::RenderText(int x,int y,char *text,unsigned long color)
{
#ifndef PLATFORM_UNIX
    SetTextColor(dc,color&0xffffff);
    TextOut(dc,x,y,text,strlen(text));
#endif
}

void* SDLRender::AddFont(HFONT font)
{
    if (firstFont==NULL)
    {
        firstFont=lastFont=new FontListElem;
        lastFont->next=NULL;
        lastFont->font=font;
        return lastFont;
    }
    lastFont->next=new FontListElem;
    lastFont=lastFont->next;
    lastFont->next=NULL;
    lastFont->font=font;
    return lastFont;
}

void SDLRender::UseFont(void* font)
{
    curFont=(FontListElem*)font;
#ifndef PLATFORM_UNIX
    if (dc) SelectObject(dc,((FontListElem*)font)->font);
#endif
}

void SDLRender::GetStats(RenderStats *stat)
{
    stat->polyDrawn=polyDrawn;
    stat->polyTotal=polyTotal;
    stat->polyReflect=polyReflect;
}

void SDLRender::BeginBitmaps()
{
    BeginScene();
}

void SDLRender::RenderBitmap(int x,int y,Art *art,int texIndex,unsigned long color)
{
    RenderBitmap(x,y,art->tex[1][texIndex],art->origSizeX[texIndex],art->origSizeY[texIndex],color);
}

void SDLRender::RenderBitmap(int x,int y,Art *art,int texIndex,int palNum,unsigned long color)
{
    RenderBitmap(x,y,art->tex[palNum][texIndex],art->origSizeX[texIndex],art->origSizeY[texIndex],color);
}

void SDLRender::RenderBitmap(int x,int y,Texture *tex,int w,int h,unsigned long color)
{
    if ((!renderEnabled)||(!sceneEnabled))
        return;
    int origin=0;
    if (tex->type!=TEXTYPE_REPEAT) origin=1;
    int sizeX=(w*scrnX)/320;
    int sizeY=(h*scrnY)/200;
    if ((sizeX==0)||(sizeY==0))
        return;
    int ofsX=x*scrnX/320;
    int ofsY=y*scrnY/200;
    int addX=(w<<16)/sizeX;
    int addY=(h<<16)/sizeY;
    int texY=(addY>>1)+(origin<<16);
    for (int j=0;j<sizeY;j++,texY+=addY)
    {
        unsigned short *scr=&((unsigned short*)screen[j+ofsY])[ofsX];
        unsigned char *texData=&((unsigned char*)tex->data)[(texY>>16)*tex->memTex.width+origin];
        unsigned short *table=(unsigned short*)colorTable[tex->pal][(color>>3)&31];
        if (scrnX==320)
        {
#if defined(PLATFORM_UNIX) && defined(__i386__)
	    __asm__ __volatile (
		"RenderBitmap_drawLoop1:	\n\t"
			"cld	\n\t"
			"cmpb $255, (%%esi)	\n\t"
			"je RenderBitmap_skip1	\n\t"
			"movzbl (%%edi), %%ebx	\n\t"
			"movw (%%edx, %%ebx, 2), %%ax	\n\t"
			"stosw	\n\t"
			"incl %%esi	\n\t"
			"loop RenderBitmap_drawLoop1	\n\t"
			"jmp RenderBitmap_endLoop	\n\t"
		"RenderBitmap_skip1:	\n\t"
			"incl %%esi	\n\t"
			"addw $2, %%di	\n\t"
			"loop RenderBitmap_drawLoop1	\n\t"
		"RenderBitmap_endLoop:		\n\t"
	    : : "S" (texData), "D" (scr), "d" (table), "c" (sizeX) : "cc");
#elif defined(PLATFORM_WIN32) && defined(__i386__)
            asm mov esi,texData
            asm mov edi,scr
            asm mov edx,table
            asm mov ecx,sizeX
            asm cld
           RenderBitmap_drawLoop1:
            asm cmp byte ptr [esi],255
            asm je RenderBitmap_skip1
            asm movzx ebx,byte ptr [esi]
            asm mov ax,[edx+ebx*2]
            asm stosw
            asm inc esi
            asm loop RenderBitmap_drawLoop1
            asm jmp RenderBitmap_endLoop
           RenderBitmap_skip1:
            asm inc esi
            asm add di,2
            asm loop RenderBitmap_drawLoop1
           RenderBitmap_endLoop:
#endif
        }
        else
        {
#if defined(PLATFORM_UNIX) && defined(__i386__)
	    __asm__ __volatile (
			"cld		\n\t"
			"movw %%bx, %%dx		\n\t"
			"shrw $1, %%dx		\n\t"
			"pushl %%ebp		\n\t"
			"movl %%eax, %%ebp		\n\t"
		"RenderBitmap_drawLoop2:		\n\t"
			"cmpb $255, (%%esi)			\n\t"
			"je RenderBitmap_skip2		\n\t"
			"movzbl (%%esi), %%eax		\n\t"
			"movw (%%ebp, %%eax, 2), %%ax		\n\t"
			"stosw		\n\t"
			"addw %%bx, %%dx		\n\t"
			"jnc RenderBitmap_next		\n\t"
			"inc %%esi		\n\t"
			"loop RenderBitmap_drawLoop2		\n\t"
			"jmp RenderBitmap_endLoop2		\n\t"
		"RenderBitmap_skip2:		\n\t"
			"addl $2, %%edi		\n\t"
			"addw %%bx, %%dx		\n\t"
			"jnc RenderBitmap_next		\n\t"
			"inc %%esi			\n\t"
		"RenderBitmap_next:		\n\t"
	    		"loop RenderBitmap_drawLoop2		\n\t"
	    	"RenderBitmap_endLoop2:		\n\t"
			"popl %%ebp				\n\t"
	    : : "S" (texData), "D" (scr), "a" (table), "b" (addX), "c"(sizeX) : "cc");
#elif defined(PLATFORM_WIN32) && defined(__i386__)
            asm mov esi,texData
            asm mov edi,scr
            asm mov eax,table
            asm mov ebx,addX
            asm mov dx,bx
            asm shr dx,1
            asm mov ecx,sizeX
            asm cld
            asm push ebp
            asm mov ebp,eax
           RenderBitmap_drawLoop2:
            asm cmp byte ptr [esi],255
            asm je RenderBitmap_skip2
            asm movzx eax,byte ptr [esi]
            asm mov ax,[ebp+eax*2]
            asm stosw
            asm add dx,bx
            asm jnc RenderBitmap_next
            asm inc esi
            asm loop RenderBitmap_drawLoop2
            asm jmp RenderBitmap_endLoop2
           RenderBitmap_skip2:
            asm add edi,2
            asm add dx,bx
            asm jnc RenderBitmap_next
            asm inc esi
           RenderBitmap_next:
            asm loop RenderBitmap_drawLoop2
           RenderBitmap_endLoop2:
            asm pop ebp
#endif
        }
    }
}

void SDLRender::RenderBitmapYFlipped(int x,int y,Art *art,int texIndex,unsigned long color)
{
    if ((!renderEnabled)||(!sceneEnabled))
        return;
    Texture *tex=art->tex[1][texIndex];
    int origin=0;
    if (tex->type!=TEXTYPE_REPEAT) origin=1;
    int sizeX=(art->origSizeX[texIndex]*scrnX)/320;
    int sizeY=(art->origSizeY[texIndex]*scrnY)/200;
    if ((sizeX==0)||(sizeY==0))
        return;
    int ofsX=x*scrnX/320;
    int ofsY=y*scrnY/200;
    int addX=(art->origSizeX[texIndex]<<16)/sizeX;
    int addY=(art->origSizeY[texIndex]<<16)/sizeY;
    int texY=(addY>>1)+(origin<<16);
    for (int j=0;j<sizeY;j++,texY+=addY)
    {
        unsigned short *scr=&((unsigned short*)screen[(sizeY-1-j)+ofsY])[ofsX];
        unsigned char *texData=&((unsigned char*)tex->data)[(texY>>16)*tex->memTex.width+origin];
        unsigned short *table=(unsigned short*)colorTable[tex->pal][(color>>3)&31];
        if (scrnX==320)
        {
#if defined(PLATFORM_UNIX) && defined(__i386__)
            __asm__ __volatile (
                "RenderBitmap2_drawLoop1:		\n\t"
                        "cld		\n\t"
                        "cmpb $255, (%%esi)		\n\t"
                        "je RenderBitmap2_skip1		\n\t"
                        "movzbl (%%edi), %%ebx		\n\t"
                        "movw (%%edx, %%ebx, 2), %%ax		\n\t"
                        "stosw		\n\t"
                        "incl %%esi		\n\t"
                        "loop RenderBitmap2_drawLoop1		\n\t"
                        "jmp RenderBitmap2_endLoop		\n\t"
                "RenderBitmap2_skip1:		\n\t"
                        "incl %%esi		\n\t"
                        "addw $2, %%di		\n\t"
                        "loop RenderBitmap2_drawLoop1		\n\t"
                "RenderBitmap2_endLoop:   		\n\t"
            : : "S" (texData), "D" (scr), "d" (table), "c" (sizeX) : "cc");
#elif defined(PLATFORM_WIN32) && defined(__i386__)
            asm mov esi,texData
            asm mov edi,scr
            asm mov edx,table
            asm mov ecx,sizeX
            asm cld
           RenderBitmap_drawLoop1:
            asm cmp byte ptr [esi],255
            asm je RenderBitmap_skip1
            asm movzx ebx,byte ptr [esi]
            asm mov ax,[edx+ebx*2]
            asm stosw
            asm inc esi
            asm loop RenderBitmap_drawLoop1
            asm jmp RenderBitmap_endLoop
           RenderBitmap_skip1:
            asm inc esi
            asm add di,2
            asm loop RenderBitmap_drawLoop1
           RenderBitmap_endLoop:
#endif
        }
        else
        {
#if defined(PLATFORM_UNIX) && defined(__i386__)
            __asm__ __volatile (
                        "cld		\n\t"
                        "movw %%bx, %%dx		\n\t"
                        "shrw $1, %%dx		\n\t"
                        "pushl %%ebp		\n\t"
                        "movl %%eax, %%ebp		\n\t"
                "RenderBitmap2_drawLoop2:		\n\t"
                        "cmpb $255, (%%esi)      		\n\t"
                        "je RenderBitmap2_skip2		\n\t"
                        "movzbl (%%esi), %%eax		\n\t"
                        "movw (%%ebp, %%eax, 2), %%ax		\n\t"
                        "stosw		\n\t"
                        "addw %%bx, %%dx		\n\t"
                        "jnc RenderBitmap2_next		\n\t"
                        "inc %%esi		\n\t"
                        "loop RenderBitmap2_drawLoop2		\n\t"
                        "jmp RenderBitmap2_endLoop2		\n\t"
                "RenderBitmap2_skip2:		\n\t"
                        "addl $2, %%edi		\n\t"
                        "addw %%bx, %%dx		\n\t"
                        "jnc RenderBitmap2_next		\n\t"
                        "inc %%esi       		\n\t"
                "RenderBitmap2_next:		\n\t"
                        "loop RenderBitmap2_drawLoop2		\n\t"
                "RenderBitmap2_endLoop2:		\n\t"
                        "popl %%ebp              		\n\t"
            : : "S" (texData), "D" (scr), "a" (table), "b" (addX), "c"(sizeX) :
 "cc");
#elif defined(PLATFORM_WIN32) && defined(__i386__)
            asm mov esi,texData
            asm mov edi,scr
            asm mov eax,table
            asm mov ebx,addX
            asm mov dx,bx
            asm shr dx,1
            asm mov ecx,sizeX
            asm cld
            asm push ebp
            asm mov ebp,eax
           RenderBitmap_drawLoop2:
            asm cmp byte ptr [esi],255
            asm je RenderBitmap_skip2
            asm movzx eax,byte ptr [esi]
            asm mov ax,[ebp+eax*2]
            asm stosw
            asm add dx,bx
            asm jnc RenderBitmap_next
            asm inc esi
            asm loop RenderBitmap_drawLoop2
            asm jmp RenderBitmap_endLoop2
           RenderBitmap_skip2:
            asm add edi,2
            asm add dx,bx
            asm jnc RenderBitmap_next
            asm inc esi
           RenderBitmap_next:
            asm loop RenderBitmap_drawLoop2
           RenderBitmap_endLoop2:
            asm pop ebp
#endif
        }
    }
}

void SDLRender::RenderScaledBitmap(int x,int y,int w,int h,Art *art,int texIndex,unsigned long color)
{
    RenderScaledBitmap(x,y,w,h,art,texIndex,1,color);
}

void SDLRender::RenderScaledBitmap(int x,int y,int w,int h,Art *art,int texIndex,int palNum,unsigned long color)
{
    if ((!renderEnabled)||(!sceneEnabled))
        return;
    int origin=0;
    if (art->tex[palNum][texIndex]->type!=TEXTYPE_REPEAT) origin=1;
    int sizeX=(w*scrnX)/320;
    int sizeY=(h*scrnY)/200;
    if ((sizeX==0)||(sizeY==0))
        return;
    int ofsX=(x*scrnX)/320;
    int ofsY=(y*scrnY)/200;
    int addX=(art->origSizeX[texIndex]<<16)/sizeX;
    int addY=(art->origSizeY[texIndex]<<16)/sizeY;
    int texY=(addY>>1)+(origin<<16);
    for (int j=0;j<sizeY;j++,texY+=addY)
    {
        unsigned short *scr=&((unsigned short*)screen[j+ofsY])[ofsX];
        unsigned char *tex=&((unsigned char *)art->tex[palNum][texIndex]->data)[(texY>>16)*art->sizeX[texIndex]+origin];
        unsigned short *table=(unsigned short*)colorTable[palNum][(color>>3)&31];
        int texX=0;
        for (int i=0;i<sizeX;i++,texX+=addX)
        {
            if (tex[texX>>16]!=255)
                scr[i]=table[tex[texX>>16]];
        }
    }
}

void SDLRender::EndBitmaps()
{
    EndScene();
}

void SDLRender::LoadPalette(int i,unsigned short *pal)
{
    if (!colorTable[i])
    {
        colorTable[i]=(void**)new unsigned short*[32];
        for (int j=0;j<32;j++)
            colorTable[i][j]=(void*)new unsigned short[256];
    }
    for (int c=0;c<256;c++)
    {
        int r=(pal[c]>>10)&31;
        int g=(pal[c]>>5)&31;
        int b=pal[c]&31;
        for (int shade=0;shade<32;shade++)
        {
            int nr=r*shade/16; if (nr>31) nr=31;
            int ng=g*shade/16; if (ng>31) ng=31;
            int nb=b*shade/16; if (nb>31) nb=31;
            ((unsigned short *)colorTable[i][shade])[c]=(nr<<11)|(ng<<6)|nb;
        }
    }
}
void SDLRender::ScreenShot()
{
}
