/*
**	scsi_info.c 1998-02-18 Mikael Forselius <mikaelf@comenius.se>
**
**	This file is subject to the terms and conditions of the GNU General Public
**	License.  See the file COPYING in the main directory of this archive
**	for more details.
**
**	Determine SCSI devices present. Does not handle machines
**	with more than one bus.
*/

#include <string.h>
#include <SCSI.h>
#include <LowMem.h>
#include "penguin_prototypes.h"
#include "scsi_info.h"

/*
 * SCSI stat bits
 */
#define kScsiStatBSY				(1 << 6)
#define kScsiStatSEL				(1 << 1)

/*
 * These commands are supported for all devices.
 */
#define kScsiCmdInquiry				0x12

/*
 * These commands are supported by direct-access devices only.
 */
#define kScsiCmdRead6				0x08

/*
 * Misc
 */
#define MAC_DEVICE_ID		7
#define MAX_DEVICE_ID		8

/* THIS IS VERY IMPORTANT - ALIGN ON 68K BOUNDARIES
 */
#pragma options align=mac68k

/*
 * Different types of SCSI devices, as from inquiry command.
 */
enum {
	kScsiDevTypeDirect					= 0,
	kScsiDevTypeSequential,
	kScsiDevTypePrinter,
	kScsiDevTypeProcessor,
	kScsiDevTypeWorm,						// Write-once, read multiple
	kScsiDevTypeCDROM,
	kScsiDevTypeScanner,
	kScsiDevTypeOptical,
	kScsiDevTypeChanger,
	kScsiDevTypeComm,
	kScsiDevTypeGraphicArts0A,
	kScsiDevTypeGraphicArts0B,
	kScsiDevTypeFirstReserved,				// Start of reserved sequence
	kScsiDevTypeUnknownOrMissing		= 0x1F,
	kScsiDevTypeMask					= 0x1F
};

/*
 * The 6-byte commands are used for most simple I/O requests.
 */

typedef struct SCSI_6_ByteCmd {				/* Six-byte command			*/
	unsigned char		opcode;				/*  0						*/
	unsigned char		lbn3;				/*  1 lbn in low 5			*/
	unsigned char		lbn2;				/*  2						*/
	unsigned char		lbn1;				/*  3						*/
	unsigned char		len;				/*  4						*/
	unsigned char		ctrl;				/*  5						*/
} SCSI_6_ByteCmd;

typedef union SCSICDB {
	SCSI_6_ByteCmd		scsi6;
} SCSICDB;

typedef struct SCSI_Inquiry_Data {			/* Inquiry returns this		*/
	unsigned char		devType;			/*  0 Device type,			*/
	unsigned char		devTypeMod;			/*  1 Device type modifier	*/
	unsigned char		version;			/*  2 ISO/ECMA/ANSI version	*/
	unsigned char		format;				/*  3 Response data format	*/
	unsigned char		length;				/*  4 Additional Length		*/ // <-- SCSI-1
	unsigned char		reserved5;			/*  5 Reserved				*/
	unsigned char		reserved6;			/*  6 Reserved				*/
	unsigned char		flags;				/*  7 Capability flags		*/
	unsigned char		vendor[8];			/*  8-15 Vendor-specific	*/
	unsigned char		product[16];		/* 16-31 Product id			*/
	unsigned char		revision[4];		/* 32-35 Product revision	*/
	unsigned char		vendorSpecific[20]; /* 36-55 Vendor stuff		*/
	unsigned char		moreReserved[40];	/* 56-95 Reserved			*/
} SCSI_Inquiry_Data;

typedef struct SCSI_Devices {
	unsigned short	devPresent;
	unsigned short	devType;
} SCSI_Devices;

/* THIS IS VERY IMPORTANT - RESET ALIGN ON 68K BOUNDARIES
 */
#pragma options align=reset

static SCSI_Devices		sSCSIDevices[MAX_DEVICE_ID];

static void				DoFindSCSIDevices(void);
static OSErr			DoPrintSCSIPartitions(int scsiID, int baseID);
static OSErr			DoTheSCSI(void);
static OSErr			DoSCSIInquiry(short targetID, SCSI_Inquiry_Data *pSID);
static OSErr			DoSCSIRead(short targetID, SCSICDB *pCDB, short cdbCount, SCSIInstr *pTIB);
static void				DoWaitSCSIBusy(void);


/*
 * show_scsi_info
 *
 */
void
ShowSCSIInfo(void)
{
	int		scsiID;
	int		devGenericCount, devDirectCount, devSeqCount;
	int		devCDROMCount, devChangerCount;

//	Gestalt('scsi', &result);
//	if ((result & hasSCSI) == 0)
//		return;

	devGenericCount = 0;
	devDirectCount = 0;
	devSeqCount = 0;
	devCDROMCount = 0;
	devChangerCount = 0;

	cprintf("########## SCSI Device Info ##########\n");

	DoFindSCSIDevices();

	// +=============-===============================================================+
	// |    Code     |  Description                                                  |
	// |-------------+---------------------------------------------------------------|
	// |     00h     |  Direct-access device (e.g. magnetic disk)                    |
	// |     01h     |  Sequential-access device (e.g. magnetic tape)                |
	// |     02h     |  Printer device                                               |
	// |     03h     |  Processor device                                             |
	// |     04h     |  Write-once device (e.g. some optical disks)                  |
	// |     05h     |  CD-ROM device                                                |
	// |     06h     |  Scanner device                                               |
	// |     07h     |  Optical memory device (e.g. some optical disks)              |
	// |     08h     |  Medium changer device (e.g. jukeboxes)                       |
	// |     09h     |  Communications device                                        |
	// |  0Ah - 0Bh  |  Defined by ASC IT8 (Graphic arts pre-press devices)          |
	// |  0Ch - 1Eh  |  Reserved                                                     |
	// |     1Fh     |  Unknown or no device type                                    |
	// +=============================================================================+

	for(scsiID = 0; scsiID < MAX_DEVICE_ID; scsiID++) {

		cprintf("SCSI device id: %02d\n", scsiID);

		if (sSCSIDevices[scsiID].devPresent) {

			if (scsiID == MAC_DEVICE_ID) {

				cprintf("  Macintosh\n");

			} else {

				switch(sSCSIDevices[scsiID].devType) {

					case kScsiDevTypeDirect:		// Direct access, harddisk  - /dev/sd[a-p][0-15]
						cprintf("  /dev/sd%c  (Hard disk)\n", (char)(devDirectCount + 'a'));
						DoPrintSCSIPartitions(scsiID, devDirectCount);
						++devDirectCount;
						break;

					case kScsiDevTypeSequential:	// Seq. access-device, tape - /dev/st[0-15]
						cprintf("  /dev/st%d  (Tape)\n", devSeqCount);
						++devSeqCount;
						break;

					case kScsiDevTypePrinter:		// Printer device           - /dev/sg[0-...]
						cprintf("  /dev/sg%d  (Printer)\n", devGenericCount);
						++devGenericCount;
						break;

					case kScsiDevTypeProcessor:		// Processor device         - /dev/sg[0-...]
						cprintf("  /dev/sg%d  (Processor)\n", devGenericCount);
						++devGenericCount;
						break;

					case kScsiDevTypeWorm:			// Write once, optical disk - /dev/sg[0-...]
						cprintf("  /dev/sg%d  (WORM)\n", devGenericCount);
						++devGenericCount;
						break;

					case kScsiDevTypeCDROM:			// CD-ROM device            - /dev/sr[0-15]
						cprintf("  /dev/sr%d  (CD-ROM)\n", devCDROMCount);
						++devCDROMCount;
						break;

					case kScsiDevTypeScanner:		// Scanner device           - /dev/sg[0-...]
						cprintf("  /dev/sg%d  (Scanner)\n", devGenericCount);
						++devGenericCount;
						break;

					case kScsiDevTypeOptical:		// Optical memory device    - /dev/sg[0-...]
						cprintf("  /dev/sg%d  (Optical)\n", devGenericCount);
						++devGenericCount;
						break;

					case kScsiDevTypeChanger:		// Changer device           - /dev/sch[0-15]
						cprintf("  /dev/sch%d (Media changer)\n", devChangerCount);
						++devChangerCount;
						break;

					default:
						cprintf("  /dev/sg%d  (Unknown device)\n", devGenericCount);
						++devGenericCount;
						break;

				}

			}

		} else
			cprintf("  (not present)\n");
	}

	cprintf("######################################\n");
}

/*
 * DoFindSCSIDevices
 *
 */
static void
DoFindSCSIDevices(void)
{
	SCSI_Inquiry_Data	sid;
	int					targetID;

	/*
	 * Assume no devices present
	 */
	for(targetID = 0; targetID < MAX_DEVICE_ID; targetID++)
		sSCSIDevices[targetID].devPresent = false;

	/*
	 * Find devices with INQUIRY command
	 */
	for(targetID = 0; targetID < MAX_DEVICE_ID; targetID++) {
		if (targetID == MAC_DEVICE_ID) {
			sSCSIDevices[targetID].devPresent = true;
			sSCSIDevices[targetID].devType = -1;
		} else {
			if (DoSCSIInquiry(targetID, &sid) == noErr) {
				sSCSIDevices[targetID].devPresent = true;
				sSCSIDevices[targetID].devType = sid.devType & kScsiDevTypeMask;
			}
		}
	}
}

/*
 * DoPrintSCSIPartitions
 *
 */
static OSErr
DoPrintSCSIPartitions(int scsiID, int baseID)
{
	SCSICDB		cdb;
	SCSIInstr	tib[2];
	Partition	part;
	long		i, partCnt, partSize;
	OSErr		theErr;
	Block0		blk0;

	// Setup SCSI command block
	memset(&cdb, 0, sizeof(cdb));
	cdb.scsi6.opcode = kScsiCmdRead6;
	cdb.scsi6.lbn1 = 0;
	cdb.scsi6.len = 1;

	// Setup SCSI manager transfer instructions
	memset(tib, 0, sizeof(tib));
	tib[0].scOpcode = scNoInc;
	tib[0].scParam1 = (long)&blk0;
	tib[0].scParam2 = 512;
	tib[1].scOpcode = scStop;

	// Read block 0 of device
	theErr = DoSCSIRead(scsiID, &cdb, sizeof(cdb.scsi6), tib);
	if (theErr == noErr) {

		// Verify validity of block 0, correct signature and driver(s) present
		if ( (blk0.sbSig == sbSIGWord) && (blk0.sbDrvrCount > 0) ) {

			// Read first partition map to determine # of partitions
			cdb.scsi6.lbn1 = 1;
			tib[0].scParam1 = (long)&part;
			theErr = DoSCSIRead(scsiID, &cdb, sizeof(cdb.scsi6), tib);
			if (theErr == noErr) {

				// Read, loop through all partitions present on device
				partCnt = part.pmMapBlkCnt;
				for(i = 1; i <= partCnt; i++) {

					cdb.scsi6.lbn1 = i;
					if (DoSCSIRead(scsiID, &cdb, sizeof(cdb.scsi6), tib) == noErr) {

						cprintf("    /dev/sd%c%c : %-32s", (char)('a'+baseID), 
							(char)(48+i), part.pmParType);
						partSize = (blk0.sbBlkSize * part.pmPartBlkCnt) / 1024L;
						if (partSize <= 1024)
							cprintf("(%ld KB)\n", partSize);
						else
							cprintf("(%.1f MB)\n", (double)partSize / 1024.0);

					} else
						cprintf("    *** Error while reading partition block %ld\n", i);

				}

			} else
				cprintf("    *** Error while reading partition block 1\n");

		}

	} else if (theErr == -1)
		cprintf("    *** Read error at SCSI device %hd\n", scsiID);

	return noErr;
}

/*
 * DoSCSIInquiry
 *
 */
static OSErr
DoSCSIInquiry(short targetID, SCSI_Inquiry_Data *pSID)
{
	unsigned char	cdb[6];
	SCSIInstr		tib[2];
	OSErr			theErr, compErr;
	short			compStat, compMsg;

	// Setup SCSI command
	memset(cdb, 0, sizeof(cdb));
	cdb[0] = kScsiCmdInquiry;	// SCSI command code for the INQUIRY command
	cdb[4] = 5;					// maximum number of bytes target should return

	memset(tib, 0, sizeof(tib));
	tib[0].scOpcode = scNoInc;
	tib[0].scParam1 = (long)pSID;
	tib[0].scParam2 = 5;
	tib[1].scOpcode = scStop;

	DoWaitSCSIBusy();

	theErr = SCSIGet();
	if (theErr == noErr) {

		theErr = SCSISelect(targetID);
		if (theErr == noErr) {

			theErr = SCSICmd((Ptr)cdb, sizeof(cdb));
			if (theErr == noErr) {

				theErr = SCSIRead((Ptr)tib);

			}

			compErr = SCSIComplete(&compStat, &compMsg, 300);

		}

	}

	if ((theErr == noErr) && (compErr != noErr))
		theErr = compErr;

	return theErr;
}

/*
 * DoSCSIRead
 *
 */
static OSErr
DoSCSIRead(short targetID, SCSICDB *pCDB, short cdbCount, SCSIInstr *pTIB)
{
	OSErr		theErr, compErr;
	short		compStat, compMsg;

	compErr = noErr;

	DoWaitSCSIBusy();

	theErr = SCSIGet();
	if (theErr == noErr) {

		theErr = SCSISelect(targetID);
		if (theErr == noErr) {

			theErr = SCSICmd((Ptr)pCDB, cdbCount);
			if (theErr == noErr) {

				theErr = SCSIRead((Ptr)pTIB);

			}

			compErr = SCSIComplete(&compStat, &compMsg, 300);		

		} else if (theErr == scCommErr) {

			// Communication error, assume device not present
			theErr = -1;

		}

	}

	if ((theErr == noErr) && (compErr != noErr))
		theErr = compErr;

	return theErr;
}

/*
 *	DoWaitSCSIBusy
 *
 *	Waits for the bus to become free
 */
static void
DoWaitSCSIBusy(void)
{
	unsigned long	timeOut;

	/*
	 * Wait for the bus to go free, 5 second timeout
	 */
	timeOut = LMGetTicks() + 300;
	while ( (SCSIStat() & (kScsiStatBSY | kScsiStatSEL)) != 0) {
		if (LMGetTicks() > timeOut)
			break;
	}
}
