/****************************************************************************
** ImagePreview class
**
** Created: Tue Feb 02 22:06:51 2004
**      by: Varol Okan using Kate
**
** This class is the encapsulation of the ImagePreview from the FileSelection
** Dialog.
** It is also used in the ImageDialog.
**
****************************************************************************/

#include <qpen.h>
#include <qfont.h>
#include <qimage.h>
#include <qpainter.h>
#include <qpopupmenu.h>
#include <qlistview.h>
#include <qwmatrix.h>
#include <qmessagebox.h>

#include "global.h"
#include "xml_dvd.h"
#include "dialogtextfont.h"
#include "dialogmatrix.h"
#include "menuobject.h"
#include "textobject.h"	// derived from menuobject

TextObject::TextObject (QWidget *pParent)
	: MenuObject (pParent)
{
	m_qsObjectType    = QString (TEXT_OBJECT);
	m_colorBackground = QColor(TRANSPARENT_COLOR);	// indicate the transparent color.
	m_iStyleStrategy  = QFont::PreferAntialias;
	m_pContextMenu    = NULL;
}

TextObject::~TextObject ()
{
	if (m_pContextMenu)
		delete m_pContextMenu;
	m_pContextMenu = NULL;
}

void TextObject::setFont (QFont &theFont)
{
	m_Font = theFont;
}

QFont &TextObject::getFont ()
{
	return m_Font;
}

void TextObject::setText (QString &theText)
{
	m_qsText = theText;
	setName (theText);
}

QString &TextObject::getText ()
{
	return m_qsText;
}

void TextObject::setTextAlign (QPoint &newAlign)
{
	m_ptTextAlign = newAlign;
}

QPoint &TextObject::getTextAlign ()
{
	return m_ptTextAlign;
}

void TextObject::setStyleStrategy (int iStyleStrategy)
{
	m_iStyleStrategy = iStyleStrategy;
	m_Font.setStyleStrategy ((QFont::StyleStrategy)iStyleStrategy);
}

void TextObject::replaceColor (QColor theColor, QColor oldColor)
{
	if (getForegroundColor () == oldColor)
		setForegroundColor (theColor);
	if (getBackgroundColor () == oldColor)
		setBackgroundColor (theColor);
}

void TextObject::setForegroundColor (QColor &theColor)
{
	m_colorForeground = theColor;
}

void TextObject::setBackgroundColor (QColor &theColor)
{
	m_colorBackground = theColor;
}

QColor &TextObject::getForegroundColor ()
{
	return m_colorForeground;
}

QColor &TextObject::getBackgroundColor ()
{
	return m_colorBackground;
}

void TextObject::drawSemiTransparent (QPainter *pPainter)
{
	QWMatrix theMatrix;
	QRect dummyRect, theRect;
	int iFlags, x, y;
	int iSourceHeight, iSourceWidth, iSourceStartX, iSourceStartY, iTargetStartX, iTargetStartY;
	QColor sourceColor, targetColor, theColor;
	QRgb sourcePixel, targetPixel;
	QImage sourceImg, targetImg;
	float fTargetTrans, fSourceTrans;

	// init variables ...
	fTargetTrans = 1.0 - modifiers()->fTransparency;
	fSourceTrans = modifiers()->fTransparency;

	QPen thePen (m_colorForeground, 2, Qt::DashDotLine);
	// Here we define the cenetered rect.
	theRect = QRect((int)-(rect().width()/ 2.0), (int)-(rect().height()/2.0), rect().width(), rect().height());

	// First we translate to the center of the rect.
	theMatrix.translate (
		(double)rect().x() + (int)(rect().width () / 2.0), 
		(double)rect().y() + (int)(rect().height() / 2.0));
	//. then we apply the other modifications ...
	theMatrix.scale  (modifiers()->fScaleX, modifiers()->fScaleY);
	theMatrix.rotate (modifiers()->fRotate);
	theMatrix.shear  (modifiers()->fShearX, modifiers()->fShearY);
	// Here we get the rect that encompasses the button for this object.
	m_boundingRect = theMatrix.mapRect(theRect);
	// And for the source we need to set the right matrix.
	theMatrix.reset  ();
	theMatrix.translate ((double)m_boundingRect.width()/2.0, (double)m_boundingRect.height()/2.0);
	theMatrix.scale  (modifiers()->fScaleX, modifiers()->fScaleY);
	theMatrix.rotate (modifiers()->fRotate);
	theMatrix.shear  (modifiers()->fShearX, modifiers()->fShearY);

	QPixmap thePixmap = QPixmap(m_boundingRect.size());
	if (thePixmap.isNull())
		return;

	thePixmap.fill (QColor (TRANSPARENT_COLOR));
	QPainter pmPainter (&thePixmap);
	//  then we actually set the world matrix 
	pmPainter.setWorldMatrix(theMatrix);

	pmPainter.setPen(Qt::NoPen);
	pmPainter.setBrush (QBrush(m_colorBackground));
	pmPainter.drawRect(theRect);
 
	m_Font.setStyleStrategy ((QFont::StyleStrategy)m_iStyleStrategy);
	pmPainter.setFont(m_Font);

	// Okay, I abused a QPoint class to hold some other infos ... so what ???
	iFlags = Qt::DontClip|Qt::WordBreak|m_ptTextAlign.x()|m_ptTextAlign.y();

	pmPainter.drawText(theRect, iFlags, m_qsText, -1, &dummyRect);
	
	// At this point we should have the text on a memory Bitmap object, and now 
	// we should copy this over, bit for bit. 
	iSourceHeight = thePixmap.height(); 
	iSourceWidth  = thePixmap.width ();
	iSourceStartX = 0;
	iSourceStartY = 0;
	iTargetStartX = pPainter->viewport().x() + m_boundingRect.x();
	iTargetStartY = pPainter->viewport().y() + m_boundingRect.y();

	// First we need to create a Pixmap which holds the Target pixels
	QPaintDevice *pPDPainter = pPainter->device();
	if (pPDPainter == 0)
		return;

	sourceImg = thePixmap.convertToImage();

	QPixmap *pPixmap = new QPixmap(iSourceWidth, iSourceHeight);
	if ( ! pPixmap )
		return;
	bitBlt (pPixmap, 0, 0, pPainter->device(), 
		iTargetStartX, iTargetStartY, iSourceWidth, iSourceHeight, Qt::CopyROP, true);
		// To get Pixel information we need a QImage however ...
	targetImg = pPixmap->convertToImage();
	delete pPixmap;

	iTargetStartX -= pPainter->viewport().x();
	iTargetStartY -= pPainter->viewport().y();
	for (y=0; y < iSourceHeight; y++)	{
		for (x=0; x < iSourceWidth; x++)	{
			sourcePixel = sourceImg.pixel(x, y );	// calculate the mix of background and foreground.
			sourcePixel &= 0x00ffffff;// filter out the transparency part of the color
			if (sourcePixel != 0x00FFFFFF)	{	// And check if this is a pixel we 
				targetPixel = targetImg.pixel(x, y);
				targetPixel &= 0x00ffffff;	// filter out the transparency part of the color
				targetColor = QColor ( targetPixel );
				sourceColor = QColor ( sourcePixel );
				theColor    = QColor (
					(int)(fSourceTrans * sourceColor.red()   + (int)(fTargetTrans * targetColor.red())),
					(int)(fSourceTrans * sourceColor.green() + (int)(fTargetTrans * targetColor.green())),
					(int)(fSourceTrans * sourceColor.blue()  + (int)(fTargetTrans * targetColor.blue()))
				);
				pPainter->setPen ( theColor );
				pPainter->drawPoint(x+iTargetStartX, y+iTargetStartY);
			}
		}
	}
}

void TextObject::drawContents (QPainter *painter)
{
	QWMatrix theMatrix;// = matrix();
	// If we have a complete transparent text we should not draw at all.
	if (modifiers()->fTransparency == 1.0)
		return;
	else if ( (modifiers()->fTransparency != 0.0) && ( ! m_bDrawRect) )
		return drawSemiTransparent (painter);
	// Here we calculate the center point of gravity (rotation)
	QPoint centerPos;
	centerPos.setX (rect().x() + (int)(rect().width () / 2.0));
	centerPos.setY (rect().y() + (int)(rect().height() / 2.0));
	// Here we define the cenetered rect.
	QRect theRect ((int)-(rect().width()/ 2.0), (int)-(rect().height()/2.0), rect().width(), rect().height());

	// From the Qt documentation ...
	// This transformation engine is a three-step pipeline
	// The first step uses the world transformation matrix.
	// The second step uses the window.
	// The third step uses the viewport.

	// First we translate to the appropriate location,
	theMatrix.translate ((double)centerPos.x(), (double)centerPos.y());

	//. then we apply the other modifications ...
	theMatrix.scale  (modifiers()->fScaleX, modifiers()->fScaleY);
	theMatrix.rotate (modifiers()->fRotate);
	theMatrix.shear  (modifiers()->fShearX, modifiers()->fShearY);
	
	QPen thePen (m_colorForeground, 2, Qt::DashDotLine);
	painter->setPen(thePen);
	
	// Here we get the rect that encompasses the button for this object.
	m_boundingRect = theMatrix.mapRect(theRect);
	if (m_bDrawRect)
		painter->drawRect(m_boundingRect);
	//  then we actually set the world matrix 
	painter->setWorldMatrix(theMatrix);

	// and continue to draw the contents ...
	if (m_colorBackground != QColor(TRANSPARENT_COLOR))	{
		// No frame wanted, we take care of that ourself if requested ...
		painter->setPen(Qt::NoPen);
		painter->setBrush (QBrush(m_colorBackground));
		painter->drawRect(theRect);
		// and re-set the PEN.
		painter->setPen(thePen);
	}
	else
		painter->setBrush(Qt::NoBrush);
 
	m_Font.setStyleStrategy ((QFont::StyleStrategy)m_iStyleStrategy);
	painter->setFont(m_Font);
	QRect boundingRect;

	// Okay, I abused a QPoint class to hold some other infos ... so what ???
	int iFlags = Qt::DontClip|Qt::WordBreak|m_ptTextAlign.x()|m_ptTextAlign.y();

	if (m_bDrawRect)
		painter->drawRect(theRect);

	painter->drawText(theRect, iFlags, m_qsText, -1, &boundingRect);
	// Here we expand the rectangle if needs be, and update again, to re-draw.
	if (m_rect.width() < boundingRect.width())	{
		m_rect.setWidth  (boundingRect.width());
		m_pParent->update ();
	}
	if (m_rect.height() < boundingRect.height())	{
		m_rect.setHeight (boundingRect.height());
		m_pParent->update ();
	}

	// Here we re-set the world matrix and translate the newly drawn ... thing ...
	theMatrix.reset();
	painter->setWorldMatrix(theMatrix);
}


void TextObject::drawContents (QPainter *pPainter, int iRenderFrameNumber, int)
{
	// Next is to set the AnimationAttributes
	for (uint t=0;t<m_listAnimationAttributes.count();t++)
		m_listAnimationAttributes[t]->setValue (iRenderFrameNumber);

	drawContents (pPainter);
}

bool TextObject::readProjectFile  (QDomNode &theNode)
{
	QDomElement theElement = theNode.toElement();
	QDomAttr attribute;

	attribute = theElement.attributeNode ( TEXT_OBJECT_TEXT );
	if (!attribute.isNull())
		m_qsText = attribute.value();

	attribute = theElement.attributeNode ( TEXT_OBJECT_TEXT_ALIGN );
	if (!attribute.isNull())	{
		// QString "x,y" - we abused a QPoint object for the Align info.
		QStringList list = QStringList::split (",", attribute.value());
		QPoint textAlign;
		if (list.count() == 2)
			textAlign = QPoint(list[0].toInt(), list[1].toInt());
		setTextAlign (textAlign);
	}

	attribute = theElement.attributeNode ( TEXT_OBJECT_FONT);
	if (!attribute.isNull())	{
		QFont theFont;
		if (theFont.fromString (attribute.value()))
			setFont (theFont);
	}

	attribute = theElement.attributeNode ( TEXT_OBJECT_BACKGROUND_COLOR );
	if (!attribute.isNull())	{
		QColor backgroundColor (attribute.value());
		setBackgroundColor (backgroundColor);
	}

	attribute = theElement.attributeNode ( TEXT_OBJECT_FOREGROUND_COLOR );
	if (!attribute.isNull())	{
		QColor foregroundColor (attribute.value());
		setForegroundColor (foregroundColor);
	}

	MenuObject::readProjectFile( theNode );
	return true;
}

bool TextObject::writeProjectFile (QDomElement &theElement)
{
	QDomDocument xmlDoc = theElement.ownerDocument();
	QDomElement textNode = xmlDoc.createElement( TEXT_OBJECT );	// <FrameObject>
	
	TextObject initialValues;
	// Here we set the attributes of the <dvdauthor> tag
	if (!getText().isNull())
		textNode.setAttribute( TEXT_OBJECT_TEXT, m_qsText );
	if (getTextAlign() != initialValues.getTextAlign())
		textNode.setAttribute( TEXT_OBJECT_TEXT_ALIGN, QString("%1,%2").arg(getTextAlign().x()).arg(getTextAlign().y()) );
	if (getFont() != initialValues.getFont())
		textNode.setAttribute( TEXT_OBJECT_FONT, getFont().toString() );
	if (getBackgroundColor() != initialValues.getBackgroundColor())
		textNode.setAttribute( TEXT_OBJECT_BACKGROUND_COLOR, getBackgroundColor().name() );
	if (getForegroundColor() != initialValues.getForegroundColor())
		textNode.setAttribute( TEXT_OBJECT_FOREGROUND_COLOR, getForegroundColor().name() );
	
	theElement.appendChild( textNode );
	// And here we write the base class ...
	return MenuObject::writeProjectFile( textNode );
}

void TextObject::createStructure (QListViewItem *pParentItem)
{
	QListViewItem *pButtonItem;
	QListViewItem *pModifierItem;

	QString qsName = tr ("Text Object");
	pButtonItem = new QListViewItem(pParentItem, qsName, m_qsText);
	pButtonItem->setExpandable (TRUE);

	// And finally we create the modifiers info ...
	if ( !((m_modifiers.fRotate == 0.0) && (m_modifiers.fShearX == 0.0) && (m_modifiers.fShearY == 0.0) &&
		   (m_modifiers.fZoom   == 1.0) && (m_modifiers.fScaleX == 1.0) && (m_modifiers.fScaleY == 1.0)) )	{
		pModifierItem = new QListViewItem(pButtonItem, tr ("Modifiers"), NULL);
		pButtonItem->setExpandable (TRUE);

		new QListViewItem(pModifierItem, tr ("Shear (x, y)"),
			QString ("%1, %2").arg(m_modifiers.fShearX).arg(m_modifiers.fShearY));
		new QListViewItem(pModifierItem, tr ("Zoom"),
			QString ("%1").arg(m_modifiers.fZoom));
		new QListViewItem(pModifierItem, tr ("Rotate"),
			QString ("%1 deg").arg(m_modifiers.fRotate));
	}

	if (m_colorBackground == QColor (TRANSPARENT_COLOR))
		new QListViewItem(pButtonItem, tr ("Background"), tr ("Transparent"));
	else
		new QListViewItem(pButtonItem, tr ("Background"),
			tr ("color(%1, %2, %3)").
				arg(m_colorBackground.red()).
				arg(m_colorBackground.green()).
				arg(m_colorBackground.blue()));

	new QListViewItem(pButtonItem, tr ("Foreground"),
		QString ("color(%1, %2, %3)").
			arg(m_colorForeground.red()).
			arg(m_colorForeground.green()).
			arg(m_colorForeground.blue()));

	new QListViewItem(pButtonItem, tr ("Geometry"),
		QString ("%1, %2, %3, %4").
		arg(rect().x()).arg(rect().y()).arg(rect().width()).arg(rect().height()));
}

bool TextObject::mouseMoveEvent (QMouseEvent *pEvent)
{
	// This will move the textObject around and around and around we go ...
	QRect theRect = rect();
	int iX, iY, iWidth, iHeight;
	iWidth  = rect().width ();
	iHeight = rect().height();
	iX = rect().x() - (m_currentMousePos.x() - pEvent->pos().x());
	iY = rect().y() - (m_currentMousePos.y() - pEvent->pos().y());
	theRect.setX(iX);
	theRect.setY(iY);
	theRect.setWidth(iWidth);
	theRect.setHeight(iHeight);

	setRect(theRect);
	m_currentMousePos = pEvent->pos();
	return false;
}

bool TextObject::mousePressEvent (QMouseEvent *pEvent)
{
	m_currentMousePos = pEvent->pos();
	if (pEvent->button() == Qt::RightButton)	{
		QPoint globalPos = pEvent->globalPos();

		QPopupMenu *pStackMenu = new QPopupMenu(m_pParent);
		pStackMenu->insertItem ( tr ("To Front")  , this, SLOT(slotToFront()));
		pStackMenu->insertItem ( tr ("To Back")   , this, SLOT(slotToBack()));
		globalPos.setX (globalPos.x() - pStackMenu->sizeHint().width()); // -100);
		pStackMenu->popup(globalPos, 1);

		if (m_pContextMenu)
			delete m_pContextMenu;
		m_pContextMenu = new QPopupMenu(m_pParent);
		m_pContextMenu->insertItem ( tr ("Edit ...")  , this,  SLOT(slotEdit()));
		m_pContextMenu->insertItem ( tr ("Matrix ..."), this,  SLOT(slotMatrix()));
		m_pContextMenu->insertItem ( tr ("Delete")    , this,  SLOT(slotDelete()));
		m_pContextMenu->insertSeparator();
		m_pContextMenu->insertItem ( tr ("Add Frame"), this,   SLOT(slotAddFrame()));
		m_pContextMenu->insertItem ( tr ("Add Image"), this,   SLOT(slotAddImage()));
		m_pContextMenu->insertItem ( tr ("Add Movie"), this,   SLOT(slotAddMovie()));
		m_pContextMenu->insertSeparator();
		m_pContextMenu->insertItem ( tr ("Define as Button") , this, SLOT(slotDefineAsButton()));
		// Blocking call
		m_pContextMenu->exec(pEvent->globalPos(), 3);
		// Here we mark that the user called a menu item thus we don't want the
		// base classes to continue with the mouse event
		delete pStackMenu;
		// Note the menu might have been already deleted in slotToFront/slotToBack
		if (m_pContextMenu)
			delete m_pContextMenu;
		m_pContextMenu = NULL;
		return true;
	}
	else
		m_bDrawRect = true;
	return false;
}

bool TextObject::mouseReleaseEvent (QMouseEvent *)
{
	m_bDrawRect = false;
	emit (signalUpdatePixmap());
	emit (signalUpdateStructure());
	return false;
}

bool TextObject::mouseDoubleClickEvent (QMouseEvent *)
{
	QFont theFont;
	DialogTextFont fontDialog (m_pParent);
	fontDialog.setFont (getFont());
	fontDialog.setText (getText());
	fontDialog.setTextAlign (getTextAlign());
	fontDialog.setBackgroundColor (getBackgroundColor());
	fontDialog.setForegroundColor (getForegroundColor());
	fontDialog.setTransparency (modifiers()->fTransparency);
	fontDialog.setAnimation ( animation () );
	fontDialog.setRect (rect());	// setRect (boundingRect());
	if (fontDialog.exec() == QDialog::Rejected)
		return false;

	setFont(fontDialog.getFont());
	QString theString = fontDialog.getText();
	setText(theString);
	setAnimation (fontDialog.getAnimation ());
	if (fontDialog.getBackgroundColor() != getBackgroundColor())	{
		// This emit signal is only caught when in ButtonDialog and the Selected, or Highlighted mask 
		// was changed (I.e. This Object is in one of those two frames).
		// Otherwise the signal is ignored.
		emit (signalMaskColorChanged(fontDialog.getBackgroundColor(), getBackgroundColor()));
		// P.s. replaceColor (QColor, QColor) of all ButtonObjects will be called, inclusive this object)
	}
	setBackgroundColor (fontDialog.getBackgroundColor());
	if (fontDialog.getForegroundColor() != getForegroundColor())	{
		// This emit signal is only caught when in ButtonDialog and the Selected, or Highlighted mask 
		// was changed (I.e. This Object is in one of those two frames).
		// Otherwise the signal is ignored.
		emit (signalMaskColorChanged(fontDialog.getForegroundColor(), getForegroundColor()));
		// P.s. replaceColor (QColor, QColor) of all ButtonObjects will be called, inclusive this object)
	}
	setForegroundColor (fontDialog.getForegroundColor());
	modifiers()->fTransparency = fontDialog.getTransparency();
	setTextAlign(fontDialog.getTextAlign());
	if (fontDialog.getFit())	{
		rect().setWidth (1);
		rect().setHeight(1);
	}
	else
		setRect(fontDialog.getRect());

	m_pParent->update ();
	emit (signalUpdatePixmap());
	emit (signalUpdateStructure());
	return false;
}

void TextObject::slotEdit ()
{
	mouseDoubleClickEvent(NULL);
}

void TextObject::slotMatrix()
{
	DialogMatrix *pDialog = new DialogMatrix(MenuObject::parent());
	pDialog->initMe(this);
	pDialog->show();
}

void TextObject::slotAddFrame ()
{
	QMessageBox::warning (NULL, tr ("Not Implemented yet."), tr ("Not Implemented yet."),
		QMessageBox::Ok ,  QMessageBox::Cancel);
}

void TextObject::slotAddImage ()
{
	QMessageBox::warning (NULL, tr ("Not Implemented yet."), tr ("Not Implemented yet."),
		QMessageBox::Ok ,  QMessageBox::Cancel);
}

void TextObject::slotAddMovie ()
{
	QMessageBox::warning (NULL, tr ("Not Implemented yet."), tr ("Not Implemented yet."),
		QMessageBox::Ok ,  QMessageBox::Cancel);
}

void TextObject::slotDefineAsButton ()
{
	// We want to convert this TextObject into a ButtonObject.
	// Thus the class TextObject is no longer the logical place for doing this, now is it ?
	// Okay we call the MenuPreview and let this handle the task.
	emit (signalDefineAsButton(this));
}

MenuObject *TextObject::clone()
{
	TextObject *pNewObject = new TextObject (MenuObject::parent());
	pNewObject->setFont (getFont());
	pNewObject->setText (getText());
	pNewObject->setTextAlign (getTextAlign());
	pNewObject->setForegroundColor (getForegroundColor());
	pNewObject->setBackgroundColor (getBackgroundColor());
	pNewObject->setStyleStrategy   ((QFont::StyleStrategy)m_iStyleStrategy);

	// The following two are not really needed, since we get those 
	// informations solely from the Normal State - objects ...
	pNewObject->setRect(rect());
	pNewObject->setModifiers(*modifiers());

	return pNewObject;
}

#include <qfont.h>
AnimationAttribute *TextObject::getSpecificAttributes (long iMaxNumberOfFrames, QString qsProperty)
{
//	QFont       m_Font;
//	QPoint      m_ptTextAlign;

	AnimationAttribute *pAnimAttr = NULL;
	static int m_iTestInt = 0;

	if (qsProperty == "text")
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_qsText, &m_qsText);
	else if (qsProperty == "font")	{
		FunctionCallback<QFont> *callMeBack = new FunctionCallback<QFont>(m_Font, &QFont::fromString);
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_Font.toString(), callMeBack);
	}
	else if (qsProperty == "textAlign")
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_iTestInt, &m_iTestInt);
	else if (qsProperty == "foregroundColor.red")	{
		FunctionCallback<QColor> *callMeBack = new FunctionCallback<QColor>(m_colorForeground, TYPE_COLOR_RED);
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_colorForeground.red(), callMeBack);
	}
	else if (qsProperty == "foregroundColor.green")	{
		FunctionCallback<QColor> *callMeBack = new FunctionCallback<QColor>(m_colorForeground, TYPE_COLOR_GREEN);
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_colorForeground.green(), callMeBack);
	}
	else if (qsProperty == "foregroundColor.blue")	{
		FunctionCallback<QColor> *callMeBack = new FunctionCallback<QColor>(m_colorForeground, TYPE_COLOR_BLUE);
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_colorForeground.blue(), callMeBack);
	}
	else if (qsProperty == "backgroundColor.red")	{
		FunctionCallback<QColor> *callMeBack = new FunctionCallback<QColor>(m_colorBackground, TYPE_COLOR_RED);
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_colorBackground.red(), callMeBack);
	}
	else if (qsProperty == "backgroundColor.green")	{
		FunctionCallback<QColor> *callMeBack = new FunctionCallback<QColor>(m_colorBackground, TYPE_COLOR_GREEN);
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_colorBackground.green(), callMeBack);
	}
	else if (qsProperty == "backgroundColor.blue")	{
		FunctionCallback<QColor> *callMeBack = new FunctionCallback<QColor>(m_colorBackground, TYPE_COLOR_BLUE);
		pAnimAttr = new AnimationAttribute (iMaxNumberOfFrames, qsProperty, m_colorBackground.blue(), callMeBack);
	}
	return pAnimAttr;
}

