/****************************************************************************
** MenuObject
**
** Created: Sun Feb 23 21:45:00 2004
**      by: Varol Okan using kate editor
**
**  This is the base class for all that is to appear onto the MenuPreview.
** Exception are background image, and the button masks (if the user
** specifies them rather then create the button manually)
**
** The MenuPreview class is the QLabel derived class where all buttons, text
** etc is to be plaeced for a DVD-Menu.
**
** The MenuObject is the base class for TextObject, ImageObject, MovieObject,
** FrameObject, and ButtonObject.
**
** The ButtonObject in turn is a container class which can hold a
** *TextObject, ImageObject, MovieObject, AND a FrameObject.
** Please refer to the buttonobject.h - file for more detail on the
** ButtonObject.
**
****************************************************************************/

#ifndef MENUOBJECT_H
#define MENUOBJECT_H

#include <qwidget.h>

#include "modifiers.h"
#include "animationattribute.h"

class QDomNode;
class QPopupMenu;
class QDomElement;
class QListViewItem;

#define HANDLE_WIDTH  9
#define HANDLE_HEIGHT 9

class MenuObject : public QObject
{
	Q_OBJECT
public:
	enum enState {
	  StateUnknown = 0,
	  StateInactive,     // No object is currently active
	  StateMoveObject,   // the object is being moved
	  StateScaleTLXY,    // the object is being scaled equally TopLeft
	  StateScaleTRXY,    // the object is being scaled equally TopRight
	  StateScaleBLXY,    // the object is being scaled equally BottomLeft
	  StateScaleBRXY,    // the object is being scaled equally BottmRight
	  StateScaleLeftX,   // the object is only scaled in X direction
	  StateScaleRightX,  // the object is only scaled in X direction
	  StateScaleTopY,    // the object is only scaled in Y direction
	  StateScaleBottomY, // the object is only scaled in Y direction
	  StateRotate
	};

	MenuObject (QWidget *pParent);
	virtual ~MenuObject();

	virtual void drawActiveFrame      ( QPainter    * );
	virtual void drawContents         ( QPainter    * ) = 0;
	virtual void drawContents         ( QPainter    *, int, int) = 0;
	virtual bool mouseMoveEvent       ( QMouseEvent * );
	virtual bool mousePressEvent      ( QMouseEvent * ) = 0;
	virtual bool mouseReleaseEvent    ( QMouseEvent * ) = 0;
	virtual bool mouseDoubleClickEvent( QMouseEvent * ) = 0;

	virtual bool readProjectFile      ( QDomNode     & );
	virtual bool writeProjectFile     ( QDomElement  & );
	virtual void createStructure      ( QListViewItem *) = 0;
	virtual bool createContextMenu    ( QPoint    ) = 0;
	virtual bool contains             ( QPoint  & );
	virtual MenuObject *clone         ( QWidget *p=NULL );
	virtual MenuObject *clone         ( MenuObject *, QWidget *p=NULL ); // those who need a parent (E.g. ShadowObject)
	virtual bool isActive             ( );
	virtual void setActive            ( bool );
	virtual void replaceColor         ( QColor, QColor ) { };
	virtual void updateShadow         ( );
	virtual void setShadow            ( MenuObject * );

	virtual void setModifiers   ( Modifiers &     );
	virtual void setRect        ( QRect &         );
	virtual QRect &rect         ( );
	void      update            (                 );
	void      setName           ( QString &       );
	void      setExtractionPath ( QString &       );
	void      setAnimation      ( QString &       );
	void      setParent         ( QWidget *       );
	void      setPosition       ( QPoint &        );
	bool      drawRect          (                 );
	void      setCurrentMousePos( QPoint  &       );
	void      animationDialog   ( QString &, int  );
	void      loadAnimationData ( QString &, long );
	enState   setObjectState    ( QPoint &        );
	void      setObjectState    ( enState         );

	virtual Modifiers *modifiers( );
	virtual MenuObject *shadow  ( );
	QString   &objectType       ( );
	QString   &name             ( );
	QString   &extractionPath   ( );	// Used to store a unique path name for this object to be extracted into (movieObject)
	QString   &animation        ( );
	QRect     &boundingRect     ( );
	QPoint     position         ( );
	QWidget   *parent           ( );
	QPoint    &currentMousePos  ( );

protected:
	// This function will return Object specific attributes for Animation purposes
	virtual AnimationAttribute *getSpecificAttributes (long, QString) = 0;

public slots:
	virtual void slotDelete     ( );
	virtual void slotEmitDelete ( );
	virtual void slotToFront    ( );
	virtual void slotToBack     ( );
	virtual void slotCut        ( );
	virtual void slotCopy       ( );
	virtual void slotAddShadow  ( );

signals:
	void signalMoveOnStack      ( MenuObject *, int );
	void signalDefineAsButton   ( MenuObject *   );
	void signalDeleteMe         ( MenuObject *   );
	void signalShadowMe         ( MenuObject *   );
	void signalAnimateMe        ( QString &, int, QString );
	void signalUpdateStructure  ( );
	void signalUpdatePixmap     ( );
	// Only used by Text, Frame, Button and only received if in ButtonDialog.
	void signalMaskColorChanged ( QColor, QColor);

protected:
	QPopupMenu   *m_pContextMenu;
	QString       m_qsObjectType;
	QString       m_qsName;
	QString       m_qsExtractionPath;
	QRect	      m_rect;
	QRect         m_boundingRect;
	QWidget      *m_pParent;
	bool          m_bDrawRect; // isActive, draw ActiveFrame (resize, rotate etc)
	MenuObject   *m_pShadow;
	Modifiers     m_modifiers;
	QString       m_qsAnimation;	// script or dataset holds also the datafilename
	QPoint        m_currentMousePos;
	enState       m_objectState;
	QValueList<AnimationAttribute *> m_listAnimationAttributes;
};

#endif	// MENUOBJECT_H
