/****************************************************************************
** Class Encode::FFmpeg implementation ...
**
**   Created : Wed Jun 17 07:53:05 2008
**        by : Varol Okan
** Copyright : (c) Varol Okan
**   License : GPL v 2.0
**
**pktdumper.c demo how to in -> enc -> out
****************************************************************************/

#include <QImage>

extern "C" {
#define __STDC_CONSTANT_MACROS
#define __STDC_LIMIT_MACROS
#include <libavformat/avformat.h>
#include <libswscale/swscale.h>
}

#define AUDIO_BUF_SIZE 192 * 1024 * 4
#define VIDEO_BUF_SIZE 1835008

#ifdef debugOut
#undef debugOut
#endif
#if 0
#define debugOut printf
#else
void dummy ( const char *, ... ) {};
#define debugOut dummy
#endif

#include "ffmpeg_enc.h"

namespace Encoder
{

QMutex FFmpeg::m_lock;

void printBuf ( short *pBuffer, int iSize )
{
  int t;
  for ( t=0; t<iSize; t++ )  {
    printf ( "[%.04X]", (unsigned short) pBuffer[t] );
  }
  printf ( "\n TOTAL = %d\n", iSize );
}

FFmpeg::Buffer::Buffer ( int iSampleSize )
{
  iSampleRate        = 0;
  iChannels          = 0;
  iSamplesPerChannel = 0;
  pBuffer            = NULL;
  pBuffer            = (short *) av_malloc ( iSampleSize * sizeof ( short ) );
}

FFmpeg::Buffer::~Buffer ( )
{
  if ( pBuffer )
    av_free ( pBuffer );
  pBuffer = NULL;
}

//////////////////////////////////////////////////
//
//  Media - Baseclass
//
//////////////////////////////////////////////////
FFmpeg::Media::Media ( )
{
  m_iAudioStreamID = -1;
  m_iVideoStreamID = -1;
  m_pAudioCodecCtx = NULL;
  m_pVideoCodecCtx = NULL;
  m_pFormatCtx     = NULL;
}

FFmpeg::Media::~Media ( )
{
  closeStream ( );
}

bool FFmpeg::Media::openStream ( QString qsStreamName )
{
  if ( qsStreamName != QString::null )
     m_qsStreamName  = qsStreamName;

  int iStreamID;
  AVCodec        *pCodec    = NULL;
  AVCodecContext *pCodecCtx = NULL;

  debugOut ( "%s : %d : streamName<%s>\n", __FUNCTION__, __LINE__, (const char *)m_qsStreamName.toUtf8 ( ) );
  // Open audio file
  if ( av_open_input_file ( &m_pFormatCtx, (const char *)m_qsStreamName.toUtf8 ( ), NULL, 0, NULL ) !=0 )
    return false; // Couldn't open file

  // Retrieve stream information
  if ( av_find_stream_info ( m_pFormatCtx ) < 0 )
    return false; // Couldn't find stream information

  // pFormatCtx->streams is just an array of pointers, of size pFormatCtx->nb_streams, 
  // so let's walk through it until we find a video stream.
  bool bHasAudio = false;
  bool bHasVideo = false;
  for  ( iStreamID=0; iStreamID < (int)m_pFormatCtx->nb_streams; iStreamID++ )  {
    if ( m_pFormatCtx->streams[iStreamID]->codec->codec_type == CODEC_TYPE_VIDEO )  {
      if ( bHasVideo )
        continue;
      // Get a pointer to the codec context for the first video stream
      pCodecCtx = m_pFormatCtx->streams[iStreamID]->codec;

      // Find the decoder for the video stream
      pCodec = avcodec_find_decoder ( pCodecCtx->codec_id );
      if ( ! pCodec ) {
        fprintf ( stderr, "Unsupported codec!\n");
        continue; // Codec not found
      }
      // Inform the codec that we can handle truncated bitstreams -- i.e.,
      // bitstreams where frame boundaries can fall in the middle of packets
      if( pCodec->capabilities  &  CODEC_CAP_TRUNCATED )
          pCodecCtx->flags     |=  CODEC_FLAG_TRUNCATED;

      // Open codec
      lockEngine ( );
      if ( avcodec_open ( pCodecCtx, pCodec ) < 0 )  {
        unlockEngine ( );
        continue; // Could not open codec
      }
      unlockEngine ( );

      bHasVideo = true;
      m_pVideoCodecCtx = pCodecCtx;
      m_iVideoStreamID = iStreamID;
    }
    else if ( m_pFormatCtx->streams[iStreamID]->codec->codec_type == CODEC_TYPE_AUDIO )  {
      if ( bHasAudio )
        continue;
      // Get a pointer to the codec context for the first audio stream
      pCodecCtx = m_pFormatCtx->streams[iStreamID]->codec;

      // Find the decoder for the audio stream
      pCodec = avcodec_find_decoder ( pCodecCtx->codec_id );
      if ( ! pCodec ) {
        fprintf ( stderr, "Unsupported codec!\n");
        continue; // Codec not found
      }

      // Open codec
      lockEngine ( );
      if ( avcodec_open ( pCodecCtx, pCodec ) < 0 )  {
        unlockEngine ( );
        continue; // Could not open codec
      }
      unlockEngine ( );
//      // Hack to correct wrong frame rates that seem to be generated by some codecs
//      if ( pCodecCtx->frame_rate > 1000 && pCodecCtx->frame_rate_base == 1 )
//           pCodecCtx->frame_rate_base = 1000;

      bHasAudio = true;
      m_pAudioCodecCtx = pCodecCtx;
      m_iAudioStreamID = iStreamID;
    }
  }
  // Didn't find an video stream
  if ( ( ! bHasVideo ) && ( ! bHasAudio ) )
    return false;  // Dont forget to close the stream though ...

  return true;
}

void FFmpeg::Media::closeStream ( )
{
  // Close the codec
  lockEngine ( );
  if ( m_pAudioCodecCtx )
    avcodec_close ( m_pAudioCodecCtx );
  if ( m_pVideoCodecCtx )
    avcodec_close ( m_pVideoCodecCtx );
  unlockEngine ( );
  // Close the file
  if ( m_pFormatCtx )
    av_close_input_file ( m_pFormatCtx );

  m_pAudioCodecCtx = NULL;
  m_pVideoCodecCtx = NULL;
  m_pFormatCtx     = NULL;
}

void FFmpeg::Media::setOutputContext ( AVCodecContext *pVideoContext, AVCodecContext *pAudioContext )
{
  m_pOutputVideo = pVideoContext;
  m_pOutputAudio = pAudioContext;
}

void FFmpeg::Media::resetStream ( )
{
}

QString FFmpeg::Media::streamName ( )
{
  return m_qsStreamName;
}

int FFmpeg::Media::streamID ( bool bAudio )
{
  if ( bAudio )
    return m_iAudioStreamID;
  return m_iVideoStreamID;
}

//////////////////////////////////////////////////
//
//  Audio - Class
//
//////////////////////////////////////////////////
FFmpeg::Audio::Audio ( QString qsFileName, unsigned int iSampleSize, AVStream *pOutputStream )
  : Media ( )
{
  m_qsStreamName  = qsFileName;
  m_iSampleSize   = iSampleSize;
  m_pOutputAudio  = pOutputStream->codec;
  m_pFormatCtx    = NULL;
  if (  m_iSampleSize < AVCODEC_MAX_AUDIO_FRAME_SIZE + FF_INPUT_BUFFER_PADDING_SIZE )
        m_iSampleSize = AVCODEC_MAX_AUDIO_FRAME_SIZE + FF_INPUT_BUFFER_PADDING_SIZE;

  m_pSampleBuffer     = new Buffer ( m_iSampleSize );
  m_pRemainingSamples = new Buffer ( m_iSampleSize );
  m_pResample = NULL;
}

FFmpeg::Audio::Audio ( unsigned int iSampleSize )
  : Media ( )
{
  m_iSampleSize   = iSampleSize;
  m_pOutputAudio  = NULL;
  if ( m_iSampleSize < AVCODEC_MAX_AUDIO_FRAME_SIZE + FF_INPUT_BUFFER_PADDING_SIZE )
       m_iSampleSize = AVCODEC_MAX_AUDIO_FRAME_SIZE + FF_INPUT_BUFFER_PADDING_SIZE;

  m_pSampleBuffer     = new Buffer ( m_iSampleSize );
  m_pRemainingSamples = new Buffer ( m_iSampleSize );
  m_pResample = NULL;
}

FFmpeg::Audio::~Audio ( )
{
  if ( m_pSampleBuffer )
    delete m_pSampleBuffer;
  m_pSampleBuffer = NULL;

  if ( m_pRemainingSamples )
    delete m_pRemainingSamples;
  m_pRemainingSamples = NULL;

//  closeStream ( );
//  if ( m_pCodecCtx )
//    delete m_pCodecCtx;
//  m_pCodecCtx = NULL;

//  if ( m_pFormatCtx )
//    delete m_pFormatCtx;
//  m_pFormatCtx = NULL;

//  av_free ( m_pSampleBuffer->pBuffer    );
//  av_free ( m_pSampleBuffer->pResampled );
//  audio_resample_close ( m_pResample );
}

void FFmpeg::Audio::closeStream ( )
{
  FFmpeg::Media::closeStream  ( );
  if ( m_pResample )
    audio_resample_close ( m_pResample );
  m_pResample = NULL;
}

FFmpeg::Buffer *FFmpeg::Audio::decode ( bool &bEOF )
{
  bEOF = false;
  if ( ! m_pAudioCodecCtx )
    return NULL;

  AVPacket packet;
  int  iRet  = av_read_frame ( m_pFormatCtx, &packet );
  if ( iRet == 0 )
    decode ( &packet  );
  else  if ( iRet < 0 )
    bEOF = true;

  return m_pSampleBuffer;
}

FFmpeg::Buffer *FFmpeg::Audio::decode ( AVPacket *pPacket )
{
  if ( ! pPacket )
    return NULL;

  unsigned int iOffset = 0;  // in Bytes
  int      iPacketSize = pPacket->size;
  uint8_t *pPacketData = pPacket->data;
  // Is this a packet from the video stream?
  if  ( pPacket->stream_index == m_iAudioStreamID )  {
    int16_t *pDecodedAudio = NULL;
    int      iDecodedSize, iLength;
    pDecodedAudio = (int16_t*) av_malloc ( m_iSampleSize );
    while  ( iPacketSize > 0 )  {
      iDecodedSize  = m_iSampleSize;
      memset ( pDecodedAudio, 0, m_iSampleSize );

      iLength = avcodec_decode_audio2 ( m_pAudioCodecCtx, pDecodedAudio, &iDecodedSize, pPacketData, iPacketSize );
      if ( iLength <= 0 )  {  // Error handling ...
        iPacketSize --;
        pPacketData ++;
      }
      else  {
        iPacketSize -= iLength;
        pPacketData += iLength;

        memcpy ( (uint8_t*)(m_pSampleBuffer->pBuffer+iOffset), pDecodedAudio, iDecodedSize );
        iOffset += iDecodedSize;
      }
    }
    av_free ( pDecodedAudio );
  }
  if ( pPacket->data )
    av_free_packet ( pPacket );

  if ( iOffset == 0 )  {
    // Invalid package ... wtf.
    memset ( m_pSampleBuffer->pBuffer, 0, AUDIO_BUF_SIZE );
    m_pSampleBuffer->iSamplesPerChannel = 0;
    return NULL;
  }

  // Calculate the number of samples in the output buffer
  // Note this is the total number, so for stereo the format is LRLRLRLR
  m_pSampleBuffer->iSampleRate        = m_pAudioCodecCtx->sample_rate;
  m_pSampleBuffer->iChannels          = m_pAudioCodecCtx->channels;  // # of input Channels
  m_pSampleBuffer->iSamplesPerChannel = (unsigned int)iOffset / ( sizeof ( short ) * getChannels ( ) );

  return m_pSampleBuffer;
}

FFmpeg::Buffer *FFmpeg::Audio::resample ( )
{
  if ( getSamplesPerChannel ( ) < 1 )
    return m_pSampleBuffer;

  short *pResampled = (short *)av_malloc ( m_iSampleSize );

  memset ( pResampled, 0, m_iSampleSize );

  int iSamples = getSamplesPerChannel ( );
  iSamples     = audio_resample ( m_pResample, pResampled, (short *)m_pSampleBuffer->pBuffer, iSamples );

  av_free ( m_pSampleBuffer->pBuffer );
  m_pSampleBuffer->pBuffer     = pResampled;
  m_pSampleBuffer->iChannels   = m_pOutputAudio->channels;
  m_pSampleBuffer->iSampleRate = m_pOutputAudio->sample_rate;
  //m_pSampleBuffer->iSamplesPerChannel = (int)( (float)iSamples / getChannels ( ) );
  m_pSampleBuffer->iSamplesPerChannel = iSamples;

  return m_pSampleBuffer;
}

FFmpeg::Buffer *FFmpeg::Audio::resample ( AVPacket * )
{
  if ( getSamplesPerChannel ( ) < 1 )
    return m_pSampleBuffer;

  short *pResampled = (short *)av_malloc ( m_iSampleSize );

  memset ( pResampled, 0, m_iSampleSize );

  int iSamples = getSamplesPerChannel ( );
  iSamples     = audio_resample ( m_pResample, pResampled, (short *)m_pSampleBuffer->pBuffer, iSamples );

  av_free (  m_pSampleBuffer->pBuffer  );
  m_pSampleBuffer->pBuffer     = pResampled;
  m_pSampleBuffer->iChannels   = m_pOutputAudio->channels;
  m_pSampleBuffer->iSampleRate = m_pOutputAudio->sample_rate;
  //m_pSampleBuffer->iSamplesPerChannel = (int)( (float)iSamples / getChannels ( ) );
  m_pSampleBuffer->iSamplesPerChannel = iSamples;

  return m_pSampleBuffer;
}

unsigned int FFmpeg::Audio::getSamplesPerChannel ( )
{
  return m_pSampleBuffer->iSamplesPerChannel;
}

unsigned int FFmpeg::Audio::getTotalSamples ( )
{
  unsigned int iTotalSamples =  getSamplesPerChannel ( ) * getChannels ( );
  return iTotalSamples;
}

unsigned int FFmpeg::Audio::getChannels ( )
{
  return m_pSampleBuffer->iChannels;
}

unsigned int FFmpeg::Audio::getBitrate ( )
{
  return m_pAudioCodecCtx ? m_pAudioCodecCtx->bit_rate : 0;
}

unsigned int FFmpeg::Audio::getSampleRate ( )
{
  return m_pSampleBuffer->iSampleRate;
}

unsigned int FFmpeg::Audio::getBytes ( )
{
  unsigned int iBytes = getTotalSamples ( ) * sizeof ( short );
  return iBytes;
}

bool FFmpeg::Audio::openStream ( QString qsStreamName )
{
  bool bRet = Media::openStream ( qsStreamName );
  if ( ! m_pAudioCodecCtx )
    return false;
  return bRet;
}

FFmpeg::Buffer *FFmpeg::Audio::decodeAudio ( bool &bEOF )
{
  int    iSamples       = 0;
  int    iSampleSize    = sizeof ( short );
  int    iOutChannels   = m_pOutputAudio->channels;
  int    iOutSampleRate = m_pOutputAudio->sample_rate;
  int    iOutFrameSize  = m_pOutputAudio->frame_size;

  if ( ! m_pResample && m_pAudioCodecCtx )
         m_pResample    = audio_resample_init ( iOutChannels, m_pAudioCodecCtx->channels, iOutSampleRate, m_pAudioCodecCtx->sample_rate );

  FFmpeg::Buffer *pBuffer = NULL;
  char *pData = (char *)av_malloc ( AUDIO_BUF_SIZE );
  memset ( pData, 0, AUDIO_BUF_SIZE );

  // Lets check if we have leftover from the previous decoding run
  if ( m_pRemainingSamples->iSamplesPerChannel > 0 )  {
    iSamples = m_pRemainingSamples->iSamplesPerChannel;
    if ( iSamples > iOutFrameSize )
         iSamples = iOutFrameSize;
    memcpy ( pData, (char *)m_pRemainingSamples->pBuffer, iSamples * iOutChannels * iSampleSize );

    m_pRemainingSamples->iSamplesPerChannel -= iSamples;
    if ( m_pRemainingSamples->iSamplesPerChannel > 0 )  {
      memcpy ( (char *)m_pRemainingSamples->pBuffer, ((char *)m_pRemainingSamples->pBuffer + iSamples * iOutChannels * iSampleSize), m_pRemainingSamples->iSamplesPerChannel );
    }
  }

  // Main Loop until we have enough samples for this frame.
  while ( iSamples  < iOutFrameSize )  {
    pBuffer = decode   ( bEOF );
    pBuffer = resample ( );

    if ( pBuffer->iSamplesPerChannel == 0 )
      break;
    memcpy ( (pData + iSamples * iOutChannels * iSampleSize), (char *)pBuffer->pBuffer, pBuffer->iSamplesPerChannel * iOutChannels * iSampleSize );

    iSamples += pBuffer->iSamplesPerChannel;
  }

  // Next we store the remaining samples in a temp buffer
  if ( iSamples - iOutFrameSize > 0 )  {
    m_pRemainingSamples->iSamplesPerChannel = iSamples - iOutFrameSize;
    memcpy ( (char *)m_pRemainingSamples->pBuffer, (pData + iOutFrameSize * iOutChannels * iSampleSize ), m_pRemainingSamples->iSamplesPerChannel * iOutChannels * iSampleSize );
  }
  else  {
    m_pRemainingSamples->iSamplesPerChannel = -1;
    iOutFrameSize -= iSamples;
  }

  av_free ( m_pSampleBuffer->pBuffer );
  m_pSampleBuffer->pBuffer            = (short *)pData;
  m_pSampleBuffer->iSamplesPerChannel = iOutFrameSize;

  return m_pSampleBuffer;
}

FFmpeg::Buffer *FFmpeg::Audio::decodeAudio ( AVPacket *pPacket )
{
  int    iSampleSize    = sizeof ( short );
  int    iOutChannels   = m_pOutputAudio->channels;
  int    iOutSampleRate = m_pOutputAudio->sample_rate;
  int    iOutFrameSize  = m_pOutputAudio->frame_size;

  // First we ensure we have a resample context if required ...
  if ( ! m_pResample && m_pAudioCodecCtx )
         m_pResample    = audio_resample_init ( iOutChannels, m_pAudioCodecCtx->channels, iOutSampleRate, m_pAudioCodecCtx->sample_rate );

  // Alloc some memory ...
  FFmpeg::Buffer *pBuffer = NULL;
  // Get the new data from the packet ...
  pBuffer = decode   ( pPacket );
  pBuffer = resample ( pPacket );
  // important, audio_resample returns the number of resampled samples. We need the total number of samples here
  // I.e. if input is mono and output is stereo, then we need to multiply by 2.
  //pBuffer->iSamplesPerChannel *= iOutChannels;

  // Next we append the new data to the remaining data ...
  char *pRemainingStart = ( (char *)m_pRemainingSamples->pBuffer + m_pRemainingSamples->iSamplesPerChannel * iOutChannels * iSampleSize );
  memcpy ( pRemainingStart, (char *)pBuffer->pBuffer, pBuffer->iSamplesPerChannel * iOutChannels * iSampleSize );
  m_pRemainingSamples->iSamplesPerChannel += pBuffer->iSamplesPerChannel;

  // Then we check if we have enough samples for at least one frame
  if ( m_pRemainingSamples->iSamplesPerChannel >= iOutFrameSize )  {
    char *pData = (char *)av_malloc ( AUDIO_BUF_SIZE );
    memset ( pData, 0, AUDIO_BUF_SIZE );

    pRemainingStart = ( (char *)m_pRemainingSamples->pBuffer + iOutFrameSize * iOutChannels * iSampleSize );
    memcpy  (  pData,   (char *)m_pRemainingSamples->pBuffer,  iOutFrameSize * iOutChannels * iSampleSize );
    m_pRemainingSamples->iSamplesPerChannel -= iOutFrameSize;
    // Should not happen but to be safe ...
    if ( m_pRemainingSamples->iSamplesPerChannel < 0 )
         m_pRemainingSamples->iSamplesPerChannel = 0;
    memcpy  ( (char *)m_pRemainingSamples->pBuffer,  pRemainingStart, m_pRemainingSamples->iSamplesPerChannel * iOutChannels * iSampleSize );

    av_free ( m_pSampleBuffer->pBuffer );
    m_pSampleBuffer->pBuffer            = (short *)pData;
    m_pSampleBuffer->iSamplesPerChannel = iOutFrameSize;

    return m_pSampleBuffer;
  }

  return NULL;
}

FFmpeg::Buffer *FFmpeg::Audio::getRemainingAudio ( AVStream *pOutputAudioStream )
{
  if ( ! pOutputAudioStream )
    return NULL;

  int iOutFrameSize = pOutputAudioStream->codec->frame_size;
  int iOutChannels  = pOutputAudioStream->codec->channels;
  int iSampleSize   = sizeof ( short );

  if ( ( m_pRemainingSamples->iSamplesPerChannel < iOutFrameSize ) || ( iOutFrameSize < 1 ) )
    return NULL;

  m_pSampleBuffer->iSamplesPerChannel      = iOutFrameSize;
  m_pRemainingSamples->iSamplesPerChannel -= iOutFrameSize;
  memcpy ( (char *)m_pSampleBuffer->pBuffer,      (char *)m_pRemainingSamples->pBuffer,  iOutFrameSize * iOutChannels * iSampleSize );
  memcpy ( (char *)m_pRemainingSamples->pBuffer, ((char *)m_pRemainingSamples->pBuffer + iOutFrameSize * iOutChannels * iSampleSize ), m_pRemainingSamples->iSamplesPerChannel * iSampleSize * iOutChannels );

  return m_pSampleBuffer;
}

//////////////////////////////////////////////////
//
//  Vid - Class
//
//////////////////////////////////////////////////
FFmpeg::Vid::Vid ( )
  : Audio ( AUDIO_BUF_SIZE )
{
  m_pPacket        = new AVPacket;
  av_init_packet   ( m_pPacket );
  m_pFrame         = NULL;
  m_pImgConvertCtx = NULL;
  m_iVideoOutSize  = 0;
  m_pVideoBuffer   = (uint8_t *) av_malloc ( VIDEO_BUF_SIZE );
  m_pAudioBuffer   = (uint8_t *) av_malloc ( AUDIO_BUF_SIZE );
  m_pOutputVideo   = NULL;
  m_pOutputAudio   = NULL;
}

FFmpeg::Vid::~Vid ( )
{
  if ( m_pPacket )
    delete m_pPacket;

  if ( m_pFrame )  {
    if ( m_pFrame->data[0] )
      av_free ( m_pFrame->data[0] );
    av_free ( m_pFrame );
    m_pFrame = NULL;
  }

  if ( m_pVideoBuffer )
    av_free ( m_pVideoBuffer );
  if ( m_pAudioBuffer )
    av_free ( m_pAudioBuffer );

  if ( m_pImgConvertCtx )
    sws_freeContext ( m_pImgConvertCtx );
  m_pImgConvertCtx = NULL;
}

bool FFmpeg::Vid::openStream ( QString qsStreamName )
{
  bool bRet = Media::openStream ( qsStreamName );
  m_iVideoOutSize = 0;
  if ( ! m_pVideoCodecCtx )
    return false;
  return bRet;
}

void FFmpeg::Vid::closeStream ( )
{
  m_iVideoOutSize = 0;
  FFmpeg::Audio::closeStream ( );
}

void FFmpeg::Vid::setSWScale ( int iTargetWidth, int iTargetHeight )
{
  if ( ! m_pVideoCodecCtx )
    return;

  // as we only generate a YUV420P picture, we must convert it to the codec pixel format if needed
  int iCWidth  = m_pVideoCodecCtx->width;
  int iCHeight = m_pVideoCodecCtx->height;
  if ( ( m_pVideoCodecCtx->pix_fmt != PIX_FMT_YUV420P ) || ( iCWidth != iTargetWidth ) || ( iCHeight != iTargetHeight ) )
    m_pImgConvertCtx = sws_getContext ( iCWidth, iCHeight, m_pVideoCodecCtx->pix_fmt, iTargetWidth, iTargetHeight, PIX_FMT_YUV420P, SWS_BICUBIC, NULL, NULL, NULL );
  else
    m_pImgConvertCtx = NULL;

  // allocate the encoded raw picture
  m_pFrame = allocPicture ( m_pVideoBuffer, PIX_FMT_YUV420P, iTargetWidth, iTargetHeight );

  if ( ! m_pFrame )
    printf ( "Could not allocate picture" );
}

AVFrame *FFmpeg::Vid::allocPicture ( uint8_t *&, int iPixFormat, int iWidth, int iHeight )
{
  // Note: Might cause memory leak ? Should be invested at some point in time
  AVFrame *pFrame = avcodec_alloc_frame ( );
  if (  !  pFrame )
    return NULL;

  int iSize = avpicture_get_size ( iPixFormat, iWidth, iHeight );
  uint8_t *pBuffer = (uint8_t *) av_malloc ( iSize );
  if (  !  pBuffer )  {
      av_free ( pFrame );
      return NULL;
  }

  avpicture_fill ( (AVPicture *)pFrame, pBuffer, iPixFormat, iWidth, iHeight );
  return pFrame;
}

SwsContext *FFmpeg::Vid::getSWScale ( )
{
  return m_pImgConvertCtx;
}

AVPacket *FFmpeg::Vid::hasRemainingAudio ( AVStream *pOutputAudioStream )
{
//  AVCodecContext *pAudioCodec = pOutputAudioStream->codec;
//  int iOutFrameSize = pAudioCodec->frame_size * pAudioCodec->channels;
//  int    iOutFrameSize  = 1536 * 2; // pAudioCodec->frame_size * iOutChannels;

  Buffer *pSamples = getRemainingAudio ( pOutputAudioStream );
  if (  ! pSamples )
     return NULL;

  // Free the packet that was previously allocated by av_read_frame
  av_init_packet ( m_pPacket );
  unsigned int iSize = getSamplesPerChannel ( ) * sizeof ( short ); //getTotalSamples ( ) * sizeof ( short );

  m_pPacket->size = pOutputAudioStream->codec->codec->encode ( pOutputAudioStream->codec, m_pAudioBuffer, iSize, (void *)pSamples->pBuffer );
  pOutputAudioStream->codec->frame_number++;

  m_pPacket->pts  = av_rescale_q ( pOutputAudioStream->codec->coded_frame->pts, pOutputAudioStream->codec->time_base, pOutputAudioStream->time_base );

  m_pPacket->flags |= PKT_FLAG_KEY;
  m_pPacket->stream_index = pOutputAudioStream->index;
  m_pPacket->data         = m_pAudioBuffer;

  return m_pPacket;
}

AVPacket *FFmpeg::Vid::getNextPacket ( AVStream *pOutputVideoStream, AVStream *pOutputAudioStream, int &iReuseFrame )
{
  // iAdjusFrame is decremented by one each time a frame is re-used
  // iAdjusrtFrame is set to 0 when a new frame is taken from the stream.

  bool bFound = false;
  int iFrameFinished;

  static int iInterleave = 3;
  iInterleave--;
  if   ( iInterleave  < 2 )  { // every other Image frame is followed one audio frame ( if we have some data )
    if ( iInterleave == 0 )
         iInterleave  = 3;
    // This function will check if we have enough data for another output audio frame
    AVPacket *pPacket = hasRemainingAudio ( pOutputAudioStream );
    if ( pPacket )
      return pPacket;
  }

  // Here we adjust the frame rate in case the source has less frames ( E.g. 10 FPS ) then the required frames ( E.g. 29.97 )
  if ( ( iReuseFrame > 1 ) && ( m_iVideoOutSize > 0 ) )  {
    m_iVideoOutSize = avcodec_encode_video ( pOutputVideoStream->codec, m_pVideoBuffer, VIDEO_BUF_SIZE, m_pFrame );
    if ( m_iVideoOutSize >= 0 )  {
      av_init_packet  ( m_pPacket );
      iReuseFrame --;
      m_pPacket->pts = av_rescale_q ( pOutputVideoStream->codec->coded_frame->pts, pOutputVideoStream->codec->time_base, pOutputVideoStream->time_base );

      if ( pOutputVideoStream->codec->coded_frame->key_frame )
           m_pPacket->flags  |= PKT_FLAG_KEY;
      m_pPacket->stream_index = pOutputVideoStream->index;
      m_pPacket->data         = m_pVideoBuffer;
      m_pPacket->size         = m_iVideoOutSize;
      return m_pPacket;
    }
  }

  // Free the packet that was previously allocated by av_read_frame
  av_free_packet ( m_pPacket );

  while ( ! bFound && av_read_frame ( m_pFormatCtx, m_pPacket ) >= 0 )  {
    if  ( m_pPacket->stream_index == m_iVideoStreamID )  {
      // Ignore this frame if source frame rate is > the target frame rate
      if ( iReuseFrame < 0 )  {
        m_iVideoOutSize = avcodec_encode_video ( pOutputVideoStream->codec, m_pVideoBuffer, VIDEO_BUF_SIZE, m_pFrame );
        iReuseFrame ++;
        continue;
      }

      int iWidth, iHeight, iNumBytes;
      uint8_t   *pBuffer   = NULL;
      AVFrame   *pFrame    = NULL;

      pFrame    = avcodec_alloc_frame ( );
      iWidth    = m_pVideoCodecCtx->width;
      iHeight   = m_pVideoCodecCtx->height;
      // Determine required buffer size and allocate buffer
      iNumBytes = avpicture_get_size ( PIX_FMT_YUV420P, iWidth, iHeight );
      pBuffer   = new uint8_t[iNumBytes];

      // Assign appropriate parts of buffer to image planes in pFrameRGB
      avpicture_fill ( (AVPicture *)pFrame, pBuffer, PIX_FMT_YUV420P, iWidth, iHeight );

      iFrameFinished = 0;
      // Decode input video frame
      int  iLen = avcodec_decode_video ( m_pVideoCodecCtx, pFrame, &iFrameFinished, m_pPacket->data, m_pPacket->size );
      if ( iFrameFinished == 0 ) // Mpeg streams might need this double encoding phase ...
           iLen = avcodec_decode_video ( m_pVideoCodecCtx, pFrame, &iFrameFinished, m_pPacket->data, m_pPacket->size );

      // Did we get a video frame?
      // Note : Instead of running the decoding again, it seems to be agreed on 
      //        that simply ignoring the iFrameFinished check. However it works for me !
      if ( iFrameFinished )  {
        int iOutSize = -1;
        if ( m_pImgConvertCtx )
          sws_scale ( m_pImgConvertCtx, pFrame->data, pFrame->linesize, 0, iHeight, m_pFrame->data, m_pFrame->linesize );
        else // av_picture_copy ( AVPicture *dst, const AVPicture *src, int pix_fmt, int width, int height);
          av_picture_copy ( (AVPicture *)m_pFrame, (AVPicture *)pFrame, PIX_FMT_YUV420P, iWidth, iHeight );

        // encode the image
        iOutSize = avcodec_encode_video ( pOutputVideoStream->codec, m_pVideoBuffer, VIDEO_BUF_SIZE, m_pFrame );

        if ( iOutSize < 0 )
          continue;
        // if zero size, it means the image was buffered
        if ( iOutSize > 0 ) {
          av_init_packet  ( m_pPacket );
          m_pPacket->pts = av_rescale_q ( pOutputVideoStream->codec->coded_frame->pts, pOutputVideoStream->codec->time_base, pOutputVideoStream->time_base );

          if ( pOutputVideoStream->codec->coded_frame->key_frame )
               m_pPacket->flags  |= PKT_FLAG_KEY;
          m_pPacket->stream_index = pOutputVideoStream->index;
          m_pPacket->data         = m_pVideoBuffer;
          m_pPacket->size         = iOutSize;
          m_iVideoOutSize = iOutSize;
          iReuseFrame     = 0; // Signals to the calling function  that we have decoded a new frame.

          delete []pBuffer;
          if ( pFrame )
            av_free ( pFrame );

          return m_pPacket;
        }
      }
    }
    else if ( m_pPacket->stream_index == m_iAudioStreamID )  {
      Buffer *pSamples = decodeAudio ( m_pPacket );
      if  ( ! pSamples )
        continue; // with the next package from stream
      unsigned int iSize = getSamplesPerChannel  ( ) * sizeof ( short );

      av_init_packet ( m_pPacket );
      m_pPacket->size = pOutputAudioStream->codec->codec->encode ( pOutputAudioStream->codec, m_pAudioBuffer, iSize, (void *)pSamples->pBuffer );
      pOutputAudioStream->codec->frame_number++;

      m_pPacket->pts  = av_rescale_q ( pOutputAudioStream->codec->coded_frame->pts, pOutputAudioStream->codec->time_base, pOutputAudioStream->time_base );

      m_pPacket->flags       |= PKT_FLAG_KEY;
      m_pPacket->stream_index = pOutputAudioStream->index;
      m_pPacket->data         = m_pAudioBuffer;

      return m_pPacket;
    }
    else  {
      ; // error. not a valid streamID
    }
  }
  return NULL;
}

QImage &FFmpeg::Vid::getFrame ( AVFrame *pFrame, int iWidth, int iHeight )
{
  if ( ( ! pFrame ) || ( ! m_pVideoCodecCtx ) )  {
    // TODO: Create Error Image ...
    m_frame = QImage ( );
    return m_frame;
  }

  uint8_t    *pBuffer   = NULL;
  SwsContext *pSwsCtx   = NULL;
  AVFrame    *pFrameRGB = avcodec_alloc_frame ( );
  int iNumBytes;

  // Determine required buffer size and allocate buffer
  iNumBytes = avpicture_get_size ( PIX_FMT_RGB32, iWidth, iHeight );
  pBuffer   = new uint8_t[iNumBytes];

  // Assign appropriate parts of buffer to image planes in pFrameRGB
  avpicture_fill ( (AVPicture *)pFrameRGB, pBuffer, PIX_FMT_RGB32, iWidth, iHeight );

  pSwsCtx = sws_getContext ( iWidth, iHeight, m_pVideoCodecCtx->pix_fmt, iWidth, iHeight, PIX_FMT_RGB32, SWS_BILINEAR, NULL, NULL, NULL);

  // Convert the image from its native format to RGB
  sws_scale ( pSwsCtx, pFrame->data, pFrame->linesize, 0, iHeight, pFrameRGB->data, pFrameRGB->linesize );

  {  // This to destroy the tmp image and force a deep copy of the bits into m_image.
    QImage tmp ( (uchar *)pBuffer, iWidth, iHeight, QImage::Format_RGB32 );
    m_frame = tmp.copy ( );
  }

  if ( pSwsCtx )
   sws_freeContext ( pSwsCtx );
  if ( pBuffer )
    delete []pBuffer;
  if ( pFrameRGB )
    av_free ( pFrameRGB );

  return m_frame;
}

QImage &FFmpeg::Vid::getFrame ( double fStreamPos )
{
  if ( ( fStreamPos < -1.0f ) || ( fStreamPos > 1.0f ) || ( ! m_pVideoCodecCtx ) )  {
    // TODO: Create Error Image ...
    m_frame = QImage ( );
    return m_frame;
  }

  AVPacket   packet;
  AVFrame   *pFrame    = NULL;
  bool bFound = false;

  int iFrameFinished;
  int iWidth, iHeight, iNumBytes;

  pFrame    = avcodec_alloc_frame ( );
  iWidth    = m_pVideoCodecCtx->width;
  iHeight   = m_pVideoCodecCtx->height;
  // Determine required buffer size and allocate buffer
  iNumBytes = avpicture_get_size ( PIX_FMT_RGB32, iWidth, iHeight );

  if ( fStreamPos < 0.0 )  {
    // get the last decoded frame, else invert the number ...
    fStreamPos *= -1.0;
    avcodec_decode_video ( m_pVideoCodecCtx, pFrame, &iFrameFinished, NULL, 0 );

    // Did we get a video frame?
    if ( iFrameFinished )  {
      getFrame ( pFrame, iWidth, iHeight );
      bFound = true;
    }
  }

  int iSeekPos  = -1;
  if ( ( fStreamPos > 0.0 ) && ( ! bFound ) )  {
    double fDuration =  (double) m_pFormatCtx->duration / AV_TIME_BASE; // can be unknown & can be incorrect (e.g. .vob files)
    if ( fDuration <= 0 )
         fDuration  = guessDuration ( m_pFormatCtx, m_iVideoStreamID );
    if ( fDuration <= 0 )  {
        ; //av_log ( NULL, AV_LOG_ERROR, "  duration is unknown: %.2f\n", duration);
    }
    AVStream *pVideoStream = m_pFormatCtx->streams[m_iVideoStreamID];
    int64_t iEndPosition = fDuration * fStreamPos / av_q2d ( pVideoStream->time_base ); // in time_base uni
    iSeekPos = reallySeek ( m_pFormatCtx, m_iVideoStreamID, iEndPosition, 0, fDuration );
  }

  while ( !bFound && av_read_frame ( m_pFormatCtx, &packet ) >= 0 )  {
    // Decode video frame
    avcodec_decode_video ( m_pVideoCodecCtx, pFrame, &iFrameFinished, packet.data, packet.size );

    // Did we get a video frame?
    if ( iFrameFinished )  {
      getFrame ( pFrame, iWidth, iHeight );
      bFound = true;
    }

    // Free the packet that was allocated by av_read_frame
    av_free_packet ( &packet );
  }

  if ( pFrame )
    av_free ( pFrame );

  return m_frame;
}

double FFmpeg::Vid::getFrameRate ( )
{
  if ( ! m_pVideoCodecCtx )
    return 3000.0 / 1001.0;
  return (double)m_pVideoCodecCtx->time_base.den / m_pVideoCodecCtx->time_base.num;
}

/*
from moviethumbnail.sf.net
try hard to seek
assume flags can be either 0 or AVSEEK_FLAG_BACKWARD
*/
int FFmpeg::Vid::reallySeek ( AVFormatContext *pFormatCtx, int index, int64_t timestamp, int flags, double duration )
{
    if ( ( flags != 0 ) && ( flags != AVSEEK_FLAG_BACKWARD ) )
      return -1;

    int ret;

    /* first try av_seek_frame */
    ret = av_seek_frame ( pFormatCtx, index, timestamp, flags );
    if (ret >= 0) { // success
        return ret;
    }

    /* then we try seeking to any (non key) frame AVSEEK_FLAG_ANY */
    ret = av_seek_frame(pFormatCtx, index, timestamp, flags | AVSEEK_FLAG_ANY);
    if (ret >= 0) { // success
        //av_log(NULL, LOG_INFO, "AVSEEK_FLAG_ANY: timestamp: %"PRId64"\n", timestamp); // DEBUG
        return ret;
    }

    /* and then we try seeking by byte (AVSEEK_FLAG_BYTE) */
    // here we assume that the whole file has duration seconds.
    // so we'll interpolate accordingly.
    AVStream *pStream = pFormatCtx->streams[index];
    int64_t duration_tb = duration / av_q2d(pStream->time_base); // in time_base unit
    double start_time = (double) pFormatCtx->start_time / AV_TIME_BASE; // in seconds
    // if start_time is negative, we ignore it; FIXME: is this ok?
    if (start_time < 0) {
        start_time = 0;
    }

    // normally when seeking by timestamp we add start_time to timestamp
    // before seeking, but seeking by byte we need to subtract the added start_time
    timestamp -= start_time / av_q2d(pStream->time_base);
    if (pFormatCtx->file_size <= 0) {
        return -1;
    }
    if (duration > 0) {
        int64_t byte_pos = av_rescale(timestamp, pFormatCtx->file_size, duration_tb);
        //av_log(NULL, LOG_INFO, "AVSEEK_FLAG_BYTE: byte_pos: %"PRId64", timestamp: %"PRId64", file_size: %"PRId64", duration_tb: %"PRId64"\n", byte_pos, timestamp, pFormatCtx->file_size, duration_tb);
        return av_seek_frame(pFormatCtx, index, byte_pos, AVSEEK_FLAG_BYTE);
    }

    return -1;
}
/*
from moviethumbnail.sf.net
return the duration. guess when unknown.
must be called after codec has been opened
*/
double FFmpeg::Vid::guessDuration ( AVFormatContext *pFormatCtx, int index )
{
    double duration = (double) pFormatCtx->duration / AV_TIME_BASE; // can be incorrect for .vob files
    if (duration > 0) {
        return duration;
    }
    AVStream *pStream = pFormatCtx->streams[index];
    double guess;

    // if stream bitrate is known we'll interpolate from file size.
    // pFormatCtx->start_time would be incorrect for .vob file with multiple titles.
    // pStream->start_time doesn't work either. so we'll need to disable timestamping.
    if ( NULL == pStream || NULL == pStream->codec )
      return 0.0;

    if (pStream->codec->bit_rate > 0 && pFormatCtx->file_size > 0) {
        guess = 0.9 * pFormatCtx->file_size / (pStream->codec->bit_rate / 8);
        if (guess > 0) {
            av_log(NULL, AV_LOG_ERROR, "  ** duration is unknown: %.2f; guessing: %.2f s from bit_rate\n", duration, guess);
            return guess;
        }
    }
    return -1;
}
/* NON Seeking version.
QImage &FFmpeg::Vid::getFrame ( double fStreamPos )
{
  if ( ( fStreamPos < 0.0f ) || ( fStreamPos > 1.0f ) || ( ! m_pVideoCodecCtx ) )  {
    // TODO: Create Error Image ...
    m_frame = QImage ( );
    return m_frame;
  }

  AVPacket   packet;
  AVFrame   *pFrame    = NULL;
  bool bFound = false;

  int iFrameFinished;
  int iWidth, iHeight, iNumBytes;

  pFrame    = avcodec_alloc_frame ( );
  iWidth    = m_pVideoCodecCtx->width;
  iHeight   = m_pVideoCodecCtx->height;
  // Determine required buffer size and allocate buffer
  iNumBytes = avpicture_get_size ( PIX_FMT_RGB32, iWidth, iHeight );

  while ( !bFound && av_read_frame ( m_pFormatCtx, &packet ) >= 0 )  {
    // Decode video frame
    avcodec_decode_video ( m_pVideoCodecCtx, pFrame, &iFrameFinished, packet.data, packet.size );

    // Did we get a video frame?
    if ( iFrameFinished )  {
      getFrame ( pFrame, iWidth, iHeight );
      bFound = true;
    }

    // Free the packet that was allocated by av_read_frame
    av_free_packet ( &packet );
  }

  if ( pFrame )
    av_free ( pFrame );

  return m_frame;
}
*/

bool FFmpeg::Vid::isValidStream ( )
{
  return ( m_pVideoCodecCtx != NULL );
}

//////////////////////////////////////////////////
//
//  FFmpeg - Encoder Class
//
//////////////////////////////////////////////////
FFmpeg::FFmpeg  ( )
      : Encoder ( )
{
  m_pOutputCtx   = NULL;
  m_pVideoStream = NULL;
  m_pAudioStream = NULL;
  m_pAudioBuffer = NULL;
  m_pVideoBuffer = NULL;
  m_pFrame       = NULL;
  m_pSamples     = NULL;
  m_pInputAudio  = NULL;
  m_iSampleSize  = 0;

  // must be called before using avcodec lib
  avcodec_init ( );

  // register all the codecs
  avcodec_register_all ( );
  av_register_all      ( );
  av_log_set_level     ( AV_LOG_QUIET );
}

FFmpeg::~FFmpeg ( )
{
  if ( m_pVideoBuffer )
    delete []m_pVideoBuffer;
  m_pVideoBuffer = NULL;

  endStream ( );
}

void FFmpeg::lockEngine ( )
{
  FFmpeg::m_lock.lock ( );
}

void FFmpeg::unlockEngine ( )
{
  FFmpeg::m_lock.unlock ( );
}

bool FFmpeg::initStream ( QString qsFileName, enVideo videoFormat, enAudio audioFormat, uint iFrames )
{
  Encoder::initStream ( qsFileName, videoFormat, audioFormat, iFrames );
  endStream ( );
  AVOutputFormat *pOutputFormat = guess_format ( "dvd", NULL, NULL );
  if ( audioFormat == afUndef )
    pOutputFormat = guess_format ( NULL, (const char *)qsFileName.toUtf8 ( ), NULL );

  if ( ! pOutputFormat )
    return false;

  pOutputFormat->video_codec = CODEC_ID_MPEG2VIDEO;
  if ( audioFormat == afUndef )
    pOutputFormat->audio_codec = CODEC_ID_NONE;
  else if (audioFormat == afAC3 )
    pOutputFormat->audio_codec = CODEC_ID_AC3;
  else
    pOutputFormat->audio_codec = CODEC_ID_MP2;

  m_pOutputCtx = av_alloc_format_context ( );
  if ( ! m_pOutputCtx )
    return false;

  m_pOutputCtx->oformat = pOutputFormat;
  snprintf ( m_pOutputCtx->filename, sizeof ( m_pOutputCtx->filename ), "%s", (const char *)qsFileName.toUtf8 ( ) );

  // add video and audio streams
  if ( ! addVideoStream ( pOutputFormat->video_codec, videoFormat, m_iVideoBitrate ) )
    return false;
  if ( ! addAudioStream ( pOutputFormat->audio_codec ) )
    return false;

  if ( av_set_parameters ( m_pOutputCtx, NULL ) < 0 )
    return false;

  //dump_format ( m_pOutputCtx, 0, (const char*)qsFileName.toUtf8 ( ), 1 );
  m_pOutputCtx->packet_size = 2048;

  // open the audio and video codecs and allocate the necessary encode buffers
  if ( m_pVideoStream  )
    OpenVideoEncoder ( );
  if ( m_pAudioStream  )
    OpenAudioEncoder ( );

  // open the output file
  if ( url_fopen ( &m_pOutputCtx->pb, (const char *)qsFileName.toUtf8 ( ), URL_WRONLY ) < 0 )  {
    printf ( "Could not open '%s'\n", (const char *)qsFileName.toUtf8 ( ) );
    return false;
  }
  // write the stream header
  m_pOutputCtx->packet_size = 2048;
  m_pOutputCtx->mux_rate    = 10080000;
  av_write_header ( m_pOutputCtx );

  // The last step we init all input audio streams ...
  bool bOkay = false;
  for ( int t=0; t<m_audioList.count ( ); t++ )  {
    Audio *pAudio = new Audio ( m_audioList[t], m_iSampleSize, m_pAudioStream );
    m_listInputAudio.append   ( pAudio );
    if ( ! bOkay )  {
      bOkay = pAudio->openStream ( );
      if ( bOkay )
        m_pInputAudio = pAudio;
    }
  }

  return true;
}

/**
 * RGBtoYUV420P function is from Gnomemeeting
 */
#define rgbtoyuv(r, g, b, y, u, v) \
  y=(uint8_t)(((int)30*r   +(int)59*g +(int)11*b)/100); \
  u=(uint8_t)(((int)-17*r  -(int)33*g +(int)50*b+12800)/100); \
  v=(uint8_t)(((int)50*r   -(int)42*g -(int)8*b+12800)/100); \

void FFmpeg::RGBtoYUV420P ( const uint8_t *pRGB, uint8_t *pYUV, uint iRGBIncrement, bool bSwapRGB, int iWidth, int iHeight,  bool bFlip )
{
  const unsigned iPlaneSize = iWidth * iHeight;
  const unsigned iHalfWidth = iWidth >> 1;

  // get pointers to the data
  uint8_t *yplane  = pYUV;
  uint8_t *uplane  = pYUV + iPlaneSize;
  uint8_t *vplane  = pYUV + iPlaneSize + (iPlaneSize >> 2);
  const uint8_t *pRGBIndex = pRGB;
  int iRGBIdx[3];
  iRGBIdx[0] = 0;
  iRGBIdx[1] = 1;
  iRGBIdx[2] = 2;
  if ( bSwapRGB )  {
    iRGBIdx[0] = 2;
    iRGBIdx[2] = 0;
  }

  for (int y = 0; y < (int) iHeight; y++) {
    uint8_t *yline  = yplane + (y * iWidth);
    uint8_t *uline  = uplane + ((y >> 1) * iHalfWidth);
    uint8_t *vline  = vplane + ((y >> 1) * iHalfWidth);

    if ( bFlip ) // Flip horizontally
      pRGBIndex = pRGB + ( iWidth * ( iHeight -1 -y ) * iRGBIncrement );

    for ( int x=0; x<iWidth; x+=2 ) {
      rgbtoyuv ( pRGBIndex[iRGBIdx[0]], pRGBIndex[iRGBIdx[1]], pRGBIndex[iRGBIdx[2]], *yline, *uline, *vline );
      pRGBIndex += iRGBIncrement;
      yline++;
      rgbtoyuv ( pRGBIndex[iRGBIdx[0]], pRGBIndex[iRGBIdx[1]], pRGBIndex[iRGBIdx[2]], *yline, *uline, *vline );
      pRGBIndex += iRGBIncrement;
      yline++;
      uline++;
      vline++;
    }
  }
}

void FFmpeg::endStream ( )
{
  CloseVideoEncoder ( );
  CloseAudioEncoder ( );

  if ( ! m_pOutputCtx )
    return;

  // write the trailer
  av_write_trailer ( m_pOutputCtx );

  // free the streams
  unsigned int t;
  for  ( t=0; t<m_pOutputCtx->nb_streams; t++ )  {
    av_freep ( &m_pOutputCtx->streams[t]->codec );
    av_freep ( &m_pOutputCtx->streams[t] );
  }

  // close the output file
#if LIBAVFORMAT_VERSION_INT >= (52<<16)
  url_fclose (  m_pOutputCtx->pb );
#else
  url_fclose ( &m_pOutputCtx->pb );
#endif

  // free the stream
  av_free ( m_pOutputCtx );
  m_pOutputCtx = NULL;

  // Free input audio streams ...
  for ( int t=0; t<m_listInputAudio.count ( ); t++ )
    delete m_listInputAudio[t];
  m_listInputAudio.clear ( );
}

bool FFmpeg::addVideoStream ( int iCodecID, enVideo videoFormat, int iBitrate )
{
  if ( iCodecID == CODEC_ID_NONE )  {
    m_pVideoStream = NULL;
    return true;
  }

  m_pVideoStream = av_new_stream ( m_pOutputCtx, 0 );
  if ( ! m_pVideoStream )
    return false;

  AVCodecContext *pVideo  = m_pVideoStream->codec;
  pVideo->codec_id        = (CodecID)iCodecID;
  pVideo->codec_type      = CODEC_TYPE_VIDEO;
  pVideo->bit_rate        = iBitrate * 1024;
  pVideo->width           = 720;
  pVideo->sample_aspect_ratio.den = 4;
  pVideo->sample_aspect_ratio.num = 3;

// The following field was introduced in 52.21.0
#if LIBAVFORMAT_VERSION_INT >= AV_VERSION_INT(52,21,0)
//#if LIBAVFORMAT_VERSION_MAJOR >= 52
//#if LIBAVFORMAT_VERSION_MINOR >= 21
  m_pVideoStream->sample_aspect_ratio.den = 4;
  m_pVideoStream->sample_aspect_ratio.num = 3;
//#endif
#endif

//  pVideo->dtg_active_format = FF_DTG_AFD_4_3; only used for decoding

  if ( videoFormat == vfPAL )  {
    pVideo->height        = 576;
    pVideo->time_base.den =  25;
    pVideo->time_base.num =   1;
    pVideo->gop_size      =  15;
  }
  else  {
    pVideo->height        =   480;
    pVideo->time_base.den = 30000;
    pVideo->time_base.num =  1001;
    pVideo->gop_size      =    18;
  }
  pVideo->pix_fmt         = PIX_FMT_YUV420P;
  pVideo->rc_buffer_size  = VIDEO_BUF_SIZE;
  pVideo->rc_max_rate     = 9 * 1024 * 1024;
  pVideo->rc_min_rate     = 0;

  return true;
}

bool FFmpeg::OpenAudioEncoder ( )
{
  AVCodecContext *pAudio = m_pAudioStream->codec;

  // find the audio encoder and open it
  AVCodec *pCodec = avcodec_find_encoder ( pAudio->codec_id );
  if ( ! pCodec )  {
    printf ( "Audio codec not found" );
    return false;
  }
  lockEngine ( );
  if ( avcodec_open ( pAudio, pCodec ) < 0 )  {
    unlockEngine ( );
    printf ("Could not open audio codec" );
    return false;
  }
  unlockEngine ( );

  m_pAudioBuffer = (uint8_t *) av_malloc ( AUDIO_BUF_SIZE );

  int iAudioInputFrameSize = pAudio->frame_size;
  // ugly hack for PCM codecs (will be removed ASAP with new PCM
  // support to compute the input frame size in samples
  if ( pAudio->frame_size <= 1 )  {
    iAudioInputFrameSize = AUDIO_BUF_SIZE / pAudio->channels;
    switch ( m_pAudioStream->codec->codec_id ) {
    case CODEC_ID_PCM_S16LE:
    case CODEC_ID_PCM_S16BE:
    case CODEC_ID_PCM_U16LE:
    case CODEC_ID_PCM_U16BE:
      iAudioInputFrameSize >>= 1;
    break;
    default:
      break;
    }
  }

  m_iSampleSize = iAudioInputFrameSize * 2 * pAudio->channels;
  m_pSamples    = (int16_t*) av_malloc ( m_iSampleSize );
  // Zero out the audio so we have silence ...
  memset ( m_pSamples, 0, m_iSampleSize );
  return true;
}

void FFmpeg::CloseAudioEncoder ( )
{
  if ( ! m_pAudioStream )
    return;
  lockEngine    ( );
  avcodec_close ( m_pAudioStream->codec );
  unlockEngine  ( );

  if ( m_pSamples )
    av_free ( m_pSamples );
  m_pSamples = NULL;

  if ( m_pAudioBuffer )
    av_free ( m_pAudioBuffer );

  m_pAudioBuffer = NULL;
  m_pAudioStream = NULL;
}

bool FFmpeg::OpenVideoEncoder ( )
{
  AVCodecContext *pVideo = m_pVideoStream->codec;

  // find the video encoder and open it
  AVCodec *pCodec = avcodec_find_encoder ( pVideo->codec_id );
  if ( ! pCodec )  {
    printf ( "Video codec not found" );
    return false;
  }

  lockEngine ( );
  if ( avcodec_open ( pVideo, pCodec ) < 0 )  {
    unlockEngine ( );
    printf ( "Could not open video codec" );
    return false;
  }
  unlockEngine ( );

  m_pVideoBuffer = (uint8_t*) av_malloc ( VIDEO_BUF_SIZE );

  // allocate the encoded raw picture
  m_pFrame = m_vid.allocPicture ( m_pVideoBuffer, pVideo->pix_fmt, pVideo->width, pVideo->height );
  if ( ! m_pFrame )  {
    printf ( "Could not allocate picture" );
    return false;
  }

  // The following settings will prevent warning messages from FFmpeg
  float fMuxPreload  = 0.5f;
  float fMuxMaxDelay = 0.7f;
  //For svcd you might set it to:
  //mux_preload= (36000+3*1200) / 90000.0; //0.44
  m_pOutputCtx->preload   = (int)( fMuxPreload  * AV_TIME_BASE );
  m_pOutputCtx->max_delay = (int)( fMuxMaxDelay * AV_TIME_BASE );

  return true;
}

void FFmpeg::CloseVideoEncoder ( )
{
  if ( ! m_pVideoStream )
    return;

  lockEngine    ( );
  avcodec_close ( m_pVideoStream->codec );
  unlockEngine  ( );

  if ( m_pFrame )  {
    if ( m_pFrame->data[0] )
      av_free ( m_pFrame->data[0] );
    av_free ( m_pFrame );
    m_pFrame = NULL;
  }

  if ( m_pVideoBuffer )
    av_free ( m_pVideoBuffer );
  m_pVideoBuffer = NULL;
  m_pVideoStream = NULL;
}

bool FFmpeg::addAudioStream ( int iCodecID )
{
  if ( iCodecID == CODEC_ID_NONE ) {
    m_pAudioStream = NULL;
    return true;
  }

  m_pAudioStream = av_new_stream ( m_pOutputCtx, 1 );
  if ( ! m_pAudioStream )
    return false;

  AVCodecContext *pAudio = m_pAudioStream->codec;
  pAudio->codec_id       = (CodecID)iCodecID;
  pAudio->codec_type     = CODEC_TYPE_AUDIO;
  pAudio->bit_rate       = 448000;
  pAudio->sample_rate    = 48000;
  pAudio->channels       = 2;

  pAudio->frame_number   = 0;
  pAudio->time_base.num  = 1;
  pAudio->time_base.den  = pAudio->sample_rate;
  m_pAudioStream->time_base.num = pAudio->time_base.num;
  m_pAudioStream->time_base.den = pAudio->time_base.den;

  return true;
}

void FFmpeg::nextInputAudio ( )
{
  bool bFound = false;
  QList<Audio *>::iterator it = m_listInputAudio.begin ( );
  while  ( it != m_listInputAudio.end ( ) )  {
    if   ( m_pInputAudio == *it++ )  {
      if ( it != m_listInputAudio.end ( ) )  {
        m_pInputAudio->closeStream ( );
        m_pInputAudio = *it;
        m_pInputAudio->openStream  ( );
        bFound = true;
        break; // out of the while loop
      }
    }
  }
//  if ( ! bFound )
//    m_pInputAudio = NULL;
}

FFmpeg::Buffer *FFmpeg::decodeAudio ( float fVolume )
{
  FFmpeg::Buffer *pBuf = NULL;
  if ( m_pInputAudio )  {
    bool bEOF;
    pBuf = m_pInputAudio->decodeAudio ( bEOF );
    if ( bEOF )
      nextInputAudio ( );
  }

  // Next we take care of the volume
  if ( pBuf )  {
    if ( ( fVolume >= 0.0f ) && ( fVolume < 1.0f ) )  {
      short *pBuffer = pBuf->pBuffer;
      for ( int t=0; t<pBuf->iSamplesPerChannel * pBuf->iChannels; t++ )  {
        *pBuffer = (short)(fVolume * *pBuffer );
         pBuffer++;
      }
    }
  }
  return pBuf;
}

bool FFmpeg::writeAudioFrame ( float fVolume )
{
  AVPacket pkt;
  av_init_packet ( &pkt );
  AVCodecContext *pAudioCodec = m_pAudioStream->codec;

  if ( m_pInputAudio )  {
    Buffer *pSamples = decodeAudio ( fVolume );
    unsigned int iSize = m_pInputAudio->getTotalSamples ( );
    pkt.size = pAudioCodec->codec->encode ( pAudioCodec, m_pAudioBuffer, iSize, (void *)pSamples->pBuffer );
  }
  else  {
    int   iSize   = m_iSampleSize * sizeof ( short );
    char *pBuffer = new char [ iSize ];
    memset ( pBuffer, 0, iSize );
    pkt.size = pAudioCodec->codec->encode ( pAudioCodec, m_pAudioBuffer, iSize, (void *)pBuffer );
    delete []pBuffer;
  }

  pAudioCodec->frame_number++;
  pkt.pts  = av_rescale_q ( pAudioCodec->coded_frame->pts, pAudioCodec->time_base, m_pAudioStream->time_base );
  pkt.flags |= PKT_FLAG_KEY;
  pkt.stream_index = m_pAudioStream->index;
  pkt.data         = m_pAudioBuffer;

  // write the compressed frame in the media file
  if ( av_write_frame ( m_pOutputCtx, &pkt ) != 0 )  { // av_interleaved_write_frame
    printf ( "Error while writing audio frame" );
    return false;
  }

  return true;
}

bool FFmpeg::writeVideoFrame ( )
{
  AVCodecContext *pVideo = m_pVideoStream->codec;

  // encode the image
  int  out_size = avcodec_encode_video ( pVideo, m_pVideoBuffer, VIDEO_BUF_SIZE, m_pFrame );
  if ( out_size < 0 )
    return false;
  // if zero size, it means the image was buffered
  if ( out_size > 0 ) {
    AVPacket pkt;
    av_init_packet ( &pkt );

    pkt.pts= av_rescale_q ( pVideo->coded_frame->pts, pVideo->time_base, m_pVideoStream->time_base );
    if ( pVideo->coded_frame->key_frame )
      pkt.flags     |= PKT_FLAG_KEY;

    pkt.stream_index = m_pVideoStream->index;
    pkt.data         = m_pVideoBuffer;
    pkt.size         = out_size;

//printf ( " ... addVideoFrame : OutVideoStream -> CurDts=%lld\n", m_pOutputCtx->streams[0]->cur_dts );
    // write the compressed frame in the media file
    int  ret = av_write_frame ( m_pOutputCtx, &pkt );
    if ( ret != 0 ) {
      printf ( "Error while writing video frame" );
      return false;
    }
  }
  return true;
}

bool FFmpeg::addImage ( QImage *pImage, int iFrames, float fVolume )
{
  if ( ! pImage )
    return false;

  AVCodecContext *pVideo = m_pVideoStream->codec;

  // Allocate buffer for FFMPeg ...
  int iWidth, iHeight;
  int iSize = pImage->width  ( ) * pImage->height ( );
  uint8_t    *pBuffer = new uint8_t [ ( ( iSize * 3 ) / 2 ) + 100 ]; // 100 bytes extra buffer
  iWidth    = pImage->width  ( );
  iHeight   = pImage->height ( );

  m_pFrame->data[0] = pBuffer;
  m_pFrame->data[1] = m_pFrame->data[0] + iSize;
  m_pFrame->data[2] = m_pFrame->data[1] + iSize / 4;
  m_pFrame->linesize[0] = iWidth;
  m_pFrame->linesize[1] = iWidth / 2;
  m_pFrame->linesize[2] = m_pFrame->linesize[1];

  // Copy data over from the QImage. Convert from 32bitRGB to YUV420P
  RGBtoYUV420P ( pImage->bits ( ), pBuffer, pImage->depth ( ) / 8, true, iWidth, iHeight );

  double fDuration = ((double) m_pVideoStream->pts.val ) * m_pVideoStream->time_base.num / m_pVideoStream->time_base.den + ((double) iFrames) * pVideo->time_base.num / pVideo->time_base.den;

  while ( true )  {
    double fDeltaVideo = (double)m_pVideoStream->time_base.num / m_pVideoStream->time_base.den;
    double fAudioPts   = m_pAudioStream ? ((double) m_pAudioStream->pts.val) * m_pAudioStream->time_base.num / m_pAudioStream->time_base.den : 0.0;
    double fVideoPts = ((double) m_pVideoStream->pts.val) * fDeltaVideo;

    if ( ( ! m_pAudioStream || fAudioPts >= fDuration ) &&
         ( ! m_pVideoStream || fVideoPts >= fDuration ) )
      break;

    // write interleaved audio and video frames
    if ( m_pAudioStream  &&  ( fAudioPts < fVideoPts ))  {  //+ fDeltaVideo ) )  {
      if ( ! writeAudioFrame ( fVolume ) )
        return false;
    }
    else {
      if ( ! writeVideoFrame ( ) )
        return false;
    }
  }

  delete pBuffer;
  m_pFrame->data[0] = NULL;
  return true;
}

bool FFmpeg::addVid ( QString qsFileName, int iTargetWidth, int iTargetHeight )
{
  // This function transcodes the VID into mpeg2 compatible material and adds it to the stream.
  // First cause of action is to ensure we have already an open stream with m_vid.
  if ( ! m_vid.isValidStream ( ) || ( m_vid.streamName ( ) != qsFileName ) )  {
    closeVid ( );
    initVid  ( qsFileName );
    if ( ! m_vid.isValidStream ( ) )  {
      closeVid ( ); // Error could not recover ...
      return false;
    }
  }
  m_vid.setSWScale ( iTargetWidth, iTargetHeight );

  double fTargetFrameRate = (double)m_pVideoStream->codec->time_base.den / m_pVideoStream->codec->time_base.num;
  double fSourceFrameRate = m_vid.getFrameRate ( ); //10.0;
  double fAdjustFrameRate =  0.0;
  int iAdjustFrameRate = 0;

  // At this point we are all set to start the transcoding of the input audio / video.
  AVPacket *pPacket = m_vid.getNextPacket ( m_pVideoStream, m_pAudioStream, iAdjustFrameRate );
  while  (  pPacket )  {
    // write the compressed frame in the media file

    //int  ret = av_interleaved_write_frame ( m_pOutputCtx, pPacket );
    int  ret = av_write_frame ( m_pOutputCtx, pPacket );
    if ( ret != 0 ) {
      printf ( "Error while writing frame\n" );
      return false;
    }

    if ( iAdjustFrameRate == 0 )  {
         // Last frame was taken from the steam. Lets see if we have to insert
         // some extra frame to adjust the frame rate.
         if ( fTargetFrameRate  / fSourceFrameRate < 1.0 )
              fAdjustFrameRate -= fSourceFrameRate / fTargetFrameRate - 1.0; // E.g. 29.97 / 25 = NTSC to PAL = -0.2
         else
              fAdjustFrameRate += fTargetFrameRate / fSourceFrameRate - 1.0; // E.g. 29.97 / 10 = AVI to NTSC =  1.997
         iAdjustFrameRate  = (int)fAdjustFrameRate;
    }
    if ( fAdjustFrameRate >= 1.0 )
         fAdjustFrameRate -= 1.0;
    else if ( fAdjustFrameRate <= -1.0 )
              fAdjustFrameRate +=  1.0;
    else if ( iAdjustFrameRate == 1 )// Between [ -1.0 .. 1.0 ]
              iAdjustFrameRate --;
    pPacket = m_vid.getNextPacket ( m_pVideoStream, m_pAudioStream, iAdjustFrameRate );
    // iAdjusFrame is decremented by one each time a frame is re-used
    // iAdjusrtFrame is set to 0 when a new frame is taken from the stream.
  }
  return true;
}

bool FFmpeg::initVid ( QString qsFileName )
{
  bool bRet = m_vid.openStream ( qsFileName );
  m_vid.setOutputContext ( m_pVideoStream->codec, m_pAudioStream->codec );
  return bRet;
}

void FFmpeg::closeVid ( )
{
  m_vid.closeStream ( );
}

QImage &FFmpeg::getVidFrame ( double fPosition )
{
  return m_vid.getFrame ( fPosition );
}

}; // end of namespace Encoder

