/****************************************************************************
** DialogRegion class
**
**   Created : Mon May 20 22:48:00 2005
**        by : Varol Okan using Kate
** Copyright : (c) Varol Okan
**   License : GPL v 2.0
**
** The image dvd_region.png has only 10 colors.
** The following order is the color table entry vs the region code.
** color[0] = Background transarency
** color[1] = Black, don't use, text
** color[2] = Yellow * Region 3 - Taiwan, Korea, the Philippines, Indonesia, Hong Kong
** color[3] = Purple * Region 6 - China
** color[4] = redish * Region 1 - The U.S., U.S. territories and Canada
** color[5] = green  * Region 4 - Mexico, South America, Central America, Australia, New Zealand, Pacific Islands, Caribbean
** color[6] = orange * Region 2 - Europe, Japan, the Middle East, Egypt, South Africa, Greenland
** color[7] = lightblue * Region 5 - Russia (okay, former Russia), Eastern Europe, India, most of Africa, North Korea, Mongolia
** color[8] = pink,  don't use, border lines
** color[9] = white, don't use, south pole
**
****************************************************************************/

#include <qlabel.h>
#include <qpixmap.h>
#include <qradiobutton.h>

#include "dialogregion.h"

RegionPreview::RegionPreview ( DialogRegion *pDialog, QWidget *pParent )
  : QLabel ( pParent )
{
  setMouseTracking ( true );
  setPaletteBackgroundColor ( QColor ( 0x93, 0xA4, 0xEE ) );
  m_pDialog = pDialog;
  QPixmap pix = QPixmap::fromMimeSource ( "dvd_regions.png" );
  setFixedSize ( pix.size ( ) );
  pParent->setFixedSize ( pix.size ( ) );
  setPixmap ( pix );
}

RegionPreview::~RegionPreview ( )
{
  setMouseTracking ( false );
}

void RegionPreview::mouseMoveEvent ( QMouseEvent *pEvent )
{
  m_pDialog->mouseMove ( pEvent->pos ( ) );
}

void RegionPreview::mousePressEvent ( QMouseEvent *pEvent )
{
  if ( pEvent->button ( ) == LeftButton )
    m_pDialog->mouseClick ( pEvent->pos ( ) );
}

DialogRegion::DialogRegion ( QWidget *pParent, int iRegionCode )
          : uiDialogRegion ( pParent )
{
  initMe ( iRegionCode );
}

DialogRegion::~DialogRegion ( )
{
}

void DialogRegion::initMe ( int iRegionCode )
{
  // First we need to get the QImage to handle the mouse events
  m_image = QImage::fromMimeSource ( "dvd_regions.png" );
  // map the color table entries with the Region Code
  m_colorIdxToRegion[0] = 0;
  m_colorIdxToRegion[1] = 0;
  m_colorIdxToRegion[2] = 3;
  m_colorIdxToRegion[3] = 6;
  m_colorIdxToRegion[4] = 1;
  m_colorIdxToRegion[5] = 4;
  m_colorIdxToRegion[6] = 2;
  m_colorIdxToRegion[7] = 5;
  m_colorIdxToRegion[8] = 0;
  m_colorIdxToRegion[9] = 0;

  // original colors
  m_colors[0][0] = 0x00000000; // all black, don't care
  m_colors[0][1] = 0xFFFA765F; // m_image.color ( m_colorToRegion[1] ); // & 0x00FFFFFF; // 4 );
  m_colors[0][2] = 0xFFFCA603; // m_image.color ( m_colorToRegion[2] ); // & 0x00FFFFFF; // 6 );
  m_colors[0][3] = 0xFFEFF30D; // m_image.color ( m_colorToRegion[3] ); // & 0x00FFFFFF; // 2 );
  m_colors[0][4] = 0xFF78BA30; // m_image.color ( m_colorToRegion[4] ); // & 0x00FFFFFF; // 5 );
  m_colors[0][5] = 0xFFA0CAF2; // m_image.color ( m_colorToRegion[5] ); // & 0x00FFFFFF; // 7 );
  m_colors[0][6] = 0xFFCE5EFF; // m_image.color ( m_colorToRegion[6] ); // & 0x00FFFFFF; // 3 );

  // mouse over colors
  m_colors[1][0] = 0x00000000; // all black, don't care
  m_colors[1][1] = 0xFFFAA68F; // m_image.color ( m_colorToRegion[1] ); // & 0x00FFFFFF; // 4 );
  m_colors[1][2] = 0xFFFCD633; // m_image.color ( m_colorToRegion[2] ); // & 0x00FFFFFF; // 6 );
  m_colors[1][3] = 0xFFFFF33D; // m_image.color ( m_colorToRegion[3] ); // & 0x00FFFFFF; // 2 );
  m_colors[1][4] = 0xFFA8EA60; // m_image.color ( m_colorToRegion[4] ); // & 0x00FFFFFF; // 5 );
  m_colors[1][5] = 0xFFD0FAF2; // m_image.color ( m_colorToRegion[5] ); // & 0x00FFFFFF; // 7 );
  m_colors[1][6] = 0xFFFE8EFF; // m_image.color ( m_colorToRegion[6] ); // & 0x00FFFFFF; // 3 );

  m_iPreviousColorIdx = 0;
  m_iSelectedRegion   = 0;

  // Set propper color table
  resetColors ( );

  QPixmap pix;
  pix.convertFromImage  ( m_image, Qt::ColorOnly | Qt::ThresholdDither | Qt::AvoidDither  );
  m_pPreview = new RegionPreview ( this, m_pLabelPreview );
  m_pPreview->setPixmap ( pix );

  setRegionCode ( iRegionCode );

  connect ( m_pRadioAll, SIGNAL ( clicked ( ) ), this, SLOT ( slotRadio ( ) ) );
  connect ( m_pRadio1,   SIGNAL ( clicked ( ) ), this, SLOT ( slotRadio ( ) ) );
  connect ( m_pRadio2,   SIGNAL ( clicked ( ) ), this, SLOT ( slotRadio ( ) ) );
  connect ( m_pRadio3,   SIGNAL ( clicked ( ) ), this, SLOT ( slotRadio ( ) ) );
  connect ( m_pRadio4,   SIGNAL ( clicked ( ) ), this, SLOT ( slotRadio ( ) ) );
  connect ( m_pRadio5,   SIGNAL ( clicked ( ) ), this, SLOT ( slotRadio ( ) ) );
  connect ( m_pRadio6,   SIGNAL ( clicked ( ) ), this, SLOT ( slotRadio ( ) ) );
}

void DialogRegion::changeSelectedRegion ( uint iColorIdx )
{
  if ( iColorIdx > 9 )
    return;

  setRegionCode ( m_colorIdxToRegion[iColorIdx] );
  changeRegion  ( iColorIdx );
}

void DialogRegion::changeRegion ( uint iColorIdx )
{
  if ( iColorIdx > 9 )
    return;

  uint iRegionCode = m_colorIdxToRegion[iColorIdx];

  updatePixmap ( iRegionCode );
  m_iPreviousColorIdx = iColorIdx;
}

void DialogRegion::updatePixmap ( uint iRegionCode )
{
  QPixmap pix;
  resetColors ( );

  const int regionToColorIdx[7] = { 0, 4, 6, 2, 5, 7, 3 };
  if ( iRegionCode != m_iSelectedRegion )
    m_image.setColor ( regionToColorIdx[iRegionCode], m_colors[1][iRegionCode] );

  pix.convertFromImage  ( m_image, Qt::ColorOnly | Qt::ThresholdDither | Qt::AvoidDither  );
  m_pPreview->setPixmap ( pix );
}

void DialogRegion::resetColors ( )
{
  const int regionToColorIdx[7] = { 0, 4, 6, 2, 5, 7, 3 };
  // reset color table
  for ( uint t=1; t<7; t++ )  {
    if ( t == m_iSelectedRegion )
      m_image.setColor ( regionToColorIdx[t], 0xFFFF0000 );
    else
      m_image.setColor ( regionToColorIdx[t], m_colors[0][t] );
  }
}

void DialogRegion::setRegionCode ( int iRegionCode )
{
  switch ( iRegionCode )  {
  case 1:
    m_pRadio1->setChecked ( true );
  break;
  case 2:
    m_pRadio2->setChecked ( true );
  break;
  case 3:
    m_pRadio3->setChecked ( true );
  break;
  case 4:
    m_pRadio4->setChecked ( true );
  break;
  case 5:
    m_pRadio5->setChecked ( true );
  break;
  case 6:
    m_pRadio6->setChecked ( true );
  break;
  default:
    m_pRadioAll->setChecked ( true );
  break;
  }
  m_iSelectedRegion = iRegionCode;
}

void DialogRegion::slotRadio ( )
{
  m_iSelectedRegion = regionalZone ( );
  resetColors  ( );
  updatePixmap ( m_iSelectedRegion );
}

int DialogRegion::regionalZone ( )
{
  // Returns the region code to the calling function.
  if ( m_pRadioAll->isOn ( ) )
    return 0;
  if ( m_pRadio1->isOn ( ) )
    return 1;
  if ( m_pRadio2->isOn ( ) )
    return 2;
  if ( m_pRadio3->isOn ( ) )
    return 3;
  if ( m_pRadio4->isOn ( ) )
    return 4;
  if ( m_pRadio5->isOn ( ) )
    return 5;
  if ( m_pRadio6->isOn ( ) )
    return 6;

  return 0;
}

void DialogRegion::mouseMove ( const QPoint &pos )
{
  if ( ( pos.x ( ) < 0 ) || ( pos.x ( ) >= m_image.width  ( ) ) )
    return;
  if ( ( pos.y ( ) < 0 ) || ( pos.y ( ) >= m_image.height ( ) ) )
    return;

  uint iIdx  = (uint)m_image.pixelIndex ( pos.x ( ), pos.y ( ) );
  if ( iIdx != (uint)m_iPreviousColorIdx )  {
    changeRegion ( iIdx );
  }
}

void DialogRegion::mouseClick ( const QPoint &pos )
{
  if ( ( pos.x ( ) < 0 ) || ( pos.x ( ) >= m_image.width  ( ) ) )
    return;
  if ( ( pos.y ( ) < 0 ) || ( pos.y ( ) >= m_image.height ( ) ) )
    return;

  uint iIdx  = (uint)m_image.pixelIndex ( pos.x ( ), pos.y ( ) );
  changeSelectedRegion ( iIdx );
}

