# This file is part of qVamps.
#
# qVamps is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
#
# qVamps is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with qVamps; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


use strict;
use warnings;


package DvdMaker;
use Qt;
use Qt::isa qw (Qt::Object);
use Qt::attributes qw (proc clock proc_running log_viewer
		       elapsed dst_space tgt_size dst_dir
		       counter default_color hilite_errors outbuf);
use Qt::slots read_output => [],
              proc_exited => [];
use QVamps qw (read_setting read_list_setting replace error tr);
use File::Path;


my $t_zero;


# DvdMaker (log_viewer, elapsed, dst_space, tgt_size, dst_dir, parent, name)
sub NEW
{
  my $this       = shift;
  my $log_viewer = shift;
  my $elapsed    = shift;
  my $dst_space  = shift;
  my $tgt_size   = shift;
  my $dst_dir    = shift;

  $this -> SUPER::NEW (@_);

  $t_zero       = Qt::Time ();
  my $proc      = Qt::Process (this, "DvdMaker::proc");
  my $clock     = Qt::Time ();
  proc          = $proc;
  clock         = $clock;
  log_viewer    = $log_viewer;
  elapsed       = $elapsed;
  dst_space     = $dst_space;
  tgt_size      = $tgt_size;
  dst_dir       = $dst_dir;
  counter       = 20;
  default_color = $log_viewer -> paletteForegroundColor ();
  hilite_errors = 0;
  outbuf        = [ "", "" ];

  $proc -> setCommunication (&Qt::Process::Stdout | &Qt::Process::Stderr);

  this -> connect ($proc, SIGNAL "readyReadStdout()", SLOT "read_output()");
  this -> connect ($proc, SIGNAL "readyReadStderr()", SLOT "read_output()");
  this -> connect ($proc, SIGNAL "processExited()", SLOT "proc_exited()");

  # setup timer for periodic 0.5s status update
  startTimer (500) if ($elapsed && $dst_space);

  $log_viewer -> clear ();
  $clock -> start ();
}


sub DESTROY
{
#  print STDERR "DvdMaker::DESTROY\n";

  append_output (0, "\n") if (outbuf -> [0]);
  append_output (1, "\n") if (outbuf -> [1]);

  update_status (1);

  removeChild (proc);
  parent () -> removeChild (this);

  proc          = undef;
  clock         = undef;
  log_viewer    = undef;
  elapsed       = undef;
  dst_space     = undef;
  default_color = undef;
  outbuf        = undef;

  SUPER -> DESTROY ();
}


sub launch
{
  my $args = shift;

  my $proc = proc;

  $proc -> setArguments ($args);

  unless ($proc -> start ())
  {
    error (sprintf '%s: "%s"',
	   &tr ("Failed launching command"), join (" ", @{$args}));

    return 0x200;
  }

  my $event_loop = Qt::Application::eventLoop ();

  proc_running = 1;

  while (proc_running)
  {
    $event_loop -> processEvents (&Qt::EventLoop::WaitForMore);
  }

  read_output ();

  unless ($proc -> normalExit ())
  {
    log_viewer -> setColor (Qt::Color (0xff, 0, 0));
    log_viewer -> append (&tr ("Command execution terminated abnormally"));
    log_viewer -> setColor (default_color);

    return 0x100;
  }

  return $proc -> exitStatus ();
}


sub launch_sh_cmd
{
  my $hilite = shift;
  my $cmd    = shift;

  hilite_errors = $hilite;
  my $lv        = log_viewer;

  $lv -> setColor (Qt::Color (0, 0, 0xff));
  $lv -> append (sprintf "%s \"%s\"\n\n", &tr ("Executing"), $cmd);
  $lv -> setColor (default_color);

  $cmd = "exec $cmd" unless ($cmd =~ /[|;]/);

  return launch ([ "/bin/sh", "-c", $cmd ]);
}


sub try_terminate
{
  proc -> tryTerminate ();
}


sub append_output
{
  my $e = shift;	# 0: stdout, 1: stderr
  my $s = shift;	# string to append

  # flush output, if string contains backspace
  my $flush = ($s =~ s/[\b]+//g);

  # replace return(s) with a newline and remove unwanted lines
  $s =~ s/\r+/\n/g;
  $s =~ s/^libdvdread: .+\n//mg;

  $s             = outbuf -> [$e] . $s;
  outbuf -> [$e] = !$flush && ($s =~ s/(.+)\z//) ? $1 : "";

  my $lv = log_viewer;

  $lv -> setColor (Qt::Color (0xff, 0, 0)) if ($e && hilite_errors);
  $lv -> append ($s);
  $lv -> setColor (default_color);
}


sub read_output : SLOT()
{
  my $proc = proc;

  while (my $s = $proc -> readStdout ())
  {
    append_output (0, $s);
  }

  while (my $s = $proc -> readStderr ())
  {
    append_output (1, $s);
  }
}


sub proc_exited : SLOT()
{
  proc_running = 0;
}


sub run_spumux
{
  my $base_path = shift;
  my $make_menu = shift;
  my $rm_temps  = shift;

  my $rc = 0;

  if ($make_menu)
  {
    my $spumux;
    my $bg_file  = parent () -> background_filename ();
    my $menu_xml = "$base_path/menu/spumux.xml";
    my $menu_mpg = "$base_path/menu.mpg";

    if ($bg_file =~ /\.mpe?g$/)
    {
      $spumux = read_setting ("/Dvdauthor/spumux_mpg");
      $spumux = replace ($spumux, { "i" => $bg_file,
				    "x" => $menu_xml,
				    "o" => $menu_mpg });
    }
    else
    {
      my $video_es = "$base_path/menu/video.mpv";
      my $prefix   = read_setting ("/Installation/prefix");
      my $audio_es = read_setting ("/Menu/Preferences/audio_es");
      $audio_es    = replace ($audio_es, { "p" => $prefix });
      $spumux      = read_setting ("/Dvdauthor/spumux_mpv");
      $spumux      = replace ($spumux, { "v" => $video_es,
					 "a" => $audio_es,
					 "x" => $menu_xml,
					 "o" => $menu_mpg });
    }

    $rc = launch_sh_cmd (0, $spumux);

    rmtree ("$base_path/menu") if ($rm_temps);
  }

  return $rc ? -1 : 0;
}


sub run_dvdauthor
{
  my $base_path = shift;
  my $rm_temps  = shift;

  my $vamps_inj = "$base_path/vamps.inj";

  rmtree ("$base_path/image");
  unlink ($vamps_inj);
  update_status (1);

  my $dvdauthor_xml = "$base_path/dvdauthor.xml";
  my $dvdauthor     = read_setting ("/Dvdauthor/command");
  $dvdauthor        = replace ($dvdauthor, { "x" => $dvdauthor_xml });

  my $rc = launch_sh_cmd (0, $dvdauthor);

  unlink ($vamps_inj) if ($rm_temps);

  return $rc ? -1 : 0;
}


sub run_burn_cmd
{
  my $base_path = shift;
  my $vol_id    = shift;

  my $burner     = read_setting ("/General/DVD_burner");
  my $burn_speed = read_setting ("/General/DVD_burn_speed");
  my $img_dir    = "$base_path/image";
  my $burn_cmds  = read_list_setting ("/Dvdauthor/burn_cmds");
  my $bcmds_item = read_setting ("/Dvdauthor/burn_cmds_item");
  my $burn_cmd   = $burn_cmds -> [$bcmds_item];
  $burn_cmd      = replace ($burn_cmd, { "d" => $burner,
					 "s" => $burn_speed,
					 "v" => $vol_id,
					 "i" => $img_dir });

  my $rc = launch_sh_cmd (1, $burn_cmd);

  return $rc ? -1 : 0;
}


sub write_script
{
  my $base_path   = shift;
  my $make_menu   = shift;
  my $disk_change = shift;
  my $rm_temps    = shift;
  my $vol_id      = shift;

  my $make_menu_bool   = $make_menu   ? "true" : "false";
  my $disk_change_bool = $disk_change ? "true" : "false";
  my $rm_temps_bool    = $rm_temps    ? "true" : "false";
  my $burner           = read_setting ("/General/DVD_burner");
  my $burn_speed       = read_setting ("/General/DVD_burn_speed");
  my $prefix           = read_setting ("/Installation/prefix");
  my $audio_es         = read_setting ("/Menu/Preferences/audio_es");
  $audio_es            = replace ($audio_es, { "p" => $prefix });
  my $bg_file          = parent () -> background_filename ();
  my $menu_ps          = $bg_file =~ /\.mpe?g$/ ? $bg_file : "";
  my $prompt           =
    &tr ("Please insert blank DVD into burner and press <RETURN> key");
  my $spumux_mpv       = read_setting ("/Dvdauthor/spumux_mpv");
  $spumux_mpv          = replace ($spumux_mpv, { "v" => '$video_es',
						 "a" => '$audio_es',
						 "x" => '$menu_xml',
						 "o" => '$menu_mpg' });
  my $spumux_mpg       = read_setting ("/Dvdauthor/spumux_mpg");
  $spumux_mpg          = replace ($spumux_mpg, { "i" => '$menu_ps',
						 "x" => '$menu_xml',
						 "o" => '$menu_mpg' });
  my $dvdauthor        = read_setting ("/Dvdauthor/command");
  $dvdauthor           = replace ($dvdauthor, { "x" => '$dvdauthor_xml' });
  my $burn_cmds        = read_list_setting ("/Dvdauthor/burn_cmds");
  my $bcmds_item       = read_setting ("/Dvdauthor/burn_cmds_item");
  my $burn_cmd         = $burn_cmds -> [$bcmds_item];
  $burn_cmd            = replace ($burn_cmd, { "d" => '$burner',
					       "s" => '$burn_speed',
					       "v" => '$vol_id',
					       "i" => '$img_path' });

  my $file = "$base_path/run_me";

  unless (open SCR, ">", $file)
  {
    error (sprintf "%s: %s: %s",
	   &tr ("Failed creating shell script"), $file, $!);

    return -1;
  }

  print SCR << "EOT";
#! /bin/sh

# flags
make_menu="$make_menu_bool"
disk_change="$disk_change_bool"
rm_temps="$rm_temps_bool"

# settings for burning
burner="$burner"
burn_speed="$burn_speed"
vol_id="$vol_id"

# paths and files
base_path="$base_path"
img_path="\$base_path/image"
dvdauthor_xml="\$base_path/dvdauthor.xml"
menu_mpg="\$base_path/menu.mpg"
vamps_inj="\$base_path/vamps.inj"
menu_path="\$base_path/menu"
menu_xml="\$menu_path/spumux.xml"
video_es="\$menu_path/video.mpv"
audio_es="$audio_es"
menu_ps="$menu_ps"

# other
prompt="$prompt"


if \$make_menu; then
  if [ -z "\$menu_ps" ]; then
    $spumux_mpv
  else
    $spumux_mpg
  fi

  \$rm_temps && rm -rf \$menu_path
fi

rm -rf \$img_path \$vamps_inj

$dvdauthor

\$rm_temps && rm -f \$vamps_inj

\$disk_change && { echo -n "\$prompt"; read x; }

$burn_cmd
EOT

  CORE::close SCR;

  chmod 0777 & ~umask (), $file;

  return 0;
}


# overload function of Qt::Object
sub timerEvent
{
  my $ev = shift;

  update_status (0);
}


sub update_status
{
  my $forced = shift;

  my $elapsed   = elapsed;
  my $dst_space = dst_space;

  return unless ($elapsed && $dst_space);

  $elapsed -> setText ($t_zero ->
		       addMSecs (clock -> elapsed ()) -> toString ("h:mm:ss"));

  my $counter = counter - 1;

  if ($forced || $counter == 0)
  {
    $counter = 20;

    $dst_space -> setText (sprintf "%d/%d MB",
			   dir_files_size (dst_dir) / 1024. / 1024., tgt_size);
  }

  counter = $counter;
}


sub dir_files_size
{
  my $dir = shift;

  return 0 unless (opendir DH, $dir);

  my $rv = 0;

  foreach my $entry (readdir (DH))
  {
    next unless (-f "$dir/$entry");

    my @stat = stat (_);
    $rv     += $stat [7];
  }

  closedir DH;

  return $rv;
}


1;
