# This file is part of qVamps.
#
# qVamps is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# qVamps is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with qVamps; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


use strict;
use warnings;


package FpcMgr;
use Qt;
use Qt::isa qw (Qt::Object);
use Qt::attributes qw (mtable_items title_combo audio_combo
		       subtitle_combo title_list_items ct);
use QVamps qw (search);
use MenuTableItems;


sub NEW
{
  my $this             = shift;
  my $parent           = shift;
  my $name             = shift;
  my $title_combo      = shift;
  my $audio_combo      = shift;
  my $subtitle_combo   = shift;
  my $title_list_items = shift;
  my $mtable_items     = shift;

  $this -> SUPER::NEW ($parent, $name);

  title_combo      = $title_combo;
  audio_combo      = $audio_combo;
  subtitle_combo   = $subtitle_combo;
  title_list_items = $title_list_items;
  mtable_items     = $mtable_items;
}


sub DESTROY
{
  parent () -> removeChild (this);

  ct               = 0;
  mtable_items     = undef;
  title_list_items = undef;
  subtitle_combo   = undef;
  audio_combo      = undef;
  title_combo      = undef;

  SUPER -> DESTROY ();
}


sub update_title
{
  my $nttn            = 0;
  my $title_combo     = title_combo;
  my $ttn             = $title_combo -> currentText ();
  my $selected_titles = title_list_items -> selected_titles ();

  # wipe out all combo box items
  $title_combo -> clear ();
  ct = 0;

  if (@{$selected_titles})
  {
    # list of selected titles not empty
    # --> create combo box items and set new title number to first in list
    $title_combo -> insertStringList ($selected_titles);
    $nttn = $title_combo -> currentText ();
    ct    = $nttn;

    if ($ttn)
    {
      # we had a title selected --> search in titles selected now
      my $i = search ($selected_titles, $ttn);

      unless ($i < 0)
      {
	# title still selected --> restore combo box selection
	$title_combo -> setCurrentItem ($i);
	ct = $ttn;

	# As user may have changed the title's audio and subtitle selection,
	# we need to update the audio and subtitle combo boxes. Even though
	# the title is still selected. Since the currently selected audio
	# and subtitle stream should be preserved in this case, returning
	# the title as negative number here indicates this situation.
	$nttn = -$ttn;
      }
    }
  }

  # adjust width of combo box
  adjust_combo_width ($title_combo);

  return $nttn;
}


sub update_audio
{
  my $ttn = shift;

  my $item;
  my $audio_combo = audio_combo;

  if ($ttn < 0)
  {
    $ttn  = -$ttn;
    $item = $audio_combo -> currentText ();
  }

  $audio_combo -> clear ();

  if ($ttn)
  {
    my $i;
    my $item_hash = mtable_items -> make_audio_item_hash ($ttn);
    my @streams   = sort { $a <=> $b } keys %{$item_hash};
    my @item_list = map $item_hash -> {$_}, @streams;

    if ($item)
    {
      $i = -1;

      for (my $j = 0; $j < @item_list; $j++)
      {
	if ($item eq $item_list [$j])
	{
	  $i = $j;
	  last;
	}
      }
    }
    else
    {
      $i = search (\@streams, mtable_items -> default_audio_stream ());
    }

    $audio_combo -> insertStringList (\@item_list);
    $audio_combo -> setCurrentItem ($i) unless ($i < 0);
  }

  adjust_combo_width ($audio_combo);
}


sub update_subtitle
{
  my $ttn = shift;

  my $item;
  my $subtitle_combo = subtitle_combo;

  if ($ttn < 0)
  {
    $ttn  = -$ttn;
    $item = $subtitle_combo -> currentText ();
  }

  $subtitle_combo -> clear ();

  if ($ttn)
  {
    my $item_hash = mtable_items -> make_subtitle_item_hash ($ttn);
    my @streams   = sort { $a <=> $b } keys %{$item_hash};
    my @item_list = map $item_hash -> {$_}, @streams;

    $subtitle_combo -> insertStringList (\@item_list);

    if ($item)
    {
      for (my $i = 0; $i < @item_list; $i++)
      {
	if ($item eq $item_list [$i])
	{
	  $subtitle_combo -> setCurrentItem ($i);
	  last;
	}
      }
    }
  }

  adjust_combo_width ($subtitle_combo);
}


sub adjust_combo_width
{
  my $combo = shift;

  $combo -> setFont ($combo -> font ());
  $combo -> updateGeometry ();
}


sub update
{
  my $ttn = update_title ();

  update_audio ($ttn);
  update_subtitle ($ttn);
}


sub title_changed
{
  my $ttn = shift;

  unless ($ttn == ct)
  {
    ct = $ttn;
    update_audio ($ttn);
    update_subtitle ($ttn);
  }
}


sub title_nr
{
  my $title_combo = title_combo;

  return 0 unless ($title_combo -> count ());

  return $title_combo -> currentItem () + 1;
}


sub audio_stream
{
  return audio_combo -> currentItem ();
}


sub subtitle_stream
{
  return audio_combo -> currentItem () - 1;
}


1;
