# This file is part of qVamps.
#
# qVamps is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# qVamps is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with qVamps; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


use strict;
use warnings;

package InstallFile;
require Exporter;
our @ISA;
@ISA = qw (Exporter IO::File);


our $call_back;


sub new
{
  my $class  = shift;
  $call_back = shift;

  return $class -> SUPER::new (@_);
}


sub print
{
  my $self = shift;

  &{$call_back} (length (join ("", @_)));

  return $self -> SUPER::print (@_);
}


package Installer;
use Carp;
use File::Path;
use File::Temp;
use Archive::Zip qw (:ERROR_CODES);
use InstCfg;


my @inf;


sub remove_zip_member
{
  my $zip      = shift;
  my $fn       = shift;
  my $manifest = shift;

  $zip -> removeMember ($fn) or croak "$fn: removeMember failed";
  ${$manifest} =~ s/^$fn\n//m;
}


sub install_file
{
  my $zip       = shift;
  my $src       = shift;
  my $dst       = shift;
  my $perm      = shift;
  my $call_back = shift;

  unlink ($dst);

  my $fh     = InstallFile -> new ($call_back, $dst, ">");
  my $member = $zip -> memberNamed ($src);

  $member -> extractToFileHandle ($fh) == AZ_OK or
    croak "$src: extractToFileHandle failed";

  # close file
  undef $fh;

  chmod ($perm, $dst) or croak "$dst: chmod failed";
  unshift @inf, $dst;
}


sub install_path
{
  my $path = shift;
  my $perm = shift;

  foreach my $dir (mkpath ($path))
  {
    chmod ($perm, $dir) or croak "$dir: chmod failed";
    unshift @inf, $dir;
  }
}


sub install_symlink
{
  my $ipath = shift;
  my $lpath = shift;
  my $src   = shift;
  my $dst   = shift;
  my $abs   = shift;

  if ($abs)
  {
    $src = "$ipath/$src" unless ($src =~ m|^/|);
    $dst = "$lpath/$dst" unless ($dst =~ m|^/|);
  }

  unlink ($dst);
  symlink ($src, $dst) or croak "$dst: symlink failed";
  unshift @inf, $dst;
}


sub patch_main_pl
{
  my $zip = shift;
  my $mfr = shift;

  my $main_pl = $zip -> contents ("script/main_switch.pl");
  $main_pl    =~ s/_SCRIPT_/$InstCfg::script/g;
  $zip -> contents ("script/main.pl", $main_pl);
  remove_zip_member ($zip, "script/main_switch.pl", $mfr);
}


sub write_par
{
  my $zip       = shift;
  my $prog_name = shift;
  my $call_back = shift;

  my ($tmpfh, $tmpfn) = File::Temp::tempfile ();

  $tmpfh = InstallFile -> new ($call_back, $tmpfn, ">");

  $zip -> writeToFileHandle ($tmpfh) == AZ_OK or
    croak "$tmpfn: writeToFileHandle failed";

  # close file
  undef $tmpfh;

  system ("$prog_name --par-options -B -O$InstCfg::binpar $tmpfn > /dev/null")
    >> 8 and croak "parl failed";

  unlink ($tmpfn);

  unshift @inf, "$InstCfg::binpar";
}


sub write_inf
{
  my $inst_path = shift;

  unshift @inf, ".InstallInfo";

  open (INF, ">", "$inst_path/.InstallInfo") or
    croak "$inst_path/.InstallInfo: $!";

  print INF join ("\n", @inf);
  print INF "\n";

  close (INF) or croak "$inst_path/.InstallInfo: $!";
}


sub run
{
  my $prog_name = shift;
  my $zip       = shift;
  my $inst_path = shift;
  my $link_path = shift;
  my $groups    = shift;
  my $call_back = shift;

  my $mani_hdle = $zip -> memberNamed ("MANIFEST");
  my $manifest  = $mani_hdle -> contents ();

  # create installation base path
  install_path ($inst_path, $InstCfg::inst_perm);

  # set working directory to installation base path
  chdir ($inst_path) or croak "$inst_path: chdir failed";

  foreach my $item (@InstCfg::items)
  {
    my $group = $item -> {group};

    for ($item -> {type})
    {
      my $src  = $item -> {src};
      my $dst  = $item -> {dst};
      my $perm = $item -> {perm};

      if (/f/)
      {
	# extract file from zip
	install_file ($zip, $src, $dst, $perm, $call_back)
	  if (!$group || grep ($_ eq $group, @{$groups}));

	remove_zip_member ($zip, $src, \$manifest);
	last;
      }

      last if ($group && !grep ($_ eq $group, @{$groups}));

      if (/d/)
      {
	# create path
	install_path ($dst, $perm);
	last;
      }

      if (/s/)
      {
	# create symbolic link
	install_symlink ($inst_path, $link_path, $src, $dst, $item -> {abs});
	last;
      }
    }
  }

  # replace script/main.pl with customized version
  patch_main_pl ($zip, \$manifest);

  # write manifest
  $mani_hdle -> contents ($manifest);

  # write binary perl archive
  write_par ($zip, $prog_name, $call_back);

  # write uninstall info
  write_inf ($inst_path);
}


1;
