ggobi <-
#
# Start a ggobi session
# data can be the name of a file, or data frame or matrix.
#
# Returns the index of the ggobi instance within 
# the session. If an error occurs, -1 is returned.
#
# see setData.ggobi
#     setDataFile.ggobi
#
function(data = NULL, args=character(0), mode=character(0))
{
 args <- c("ggobi", c("--keepalive", as.character(args)))

 if(!missing(data) && is.character(data)) {
  args <- c(args, as.character(mode), data)
  data <- NULL
 }

 ok <- .Call(.ggobi.symbol("init"), args, TRUE)

 if(!is.null(ok)) {
   # So we didn't get an error.
   # Set this new ggobi to be default instance
   # so that subsequent commands will operate
   # on it.
  setDefaultGGobi(ok) 

   #
  if(!missing(data) && !is.null(data))
    setData.ggobi(data, .gobi = ok)
 }

 return(ok)
}

setData.ggobi <-
function(data, ..., .gobi = getDefaultGGobi())
{
 if(is.character(data)) {
  ok <- setDataFile.ggobi(data, ..., .gobi = .gobi)
 } else
  ok <- setDataFrame.ggobi(as.data.frame(data), ..., .gobi = .gobi)

 return(ok)
}


setDataFile.ggobi <-
function(file, mode = "unknown", add = T, .gobi = getDefaultGGobi())
{
 if(is.character(file)) {
   modes <- getDataModes.ggobi()
   mode <- match(mode, modes)
   if(is.na(mode)) {
     mode <- modes["Unknown"]
   } else
     modes <- modes[mode]
 }

 num <- .GGobiCall("setFile", as.character(file), as.integer(mode), as.logical(add), .gobi = .gobi)
 if(num > -1) {
   getDatasetReference.ggobi(num, .gobi)
 } else
   NULL
}


getNumDatasets.ggobi <-
function(.gobi = getDefaultGGobi())
{
 .GGobiCall("getNumDatasets", .gobi=.gobi)
}

getFileNames.ggobi <-
#
# Get the name and mode of the file
# associated with the current dataset.
function(auxillary = FALSE, .gobi = getDefaultGGobi())
{
  .GGobiCall("getFileNames", as.logical(auxillary), .gobi = .gobi)
}

setDataFrame.ggobi <-
function(data, name = description, description = deparse(sys.call()[[2]]), 
            add = TRUE, id = rownames(data), .data = NULL, .gobi = getDefaultGGobi())
{

 # description was 

 n <- dimnames(data)
 if(!inherits(.gobi, "ggobi")) {
   .gobi <- getGGobi(as.integer(.gobi))
 }

  if(length(id) != nrow(data)) {
    id <- paste(seq(1, nrow(data)))
  }

# .gobi <- as.numeric(unclass(.gobi)[["ref"]])
 num <- .Call(.ggobi.symbol("setData"),
           as.numeric(as.matrix(data)),
           n[[1]], n[[2]], dim(data), 
           as.character(description), as.character(name), as.logical(add),
           as.character(id), .data, .gobi = .gobi)$num

 getDatasetReference.ggobi(num, .gobi)
}

getDatasetReference.ggobi <-
function(which, .gobi = getDefaultGGobi())
{
 if(is.character(which)) {
   id <- match(which, getDatasetNames.ggobi(.gobi))
   if(any(is.na(id)))
     stop(paste("Unrecognized dataset name", which[is.na(id)]))

   which <- id
 }

 refs <- .GGobiCall("getDatasetReference", as.integer(which-1), .gobi=.gobi)

  # if the user just asked for one, then return not a list,
  # but the actual ggobiDataset object. i.e. remove the list.
 if(length(which) == 1 & length(refs) == 1) {
   refs <- refs[[1]]
 }

 refs
}

getDatasetReference <-
function(which,...)
{
 UseMethod("getDatasetReference")
}

getDatasetReference.ggobiDisplayDescription <-
function(which)
{
  which[["dataset"]]
}

getDatasetReference.ggobiDisplay <-
function(which)
{
  getDisplayDataset.ggobi(which)
}

getDisplayDataset.ggobi <-
function(dpy, .gobi = getDefaultGGobi())
{
 .GGobiCall("getDisplayDataset", dpy, .gobi = .gobi)
}


getCurrentDisplayType.ggobi <-
#
# Returns a description of the type
# of display which is the active or current
# one.
#
function(.gobi = getDefaultGGobi())
{
  .GGobiCall("getCurrentDisplayType", .gobi = .gobi)
}


getDisplayTypes.ggobi <-
function()
{
 .Call(.ggobi.symbol("getDisplayTypes"))
}

getViewTypes.ggobi <-
# Returns a named integer vector
# identifying the  symbolic constants
# for the different types of plots, indexed
# by the descriptive name of the plot type.
function()
{
  .Call(.ggobi.symbol("getViewTypes"))
}

getDisplayOptions.ggobi <-
#
# Retrieve the display options for the 
# specified display. The default is to get
# the template default options which are used when
# creating a new display/plot.
# Alternatively, one can specify an integer which identifies
# the display (window) and its sub-plots.
# These are stored in ordered and can be examined using
# the hierarchical display tree accessed from the Plots
# menu item.

#
# which - the display number within the specified ggobi instance.
#

function(which = 1, .gobi = getDefaultGGobi())
{
  .GGobiCall("getDisplayOptions", as.integer(which-1), .gobi = .gobi)
}


setDisplayOptions.ggobi <-
#
# Changes the settings of the display options for a
# collection of plots or the template options for future
# plots.
#
#
# The argument `which' identifies the GGobi display (this is an index
# into the ordered list of displays viewable via the hierarchical dislay
# tree). If this is negative, the Default options used when creating new
# plots will be modified.
#
# The return value is the current settings for the specified options.
#
# Need to handle the case where the user specifies a single vector of
# logicals. And named vectors.
#
#
function(points, directed, undirected, segments, missings,  axes, 
           display = 1, .gobi = getDefaultGGobi())
{
  current <- getDisplayOptions.ggobi(display, .gobi=.gobi)
  old <- current

  if(!missing(points))
    current["Points"] <- as.logical(points)

  if(!missing(segments))
    current["Segments"] <- as.logical(segments)

  if(!missing(directed))
    current["Directed segments"] <- as.logical(directed)

  if(!missing(undirected))
    current["Undirected segments"] <- as.logical(undirected)

  if(!missing(missings))
    current["Missing Values"] <- as.logical(missings)

  if(!missing(axes))
    current["Axes"] <- as.logical(axes)

  .GGobiCall("setDisplayOptions", as.integer(display - 1), current, .gobi = .gobi)

 return(old)
}



getActivePlot.ggobi <-
function(.gobi = getDefaultGGobi())
{
 as.integer(.GGobiCall("getActivePlot", .gobi = .gobi)+1)
}

setActivePlot.ggobi <-
function(display, plot=1, .gobi = getDefaultGGobi())
{
 .GGobiCall("setActivePlot", as.integer(c(display, plot)-1), .gobi = .gobi)
}

getDisplays.ggobi <-
function(describe = F, .gobi = getDefaultGGobi())
{
 .GGobiCall("getDisplays", as.logical(describe), .gobi = .gobi)
}


close.ggobiDisplayDescription <-
function(con, ...)
{
 close(con[["display"]])
}

close.ggobiDisplay <-
function(con, ...)
{
  .GGobiCall("closeDisplay", con[["ref"]], .gobi = con[["ggobi"]])
}

getDatasetNames.ggobi <-
function(.gobi = getDefaultGGobi())
{
 .GGobiCall("getDatasetNames", .gobi=.gobi)
}

getDescription.ggobi <-
#
# Get a description of the global state of the GGobi session.
# 
#
function(.gobi = getDefaultGGobi())
{
  .GGobiCall("getDescription", .gobi = .gobi)
}



getVariableNames.ggobi <-
function(tform = FALSE, .data = 1, .gobi = getDefaultGGobi())
{
  if(!inherits(.gobi, "ggobi"))
   .gobi <- getGGobi(.gobi = .gobi)

  if(mode(.data) == "numeric")
    .data <- as.integer(.data - 1)

  .GGobiCall("getVariableNames", as.logical(tform), .data, .gobi = .gobi)
}

datasetIndex.ggobi <-
function(.data, .gobi = getDefaultGGobi())
{
 if(is.integer(.data))
   return(.data)

 if(mode(.gobi) == "numeric")
   .gobi <- getGGobi(.gobi)
 
 index <- match(as.character(.data), names(.gobi))
 names(index) <- .data

 return(index)
}


getGlyphTypes.ggobi <-
function()
{
 .Call(.ggobi.symbol("getGlyphTypes"))
}

getGlyphSizes.ggobi <-
function()
{
 .Call(.ggobi.symbol("getGlyphSizes"))
}


getGlyphs.ggobi <-
function(which=NULL, .data = 1, .gobi = getDefaultGGobi())
{
 if(mode(.data) == "numeric")
   .data <- as.integer(.data - 1)

 if(!is.null(which)) {
   which <- as.integer(which-1)
 }

 .GGobiCall("getCaseGlyphs", which, .data, .gobi = .gobi)
}

setGlyphs.ggobi <-
function(types, sizes, which, .data=1, .gobi = getDefaultGGobi())
{
 if(mode(.data) == "numeric")
    .data <- as.integer(.data - 1)

 if(is.character(types)) {
    # map the glyph types to integers
   types <- mapGlyphType(types)
 }

 if(missing(which))
   which <- seq(1, length = length(types))

 if(missing(sizes)) {
   sizes <- rep(-1, length(types))
 }

 m <- max(length(which), length(sizes), length(types))
 which <- rep(which, length.out = m)
 sizes <- rep(sizes, length.out = m)
 types <- rep(types, length.out = m)

 if(any( c(length(which), length(sizes), length(which)) == 0))
   stop("Non-zero glyph attributes needed")

 .GGobiCall("setCaseGlyphs", as.integer(types), as.integer(sizes), as.integer(which-1), .data, .gobi = .gobi)
}


getColors.ggobi <-
function(which = NULL, .data = 1, .gobi = getDefaultGGobi())
{
 if(mode(.data) == "numeric")
   .data <- as.integer(.data - 1)

 .GGobiCall("getCaseColors", as.integer(which), .data,  .gobi = .gobi)
}

setColors.ggobi <-
function(colors, which = 1:length(colors), .data=1, .gobi = getDefaultGGobi())
{
 if(mode(colors) == "numeric")
   colors <- as.integer(colors)
 else {
   tmp <-  resolveColors.ggobi(as.character(colors), .gobi = .gobi)
   if(any(is.na(tmp)))
     stop(paste("Unspecified color(s)", colors[is.na(tmp)]))

    colors <- tmp
 }

 if(mode(.data) == "numeric")
   .data <- as.integer(.data - 1)

 if(any(colors < 1) || any(colors > 65535))
    stop("All color indices must be positive and less than 65536.")
 
 old <- .GGobiCall("setCaseColors", old=colors, as.integer(which-1), .data, .gobi = .gobi)
 invisible(old)
}

resolveColors.ggobi <-
function(..., .gobi = getDefaultGGobi())
{
  stop("Color names are not currently supported in ggobi")
  ids <- rownames(getColorMap.ggobi(.gobi=.gobi))
  match(c(...), ids)
}

getColorMap.ggobi <-
function(.gobi = getDefaultGGobi())
{
 m <- .GGobiCall("getColorMap", .gobi = .gobi)
 m <- m/65535

  # Now fix up the missing names for the colors
  # and give them their index values.
 n <- dimnames(m)[[1]]
 n[n==""] <- (1:length(n))[n==""]
 
 dimnames(m)[[1]] <- n
 m
}

setColorMap.ggobi <-
function(vals, .gobi = getDefaultGGobi(), scale=65535)
{
 if(!is.matrix(vals) | ncol(vals) != 3)
  stop("Color map must be an Nx3 matrix of RGB values")
 
 colorNames <- rownames(vals)
 .GGobiCall("setColorMap", as.numeric(vals * scale), colorNames, .gobi = .gobi)
}



close.ggobi <-
function(con, ...)
{
  if(missing(con))
     con <- getDefaultGGobi()
  
  ok <- .GGobiCall("close", .gobi = con)
  if(ok) {
    cur <-getDefaultGGobi()
    if(unclass(con)$ref == unclass(cur)$ref) {
      setDefaultGGobi(getNumGGobis()) 
    }
  }

 ok
}


getNumGGobis <-
function()
{
 .C(.ggobi.symbol("getNumGGobiInstances"), as.integer(-1))[[1]]
}

isValid.ggobi <-
function(.gobi)
{
 .GGobiCall("isValid", .gobi=.gobi)
}

getGGobi <-
function(...)
{
 args <- list(...)
 if(length(args) > 0) {
   which <- sapply(args, as.integer)
 } else {
   which <- as.integer(seq(1,length=getNumGGobis()))
 }

 if(length(which)) {
   v <- .Call("RS_GGOBI_getGGobi", which)
   if(!is.null(v) & length(which) == 1) {
    v <- v[[1]]
   }

   v
 } else
   NULL
}


setMode.ggobi <-
function(name, .gobi = getDefaultGGobi())
{
 old <- getMode.ggobi(.gobi) 
 .GGobiCall("setMode",  as.character(name), .gobi=.gobi)

 old
}

getMode.ggobi <-
function(.gobi = getDefaultGGobi())
{
 .GGobiCall("getModeName", .gobi=.gobi)
}

getModeNames.ggobi <-
function()
{
  .Call(.ggobi.symbol("getModeNames"))
}

getDataModes.ggobi <-
function()
{
 .Call(.ggobi.symbol("getDataModes"))
}


gdk.flush <-
function()
{
  .C(.ggobi.symbol("flush"))
}


getHiddenCases.ggobi <-
function(.data = 1, .gobi = getDefaultGGobi())
{
 if(mode(.data) == "numeric")
   .data <- as.integer(.data - 1)

 .GGobiCall("getCasesHidden", .data = .data, .gobi=.gobi)
}

setHiddenCases.ggobi <-
function(vals, which = 1:length(vals), .data = 1, .gobi = getDefaultGGobi())
{
 vals <- as.logical(vals)
 if(mode(.data) == "numeric")
   .data <- as.integer(.data - 1)

 .GGobiCall("setCasesHidden", vals, as.integer(which-1), .data = .data, .gobi=.gobi)
}
