/*
 * Copyright (c) 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Modified to PostgreSQL 7.3 and 7.4 by Fernando Nasser
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.explain;

import java.io.*;
import java.sql.*;
import java.util.*;
import java.text.*;

/**
 * EXPLAIN a query. Should only be used by classes in the
 * com.redhat.rhdb.explain package. The query should have been passed
 * through Explain.query(). Meant to be used if the connection is to a
 * PostgreSQL 7.3+ database.
 *
 * @author <a href="mailto:liams@redhat.com">Liam Stewart</a>
 * @author <a href="mailto:fnasser@redhat.com">Fernando Nasser</a>
 * @version 1.2.0
 */
public class ExplainParserV73 implements ExplainParser
{
	private Explain explain;
	private Statement stmt;
	private	double total_cost = 0;
	private	double total_runtime = 0;

	public ExplainParserV73(Explain exp)
	{
		this.explain = exp;
	}

	public void explain(Connection con, boolean analyze) throws ExplainException
	{
		String dump = null;
		StringReader strreader;
		StringWriter strwriter, planwriter;
		PrintWriter printer;
		StringTokenizer strtok;
		String str = null;
                String token;
                ResultSet rs;
		ExplainTreeNode root;
		ExplainTreeNode toproot = null;
		boolean explain_analyze;

		if (con == null || explain.getQuery() == null)
			throw new ExplainException("Null connection or query");

		try {
			if (con.isClosed())
				throw new ExplainException("Connection is closed");

			// Attempt EXPLAIN ANALYZE?
			String q;
			if (analyze)
			{
				q = "EXPLAIN ANALYZE VERBOSE " + explain.getQuery();
				explain_analyze = true;
			}
			else
			{
				q = "EXPLAIN VERBOSE " + explain.getQuery();
				explain_analyze = false;
			}
			
			boolean hasPlannerOptions = !explain.getNewPlannerOptions().trim().equals("");
			
			// Set planner options, if any specified
			Statement utilstmt = con.createStatement();
			if (hasPlannerOptions)
			{
				utilstmt.executeUpdate(explain.getNewPlannerOptions());
			}

			stmt = con.createStatement();

			// Turn off auto-commit mode. explain analyze actually
			// runs the query so if it as insert, update, or delete,
			// modifications will be made. we don't want that.. A
			// rollback is performed after the explain is done.
			// 
			// If the connection is already out of auto-commit mode,
			// rollback before doing the explain. After the explain,
			// the auto-commit mode will be restored to whatever it
			// was before the explain.
			boolean autoCommitState = con.getAutoCommit();

			if (explain_analyze)
			{
				if (autoCommitState)
					con.setAutoCommit(false);
				else
					con.rollback();
			}

			// Now execute (explain) the query itself
			rs = stmt.executeQuery(q);

			if (explain_analyze)
			{
				con.rollback();
				con.setAutoCommit(autoCommitState);
			}

			// Reset planner options to defaults if we changed any
			if (hasPlannerOptions)
			{
				utilstmt.executeUpdate(explain.getResetCommands());
			}

			// Parse the first row. Ignore all 'bad' rows.
 			while (rs.next())
			{
				// Parse the first part (the dump)
				// The form of the dump is:
				//   .A '{'
				//   <dump - may contain pairs of '{}'>
                                //   .A '}'

				// first row
				str = rs.getString(1);

				if (str.trim().startsWith("{"))
                                {
                                        dump = str;
					break;
                                }
				else
				{
					System.err.println("Ignoring: " + str);
					rs.next();
				}
			}
                        
			if (rs.isAfterLast())
				throw new ExplainException("Unexpected EXPLAIN output");
                        
			// If we get here, we know that first row has the first "{"

			// look for the "}" that closes the query dump
                        int braces = countChars(str, '{');
                        braces -= countChars(str, '}');
                        while ((braces > 0) && (rs.next()))
                        {
                            str = rs.getString(1);
			    dump += "\n" + str;
                            braces += countChars(str, '{');
                            braces -= countChars(str, '}');
                        }

			if (rs.isAfterLast())
				throw new ExplainException("Got nothing when QUERY PLAN expected");

			// Parse the second part (the plan)
			// Form:
			//   <blank line>
			//   <root>
			//     ->  <child>
			//       ->  <child of child> ...
			//     ->  <child>

			// first row
                        rs.next();
			str = rs.getString(1);
			strtok = new StringTokenizer(str);
			if (strtok.countTokens() != 0) {
				throw new ExplainException("Bad number of tokens on first plan line: " + strtok.countTokens());
			}

			// now for the tree...
			planwriter = new StringWriter();
			printer = new PrintWriter(planwriter);

			HashMap ret = buildExplainTree(rs, printer, toproot, explain_analyze);
			
			calcNodeCost(ret, (ExplainTreeNode) ret.get("root"));
			
			explain.setExplainTree(new ExplainTree((ExplainTreeNode) ret.get("root")));
			explain.setOptions();
			explain.setPlan(planwriter.toString());
			explain.setDump(dump);
			
			// done
		} catch (SQLException e) {
			if ((e.getSQLState() != null) && (e.getSQLState().equals("08S01")))
				throw new ExplainException("Communication link error: " + e.getMessage());
			else
				throw new ExplainException("SQL error: " + e.getMessage());
		} catch (IOException e) {
			throw new ExplainException("I/O error: " + e.getMessage());
		} catch (ExplainException e) {
			throw new ExplainException("Processing error: " + e.getMessage());
		} catch (Exception e) {
			e.printStackTrace();
			throw new ExplainException("Other error: " + e);
		}
	}
	
	public void cancel() {
		try {
			stmt.cancel();
		} catch (SQLException e) {
			System.out.println("exception when trying to cancel() = " + e);
		}
	}

	/*
	 * Recursively calculate the node cost by subtracting the sum of the child costs
	 * from the total cost for the node.  Do the same for time, if available.
	 */
	private void calcNodeCost(HashMap hm, ExplainTreeNode pnode) {
		NumberFormat formatter = NumberFormat.getNumberInstance();
		int numChildren = pnode.getChildCount();
		
		if (pnode.getDetail(ExplainTreeNode.TOTAL_COST) != null)
		{
			try {
				String text = pnode.getDetail(ExplainTreeNode.TOTAL_COST);
				double node_cost = formatter.parse(text).doubleValue();
				for (int i = 0; i < numChildren; i++) {
					text = pnode.getChild(i).getDetail(ExplainTreeNode.TOTAL_COST);
					if (text != null)
						node_cost -= formatter.parse(text).doubleValue();
				}
				pnode.setDetail(ExplainTreeNode.NODE_COST, formatter.format(node_cost));
			} catch (ParseException pe) {
				pe.printStackTrace();	// And do not generate the NODE_COST
			}
		}

		if (pnode.getDetail(ExplainTreeNode.TOTAL_TIME) != null)
		{
			try {
				String text = pnode.getDetail(ExplainTreeNode.TOTAL_TIME);
				double node_time = formatter.parse(text).doubleValue();
				for (int i = 0; i < numChildren; i++) {
					text = pnode.getChild(i).getDetail(ExplainTreeNode.TOTAL_TIME);
					if (text != null)
						node_time -= formatter.parse(text).doubleValue();
				}
				pnode.setDetail(ExplainTreeNode.NODE_TIME, formatter.format(node_time));
			} catch (ParseException pe) {
				pe.printStackTrace();	// And do not generate the NODE_TIME
			}
		}
			
		for (int i = 0; i < numChildren; i++) {
			calcNodeCost(hm, pnode.getChild(i));
		}
	}

	/*
	 * Build an ExplainTree. Return the ExplainTree in a HashMap,
	 * which can be accessed with the ROOT key. The last line read is
	 * also stored in the HashMap. It can be accessed with the LASTLINE
	 * key.
	 */
	private HashMap buildExplainTree(ResultSet rs, PrintWriter planwriter, ExplainTreeNode toproot, boolean explain_analyze) 
        throws IOException, SQLException
	{
		HashMap rv = new HashMap();
		rs.next();
		String str = rs.getString(1);
		planwriter.println(str);
		ExplainTreeNode root = createNode(str, explain_analyze);
		rv.put("root", root);

		if (toproot == null)
			toproot = root;

		int level = 0;
		ExplainTreeNode parent = root;
		ExplainTreeNode child = root;
		ArrayList levels = new ArrayList(10);
		levels.add(level, new Integer(whitespaceBeforeArrow(str)));

		str = null;

		while (rs.next())
		{
			if (str == null)
			{
				str = rs.getString(1);
				if (str == null)
					break;
				planwriter.println(str);
			}
			// else str was read last time and needs to be processed again
			// (usually since it was passed back from another call to
			// buildExplainTree
			
			if (str.trim().length() == 0) {
				str = null;
				continue;
			}

			if (str.trim().startsWith("SubPlan") || str.trim().startsWith("InitPlan")) {
				ExplainTreeNode old = parent;
				parent = child;
				child = createNode(str, explain_analyze);
				parent.addChild(child);
				child.setParent(parent);
				
				HashMap ret = buildExplainTree(rs, planwriter, toproot, explain_analyze);
				ExplainTreeNode n = (ExplainTreeNode) ret.get("root");
				str = (String) ret.get("lastline");
				
				child.addChild(n);
				n.setParent(child);
				parent = old;
			} else if (str.trim().startsWith("Total runtime:")) {
				String s = str.trim();
				s = s.substring(s.indexOf(':') + 2);
				StringTokenizer detailtok = new StringTokenizer(s, ".");
				String intpart = detailtok.nextToken();
				try {
					int ivalue = Integer.parseInt(intpart);
					NumberFormat formatter = NumberFormat.getNumberInstance();
					toproot.setDetail(ExplainTreeNode.TOTAL_RUNTIME, formatter.format(ivalue) + "." + detailtok.nextToken());
				} catch (NumberFormatException pe) {
					toproot.setDetail(ExplainTreeNode.TOTAL_RUNTIME, intpart + "." + detailtok.nextToken());
				}
				str = null;
			} else {
				int n = whitespaceBeforeArrow(str);
				
				if (n < 0) {
					// It is probably a qualifier expression
					// so we add this information to the last node
					addQual(child, str);
                                        
					str = null;
					continue;
				}
				
				int p = ((Integer) levels.get(level)).intValue();

				if (n > p) {
					levels.add(++level, new Integer(n));
					parent = child;
				} else if (n == p) {
					// same level
				} else if (n < p) {
					// back up to appropriate level
					while (n != p && level != 0) {
						level--;
						p = ((Integer) levels.get(level)).intValue();
						parent = parent.getParent();
					}
					if (level == 0)
						break;
				}
				
				child = createNode(str, explain_analyze);
				parent.addChild(child);
				child.setParent(parent);
				
				str = null;
			}
		}

		rv.put("lastline", str);

		return rv;
	}
	
	/*
	 * Assuming (worthless) initial indentation and no "->", a plan
	 * tree node has the following form:
	 *
	 *   [algorithm] (key1=val1 key2=val2 ...)
	 *
	 * [algorithm] can contain (and usually does contain) more than
	 * one word. Examples: "Seq Scan on hosts", "Merge Join", "Sort"
	 *
	 * -> More error checking/handling is needed.
	 */
	private ExplainTreeNode createNode(String str, boolean explain_analyze) {
		StringTokenizer strtok;
		StringBuffer name = new StringBuffer();
		String type, token, first = null;

		ExplainTreeNode node = new ExplainTreeNode();

		str = stripCruft(str);

		// special "nodes"
		if (str.startsWith("SubPlan"))
		{
			node.setName("SubPlan");
			node.setType(ExplainTreeNode.NODE_SUBPLAN);
			return node;
		}
		else if (str.startsWith("InitPlan"))
		{
			node.setName("InitPlan");
			node.setType(ExplainTreeNode.NODE_INITPLAN);
			return node;
		}

		int detail_start = str.indexOf('(');
		int detail_end = str.indexOf(')', detail_start);
		int analyze_start = -1;
		int analyze_end = -1;

		if (explain_analyze)
		{
			analyze_start = str.indexOf('(', detail_end);
			analyze_end = str.indexOf(')', analyze_start);
		}

		String algo = str.substring(0, detail_start);

		strtok = new StringTokenizer(algo);

		first = strtok.nextToken();
		if (first.equals("Result"))
		{
			node.setType(ExplainTreeNode.NODE_RESULT);
			node.setDetail(ExplainTreeNode.TYPE, first);
			
			name.append(first);
		}
		else if (first.equals("Append"))
		{
			node.setType(ExplainTreeNode.NODE_APPEND);
			node.setDetail(ExplainTreeNode.TYPE, first);

			name.append(first);
		}
		else if (first.equals("Hash"))
		{
			if (strtok.hasMoreTokens())
			{
				String next = strtok.nextToken();

				if (next.equals("Join"))
				{
					node.setType(ExplainTreeNode.NODE_JOIN);
					node.setDetail(ExplainTreeNode.TYPE, next);
					node.setDetail(ExplainTreeNode.SUBTYPE, first + " " + next);
					name.append(first + " " + next);
				}
				else
				{
					node.setType(ExplainTreeNode.NODE_UNKNOWN);
					node.setDetail(ExplainTreeNode.TYPE, "unknown: " + first);
					name.append("???");
				}
			}
			else
			{
				node.setType(ExplainTreeNode.NODE_HASH);
				node.setDetail(ExplainTreeNode.TYPE, first);
				name.append(first);
			}
		}
		else if (first.equals("Nested") ||
				 first.equals("Merge"))
		{
			String next = strtok.nextToken();

			node.setType(ExplainTreeNode.NODE_JOIN);
			node.setDetail(ExplainTreeNode.TYPE, "Join");
			node.setDetail(ExplainTreeNode.SUBTYPE, first + " " + next);

			name.append(first + " " + next);
		}
		else if (first.equals("Seq")   ||
				 first.equals("Function") ||
				 first.equals("Index") ||
				 first.equals("Tid")   ||
				 first.equals("Subquery"))
		{
			node.setType(ExplainTreeNode.NODE_SCAN);
			node.setDetail(ExplainTreeNode.TYPE, "Scan");
			
			name.append(first + " " + strtok.nextToken());

			if (first.equals("Seq"))
			{
				node.setDetail(ExplainTreeNode.SUBTYPE, "Sequential Scan");
			}
			else if (first.equals("Tid"))
			{
				node.setDetail(ExplainTreeNode.SUBTYPE, "Tid Scan");
			}
			else if (first.equals("Function"))
			{
				node.setDetail(ExplainTreeNode.SUBTYPE, "Function Scan");
			}
			else if (first.equals("Index"))
			{
				node.setDetail(ExplainTreeNode.DIRECTION, "forward");
				node.setDetail(ExplainTreeNode.SUBTYPE, "Index Scan");
			}
			else if (first.equals("Subquery"))
			{
				StringBuffer text = new StringBuffer();

				while (strtok.hasMoreTokens())
				{
					text.append(strtok.nextToken() + " ");
				}

				node.setDetail(ExplainTreeNode.SUBQUERY_ON, text.toString());
				node.setDetail(ExplainTreeNode.SUBTYPE, "Subquery Scan");
			}
			
			while (strtok.hasMoreTokens()) {
				token = strtok.nextToken();
				
				if (token.equals("Backward"))
				{
					node.setDetail(ExplainTreeNode.DIRECTION, "backward");
				}
				else if (token.equals("using"))
				{
					StringBuffer indexes = new StringBuffer();
					String next = null;
					
					while (strtok.hasMoreTokens())
					{
						next = strtok.nextToken();
						if (next.indexOf(',') == -1)
							break;
						indexes.append(next + " ");
					}
					indexes.append(next);
					
					node.setDetail(ExplainTreeNode.INDEXES, indexes.toString());
				}
				else if (token.equals("on"))
				{
					node.setDetail(ExplainTreeNode.TABLE, strtok.nextToken());
					if (strtok.hasMoreTokens())
					{
						node.setDetail(ExplainTreeNode.TABLE_ALIAS, strtok.nextToken());
					}
				}
			}
		}
		else if (first.equals("Materialize"))
		{
			node.setType(ExplainTreeNode.NODE_MATERIALIZE);
			node.setDetail(ExplainTreeNode.TYPE, first);

			name.append(first);
		}
		else if (first.equals("Sort"))
		{
			node.setType(ExplainTreeNode.NODE_SORT);
			node.setDetail(ExplainTreeNode.TYPE, first);

			name.append(first);
		}
		else if (first.equals("Group"))
		{
			node.setType(ExplainTreeNode.NODE_GROUP);
			node.setDetail(ExplainTreeNode.TYPE, first);

			name.append(first);
		}
		else if ((first.equals("Aggregate")) ||
				 (first.equals("GroupAggregate")) ||
				 (first.equals("HashAggregate")))
		{
			node.setType(ExplainTreeNode.NODE_AGGREGATE);
			node.setDetail(ExplainTreeNode.TYPE, first);

			name.append(first);
		}
		else if (first.equals("Unique"))
		{
			node.setType(ExplainTreeNode.NODE_UNIQUE);
			node.setDetail(ExplainTreeNode.TYPE, first);

			name.append(first);
		}
		else if (first.equals("SetOp"))
		{
			node.setType(ExplainTreeNode.NODE_SETOP);
			node.setDetail(ExplainTreeNode.TYPE, first);

			String next = strtok.nextToken();
			node.setDetail(ExplainTreeNode.SUBTYPE, next);
			name.append(next);

			if (strtok.countTokens() == 3 &&
				strtok.nextToken().equals("All"))
			{
				node.setDetail(ExplainTreeNode.SETOP_ALL, "yes");
			}
			else
			{
				node.setDetail(ExplainTreeNode.SETOP_ALL, "no");
			}
		}
		else if (first.equals("Limit"))
		{
			node.setType(ExplainTreeNode.NODE_LIMIT);
			node.setDetail(ExplainTreeNode.TYPE, first);

			name.append(first);
		}
		else
		{
			node.setType(ExplainTreeNode.NODE_UNKNOWN);
			node.setDetail(ExplainTreeNode.TYPE, "unknown: " + first);

			name.append("???");
		}
		
		node.setName(name.toString());

		// details
		String details = str.substring(detail_start + 1, detail_end);
		strtok = new StringTokenizer(details);
		NumberFormat formatter = NumberFormat.getNumberInstance();
		NumberFormat pformatter = NumberFormat.getPercentInstance();

		while (strtok.hasMoreTokens()) {
			String str2 = strtok.nextToken();
			StringTokenizer detailtok = new StringTokenizer(str2, "=");
			String key = detailtok.nextToken();
			String value = detailtok.nextToken();
			
			if (key.equals("cost")) {
				detailtok = new StringTokenizer(value, ".");
				try {
					double cost = Double.parseDouble(detailtok.nextToken()) + (Double.parseDouble(detailtok.nextToken()) / 100);
					node.setDetail(ExplainTreeNode.STARTUP_COST, formatter.format(cost));
				} catch (NumberFormatException pe) {
					node.setDetail(ExplainTreeNode.STARTUP_COST, value.substring(0, value.indexOf(".") + 2));
				}
				try {
					double cost = Double.parseDouble(detailtok.nextToken()) + (Double.parseDouble(detailtok.nextToken()) / 100);
					if (total_cost == 0) {
						// If it is the root, save total cost
						total_cost = cost;
					}
					if (total_cost > 0)
						// Can the total cost be zero? 
						node.setDetail(ExplainTreeNode.TOTAL_COST, formatter.format(cost) +
										"  (" + pformatter.format(cost / total_cost) + ")");
					else
						node.setDetail(ExplainTreeNode.TOTAL_COST, formatter.format(cost));
				} catch (NumberFormatException pe) {
					if (total_cost == 0)
						total_cost = -1;  // If could not get the total cost stop trying
					node.setDetail(ExplainTreeNode.TOTAL_COST, value.substring(value.indexOf(".") + 4));
				}
			} else if (key.equals("rows") || key.equals("width")) {
				try {
					double ivalue = Double.parseDouble(value);
					node.setDetail(key, formatter.format(ivalue));
				} catch (NumberFormatException pe) {
					node.setDetail(key, value);
				}
			} else {
				node.setDetail(key, value);
			}
		}

		// analyze
		if (explain_analyze && analyze_start >= 0)
		{
			String analyze = str.substring(analyze_start + 1, analyze_end);
			strtok = new StringTokenizer(analyze);

			while (strtok.hasMoreTokens())
			{
				String str2 = strtok.nextToken();

				// Check first for a "never executed" node
				if (str2.equals("never"))
				{
					node.setDetail(ExplainTreeNode.ACTUAL_NOTE, analyze);
					// No other info for this node
					break;
				}

				if (str2.equals("actual"))
					str2 = strtok.nextToken();

				StringTokenizer detailtok = new StringTokenizer(str2, "=");
				String key = detailtok.nextToken();
				String value = detailtok.nextToken();

				if (key.equals("time")) {
					detailtok = new StringTokenizer(value, ".");
					try {
						double time = Double.parseDouble(detailtok.nextToken()) + (Double.parseDouble(detailtok.nextToken()) / 100);
						node.setDetail(ExplainTreeNode.STARTUP_TIME, formatter.format(time));
					} catch (NumberFormatException pe) {
						node.setDetail(ExplainTreeNode.STARTUP_TIME, value.substring(0, value.indexOf(".") + 2));
					}
					try {
						double time = Double.parseDouble(detailtok.nextToken()) + (Double.parseDouble(detailtok.nextToken()) / 100);
						if (total_runtime == 0) {
							// If it is the root, save total runtime
							total_runtime = time;
						}
						if (total_runtime > 0)
							// Can the total runtime be zero? 
							node.setDetail(ExplainTreeNode.TOTAL_TIME, formatter.format(time) +
										"  (" + pformatter.format(time / total_runtime) + ")");
						else
							node.setDetail(ExplainTreeNode.TOTAL_TIME, formatter.format(time));
					} catch (NumberFormatException pe) {
						if (total_runtime == 0)
							total_runtime = -1;  // If could not get the total runtime stop trying
						node.setDetail(ExplainTreeNode.TOTAL_TIME, value.substring(value.indexOf(".") + 4));
					}
				} else if (key.equals("rows")) {
					try {
				    	double ivalue = Double.parseDouble(value);
						node.setDetail(ExplainTreeNode.ACTUAL_ROWS, formatter.format(ivalue));
					} catch (NumberFormatException pe) {
						node.setDetail(ExplainTreeNode.ACTUAL_ROWS, value);
					}
				} else if (key.equals("loops")) {
					try {
				    	double ivalue = Double.parseDouble(value);
						node.setDetail(ExplainTreeNode.ACTUAL_LOOPS, formatter.format(ivalue));
					} catch (NumberFormatException pe) {
						node.setDetail(ExplainTreeNode.ACTUAL_LOOPS, value);
					}
				} else {
					node.setDetail(key, value);
				}
			}
		}

		return node;
	}
	
	/*
	 * Assuming (worthless) initial indentation and no "->", a plan
	 * tree node has the following form:
	 *
	 *   [algorithm] (key1=val1 key2=val2 ...)
	 *
	 * [algorithm] can contain (and usually does contain) more than
	 * one word. Examples: "Seq Scan on hosts", "Merge Join", "Sort"
	 *
	 * -> More error checking/handling is needed.
	 */
	private void addQual(ExplainTreeNode node, String str) {
		String first = str.trim();
                
		if (first.startsWith("Filter:"))
		{
			node.setDetail(ExplainTreeNode.FILTER, 
						   first.substring(first.indexOf(':') + 2));
		}
		else if (first.startsWith("Join Filter:") &&
				 (node.getDetail(ExplainTreeNode.SUBTYPE).equals("Nested Loop") ||
				  node.getDetail(ExplainTreeNode.SUBTYPE).equals("Merge Join") ||
				  node.getDetail(ExplainTreeNode.SUBTYPE).equals("Hash Join")))
		{
			node.setDetail(ExplainTreeNode.JOIN_FILTER, 
						   first.substring(first.indexOf(':') + 2));
		}
		else if (first.startsWith("One-Time Filter:") &&
				 node.getDetail(ExplainTreeNode.TYPE).equals("Result"))
		{
			node.setDetail(ExplainTreeNode.ONE_TIME_FILTER, 
						   first.substring(first.indexOf(':') + 2));
		}
		else if (first.startsWith("Index Cond:") &&
				 node.getDetail(ExplainTreeNode.SUBTYPE).equals("Index Scan"))
		{
			node.setDetail(ExplainTreeNode.INDEX_COND, 
						   first.substring(first.indexOf(':') + 2));
		}
		else if (first.startsWith("Merge Cond:") &&
				 node.getDetail(ExplainTreeNode.SUBTYPE).equals("Merge Join"))
		{
			node.setDetail(ExplainTreeNode.MERGE_COND, 
						   first.substring(first.indexOf(':') + 2));
		}
		else if (first.startsWith("Hash Cond:") &&
				 node.getDetail(ExplainTreeNode.SUBTYPE).equals("Hash Join"))
		{
			node.setDetail(ExplainTreeNode.HASH_COND, 
						   first.substring(first.indexOf(':') + 2));
		}
		else if (first.startsWith("Sort Key:") &&
				 node.getDetail(ExplainTreeNode.TYPE).equals("Sort"))
		{
			node.setDetail(ExplainTreeNode.SORT_KEY, 
						   first.substring(first.indexOf(':') + 2));
		}
		else
		{
			System.err.println("Bad or misplaced qualifier.\nIgnoring: " + str);   
		}
	}
        
	/*
	 * count the amount of whitespace before the "->"
	 */
	private static int whitespaceBeforeArrow(String str) {
		int n = str.indexOf("->");

		return n;
	}

	/*
	 * Strip all leading and trailing cruft from a string.
	 * This includes all leading and trailing whitespace and
	 * the arrow ("->"). All whitespace after the arrow and before the
	 * next non-whitespace character is also stripped.
	 */
	private static String stripCruft(String str) {
		str = str.trim();
		if (str.startsWith("->")) {
			str = str.substring(2);
			str = str.trim();
		}

		return str;
	}
        
	/*
	 * Count the occurences of a char in a string.
	 */
	private static int countChars(String str, char c) {
                int count = 0;
                int ind = str.indexOf(c);

                while (ind >= 0)
                {
                    count++;
                    ind++;
                    ind = str.indexOf(c, ind);
                }

		return count;
	}

}
