/*
 * Copyright (c) 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Fernando Nasser
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.vise;

import java.util.StringTokenizer;

/**
 * The <code>PlannerOptions</code> class encapsulates all the
 * possible planner options that can be set by the user.
 *
 * @author <a href="mailto:fnasser@redhat.com">Fernando Nasser</a>
 * @version 1.2.0
 */

public class PlannerOptions
{
	// A null value means that the default should be used
	// That is why all options are stored as objects
	// We make these protected instead of having accessors for clarity
	// (see Bloch, "Effective Java")
	protected Boolean enable_seqscan;
	protected Boolean enable_indexscan;
	protected Boolean enable_tidscan;
	protected Boolean enable_sort;
	protected Boolean enable_hashagg;
	protected Boolean enable_nestloop;
	protected Boolean enable_mergejoin;
	protected Boolean enable_hashjoin;
	protected Boolean geqo;
	protected Integer geqo_threshold;
	protected Integer geqo_pool_size;
	protected Integer geqo_effort;
	protected Integer geqo_generations;
	protected Float random_page_cost;
	protected Float cpu_tuple_cost;
	protected Float cpu_index_tuple_cost;
	protected Float cpu_operator_cost;
	protected Float geqo_selection_bias;

	// One object of this class filled with the backend's defaults
	// by the Connection Model.  Null if this information is not available,
	// in which case we cannot generate anything with getResetCommands	
	private PlannerOptions defaults;
	
	/**
	 * Creates a new <code>PlannerOptions</code> instance.
	 *
	 */
	public PlannerOptions() {
		// All fields automatically set to null as they are all objects
		
		// Note: Without the defaults, getResetCommands cannot be used
		defaults = null;
	}
	
	/**
	 * Creates a new <code>PlannerOptions</code> instance
	 * and inform it of what are the backend defaults
	 * so it can generate reset commands.
	 *
	 * @param defs <code>PlannerOptions</code> value
	 */
	public PlannerOptions(PlannerOptions defs) {
		// All fields automatically set to null as they are all objects

		this.defaults = defs;
	}
	
	/**
	 * Creates a new <code>PlannerOptions</code> instance
	 * from the SET commands generated by the toString method.
	 *
	 * @param sets <code>String</code> value
	 */
	public PlannerOptions(PlannerOptions defs, String sets) {
		this.defaults = defs;
		
		if (sets.trim().equals(""))
			return;
			
		// Use stringtokenizer to extract the settings
		StringTokenizer st = new StringTokenizer(sets, ";");
		
		// Loop through all SET commands
		while (st.hasMoreTokens())
		{
			String setcmd = st.nextToken().trim();
			if (setcmd.equals(""))
				break;	// Last token is empty due to the last ';'
			StringTokenizer wst = new StringTokenizer(setcmd);
			
			// The second token is the variable and the forth the value
			wst.nextToken();
			String var = wst.nextToken();
			wst.nextToken();
			String value = wst.nextToken();

			// Set the variable to the value
			if (var.equals("enable_seqscan"))
				enable_seqscan = Boolean.valueOf(value);
			else if (var.equals("enable_indexscan"))
				enable_indexscan = Boolean.valueOf(value);
			else if (var.equals("enable_tidscan"))
				enable_tidscan = Boolean.valueOf(value);
			else if (var.equals("enable_sort"))
				enable_sort = Boolean.valueOf(value);
			else if (var.equals("enable_hashagg"))
				enable_hashagg = Boolean.valueOf(value);
			else if (var.equals("enable_nestloop"))
				enable_nestloop = Boolean.valueOf(value);
			else if (var.equals("enable_mergejoin"))
				enable_mergejoin = Boolean.valueOf(value);
			else if (var.equals("enable_hashjoin"))
				enable_hashjoin = Boolean.valueOf(value);
			else if (var.equals("geqo"))
				geqo = Boolean.valueOf(value);
			else if (var.equals("geqo_threshold"))
				try {
					geqo_threshold = Integer.decode(value);
				} catch (NumberFormatException nfe) {}
			else if (var.equals("geqo_pool_size"))
				try {
					geqo_pool_size = Integer.decode(value);
				} catch (NumberFormatException nfe) {}
			else if (var.equals("geqo_effort"))
				try {
					geqo_effort = Integer.decode(value);
				} catch (NumberFormatException nfe) {}
			else if (var.equals("geqo_generations"))
				try {
					geqo_generations = Integer.decode(value);
				} catch (NumberFormatException nfe) {}
			else if (var.equals("random_page_cost"))
				try {
					random_page_cost = Float.valueOf(value);
				} catch (NumberFormatException nfe) {}
			else if (var.equals("cpu_tuple_cost"))
				try {
					cpu_tuple_cost = Float.valueOf(value);
				} catch (NumberFormatException nfe) {}
			else if (var.equals("cpu_index_tuple_cost"))
				try {
					cpu_index_tuple_cost = Float.valueOf(value);
				} catch (NumberFormatException nfe) {}
			else if (var.equals("cpu_operator_cost"))
				try {
					cpu_operator_cost = Float.valueOf(value);
				} catch (NumberFormatException nfe) {}
			else if (var.equals("geqo_selection_bias"))
				try {
					geqo_selection_bias = Float.valueOf(value);
				} catch (NumberFormatException nfe) {}
			else
				// Ignore unknown variables (should never happen)
				System.out.println("Unknown Planner Option " + var);
		}
		// All other fields automatically set to null as they are all objects
	}
	
	/**
	 * Gets the planner option set commands.
	 *
	 * @return a <code>String</code> value
	 */
	public String toString()
	{
		StringBuffer buff = new StringBuffer(240);
		
		if (enable_seqscan != null)
		{
			if (enable_seqscan.booleanValue() == true)
				buff.append("SET enable_seqscan TO TRUE; ");
			else
				buff.append("SET enable_seqscan TO FALSE; ");
		}
		
		if (enable_indexscan != null)
		{
			if (enable_indexscan.booleanValue() == true)
				buff.append("SET enable_indexscan TO TRUE; ");
			else
				buff.append("SET enable_indexscan TO FALSE; ");
		}
		
		if (enable_tidscan != null)
		{
			if (enable_tidscan.booleanValue() == true)
				buff.append("SET enable_tidscan TO TRUE; ");
			else
				buff.append("SET enable_tidscan TO FALSE; ");
		}
		
		if (enable_sort != null)
		{
			if (enable_sort.booleanValue() == true)
				buff.append("SET enable_sort TO TRUE; ");
			else
				buff.append("SET enable_sort TO FALSE; ");
		}
		
		if (enable_hashagg != null)
		{
			if (enable_hashagg.booleanValue() == true)
				buff.append("SET enable_hashagg TO TRUE; ");
			else
				buff.append("SET enable_hashagg TO FALSE; ");
		}
		
		if (enable_nestloop != null)
		{
			if (enable_nestloop.booleanValue() == true)
				buff.append("SET enable_nestloop TO TRUE; ");
			else
				buff.append("SET enable_nestloop TO FALSE; ");
		}
		
		if (enable_mergejoin != null)
		{
			if (enable_mergejoin.booleanValue() == true)
				buff.append("SET enable_mergejoin TO TRUE; ");
			else
				buff.append("SET enable_mergejoin TO FALSE; ");
		}
		
		if (enable_hashjoin != null)
		{
			if (enable_hashjoin.booleanValue() == true)
				buff.append("SET enable_hashjoin TO TRUE; ");
			else
				buff.append("SET enable_hashjoin TO FALSE; ");
		}
		
		if (geqo != null)
		{
			if (geqo.booleanValue() == true)
				buff.append("SET geqo TO TRUE; ");
			else
				buff.append("SET geqo TO FALSE; ");
		}
		
		if (geqo_threshold != null)
		{
			buff.append("SET geqo_threshold TO " + geqo_threshold.toString() + "; ");
		}
		
		if (geqo_pool_size != null)
		{
			buff.append("SET geqo_pool_size TO " + geqo_pool_size.toString() + "; ");
		}
		
		if (geqo_effort != null)
		{
			buff.append("SET geqo_effort TO " + geqo_effort.toString() + "; ");
		}
		
		if (geqo_generations != null)
		{
			buff.append("SET geqo_generations TO " + geqo_generations.toString() + "; ");
		}
		
		if (random_page_cost != null)
		{
			buff.append("SET random_page_cost TO " + random_page_cost.toString() + "; ");
		}
		
		if (cpu_tuple_cost != null)
		{
			buff.append("SET cpu_tuple_cost TO " + cpu_tuple_cost.toString() + "; ");
		}
		
		if (cpu_index_tuple_cost != null)
		{
			buff.append("SET cpu_index_tuple_cost TO " + cpu_index_tuple_cost.toString() + "; ");
		}
		
		if (cpu_operator_cost != null)
		{
			buff.append("SET cpu_operator_cost TO " + cpu_operator_cost.toString() + "; ");
		}
		
		if (geqo_selection_bias != null)
		{
			buff.append("SET geqo_selection_bias TO " + geqo_selection_bias.toString() + "; ");
		}

		return buff.toString();
	}
	
	/**
	 * Gets the planner option reset commands, necessary to restore the
	 * options to what they were before we changed them (default values)
	 *
	 * @return a <code>String</code> value
	 */
	public String getResetCommands()
	{
		StringBuffer buff = new StringBuffer(240);
		
		// For the boolean ones we can guess what the default was
		// because if it was set it was because it was not set
		// and vice-versa
		if (enable_seqscan != null)
		{
			if (enable_seqscan.booleanValue() == true)
				buff.append("SET enable_seqscan TO FALSE; ");
			else
				buff.append("SET enable_seqscan TO TRUE; ");
		}
		
		if (enable_indexscan != null)
		{
			if (enable_indexscan.booleanValue() == true)
				buff.append("SET enable_indexscan TO FALSE; ");
			else
				buff.append("SET enable_indexscan TO TRUE; ");
		}
		
		if (enable_tidscan != null)
		{
			if (enable_tidscan.booleanValue() == true)
				buff.append("SET enable_tidscan TO FALSE; ");
			else
				buff.append("SET enable_tidscan TO TRUE; ");
		}
		
		if (enable_sort != null)
		{
			if (enable_sort.booleanValue() == true)
				buff.append("SET enable_sort TO FALSE; ");
			else
				buff.append("SET enable_sort TO TRUE; ");
		}
		
		if (enable_hashagg != null)
		{
			if (enable_hashagg.booleanValue() == true)
				buff.append("SET enable_hashagg TO FALSE; ");
			else
				buff.append("SET enable_hashagg TO TRUE; ");
		}
		
		if (enable_nestloop != null)
		{
			if (enable_nestloop.booleanValue() == true)
				buff.append("SET enable_nestloop TO FALSE; ");
			else
				buff.append("SET enable_nestloop TO TRUE; ");
		}
		
		if (enable_mergejoin != null)
		{
			if (enable_mergejoin.booleanValue() == true)
				buff.append("SET enable_mergejoin TO FALSE; ");
			else
				buff.append("SET enable_mergejoin TO TRUE; ");
		}
		
		if (enable_hashjoin != null)
		{
			if (enable_hashjoin.booleanValue() == true)
				buff.append("SET enable_hashjoin TO FALSE; ");
			else
				buff.append("SET enable_hashjoin TO TRUE; ");
		}
		
		if (geqo != null)
		{
			if (geqo.booleanValue() == true)
				buff.append("SET geqo TO FALSE; ");
			else
				buff.append("SET geqo TO TRUE; ");
		}
		
		// For the numeric ones we need to know what the backend defaults were
		// as we can't infer it from the changed values
		
		// Without the defaults we can't go any further
		if (defaults == null)
		{
			// We should not be called like that...
			System.out.println("Could not generate all reset commands: no defaults!");
			Exception ex = new Exception();
			ex.printStackTrace();
			return buff.toString();
		}
		
		if (geqo_threshold != null)
		{
			buff.append("SET geqo_threshold TO " + defaults.geqo_threshold.toString() + "; ");
		}
		
		if (geqo_pool_size != null)
		{
			buff.append("SET geqo_pool_size TO " + defaults.geqo_pool_size.toString() + "; ");
		}
		
		if (geqo_effort != null)
		{
			buff.append("SET geqo_effort TO " + defaults.geqo_effort.toString() + "; ");
		}
		
		if (geqo_generations != null)
		{
			buff.append("SET geqo_generations TO " + defaults.geqo_generations.toString() + "; ");
		}
		
		if (random_page_cost != null)
		{
			buff.append("SET random_page_cost TO " + defaults.random_page_cost.toString() + "; ");
		}
		
		if (cpu_tuple_cost != null)
		{
			buff.append("SET cpu_tuple_cost TO " + defaults.cpu_tuple_cost.toString() + "; ");
		}
		
		if (cpu_index_tuple_cost != null)
		{
			buff.append("SET cpu_index_tuple_cost TO " + defaults.cpu_index_tuple_cost.toString() + "; ");
		}
		
		if (cpu_operator_cost != null)
		{
			buff.append("SET cpu_operator_cost TO " + defaults.cpu_operator_cost.toString() + "; ");
		}
		
		if (geqo_selection_bias != null)
		{
			buff.append("SET geqo_selection_bias TO " + defaults.geqo_selection_bias.toString() + "; ");
		}

		return buff.toString();
	}
}
