#ifndef __LRT_FILE_EPOC__
#define __LRT_FILE_EPOC__

#include "rtfile.h"

#include <f32file.h>

#include "rtetools.h"

#include "rtcollect.h"
#include "rtstring.h"
#include "rtchar.h"

namespace lrt {

const int File::minNameCount = 1;
const char File::separatorChar = '\\';
const String File::separator = "\\";
const bool File::caseSensitive = false;

const TTime TIMEepoch(TDateTime(1970, EJanuary, 0, 0, 0, 0, 0));

//////// Private Helper Functions /////////////
// checks if the drive specified in the given file name exists
bool driveExists(const String& fname, RFs& session)
{
	if(fname.length() == 0) return false;
	int driveNum = Char::lowerCase(fname[0]) - 'a';
	if((driveNum < 0) || (driveNum >= 26)) 
		return false;
	TDriveList drives;
	if(session.DriveList(drives) < 0)
		return false;
	return drives[driveNum];
}

// copies the file attributes from the entry to the file
void File::setAttribs(TEntry& entry)
{
	attribs->exists = ATTRIB_TRUE;
	attribs->canRead = ATTRIB_TRUE;
	attribs->canWrite = (entry.IsReadOnly() ? ATTRIB_FALSE : ATTRIB_TRUE);
	attribs->isFolder = (entry.IsDir() ? ATTRIB_TRUE : ATTRIB_FALSE);
	attribs->size = entry.iSize;
	TTimeIntervalSeconds ti;
	entry.iModified.SecondsFrom(TIMEepoch, ti);
	attribs->lastModified.sec = ti.Int();
}

/////// Real functions /////////////////

void File::initialize(const String &executableFileName)
{
	currentFolder = new File("C:\\");
	currentFolder->attribs->isFolder = ATTRIB_TRUE;
	executableFile = new File(executableFileName);
	File execFolder = executableFile->getParentFile();
	homeFolder = new File(execFolder);
	settingsFolder = new File(execFolder);
	delete currentFolder;
	currentFolder = new File(execFolder);
}

// Same as in Windows
bool File::resolveExtra(File *file, const String &fname) const
{
	if(fname.startsWith("\\"))
	{
		Array<String> *nNames = new Array<String>(file->names->length() + 1);
		(*nNames)[0] = (*names)[0];
		for(int i = 0; i < file->names->length(); i++)
			(*nNames)[i+1] = (*(file->names))[i];
		file->set(nNames);
		return true;
	}
	return false;
}

bool File::createSingleFolder(const File& folder)
{
	String fname = folder.getName();
	RFs session;
	if(session.Connect() < 0) return false;
	bool ret;
	ret = (session.MkDir(ESTRING(fname)) >= 0);
	session.Close();
	return ret;
}

// How?
void File::fetchAttributes()
{
	RFs session;
	if(session.Connect() < 0) return;
	unsigned int attr = 0;
	String fname = getName();

	if(names->length() == 1) // volumes must be handled separately, Att() fails for them
	{
		attribs->exists = (driveExists(fname, session) ? ATTRIB_TRUE : ATTRIB_FALSE);
		attribs->canRead = ATTRIB_FALSE;
		attribs->canWrite = ATTRIB_FALSE;
		return;
	}

	int ret = session.Att(ESTRING(fname), attr);
	if(ret == KErrNotFound) // file doesn't exist
	{
		attribs->exists = ATTRIB_FALSE;
		attribs->canRead = ATTRIB_FALSE;
		attribs->canWrite = ATTRIB_TRUE;
	}
	else if(ret < 0) { // Att() fails for volume identifiers (ex: "C:\"), try to catch that
		System::println("Att() failed for " + fname); 
		return; 
	}

	else { // file exists
		TEntry entry;
		if(!session.Entry(ESTRING(fname), entry)) // success
		{
			setAttribs(entry);
		}
		else 
		{
			attribs->exists = ATTRIB_TRUE;
			attribs->canRead = ATTRIB_TRUE;
			attribs->canWrite = ((attr & KEntryAttReadOnly) ? ATTRIB_FALSE : ATTRIB_TRUE);
			attribs->isFolder = ((attr & KEntryAttDir) ? ATTRIB_TRUE : ATTRIB_FALSE);
		}
	}
	session.Close();
}

// Same as in Windows
bool File::isAbsolute(const String &fname)
{
	if(fname.length() < 2) return false;
	if((fname[1] == ':') && Char::isLetter(fname[0]))
		return true;
	return false;

}

String File::getAbsolutePrefix()
{
  return String();
}

// How?
Array<File> File::listImpl(const IFilenameFilter *filter)
{
	bool acceptAll = false;
	if(filter == 0) 
		acceptAll = true;
	
	File parentFile = (isFolder() ? (*this) : getParentFile());
	String search = (isFolder() ? getName() : parentFile.getName());
	search += "*";

	Vector<File> ret(0);

	RFs session;
	if(session.Connect() < 0) { System::println("Can't connect!"); return ret; }
	CDir *dirList;
	
    if(session.GetDir(ESTRING(search),
		KEntryAttMaskSupported,ESortByName,dirList) < 0) {
		System::println("Can't GetDir!"); delete[] dirList; return ret; }
    for(int i = 0; i < dirList->Count(); i++)
	{
		TEntry curEntry = (*dirList)[i];
		String file = MYSTRING_C(curEntry.iName);
		if((file == ".") || (file == "..")) // don't include them
			continue;
		if(curEntry.IsDir()) file += separatorChar;
		if(acceptAll || filter->accept(parentFile, file))
		{
			File r = File(parentFile, file);
			r.setAttribs(curEntry);
		    ret += r;
		}
	}
	
	delete dirList;
	session.Close();

	return ret;

}

} // namespace


#endif
