/** \file
  * Contains the class <tt>File</tt>, which implements access to the
  * file system (listing folders, querying for attributes) and some associated
  * classes (file name filters like <tt>StarFilenameFilter</tt>).
  */
  
#ifndef __LRT_FILE__
#define __LRT_FILE__

#include "rtcollect.h"

#ifdef __SYMBIAN32__
class TEntry;
#endif

namespace lrt {

class File;
class String;
class InputStream;
class OutputStream;
class FileAttributes;

/** The interface IFilenameFilter can be implemented by applications which wish to filter folder
  * listings. The only method which needs to be overridden is <tt>accept()</tt>.
  */
class IFilenameFilter 
{
public:
	/** Destroys the filename filter. */
	virtual ~IFilenameFilter() {}
	/** Defines if a given file should be included in the folder listing.
	  * This method must be implemented by all filename filters.
	  * @param parent The parent folder's file descriptor for the folder which is being listed.
	  * @param child The current child file name.
	  * @return <tt>true</tt> if and only if the file described by <tt>child</tt> should be
	  *         included in the folder listing.
	  */
	virtual bool accept(const File &parent, const String &child) const = 0;
};

/** The class File defines a file descriptor in a system-independent manner.
  * It provides methods to query file attributes, open files, and list folders.
  * <p>
  * When using libRT to open files, you should specify their names separated with
  * forward slashes '<tt>/</tt>'. The File class will automatically convert them 
  * to the system dependent file name separator.  Also, when constructing a 
  * <tt>File</tt> which denotes a folder, you must append a slash to its name. 
  * <p>
  * <b>Examples: </b><br>
  * "<tt>c:/Programs</tt>" means the file called "<tt>Programs</tt>" in folder "<tt>c:</tt>".<br>
  * "<tt>/usr/bin/</tt>" means the folder "<tt>bin</tt>" in folder "<tt>usr</tt>",
  * specified as an absolute path.<br>
  * "<tt>test/test.txt</tt>" means the file "<tt>test.txt</tt>" in the subfolder 
  * "<tt>test</tt>" of the current folder (see getCurrentFolder() for this).<br>
  * "<tt>../xyz.txt</tt>" means the file "<tt>xyz.txt</tt>", one folder up from the
  * current folder. <br>
  */
class File
{
public:
	/** The character used to separate path components in <tt>File</tt>s. 
	  * For example, it is '<tt>/</tt>' in Unix, and '<tt>\\</tt>' in Windows. */
	static const char separatorChar;
	/** The character used to separate path components in <tt>File</tt>s.
	  * Represented as a String for your convenience. */
	static const String separator;
	/** Are file names case sensitive? */
	static const bool caseSensitive;

	/** Sets the "current folder" to the given new folder. The current folder is
	  * the one, to which all relative paths are resolved. <br>
	  * Example: If the current folder is <tt>c:\\Programs\\</tt> and you open 
	  * file <tt>../test.dat</tt>, in fact, <tt>c:\\test.dat</tt> will be opened. */
	static void setCurrentFolder(const File &newFolder);
	/** Returns the "current folder". All relative paths are resolved to this
	  * folder. */
    static const String getCurrentFolder();
	/** Returns the executable program's file name (if available). Note that the
	  * <tt>rtMain()</tt> method is <b>not</b> passed the executable file name 
	  * within <tt>args</tt>. */
	static const File getExecutableFile();
	/** Returns a file descriptor for the current user's home folder. 
	  * Under Unix systems, this is the contents of the $HOME environment variable, 
	  * under Windows, the "My Documents" folder is returned. */
	static const File getHomeFolder();
	/** Returns a file descriptor for the current user's LRT settings folder. 
	  * This folder may be used to store application preferences and settings files. 
	  * Under Unix, this is the $HOME/.lrt/ folder, on Windows it is either the 
	  * "Application Data" folder (W2K etc.) or the program's folder. Note that you 
	  * may have to create the settings folder first using createFolder(). */
	static const File getSettingsFolder();
	/** Creates the folder as given by its file descriptor. If the given file descriptor
	  * is a file, its parent is created instead. 
	  * @return <tt>true</tt> if the folder already existed or could be created 
	  *         successfully, <tt>false</tt> if it couldn't be created.
	  */
	static bool createFolder(const File& folder);
	/** Initializes the <tt>File</tt> class. You should never call this method 
	  * from applications. <br>
	  * Implementation note: Must ensure that all static Files are set up to something. */
	static void initialize(const String &executableFileName);
	/** Finalizes the <tt>File</tt> class. As this method is automatically called 
	  * by <tt>System::exit()</tt>, you should never have to call it yourself from
	  * applications. */
	static void finalize();
	/** Checks if the given file name is absolute. */
	static bool isAbsolute(const String &fname);

	/** Creates an invalid file descriptor. */
	File();
	/** Creates a file descriptor for the given file name. If the given name is not
	  * absolute, it is resolved against the current folder. */
	File(const String &name);
	/** Creates a file descriptor by copying over the given one. */
	File(const File &);
	/** Creates a file descriptor for the given file name. If the name is not absolute,
	  * it is resolved against the given parent file descriptor. 
	  * @param parent The parent file descriptor. 
	  * @param child The child file name. */
	File(const File &parent, const String &child);
	/** Copies all data over from the given file descriptor. */
	File &operator=(const File &);
	~File();
	/** Returns the absolute file name, using the system's path separator, of this file.
	  */
	String getName() const;
	/** Returns the file name, relative to the folder in which it is contained. 
	  * (This is the local file name without any folders prefixed.)
	  */
	String getFileName() const;
	/** Returns a local file name, using the system's path separator, of this file.
	  * @param relFile (optional) The folder against which the localization should be
	  *                performed. 
	  */
	String getLocalName(const File& relFile = (*currentFolder)) const;

	/** Returns the absolute file name, using the system's path separator, of this
	  * file's parent folder. */
	String getParent() const;
	/** Returns a file descriptor for this file's parent folder. If this file descriptor
	  * already denotes a root folder, it is returned unchanged. */
	File getParentFile() const;

	/** Returns <tt>true</tt> if this file descriptor denotes a folder. */
	bool isFolder() const;
	/** Returns <tt>true</tt> if this file descriptor denotes a normal file (and not 
	  * a folder). */
	bool isFile() const;
	/** Returns <tt>true</tt> if this file exists. */
	bool exists();
	/** Returns <tt>true</tt> if the application can write to the file denoted by
	  * this file descriptor. */
	bool canWrite();
	/** Returns <tt>true</tt> if the application can read from the file denoted by
	  * this file descriptor. */
	bool canRead();
	/** Returns the file's size, or -1 if it does not exist. */
	int getSize();
	/** Returns the file's last modification date, or the epoch if it doesn't exist. */
	Time getLastModified();

	/** Opens this file for reading, if it exists. You should always call <tt>fail()</tt>
	  * on the returned stream before using it to check if it was opened properly. 
	  * @param textMode If you want to read text data (as opposed to binary data) 
	  *                 from the stream, you should set this flag. */
	InputStream *openRead(bool textMode);
	/** Opens this file for overwriting, if it exists and can be written to. */
	OutputStream *openWrite();
	/** Opens this file for appending, if it exists and can be written to. */
	OutputStream *openAppend();

	/** Lists all files and folders contained in this folder. If this file descriptor 
	  * does not denote a folder, the contents of its parent folder are returned instead.
	  * If this file doesn't exist, a zero-length array is returned. <br>
	  * You should not assume that the returned array is in any particular order. */
	Array<File> list();
	/** Lists all files and folders contained in this folder which match the given
	  * filename filter. <br>
	  * You should not assume that the returned array is in any particular order. */
	Array<File> list(const IFilenameFilter*);

private:
	static File *executableFile;
	static File *homeFolder;
	static File *settingsFolder;
	static File *currentFolder;
	static const int minNameCount;

	static String getAbsolutePrefix();

	// sysdep implementation for createFolder() - systems generally only support
	// creating one subfolder at once
	static bool createSingleFolder(const File& folder);

	enum Attrib { ATTRIB_TRUE, ATTRIB_FALSE, ATTRIB_UNKNOWN };
    
	/// A file's attributes.
	class FileAttributes {
	public:
		FileAttributes();
		FileAttributes(const FileAttributes &);

		Attrib exists, isFolder, canWrite, canRead;
		int size;
		Time lastModified;
	};

	File(Array<String>* names, FileAttributes* attribs);

	void fetchAttributes();
	void resolve(File *file, const String& fname) const;
	bool resolveExtra(File *file, const String& fname) const; // system-dependent resolve operations
	//! "." and ".." must not be included into the folder listing!
	//! If your system's list function already delivers file attributes, feel free to set them.
	Array<File> listImpl(const IFilenameFilter *);
	
	void set(Array<String>*);
	void set(Array<String>*, FileAttributes*);

#ifdef __SYMBIAN32__
	void setAttribs(TEntry& entry);
#endif

	Array<String>* names;
	FileAttributes* attribs;

	friend class File::FileAttributes;
};

/** A filename filter which can create folder listings for queries using star
  * patterns. Examples are "<tt>*.txt</tt>" and "<tt>c:\\Programs\\*.exe</tt>".
  * @see getFiles()
  */
class StarFilenameFilter : public IFilenameFilter {

public:
	/** Returns an array including all the files which match the given query.
	  * The query may include any number of star characters '<tt>*</tt>', which
	  * are matched with any number of characters. Question marks, which are matched
	  * by exactly one character, are also allowed. 
	  * This method is normally preferable to using <tt>File::list()</tt> with
	  * a self-constructed <tt>StarFilenameFilter</tt> as it can handle paths
	  * in the query, too. <p>
	  * <b>Example queries:</b> "<tt>*.txt</tt>", "<tt>c:\\Programs\\*.exe</tt>".<br>
	  * <b>Note:</b> Stars in the path part of the query (such as "<tt>c:\\P*\\hello.txt</tt>")
	  * are not allowed. */
	static Array<File> getFiles(const String& query);

	/** Filters the folder listing. */
	virtual bool accept(const File& parent, const String& child) const;

	virtual ~StarFilenameFilter();

	/** Creates a StarFilenameFilter for a given query, which may include any number
	  * of star characters. 
	  * <b>Note:</b> Unless you are sure that <tt>query</tt> doesn't include 
	  * a path part, use <tt>getFiles()</tt> instead. 
	  * @see getFiles() */
	StarFilenameFilter(const String& query);
private:
	Vector<String> queryComps;
	// like a.indexOf(b, pos) but with question marks
	int findQM(const String& a, const String& b, int fromPos = 0) const;
};



} // namespace


#endif
