/////////////////////// rtmath.h /////////////////////////
/** \file
  * Contains the class Math, which provides static helper functions
  * and mathematical constants.
  * @see Math
  */

#ifndef __LRT_MATH__
#define __LRT_MATH__

namespace lrt {

/** Class Math provides some static helper functions to compute often-used mathematical functions
  * in a system-independent manner.
  * Currently under construction.
  */
class Math  
{
public:
	/** The largest positive value which is representable as a 32-bit integer. */
	static const int MAX_INT;
	/** The largest negative value which is representable as a 32-bit integer. */
	static const int MIN_INT;

	/** The most possibly exact representation of the number Pi. */
	static const double PI;
	/** The most possibly exact representation of the number <b>e</b> (the
	  * base of the natural logarithm). */
	static const double E;

	
	/** Returns the bigger one of the given integers. 
	  * Note: This function is inline and will be translated into the usual code.
	  * So please use it to enhance code readability. */
	static inline int max(int num1, int num2);
	/** Returns the smaller one of the given integers. 
	  * Note: This function is inline and will be translated into the usual code.
	  * So please use it to enhance code readability. */
	static inline int min(int num1, int num2);
	/** Returns the absolute value of the given integer. 
	  * Note: This function is inline and will be translated into the usual code.
	  * So please use it to enhance code readability. */
	static inline int abs(int num);

	/** Creates a pseudorandom number between (and including) the values <tt>from</tt>
	  * and <tt>to</tt>.<br>
	  * If you do not use seed() or randomize() before using rand(), it will
	  * initialize itself to return randomized numbers.
	  */
	static int rand(int from, int to);

	/** Creates reproducable pseudorandom numbers by seeding the random number
	  * generator with a fix number. 
	  */
	static void seed(unsigned long seed);

	/** Creates unreproducable, but "really random" numbers by seeding the random
	  * number generator with a number from a random source (usually the system timer).
	  */
	static void randomize();


	/// Computes the square root of num. 
	static double sqrt(double num);
	static double log10(double num);
	static double ln(double num);
	static double exp(double num);
	static double sin(double num);
	static double cos(double num);
	static double tan(double num);

private:
	static bool randomInit;
	Math();
	~Math();
};

} // namespace


#include "rtmath.inline.cpp"

#endif

