//////////////////////// rtsound.h //////////////////////
/** \file
  * Contains the class Sound, a simple digital sound playback mechanism.
  * Note that sounds are currently not supported on all platforms.
  * @see Sound
  */

#ifndef __LRT_SOUND__
#define __LRT_SOUND__

#include "rtstring.h"

namespace lrt {

/** Implements a simple sound playing mechanism. 
  * For the purpose of this Sound class, sounds are represented by logical sound 
  * names, i.e. file names without extension (such as "<tt>start</tt>"). Whenever
  * an application requests to play a given sound name, libRT looks for the sound 
  * file (with a system specific extension) first in the current folder, then in 
  * the executable file folder, and finally in the subfolder <tt>sounds/</tt> of 
  * the executable file folder. */
class Sound  
{
public:
	/** Returns whether sound playing is supported on this platform. */
	static bool isSupported();
	/** Returns <tt>true</tt> if the sound given by its logical name is found. */
	static bool isPresent(const String& logicalSoundName);
	/** Plays a sound. 
	  * @param logicalSoundName The logical name of the requested sound. 
	  * @param sync (optional) If the sound should be played synchronously
	  *				(normally not).
	  *				Synchronous playing means that the play() method only returns when the 
	  *				sound has finished playing. Asynchronous playing means that the play() 
	  *				method returns immediately. 
	  * @param stop (optional) Stop other playing sounds, if that is needed to play this sound.
	  *				If <tt>stop</tt> is not given and other sounds are playing, this sound
	  *				will not be played. */
	static bool play(const String& logicalSoundName, bool sync = false, bool stop = true);
	/** Returns this platform's sound file extension. (For example: "<tt>.wav</tt>" for 
	  * Windows.) */
	static String getExtension();

private:
	Sound() {}
	~Sound() {}
	static String resolve(const String& logicalSoundName);

};

} // namespace

#endif
