////////////////////////// rtstring.h ///////////////////
/** \file
  * Contains the class String, libRT's string implementation.
  * String contains many powerful methods to convert strings, create substrings,
  * search substrings and compare strings (among others). <br>
  * The String methods are quite similar (but not the same) to Java's String class.
  */

#ifndef __LRT_STRING__
#define __LRT_STRING__

#include "rtcollect.h"
#include "rtmath.h"

namespace lrt {

/** A sequence of characters. 
  * The String class of libRT implements many methods for comfortable handling of strings, 
  * including conversion from and to numbers, case conversions, substrings, searches, 
  * comparisons, etc.
  */
class String : public Vector<char>  
{
public:
	/** Creates an empty string. */
	String();

	/** Creates a string from an C-style character pointer.
	  * This constructor is also used to implicitely convert all string literals to libRT
	  * Strings.
	  */
	String(const char * cStr);

	/** Creates a string consisting of <tt>length</tt> equal characters, namely <tt>chr</tt>.
	  * Example: String('a', 7) becomes "aaaaaaa"
	  */
	String(const char chr, int length);
//	String(const short * ucStr);
	
    /** Creates a string representing a given integer number.
	  * @param num The number to represent.
	  * @param radix (optional) The radix to use for number conversion (e.g. 2 for binary, 10 for decimal).
	  *              radix may not be smaller than 2, nor larger than Char::MAX_RADIX.
	  * @param length (optional) The length the returned String should have. 
	  *                  Places not used by the number conversion are filled with spaces. 
	  */
	String(int num, int radix = 10, int length = 0);

    /** Creates a string representing a given floating-point number.
	  * @param num The number to represent.
	  * @param minLength (optional) The minimum length the returned String is required to have. 
	  *                  The places are all filled with numbers. 
	  */
	explicit String(double num, int minLength = 13);

	/** Creates a copy of String str. */
	String(const String& str);

/*	String operator += (const char *); // impl. Konv. per Konstruktor */
	
	/** Appends a single character (or character literal) to this string. 
	  * @return A reference to this string. 
	  */
	String &operator += (const char);
    
	/** Appends the numerical (decimal) representation of a number to this string.
	  * @return A reference to this string. 
	  */
	String &operator += (const int);    // impl. Konv. per Konstruktor tut nicht

	/** Appends another String to this String.
	  * @return A reference to this string. 
	  */
	String &operator += (const String&);

	/** Clears this String, then copies the given String's contents over to this one.
	  * @return A reference to this string. */
	String& operator= (const String&);

    /** Returns a constant reference to the character at position <tt>index</tt>,
	  * if <tt>index</tt> is not contained within this string, exits the application. */
//	const char& operator[](int index) const;

    /** Returns a modifiable reference to the character at position <tt>index</tt>,
	  * if <tt>index</tt> is not contained within this string, appends enough zeroes to it. */
//    char& operator[](int index);

	/** Returns a new string like this one, but without whitespace at its start and end.
	  * For a definition of whitespace, 
	  * @see Char::isSpace()
	  */
	String trim() const;
	
	/** Returns a new string which represents a part of this string.
	  * @param start Begin index of the new string.
	  * @param end (optional) Position <b>after</b> the end index of the new string. If
	  *            not specified, the new string extends until the end of this string.
	  */
	String substring(int start, int end = Math::MAX_INT) const;

	/** Returns a new string which is the lowercase equivalent of this string. */
	String lowerCase() const;

	/** Returns a new string which is the uppercase equivalent of this string. */
    String upperCase() const;
	
	// OBSOLETE. Now in Array<>. However, we still need it so that this method is selected,
	//           and not the next one (where a String would be constructed from (int)ch)
	/** Finds the first occurence of a given character in this string. 
	  * @param ch The character you'd like to search for. 
	  * @param fromIndex (optional) The position where the search starts. 
	  *                  If fromIndex is not specified, the whole String will be searched. 
	  * @return The index of the first occurence of <tt>ch</tt>, or <tt>-1</tt> if
	  *         <tt>ch</tt> is not present in this String. */
	inline int indexOf(char ch, int fromIndex = 0) const { return Array<char>::indexOf(ch, fromIndex); }
	/** Finds the first occurence of a given substring in this string. 
	  * @param str The string you'd like to search for. 
	  * @param fromIndex (optional) The position where the search starts. 
	  *                  If fromIndex is not specified, the whole String will be searched. 
	  * @return The index of the first occurence of <tt>str</tt>, or <tt>-1</tt> if
	  *         <tt>str</tt> is not present in this String. */
	int indexOf(const String &str, int fromIndex = 0) const;
	/** Finds the first occurence of a given character in this string, searching backwards
	  * from the end of this string. 
	  * @param ch The character you'd like to search for. 
	  * @param fromIndex (optional) The position where the search starts. 
	  *                  If fromIndex is not specified, the whole String will be searched. 
	  * @return The index of the last occurence of <tt>ch</tt>, or <tt>-1</tt> if
	  *         <tt>ch</tt> is not present in this String. */
	int lastIndexOf(char ch, int fromIndex = Math::MAX_INT) const;
	/** Finds the first occurence of a given substring in this string, searching backwards
	  * from the end of this string. 
	  * @param str The substring you'd like to search for. 
	  * @param fromIndex (optional) The position where the search starts. 
	  *                  If fromIndex is not specified, the whole String will be searched. 
	  * @return The index of the last occurence of <tt>str</tt>, or <tt>-1</tt> if
	  *         <tt>str</tt> is not present in this String. */
	int lastIndexOf(const String &str, int fromIndex = Math::MAX_INT) const;
	
	/** Returns <tt>true</tt> if all characters of this string are exactly equal to 
	  * the given string. */
	inline bool operator == (const String&) const;
	/** Returns <tt>true</tt> if any character, or the length, of this string
	  * differ from the given string. */
	inline bool operator != (const String&) const;
	/** Returns <tt>true</tt> if this string should be ordered before the given string. 
	  * This operator is case sensitive, and will not work well for non-English text. */
	inline bool operator < (const String&) const;
	/** Compares this string with another string <tt>str</tt>. 
	  * This method is case sensitive, and will not work well for non-English text.
	  * @return A negative number, zero, or a positive number as this string is less than,
	  *         equal or more than <tt>str</tt>. */
	inline int compare(const String& str) const;
	/** Compares this string with another string <tt>str</tt>, ignoring case. 
	  * This method is case insensitive, and will not work well for non-English text.
	  * @return A negative number, zero, or a positive number as this string is less than,
	  *         equal or more than <tt>str</tt>. */
	inline int compareIgnoreCase(const String& str) const;
	/** Compares two strings. 
	  * This method is case sensitive, and will not work well for non-English text.
	  * @return A negative number, zero, or a positive number as the first string is less than,
	  *         equal or more than the second. */
	static inline int compare(const String& str1, const String& str2);
	/** Compares two strings, ignoring case. 
	  * This method is case insensitive, and will not work well for non-English text.
	  * @return A negative number, zero, or a positive number as the first string is less than,
	  *         equal or more than the second. */
	static inline int compareIgnoreCase(const String& str1, const String& str2);
	/** Returns <tt>true</tt> if this string starts with the given substring <tt>str</tt>.*/
	inline bool startsWith(const String &str) const;
	/** Returns <tt>true</tt> if this string ends with the given substring <tt>str</tt>.*/
	inline bool endsWith(const String &str) const;
	
	/** Returns a new string, in which all occurencies of <tt>oldCh</tt> are replaced 
	  * by <tt>newCh</tt>. */
	String replace(char oldCh, char newCh) const;
	/** Returns a new string, in which all occurencies of <tt>oldStr</tt> have been replaced
	  * by <tt>newStr</tt>. */
	String replace(const String &oldStr, const String &newStr) const;
	/** Flags for split().*/
	enum SplitFlags {
		NONE = 0,
		CREATE_EMPTY = 1, ///< create empty parts between, if two separators follow immediately. Use this for CSV or similar data.
	};
	/** Splits this string into parts. 
	  * The string is split into portions wherever one of the <tt>separators</tt> is found. 
	  * However, matching pairs of <tt>combiners</tt> can be used to group words together.
	  * There is automatically any whitespace removed from the returned parts (as defined
	  * by <tt>Char::isSpace()</tt>).
	  * The separators themselves are not included in the returned parts. 
	  * <p>
	  * <b>Example:</b> The string "<tt>"Hello, my ",dear , world!</tt>", using the
	  * separators "<tt> </tt>" and "<tt>,</tt>" and the combiner "<tt>"</tt>", will be split
	  * into the following parts: "<tt>Hello, my</tt>", "<tt>dear</tt>", "<tt>world!</tt>".
	  *
      * @param separators (optional) A list of all separators which may be used to split the string. 
	  * @param combiners (optional) A list of all combiners which may be used to group words together. 
	  * @param maxParts (optional) The maximum number of parts the string will be split into.
	  *                 Normally, this value is unlimited. 
	  * @param flags (optional) More flags to customize the split operation. 
	  * */
	Array<String> split(const String &separators = ", ", const String &combiners = "\"\'", 
		int maxParts = -1, SplitFlags flags = NONE) const;
	
	/** Joins an array of <tt>String</tt>s together. 
	  * @param arr The array to be joined together. 
	  * @param joiner (optional) The joiner string which is inserted between every two array 
	  *               entries. Useful examples: "<tt>, </tt>" or "<tt>\\n</tt>". 
	  * @param upTo (optional) The maximum array index which is included in the returned string. */
	static String join(const Array<String> &arr, const String &joiner = ", ", const int upTo = Math::MAX_INT);

	/** Returns a constant pointer to this string's data. This string is zero-terminated before 
	  * returning the pointer, but any subsequent String modifications (through the String 
	  * methods, as you cannot modify the string using the returned pointer) will probably 
	  * make the pointer useless. <br>
	  * <b>Warning:</b> If this String is destroyed, the pointer becomes invalid. So do never
	  * call this method on temporary objects! */
	const char *cStr() const;
	/** Parses this String as an integer. 
	  * As many numbers as possible are parsed from the beginning of this String. If there
	  * are no valid numbers at the beginning of this String (considering the given 
	  * <tt>radix</tt>), the given default value is instead returned. 
	  * @param def The default value, which is returned if this String cannot be interpreted
	  *            as a number. 
	  * @param radix (optional) The radix which is used to convert the String. Valid values
	  *              are from 2 up to <tt>Char::MAX_RADIX</tt>. */
	int intValue(int def, int radix = 10) const;
	/** Parses this String as a floating-point value.
	  * As many numbers as possible are parsed from the beginning of this String. If there
	  * are no valid numbers at the beginning of this String, the given default value is 
	  * instead returned. 
	  * @param def The default value, which is returned if this String cannot be interpreted
	  *            as a number. 
	  */
	double doubleValue(double def) const;

	virtual ~String();

private:
	String(const String& str, int start, int end);
	int compare(int thisPos, const String& other, int otherPos, int length) const;
	int compareIgnoreCase(int thisPos, const String& other, int otherPos, int length) const;
	int findNextOf(const String &what, int start) const;
	static int cStrLength(const char *);
	
};

/** Returns a new String, which is the concatenation of the two given Strings. */
String operator + (const String&, const String&);

} // namespace


#include "rtstring.inline.cpp"

#endif
