/** @file robloader.cpp
  * Loads robots from source files (.ROB).
  */

#include "robloader.h"
#include "robbase.h"
#include "robvars.h"
#include "robstrings.h"
#include "robinstr.h"

#include <rtstring.h>
#include <rtcollect.h>
#include <rtlist.h>
#include <rtmap.h>
#include <rtmath.h>

using namespace lrt;

namespace rt {

RobLoader::RobLoader(const ErrorHandler* handler) : 
	ProgramLoader(Globals(handler), handler, ""), 
	isExample(true), labels(false)
{
}

RobLoader::~RobLoader()
{
}

bool RobLoader::canLoad(const String& filename)
{
	if(filename.lowerCase().endsWith(".rob"))
		return true;
	else return false;
}

String RobLoader::getFormatName()
{
	return "Robot Source Files (*.rob)";
}


ProgramLoader* RobLoader::create(const String& filename, const Globals& glob)
{
	RobLoader* ret = new RobLoader(this, filename, glob);
	return ret;
}

RobLoader::LoadReturnType RobLoader::load()
{
	LoadReturnType ret;
	if(isExample) {
		ret.success = false;
		ret.message = "Example loaders cannot really load programs!";
		return ret;
	}
	/// load bot...
	if(!headers) headers = new StringMap<Program::HeaderField>;
	if(!banks) banks = new Vector<Bank*>(0);

	DefineInputStream* in = new DefineInputStream(new FileInputStream(filename, true), 
		DefineInputStream::whitespace + "(),", ';');
	in->setIgnoreComments(true);
	in->setAutoTrim(true);
	if(in->fail()) { delete in; return error(rlFailNoFile, 0); }
	
	RobLoadReturnType rret = doLoadHeaders(in);
	if(rret == rlOK)
		rret = doLoad(in);

	ret = error(rret, in);
	in->close();
	delete in;

	return ret;
}

RobLoader::LoadReturnType RobLoader::loadHeaders()
{
	LoadReturnType ret;
	if(isExample) {
		ret.success = false;
		ret.message = "Example loaders cannot really load programs!";
		return ret;
	}
	/// load headers...
	if(!headers) headers = new StringMap<Program::HeaderField>(false);

	DefineInputStream* in = new DefineInputStream(new FileInputStream(filename, true), 
		DefineInputStream::whitespace + "(),", ';');
	in->setIgnoreComments(true);
	in->setAutoTrim(true);
	if(in->fail()) { delete in; return error(rlFailNoFile, 0); }
	
	RobLoadReturnType rret = doLoadHeaders(in);

	ret = error(rret, in);
	in->close();
	delete in;

	return ret;
}

RobLoader::RobLoader(const RobLoader* example, const String& filename, const Globals& glob) :
	ProgramLoader(glob, example->handler, filename), isExample(false),
	labels(false)
{
}

RobLoader::LoadReturnType RobLoader::error(RobLoadReturnType num, DefineInputStream* in)
{
	LoadReturnType ret;
	ret.success = (num == rlOK);
	ret.message = "";
	if(in) ret.message += String(in->getLineNumber()) + ": ";
	ret.message += robLoadFailMsg[num];
	return ret;
}

RobLoadReturnType RobLoader::convertParamDest(const String& word, Op& op)
{
  String myWord = word.lowerCase(); // all comparisons are lowercase
  
  op.isRemote = false; // most ops are not remote
  
  if(myWord.startsWith("##"))
  {
	  if(!glob.enableRC3Instr) return rlFailRC3Required;
	  op.type = opTypeArray;
	  op.ptr = myWord.substring(2).intValue(-1);
	  if((op.ptr < 1) || (op.ptr > glob.maxVars))
		  return rlFailUnknownParam;
	  return rlOK;
  }
  else if(myWord.startsWith("#")) 
  {
	  op.type = opTypeActive;
	  if(myWord == "#active")
		  op.ptr = botActive;
	  else if(myWord == "#pub")
	  {
		  if(!glob.enableRC3Instr) return rlFailRC3Required;
		  op.type = opTypeGlobal;
		  op.ptr = simPub;
	  }
	  else {
		  op.type = opTypeUnnamedVar;
		  op.ptr = myWord.substring(1).intValue(-1);
		  if((op.ptr < 1) || (op.ptr > glob.maxVars))
			  return rlFailUnknownParam;
		  return rlOK;
	  }
  }
  else if(myWord.startsWith("%"))
  {
	  op.isRemote = true;
	  op.type = opTypeActive;
	  if(myWord == "%active")
		  op.ptr = botActive;
	  else
		  return rlFailUnknownParam;
  }
  else
	return rlFailUnknownParam;
  return rlOK;
}

RobLoadReturnType RobLoader::convertParamSrc(const String& word, Op& op)
{
  String myWord = word.lowerCase(); // all comparisons are lowercase
  
  op.isRemote = false; // most ops are not remote
  
  if(myWord.startsWith("##"))
  {
	  if(!glob.enableRC3Instr) return rlFailRC3Required;
	  op.type = opTypeArray;
	  op.ptr = myWord.substring(2).intValue(-1);
	  if((op.ptr < 1) || (op.ptr > glob.maxVars))
		  return rlFailUnknownParam;
	  return rlOK;
  }
  else if(myWord.startsWith("#")) 
  {
	  op.type = opTypeActive;
	  if(myWord == "#active")
		  op.ptr = botActive;
	  else if((myWord == "#pub"))
	  {
		  if(!glob.enableRC3Instr) return rlFailRC3Required;
		  op.type = opTypeGlobal;
		  op.ptr = simPub;
	  }
	  else {
		  op.type = opTypeUnnamedVar;
		  op.ptr = myWord.substring(1).intValue(-1);
		  if((op.ptr < 1) || (op.ptr > glob.maxVars))
			  return rlFailUnknownParam;
		  return rlOK;
	  }
  }
  else if(myWord.startsWith("$")) 
  {
	  op.type = opTypeNamedVar;
	  if(myWord == "$banks")
		  op.ptr = botNumBanks;
	  else if(myWord == "$mobile")
		  op.ptr = botMobile;
	  else if(myWord == "$instrset")
		  op.ptr = botInstrSet;
	  else if(myWord == "$generation")
	  {
		  if(!glob.enableRC3Instr) return rlFailRC3Required;
		  op.ptr = botGeneration;
	  }
	  else if(myWord == "$instrpos")
	  {
		  if(!glob.enableRC3Instr) return rlFailRC3Required;
		  op.type = opTypeTask;
		  op.ptr = taskCurInstr;
	  }
	  else if(myWord == "$fields") 
	  {
		  op.type = opTypeFields;
		  op.ptr = simFields;
	  }
	  else if(myWord == "$age")
	  {
		  if(!glob.enableRC3Instr) return rlFailRC3Required;
		  op.type = opTypeAge;
		  op.ptr = 0;
	  }
	  else if(myWord == "$elimtrigger")
	  {
		  if(!glob.enableRC3Instr) return rlFailRC3Required;
		  op.type = opTypeGlobal;
		  op.ptr = simElimTrigger;
	  }
	  else if(myWord == "$time")
	  {
		  if(!glob.enableRC3Instr) return rlFailRC3Required;
		  op.type = opTypeGlobal;
		  op.ptr = simTime;
	  }
	  else if(myWord == "$maxvars")
	  {
		  if(!glob.enableRC3Instr) return rlFailRC3Required;
		  op.type = opTypeGlobal;
		  op.ptr = simMaxVars;
	  }
	  else if(myWord == "$maxtasks")
	  {
		  if(!glob.enableMultitasking) return rlFailMuTaRequired;
		  op.type = opTypeGlobal;
		  op.ptr = simMaxTasks;
	  }
	  else if(myWord == "$maxgeneration")
	  {
		  if(!glob.enableRC3Instr) return rlFailRC3Required;
		  op.type = opTypeGlobal;
		  op.ptr = simMaxGeneration;
	  }
	  else if(myWord == "$maxmybots")
	  {
		  if(!glob.enableRC3Instr) return rlFailRC3Required;
		  op.type = opTypeGlobal;
		  op.ptr = simMaxMybots;
	  }
	  else if(myWord == "$maxlifetime")
	  {
		  if(!glob.enableRC3Instr) return rlFailRC3Required;
		  op.type = opTypeGlobal;
		  op.ptr = simMaxLifetime;
	  }
	  else if(myWord == "$timeout")
	  {
		  if(!glob.enableRC3Instr) return rlFailRC3Required;
		  op.type = opTypeGlobal;
		  op.ptr = simTimeout;
	  }
	  else if(myWord == "$randnum")
	  {
		  if(!glob.enableRC3Instr) return rlFailRC3Required;
		  op.type = opTypeRandom;
		  op.ptr = 0;
	  }
	  else if(myWord == "$id")
	  {
		  if(!glob.enableRC3Instr) return rlFailRC3Required;
		  op.type = opTypeProgram;
		  op.ptr = progId;
	  }
	  else if(myWord == "$mybots")
	  {
		  if(!glob.enableRC3Instr) return rlFailRC3Required;
		  op.type = opTypeProgram;
		  op.ptr = progMybots;
	  }
	  else if(myWord == "$otherbots")
	  {
		  if(!glob.enableRC3Instr) return rlFailRC3Required;
		  op.type = opTypeProgram;
		  op.ptr = progOtherbots;
	  }
	  else if(myWord == "$tasks")
	  {
		  if(!glob.enableMultitasking) return rlFailMuTaRequired;
		  op.ptr = botNumTasks;
	  }
	  else
		  return rlFailUnknownParam;
  }
  else if(myWord.startsWith("%"))
  {
	  op.isRemote = true;
	  op.type = opTypeNamedVar;
	  if(myWord == "%active")
	  {
		  op.type = opTypeActive;
		  op.ptr = botActive;
	  }
	  else if(myWord == "%banks")
		  op.ptr = botNumBanks;
	  else if(myWord == "%mobile")
		  op.ptr = botMobile;
	  else if(myWord == "%instrset")
		  op.ptr = botInstrSet;
	  else if(myWord == "%age")
	  {
		  if(!glob.enableRC3Instr) return rlFailRC3Required;
		  op.type = opTypeAge;
		  op.ptr = 0;
	  }
	  else
		  return rlFailUnknownParam;
  }
  else if(myWord.startsWith("@@"))
  {
	  if(!glob.enableRC3Instr) return rlFailRC3Required;
	  op.type = opTypeAbsLabel;
	  if(labels.isSet(myWord.substring(2)))
	    op.ptr = labels.get(myWord.substring(2));
	  else
		return rlFailUndefinedLabel;
  }
  else if(myWord.startsWith("@"))
  {
	  op.type = opTypeLabel;
	  if(labels.isSet(myWord.substring(1)))
	    op.ptr = labels.get(myWord.substring(1));
	  else
		return rlFailUndefinedLabel;
  }
  else 
  {
	  op.type = opTypeNumber;
	  int num = myWord.intValue(Math::MIN_INT);
	  if(num == Math::MIN_INT) 
		  return rlFailUnknownParam;
	  op.ptr = (rint)num;
  }
  return rlOK;
}

RobLoadReturnType RobLoader::mkOldHeader(DefineInputStream* in, const String& name)
{
	String text = in->getLine(false); // if the rest-of-line is empty, still use it
	headers->put(name, Program::HeaderField(name, text, headerPublished));
	return rlOK;
}

RobLoadReturnType RobLoader::mkNewHeader(DefineInputStream* in, HeaderType type)
{
	String name; in->getWord(name, DefineInputStream::whitespace);
	String value = in->getLine(false); // if the rest-of-line is empty, still use it
	headers->put(name, Program::HeaderField(name, value, type));
	return rlOK;
}

/// after finishing, file position is before first "Bank" or "Define" in file
RobLoadReturnType RobLoader::doLoadHeaders(DefineInputStream* in)
{
  String word; in->mark(); in->getWord(word, DefineInputStream::whitespace);
  do{
	word = word.lowerCase();   // all string comparisons are lower case here
	if((word == "name") || (word == "author") || (word == "country"))
	{
		RobLoadReturnType ret = mkOldHeader(in, word);
		if(ret) return ret;
		in->mark(); in->getWord(word, DefineInputStream::whitespace);
	}
	else if(word == "published")
	{
		RobLoadReturnType ret = mkNewHeader(in, headerPublished);
		if(ret) return ret;
		in->mark(); in->getWord(word, DefineInputStream::whitespace);
	}
	else if(word == "secret")
	{
		RobLoadReturnType ret = mkNewHeader(in, headerSecret);
		if(ret) return ret;
		in->mark(); in->getWord(word, DefineInputStream::whitespace);
	}
	else if(word == "internal")
	{
		RobLoadReturnType ret = mkNewHeader(in, headerInternal);
		if(ret) return ret;
		in->mark(); in->getWord(word, DefineInputStream::whitespace);
	}
	else if((word == "bank") || (word == "define"))
	{
		in->reset();
		break;
	}
	else
	  return rlFailExpectBank;
  }while(true);
  return rlOK;
}

RobLoadReturnType RobLoader::doLoad(DefineInputStream* in)
{
  in->setEnableDefines(true); // enable defines
  
  in->mark(); // store current position (after headers)
  RobLoadReturnType ret = getLabels(in);
  if(ret) return ret;
  in->reset(); // restore position after headers

  String word; 
  ret = in->getWord(word); // read first "Bank"
  if(ret) return ret;
  word = word.lowerCase();

  Bank *newBank = 0;

  // read banks
  for( rint i=0; !in->eos(); i++ )
  {
      if( word == "bank" )
      {
		  if(i >= glob.maxBanks)
			  return rlFailTooManyBanks;
          String bankName = in->getLine(false);
		  if(bankName == "")
			  bankName = "Unnamed Bank";
		  newBank = new Bank(owner, bankName);
      }
      else // really strange
		  return rlFailExpectBank;

      // **********************
      // Read instructions

      int j; // Instruktionsanzahl
      for( j=0; true; j++ )
      {
		  if(j >= glob.maxInstr)
			  return rlFailTooManyInstrs;
          ret = in->getWord(word);
		  if(ret) return ret;
		  word = word.lowerCase(); // all String comparisons are lower-case
		  // (word == "") means EOF
		  if((word == "") || (word == "bank")) // end of bank
			  break;

		  if( word[0] == '@' ) // label
          {
              in->skipRestOfLine();
              j--;       // j must NOT be increased!
              continue;
          }

          // it's an instruction
          if( word == "die" )
          {
              newBank->instr += new InstrDie(glob);
              in->skipRestOfLine();
              continue;
          }

          if( word == "move" )
          {
              newBank->instr += new InstrMove(glob);
              in->skipRestOfLine();
              continue;
          }

          if( word == "turn" )
          {
			  Array<Op> ops(1);
			  bool dest[] = {false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrTurn(glob, ops);

              continue;
          }

          if( word == "scan" )
          {
			  Array<Op> ops(1);
			  bool dest[] = {true};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrScan(glob, ops);

              continue;
          }

          if( word == "farscan" )
          {
			  if(!glob.enableRC3Instr) return rlFailRC3Required;
			  Array<Op> ops(3);
			  bool dest[] = {true, true, false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrFarscan(glob, ops);

              continue;
          }

          if( word == "set" )
          {
			  Array<Op> ops(2);
			  bool dest[] = {true, false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrSet(glob, ops);

              continue;
          }

          if( word == "add" )
          {
			  Array<Op> ops(2);
			  bool dest[] = {true, false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrAdd(glob, ops);

              continue;
          }

          if( word == "sub" )
          {
			  Array<Op> ops(2);
			  bool dest[] = {true, false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrSub(glob, ops);

              continue;
          }

          if( word == "mul" )
          {
			  if(!glob.enableRC3Instr) return rlFailRC3Required;
			  Array<Op> ops(2);
			  bool dest[] = {true, false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrMul(glob, ops);

              continue;
          }

          if( word == "div" )
          {
			  if(!glob.enableRC3Instr) return rlFailRC3Required;
			  Array<Op> ops(2);
			  bool dest[] = {true, false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrDiv(glob, ops);

              continue;
          }

          if( word == "mod" )
          {
			  if(!glob.enableRC3Instr) return rlFailRC3Required;
			  Array<Op> ops(2);
			  bool dest[] = {true, false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrMod(glob, ops);

              continue;
          }

          if( word == "min" )
          {
			  if(!glob.enableRC3Instr) return rlFailRC3Required;
			  Array<Op> ops(2);
			  bool dest[] = {true, false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrMin(glob, ops);

              continue;
          }

          if( word == "max" )
          {
			  if(!glob.enableRC3Instr) return rlFailRC3Required;
			  Array<Op> ops(2);
			  bool dest[] = {true, false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrMax(glob, ops);

              continue;
          }

          if( word == "create" )
          {
			  Array<Op> ops(3);
			  bool dest[] = {false, false, false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrCreate(glob, ops);

              continue;
          }

          if( word == "random" )
          {
			  if(!glob.enableRC3Instr) return rlFailRC3Required;
			  Array<Op> ops(3);
			  bool dest[] = {true, false, false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrRandom(glob, ops);

              continue;
          }

          if( word == "trans" )
          {
			  Array<Op> ops(2);
			  bool dest[] = {false, false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrTrans(glob, ops);

              continue;
          }

          if( word == "rtrans" )
          {
			  if(!glob.enableRC3Instr) return rlFailRC3Required;
			  Array<Op> ops(2);
			  bool dest[] = {false, false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrRtrans(glob, ops);

              continue;
          }

          if( word == "jump" )
          {
			  Array<Op> ops(1);
			  bool dest[] = {false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrJump(glob, ops);

              continue;
          }

          if( word == "ajump" )
          {
			  if(!glob.enableRC3Instr) return rlFailRC3Required;
			  Array<Op> ops(1);
			  bool dest[] = {false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrAjump(glob, ops);

              continue;
          }

          if( word == "bjump" )
          {
			  Array<Op> ops(2);
			  bool dest[] = {false, false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrBjump(glob, ops);

              continue;
          }

          if( word == "sleep" )
          {
			  if(!glob.enableRC3Instr) return rlFailRC3Required;
			  Array<Op> ops(1);
			  bool dest[] = {false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrSleep(glob, ops);

              continue;
          }

          if( word == "comp" )
          {
			  Array<Op> ops(2);
			  bool dest[] = {false, false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrComp(glob, ops);

              continue;
          }

          if( word == "lcomp" )
          {
			  if(!glob.enableRC3Instr) return rlFailRC3Required;
			  Array<Op> ops(2);
			  bool dest[] = {false, false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrLcomp(glob, ops);

              continue;
          }

          if( word == "gcomp" )
          {
			  if(!glob.enableRC3Instr) return rlFailRC3Required;
			  Array<Op> ops(2);
			  bool dest[] = {false, false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrGcomp(glob, ops);

              continue;
          }

          if( word == "ncomp" )
          {
			  if(!glob.enableRC3Instr) return rlFailRC3Required;
			  Array<Op> ops(2);
			  bool dest[] = {false, false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrNcomp(glob, ops);

              continue;
          }

          if( word == "init" )
          {
			  if(!glob.enableMultitasking) return rlFailMuTaRequired;
			  Array<Op> ops(2);
			  bool dest[] = {false, false};
			  ret = readOps(in, ops, dest);
			  if(ret) return ret;
			  newBank->instr += new InstrInit(glob, ops);

              continue;
          }

          if( word == "break" )
          {
			  if(!glob.enableMultitasking) return rlFailMuTaRequired;
              newBank->instr += new InstrBreak(glob);
              in->skipRestOfLine();
              continue;
          }

          if( word == "seize" )
          {
			  if(!glob.enableMultitasking) return rlFailMuTaRequired;
              newBank->instr += new InstrSeize(glob);
              in->skipRestOfLine();
              continue;
          }

          if( word == "resume" )
          {
			  if(!glob.enableMultitasking) return rlFailMuTaRequired;
              newBank->instr += new InstrResume(glob);
              in->skipRestOfLine();
              continue;
          }

          if( word == "quit" )
          {
			  if(!glob.enableMultitasking) return rlFailMuTaRequired;
              newBank->instr += new InstrQuit(glob);
              in->skipRestOfLine();
              continue;
          }

		  // it's none of these instructions
          return rlFailUnknownCommand;
      }
	  (*banks) += newBank;
  }
  
  return rlOK;
}

RobLoadReturnType RobLoader::getLabels(DefineInputStream* in)
{
  String word;
  RobLoadReturnType ret = in->getWord(word); // should be "bank"
  if(ret) return ret;
  in->skipRestOfLine(); // skip identifier of first bank
  rint curInstr = 1;
  while(!in->eos())
  {
      RobLoadReturnType ret = in->getWord(word);
	  if(ret) return ret;
	  if(in->eos()) break;

      // is it the beginning of a new bank?
      if(!word.compareIgnoreCase("bank"))
      {
          curInstr = 1;
		  in->skipRestOfLine();
          continue;
      }

      // is it a label?
      if( word[0] == '@' )
      {
		  String lname = word.substring(1);
		  if(labels.isSet(lname))
			  return rlFailRedefinedLabel;
		  labels.put(lname, curInstr);
		  in->skipRestOfLine();
          continue;
      }
      else
      {
          // must be a normal command
          curInstr++;
		  in->skipRestOfLine();
      }
  }
  return rlOK;
}

RobLoadReturnType RobLoader::readOps(DefineInputStream* in, Array<Op>& ops, bool* dest)
{
	RobLoadReturnType ret;
	for(int i = 0; i < ops.length(); i++)
	{
		String word;
		ret = in->getWord(word);
		if(ret) return ret;
		if(dest[i]) ret = convertParamDest(word, ops[i]);
		else ret = convertParamSrc(word, ops[i]);
		if(ret) return ret;

		if(i != ops.length() - 1) {
			ret = in->getWord(word);
			if(ret) return ret;
			if(word != ",")
				return rlFailExpectComma;
		}
	}
	in->skipRestOfLine();
	return rlOK;
}


} // namespace

