#include <wx/wx.h>

#include "robwxhist.h"
#include "robwxvis.h"
#include "robwxstruct.h"

/*
class HistoryCtrl : public wxControl 
{
public:
	HistoryCtrl(wxWindow* parent, wxWindowID id);
	~HistoryCtrl();

	void SetMaximum(int max);
	void ClearHistory();
	void UpdateFrom(...);

	DECLARE_EVENT_TABLE()
protected:
	wxSize DoGetBestSize();
private:
	void EnsureSize(const wxSize&);

	void OnSize(wxSizeEvent&);

	int max;
	int screenPos, bmpPos;
	wxBitmap* buf;
};
*/

BEGIN_EVENT_TABLE(HistoryCtrl, wxControl)
  EVT_SIZE(HistoryCtrl::OnSize)
  EVT_ERASE_BACKGROUND(HistoryCtrl::OnEraseBackground)
  EVT_PAINT(HistoryCtrl::OnPaint)
END_EVENT_TABLE()

HistoryCtrl::HistoryCtrl(wxWindow* parent, wxWindowID id, const wxPoint& pos, const wxSize& size)
	: wxControl(parent, id, pos, size, wxSUNKEN_BORDER),
	  max(255), screenPos(0), bmpPos(0), minInterval(50), lastCycle(-10000)
{
	SetBackgroundColour(wxSystemSettings::GetColour(wxSYS_COLOUR_WINDOW));
	SetToolTip("History");
	buf = new wxBitmap(size.x * 2, size.y);
	ClearHistory();
}

HistoryCtrl::~HistoryCtrl()
{
	delete buf;
}


// data stuff

void HistoryCtrl::SetMaximum(int max)
{
	this->max = max;
	ClearHistory();
}

void HistoryCtrl::ClearHistory()
{
	lastCycle = -1000; 
	screenPos = 0;
	bmpPos = 0;
	Clear();
	wxMemoryDC trgDC;
	trgDC.SelectObject(*buf);
	wxBrush bgBrush(GetBackgroundColour(), wxSOLID);
	trgDC.SetBackground(bgBrush);
	trgDC.Clear();
}

void HistoryCtrl::UpdateFrom(RoboTourEvent& evt)
{
	if(!evt.tourData.length())
		return;
	if(evt.simData.cycle > lastCycle && evt.simData.cycle < lastCycle + minInterval)
		return;
	lastCycle = evt.simData.cycle;
	lrt::Vector<TourInfo>& data = evt.tourData;

	int height = GetClientSize().GetHeight();
	// clear current bitmap line
	wxMemoryDC bmpDC;
	bmpDC.SelectObject(*buf);
	wxPen pen(GetBackgroundColour(), 1, wxSOLID);
	bmpDC.SetPen(pen);
	bmpDC.DrawLine(bmpPos, 0, bmpPos, height);
	
	// draw points for each program
	for(int p = 0; p < data.length(); p++)
	{
		TourInfo ti = data[p];
		if(ti.playerNum < 0) continue;

		pen.SetColour(RobVisFrame::theVisFrame->GetBotPainter().GetProgramColour(ti.playerNum));
		bmpDC.SetPen(pen);
		int h = ti.numBots * height / max;
		h = height - h;
		bmpDC.DrawLine(bmpPos, h-1, bmpPos, h+1);
	}

	// advance to next position
	screenPos++;
	if(screenPos >= GetClientSize().GetWidth()) // walking out of window
		screenPos--; 
	bmpPos++;
	if(bmpPos >= buf->GetWidth()) // walking out of bitmap
		bmpPos = 0; 

	// draw red line on next position
	pen.SetColour(*wxRED);
	bmpDC.SetPen(pen);
	bmpDC.DrawLine(bmpPos, 0, bmpPos, height);

	// clean up operations on bmpDC
	bmpDC.SelectObject(wxNullBitmap); 

	// blit to screen
	wxClientDC dc(this);
	Paint(dc);
}


// paint stuff
void HistoryCtrl::OnPaint(wxPaintEvent& evt)
{
	wxPaintDC dc(this);
	Paint(dc);
}

void HistoryCtrl::Paint(wxDC& dc)
{
	wxSize sz = GetClientSize();
	wxMemoryDC bmpDC;
	bmpDC.SelectObject(*buf);
	int bmpStart = bmpPos - screenPos; // window origin in the bitmap
	int bmpEnd = bmpStart + sz.GetWidth(); // window end in the bitmap
	int winPos = 0; 
	int winEnd = sz.GetWidth();
	if(bmpStart < 0) {
		int leftLen = -bmpStart; // length of history which wraps left out of the bitmap
		dc.Blit(0, 0, leftLen, buf->GetHeight(), &bmpDC, bmpStart + buf->GetWidth(), 0);
		winPos += leftLen;
	}
	if(bmpEnd > buf->GetWidth()) {
		int rightLen = bmpEnd - buf->GetWidth(); // length of history which wraps right out of the bitmap
		dc.Blit(winEnd - rightLen, 0, rightLen, buf->GetHeight(), &bmpDC, 0, 0);
		winEnd -= rightLen;
	}
	int midLen = winEnd - winPos;
	dc.Blit(winPos, 0, midLen, buf->GetHeight(), &bmpDC, bmpStart + winPos, 0);
}


// size stuff
void HistoryCtrl::EnsureSize(const wxSize& sz)
{
	if(buf->GetWidth() >= sz.GetWidth())
		return; // nothing to do

	// need to resize. 
	wxBitmap* newBuf = new wxBitmap(sz.GetWidth() * 2, buf->GetHeight());
	wxMemoryDC srcDC;
	srcDC.SelectObject(*buf);
	wxMemoryDC trgDC;
	trgDC.SelectObject(*newBuf);
	int restX = buf->GetWidth() - bmpPos;
	wxBrush bgBrush(GetBackgroundColour(), wxSOLID);
	trgDC.SetBackground(bgBrush);
	trgDC.Clear();
	trgDC.Blit(0, 0, restX, buf->GetHeight(), &srcDC, bmpPos, 0);
	trgDC.Blit(restX, 0, bmpPos, buf->GetHeight(), &srcDC, 0, 0);

	delete buf;
	buf = newBuf; 
	bmpPos = 0; 
}

wxSize HistoryCtrl::DoGetBestSize()
{
	return wxSize(300, 150);
}

void HistoryCtrl::OnSize(wxSizeEvent& evt)
{
	EnsureSize(GetClientSize());
	if(screenPos >= GetClientSize().GetWidth())
		screenPos = GetClientSize().GetWidth() - 1; 
}

