/*
 * robinstr.h
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

/** @file
  * Declares concrete instruction classes.
  */

#ifndef __RT_INSTR__
#define __RT_INSTR__

#include <rtcollect.h>
#include <rtstring.h>

#include "robvars.h" // everyone include this!
#include "robbase.h"

// abstract base class Instr is in robbase.h
namespace rt {

/*******/
/* DIE */
/*******/

class InstrDie : public Instr{

public:

  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrDie(const Globals& glob);
  ~InstrDie();
};

/********/
/* MOVE */
/********/

class InstrMove : public Instr{

public:

  virtual rint getNumCycles(Task* task);
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrMove(const Globals& glob);
  ~InstrMove();
};

/********/
/* TURN */
/********/

class InstrTurn : public Instr{

public:

  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrTurn(const Globals& glob, const lrt::Array<Op>& ops);

  ~InstrTurn();
};

/********/
/* SCAN */
/********/

class InstrScan : public Instr{

public:

  rint getNumCycles(Task* task);
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrScan(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrScan();
};

/***********/
/* FarScan */
/***********/

class InstrFarscan : public Instr{

public:

  rint getNumCycles(Task* task);
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrFarscan(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrFarscan();
};

/*******/
/* SET */
/*******/

class InstrSet : public Instr{

public:

  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrSet(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrSet();
};

/*******/
/* MIN */
/*******/

class InstrMin : public Instr{

public:

  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrMin(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrMin();
};


/*******/
/* MAX */
/*******/

class InstrMax : public Instr{

public:

  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrMax(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrMax();
};


/*******/
/* ADD */
/*******/

class InstrAdd : public Instr{

public:

  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrAdd(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrAdd();
};


/*******/
/* SUB */
/*******/

class InstrSub : public Instr{

public:
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();
  InstrSub(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrSub();
};


/*******/
/* MUL */
/*******/

class InstrMul : public Instr{

public:
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrMul(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrMul();
};


/*******/
/* DIV */
/*******/

class InstrDiv : public Instr{

public:
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrDiv(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrDiv();
};


/*******/
/* MOD */
/*******/

class InstrMod : public Instr{

public:
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrMod(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrMod();
};


/*********/
/* TRANS */
/*********/

class InstrTrans : public Instr{

public:
  virtual rint getNumCycles(Task* task);
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrTrans(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrTrans();
};

/**********/
/* RTRANS */
/**********/

class InstrRtrans : public Instr{

public:
  virtual rint getNumCycles(Task* task);
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrRtrans(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrRtrans();
};

/*********/
/* AJUMP */
/*********/

class InstrAjump : public Instr{

public:
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrAjump(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrAjump();
};


/********/
/* JUMP */
/********/

class InstrJump : public Instr{

public:
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrJump(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrJump();
};


/*********/
/* BJUMP */
/*********/

class InstrBjump : public Instr{

public:
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrBjump(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrBjump();
};


/*********/
/* SLEEP */
/*********/

class InstrSleep : public Instr{

public:
  virtual rint getNumCycles(Task* task);
  virtual ExecReturnType exec(Task* task);
  virtual TimingMode getTimingMode();

  lrt::String getName();

  InstrSleep(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrSleep();
};


/********/
/* COMP */
/********/

class InstrComp : public Instr{

public:
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrComp(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrComp();
};


/*********/
/* NCOMP */
/*********/

class InstrNcomp : public Instr{

public:
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrNcomp(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrNcomp();
};


/*********/
/* LCOMP */
/*********/

class InstrLcomp : public Instr{

public:
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrLcomp(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrLcomp();
};


/*********/
/* GCOMP */
/*********/

class InstrGcomp : public Instr{

public:
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrGcomp(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrGcomp();
};


/**********/
/* CREATE */
/**********/

class InstrCreate : public Instr{

public:
  virtual rint getNumCycles(Task* task);
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrCreate(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrCreate();
};

/**********/
/* RANDOM */
/**********/

class InstrRandom : public Instr{

public:
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrRandom(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrRandom();
};

////////////////////////////// Multitasking /////////////////////////

/*********/
/* BREAK */
/*********/

class InstrBreak : public Instr{

public:
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrBreak(const Globals& glob);
  ~InstrBreak();
};

/*********/
/* INIT  */
/*********/

class InstrInit : public Instr{

public:
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrInit(const Globals& glob, const lrt::Array<Op>& ops);
  ~InstrInit();
};

/*********/
/* SEIZE */
/*********/

class InstrSeize : public Instr{

public:
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrSeize(const Globals& glob);
  ~InstrSeize();
};

/**********/
/* RESUME */
/**********/

class InstrResume : public Instr{

public:
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrResume(const Globals& glob);
  ~InstrResume();
};

/********/
/* QUIT */
/********/

class InstrQuit : public Instr{

public:
  virtual ExecReturnType exec(Task* task);

  lrt::String getName();

  InstrQuit(const Globals& glob);
  ~InstrQuit();
};





} // namespace


#endif

