/*
 * robwxdlgs.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

// Implements auxiliary dialogs for the visualization

#include <wx/wx.h>
#include <wx/listctrl.h>
#include <wx/config.h>
#include <wx/colordlg.h>

#include "robstrings.h"

#include "robwxdlgs.h"
#include "robotourdlg_wdr.h"
#include "robwxvis.h"
#include "robwxutil.h"

using namespace lrt;
using namespace rt;

// Extension to the WDR resource
// CreateTourSetupBitmap bitmap list indices
enum {
  TBMP_ROBADD,
  TBMP_ROBDEL,
  TBMP_ROBUP,
  TBMP_ROBDOWN,
  TBMP_OPTSETSEL,
  TBMP_SRCROB,
  TBMP_BINROB
};


///////////////////////////////////////////////////////////////////////
////////////////////// TourSetupDialog ////////////////////////////////

BEGIN_EVENT_TABLE(TourSetupDialog, wxDialog)
  EVT_BUTTON(TDLG_ROBADD,    TourSetupDialog::OnAdd)
  EVT_BUTTON(TDLG_ROBDEL,    TourSetupDialog::OnRemove)
  EVT_BUTTON(TDLG_ROBUP,     TourSetupDialog::OnMoveUp)
  EVT_BUTTON(TDLG_ROBDOWN,   TourSetupDialog::OnMoveDown)
  EVT_BUTTON(TDLG_OPTSETSEL, TourSetupDialog::OnOptionSet)
  EVT_BUTTON(wxID_OK, TourSetupDialog::OnOK)
END_EVENT_TABLE()

////////////////////// conversion functions

TourSetupInfo::TourType SelToTourType(int sel)
{
	switch(sel) {
	  case 1:
		return TourSetupInfo::Charts;
	  case 2:
		return TourSetupInfo::AllInOne;
	  case 0:
	  default:
		return TourSetupInfo::Single;
	}
}
int TourTypeToSel(TourSetupInfo::TourType tt)
{
	switch(tt) {
	  case TourSetupInfo::Charts:
		return 1;
	  case TourSetupInfo::AllInOne:
		return 2;
	  case TourSetupInfo::Single:
	  default:
		return 0;
	}
}


////////////////////// real implementation //////////////

TourSetupDialog::TourSetupDialog(TourSetupInfo* data)
	: wxDialog(RobVisFrame::theVisFrame, -1, _("RoboTour - New Tournament"), 
		wxDefaultPosition, wxDefaultSize, wxDEFAULT_DIALOG_STYLE | wxRESIZE_BORDER), 
      data(data), assemblyGlob(this)
{
	assemblyGlob.getGlobals("assemble.rco"); // embedded resource file

	CreateTourSetupDialog(this);
	Wx::RestoreRect(this, wxT("/GUI/TsdWindowRect"));
	wxConfigBase* cfg = wxConfig::Get(); wxString sTmp; long iTmp;

	wxImageList* robimagelist = new wxImageList(16,16);
	robimagelist->Add(CreateTourSetupBitmap(TBMP_SRCROB));
	robimagelist->Add(CreateTourSetupBitmap(TBMP_BINROB));

	wxListCtrl* robctrl = (wxListCtrl*) FindWindow(TDLG_ROBOTS);
	robctrl->InsertColumn(0, _("Program"), wxLIST_FORMAT_LEFT, -2);
	robctrl->InsertColumn(1, _("Author"), wxLIST_FORMAT_LEFT, -2);
	robctrl->InsertColumn(2, _("File"), wxLIST_FORMAT_LEFT, -2);
	robctrl->AssignImageList(robimagelist, wxIMAGE_LIST_SMALL);
	for(int rf = 0; rf < data->files.length(); rf++)
		AddBot(Wx::Conv(data->files[rf]));
	Wx::RestoreColumns(robctrl, wxT("/GUI/TsdRobColumns"));

	wxTextCtrl* osetctrl = (wxTextCtrl*) FindWindow(TDLG_OPTSET);
	osetctrl->SetValue(Wx::Conv(data->optionFile));
	if(cfg->Read(wxT("/Options/OptionSet"), &sTmp)) osetctrl->SetValue(sTmp);

	wxChoice* ttctrl = (wxChoice*) FindWindow(TDLG_TOURTYPE);
	ttctrl->SetSelection(TourTypeToSel(data->type));
	if(cfg->Read(wxT("/Options/TourType"), &iTmp)) ttctrl->SetSelection(iTmp);
	
	wxSpinCtrl* repctrl = (wxSpinCtrl*) FindWindow(TDLG_REPEATS);
	repctrl->SetValue(data->numRepeats);
	if(cfg->Read(wxT("/Options/RepeatCount"), &iTmp)) repctrl->SetValue(iTmp);
	
	wxCheckBox* soundctrl = (wxCheckBox*) FindWindow(TDLG_SOUND);
	if(data->haveSound)
		soundctrl->SetValue(data->sound);
	else
		soundctrl->Disable();
	if(cfg->Read(wxT("/Options/Sound"), &iTmp)) soundctrl->SetValue(iTmp);

	wxCheckBox* profctrl = (wxCheckBox*) FindWindow(TDLG_PROFILE);
	wxChoice* proftypectrl = (wxChoice*) FindWindow(TDLG_PROFILETYPE);
	if(data->haveProfiler) {
		profctrl->SetValue(data->profile);
		proftypectrl->SetSelection(data->profileType);
	}
	else {
		profctrl->Disable(); proftypectrl->Disable();
	}

	cfg->Read(wxT("/Options/RobotPath"), &robotPath);
}

void TourSetupDialog::handleLoadError(const String& affectedFile, const String& message) const
{
	loadErrors += "\n" + message;
}

wxString TourSetupDialog::GetItemText(wxListCtrl* ctrl, long row, int col)
{
	wxListItem item;
	item.SetId(row);
	item.SetColumn(col);
	item.SetMask(wxLIST_MASK_TEXT);
	ctrl->GetItem(item);
	return item.GetText();
}

Vector<long> TourSetupDialog::GetSelectedItems(wxListCtrl* ctrl)
{
	Vector<long> ret;
	long item = -1;
	while(true){
		item = ctrl->GetNextItem(item, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
		if(item < 0) break;
		ret += item;
	}
	return ret;
}

void TourSetupDialog::GetItemInfo(wxListCtrl* ctrl, long row, int col, wxListItem& info)
{
	info.SetId(row);
	info.SetColumn(col);
	info.SetMask(wxLIST_MASK_TEXT | wxLIST_MASK_IMAGE);
	ctrl->GetItem(info);
}

void TourSetupDialog::SelectOnly(wxListCtrl* ctrl, long row)
{
	long ic = ctrl->GetItemCount();
	for(long r = 0; r < ic; r++)
	{
		if(r == row)
			ctrl->SetItemState(r, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
		else
			ctrl->SetItemState(r, 0, wxLIST_STATE_SELECTED);
	}
}

void TourSetupDialog::AddBot(const wxString& botFile)
{
	// try to assemble robot
	loadErrors.clear();
	ProgramLoader* matchingLoader = Program::getLoader(Wx::Conv(botFile)); 
	const ErrorHandler* oldHandler = matchingLoader->getErrorHandler(); 
	matchingLoader->setErrorHandler(this); 
	Program* prog = Program::create(Wx::Conv(botFile), assemblyGlob, -1);
	matchingLoader->setErrorHandler(oldHandler); 
	if(prog == 0 || loadErrors.length())
	{
		wxString msg; msg.Printf(_("Cannot load robot '%s'"), botFile.c_str());
		if(loadErrors.length())
			msg += _("\nLoad error: ") + Wx::Conv(loadErrors);
		wxMessageBox(msg, _("Load error"), wxOK | wxICON_ERROR, this);
		return;
	}
	// OK, program loaded, get bot info
	wxString progname = Wx::Conv(prog->headers["name"].value) + wxT(" ") + Wx::Conv(prog->headers["version"].value);
	wxString author = Wx::Conv(prog->headers["author"].value);
	// done with the program
	delete prog;

	wxListCtrl* robctrl = (wxListCtrl*) FindWindow(TDLG_ROBOTS);
	int imgIndex = 0; // ROB
	if(!(botFile.Lower().AfterLast('.') == wxT("rob")))
		imgIndex = 1; // RBI
	long rowId = robctrl->GetItemCount();
	robctrl->InsertItem(rowId, progname, imgIndex);
	robctrl->SetItem(rowId, 1, author);
	robctrl->SetItem(rowId, 2, botFile);
}

void TourSetupDialog::SwapBots(long b1, long b2)
{
	wxListCtrl* robctrl = (wxListCtrl*) FindWindow(TDLG_ROBOTS);
	wxListItem t1, t2;
	for(int col = 0; col < 3; col++)
	{
		GetItemInfo(robctrl, b1, col, t1);
		GetItemInfo(robctrl, b2, col, t2);
		t1.SetId(b2);
		t2.SetId(b1);
		robctrl->SetItem(t1);
		robctrl->SetItem(t2);
	}
}

////////////////////////////////////////// events

void TourSetupDialog::OnAdd(wxCommandEvent& evt)
{
	wxFileDialog fd(this, _("Select robots"), robotPath, wxT(""), _("All robots files|*.rob;*.rbi|Robot sources (*.rob)|*.rob|Robot binaries (*.rbi)|*.rbi"),
		wxOPEN | wxMULTIPLE);
	if(fd.ShowModal() == wxID_OK)
	{
		robotPath = fd.GetDirectory();
		wxArrayString paths;
		fd.GetPaths(paths);
		for(size_t f = 0; f < paths.GetCount(); f++)
			AddBot(paths[f]);
	}
}

void TourSetupDialog::OnRemove(wxCommandEvent&)
{
	wxListCtrl* robctrl = (wxListCtrl*) FindWindow(TDLG_ROBOTS);
	Vector<long> delItems = GetSelectedItems(robctrl);
	if(!delItems.length())
		wxBell();
	for(int it = delItems.length() - 1; it >= 0; it--)
		robctrl->DeleteItem(delItems[it]);
}

void TourSetupDialog::OnMoveUp(wxCommandEvent&)
{
	wxListCtrl* robctrl = (wxListCtrl*) FindWindow(TDLG_ROBOTS);
	Vector<long> sel = GetSelectedItems(robctrl);
	long itemCount = robctrl->GetItemCount();
	
	if(!sel.length() || sel[0] == 0 || itemCount < 2) {
		wxBell(); return; 
	}

	SwapBots(sel[0], sel[0] - 1);
	SelectOnly(robctrl, sel[0] - 1);
}

void TourSetupDialog::OnMoveDown(wxCommandEvent&)
{
	wxListCtrl* robctrl = (wxListCtrl*) FindWindow(TDLG_ROBOTS);
	Vector<long> sel = GetSelectedItems(robctrl);
	long itemCount = robctrl->GetItemCount();
	
	if(!sel.length() || sel[0] == itemCount-1 || itemCount < 2) {
		wxBell(); return; 
	}

	SwapBots(sel[0], sel[0] + 1);
	SelectOnly(robctrl, sel[0] + 1);
}

void TourSetupDialog::OnOptionSet(wxCommandEvent& evt)
{
	wxTextCtrl* osetctrl = (wxTextCtrl*) FindWindow(TDLG_OPTSET);
	wxString osetfile = osetctrl->GetValue();

	wxFileDialog fd(this, _("Select option set"), osetfile, wxT(""), _("Option set files (*.rco)|*.rco"), wxOPEN);
	if(fd.ShowModal() == wxID_OK)
	{
		osetfile = fd.GetPath();
		osetctrl->SetValue(osetfile);
	}
}

void TourSetupDialog::OnOK(wxCommandEvent& evt)
{
	data->ok = true;

	Wx::StoreRect(this, wxT("/GUI/TsdWindowRect"));
	wxConfigBase* cfg = wxConfig::Get();

	wxListCtrl* robctrl = (wxListCtrl*) FindWindow(TDLG_ROBOTS);
	for(int rf = 0; rf < robctrl->GetItemCount(); rf++)
	{
		data->files += Wx::Conv(GetItemText(robctrl, rf, 2));
	}
	Wx::StoreColumns(robctrl, wxT("/GUI/TsdRobColumns"));

	wxTextCtrl* osetctrl = (wxTextCtrl*) FindWindow(TDLG_OPTSET);
	data->optionFile = Wx::Conv(osetctrl->GetValue());
	cfg->Write(wxT("/Options/OptionSet"), osetctrl->GetValue());

	wxChoice* ttctrl = (wxChoice*) FindWindow(TDLG_TOURTYPE);
	data->type = SelToTourType(ttctrl->GetSelection());
	cfg->Write(wxT("/Options/TourType"), ttctrl->GetSelection());

	wxSpinCtrl* repctrl = (wxSpinCtrl*) FindWindow(TDLG_REPEATS);
	data->numRepeats = repctrl->GetValue();
	cfg->Write(wxT("/Options/RepeatCount"), repctrl->GetValue());

	if(data->haveSound) {
		wxCheckBox* soundctrl = (wxCheckBox*) FindWindow(TDLG_SOUND);
		data->sound = soundctrl->IsChecked();
		cfg->Write(wxT("/Options/Sound"), soundctrl->IsChecked());
	}
	if(data->haveProfiler) {
		wxCheckBox* profctrl = (wxCheckBox*) FindWindow(TDLG_PROFILE);
		data->profile = profctrl->IsChecked();
		wxChoice* proftypectrl = (wxChoice*) FindWindow(TDLG_PROFILETYPE);
		data->profileType = proftypectrl->GetSelection(); //?
	}

	cfg->Write(wxT("/Options/RobotPath"), robotPath);
	cfg->Flush();

	wxDialog::OnOK(evt);
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
BEGIN_EVENT_TABLE(BotErrFilterDialog, wxDialog)
  EVT_CHECKLISTBOX(-1, BotErrFilterDialog::OnToggleSpecific)
  EVT_CHECKBOX(-1, BotErrFilterDialog::OnToggleAny)
  EVT_BUTTON(wxID_CLOSE, BotErrFilterDialog::OnCloseButton)
  EVT_CLOSE(BotErrFilterDialog::OnCloseWindow)
END_EVENT_TABLE()

BotErrFilterDialog::BotErrFilterDialog()
	: wxDialog(RobVisFrame::theVisFrame, -1, _("Robot message filter"), 
		   wxDefaultPosition, wxDefaultSize, wxDEFAULT_DIALOG_STYLE | wxRESIZE_BORDER),
	  allowAny(true), allowSpecific(-failLast+1)
{
	wxSizer* sz = new wxBoxSizer(wxVERTICAL);
	anyCB = new wxCheckBox(this, -1, _("Report the following robot errors:"));
	sz->Add(anyCB, 0, wxALIGN_LEFT | wxALL, 3);
	specificCLB = new wxCheckListBox(this, -1);
	sz->Add(specificCLB, 1, wxEXPAND | wxALL, 3);
	sz->Add(new wxStaticLine(this, -1), 0, wxEXPAND | wxALL, 3);
	sz->Add(new wxButton(this, wxID_CLOSE, _("Close")), 0, wxALIGN_RIGHT | wxALL, 3);

	for(int en = -1; en >= failLast; en--)
		specificCLB->Append(Wx::Conv(lrt::String(rt::getFailMsg((rt::ExecReturnType)en))));

	SetSizer(sz);
	SetSize(300, 300);
	SetSizeHints(200, 200);

	LoadSettings();
}

void BotErrFilterDialog::LoadSettings()
{
	wxConfigBase* cfg = wxConfig::Get();
	allowAny = (bool) cfg->Read(wxT("/GUI/ShowAnyBotError"), 1);
	anyCB->SetValue(allowAny);
	specificCLB->Enable(allowAny);

	lrt::String str = Wx::Conv(cfg->Read(wxT("/GUI/ShowThisBotError"), wxT("")));
	lrt::Array<lrt::String> arr = str.split(",", "");
	int term = lrt::Math::min(arr.length(), -failLast+1);
	for(int c = 0; c < term; c++) {
		allowSpecific[c] = (bool) arr[c].intValue(0);
		if(c) specificCLB->Check(c-1, allowSpecific[c]);
	}
}

void BotErrFilterDialog::StoreSettings()
{
	wxConfigBase* cfg = wxConfig::Get();
	cfg->Write(wxT("/GUI/ShowAnyBotError"), (long)allowAny);

	lrt::String str;
	for(int c = 0; c < allowSpecific.length(); c++)
		str += String((int) allowSpecific[c]) + ",";
	cfg->Write(wxT("/GUI/ShowThisBotError"), Wx::Conv(str));
	cfg->Flush();
}

void BotErrFilterDialog::OnToggleAny(wxCommandEvent& evt)
{
	if(evt.IsChecked())
	{
		specificCLB->Enable();
		allowAny = true;
	}
	else
	{
		specificCLB->Disable();
		allowAny = false;
	}
}

void BotErrFilterDialog::OnToggleSpecific(wxCommandEvent& evt)
{
	allowSpecific[evt.GetInt()+1] = specificCLB->IsChecked(evt.GetInt());
}


////////////////////////////////////////////////////////////////////////
////////////////// ProgramColourDialog

class BotColourComponent : public wxControl {
public:
	BotColourComponent(wxWindow* parent, wxWindowID id, BotPainter* painter) 
		: wxControl(parent, id), painter(painter), botSize(20)
	{
	}
	wxSize DoGetBestSize() const
	{
		return wxSize(botSize * painter->GetProgramColourCount(), botSize);
	}

	DECLARE_EVENT_TABLE()
private:
	void OnClick(wxMouseEvent& evt)
	{
		int num = evt.GetX() / botSize; 
		if(num < 0 || num >= painter->GetProgramColourCount())
			return; 
		wxColour newCol = wxGetColourFromUser(this, painter->GetProgramColour(num));
		if(!newCol.Ok())
			return;
		painter->SetProgramColour(num, newCol);
		wxClientDC dc(this);
		UpdateSingle(num, dc);
	}
	void OnPaint(wxPaintEvent& evt) 
	{
		wxSize sz = GetSize();
		botSize = sz.x / painter->GetProgramColourCount();

		wxPaintDC dc(this);
		dc.BeginDrawing();
		for(int i = 0; i < painter->GetProgramColourCount(); i++)
			UpdateSingle(i, dc);
		dc.EndDrawing();
	}
	void UpdateSingle(int num, wxDC& dc)
	{
		BotInfo info;
		info.program = num;
		info.type = 2;
		info.active = true;
		info.head[0] = num;
		painter->DrawBot(info, dc, wxPoint(num * botSize, 0), botSize);
	}

	int botSize; 
	BotPainter* painter; 
};

BEGIN_EVENT_TABLE(BotColourComponent, wxControl)
  EVT_PAINT(BotColourComponent::OnPaint)
  EVT_LEFT_UP(BotColourComponent::OnClick)
END_EVENT_TABLE()

ProgramColourDialog::ProgramColourDialog(const BotPainter& prevSettings)
: wxDialog(RobVisFrame::theVisFrame, -1, _("Change program colours"), wxDefaultPosition, 
		   wxDefaultSize, wxDEFAULT_DIALOG_STYLE | wxRESIZE_BORDER), painter(new BotPainter(prevSettings))
{
	wxSizer* sz = new wxBoxSizer(wxVERTICAL);
	BotColourComponent* botCC = new BotColourComponent(this, -1, painter);
    sz->Add(1, 1, 1);
	sz->Add(botCC, 0, wxEXPAND | wxADJUST_MINSIZE | wxALL, 5);
    sz->Add(1, 1, 1);
	sz->Add(new wxStaticLine(this, -1), 0, wxEXPAND | wxALL, 3);
	wxSizer* bsz = new wxBoxSizer(wxHORIZONTAL);
	bsz->Add(1, 1, 1); 
	bsz->Add(new wxButton(this, wxID_OK, _("OK")), 0, wxALL, 3);
	bsz->Add(new wxButton(this, wxID_CANCEL, _("Cancel")), 0, wxALL, 3);
	sz->Add(bsz, 0, wxEXPAND);

	SetSizerAndFit(sz);
	sz->SetSizeHints(this);
	CentreOnParent();
}

ProgramColourDialog::~ProgramColourDialog() 
{ 
	delete painter; 
}



//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
BEGIN_EVENT_TABLE(DebugDialog, wxDialog)
  EVT_LIST_ITEM_SELECTED(DD_BANKS, DebugDialog::OnBankList)
  EVT_COMMAND_SCROLL(DD_TASK, DebugDialog::OnTask)
  EVT_CHECKBOX(DD_FOLLOW, DebugDialog::OnFollow)
  EVT_CLOSE(DebugDialog::OnCloseWindow)
END_EVENT_TABLE()

	/*
	// wxComponentPointers
	wxCheckBox* followCB;
	wxSlider* taskSL;

	ImprovedListCtrl* bankLC;
	wxListBox* instrLB;
	ImprovedListCtrl* unnamedVarLC;
	lrt::Array<wxStaticText*> namedVarT;
	wxPanel* colourP;
	// customPanel* taskViewP;
	wxStaticText* programT;
	wxStaticText* taskNumT;
	wxStaticText* curInstrT;
	wxStaticText* curBankT;
	wxStaticText* curInstrTextT;
	*/

DebugDialog::DebugDialog(DebugInfo& initialInfo) 
	: wxDialog(RobVisFrame::theVisFrame, -1, _("RoboTour Debugger"), wxDefaultPosition, wxDefaultSize, 
			   wxDEFAULT_DIALOG_STYLE | wxRESIZE_BORDER | wxSYSTEM_MENU), namedVarT(rt::botVarLength), 
	  curBank(0), curTask(0), follow(false)
{
	new wxPanel(this, DD_BOTPAINT, wxDefaultPosition, wxSize(16, 16));
	CreateDebugDialog(this);

	FindWindows();
	SetupColumns();
	UpdateBankImageList();
	LoadSettings();

	HandleUpdate(initialInfo, true);
	Show(true);
}

void DebugDialog::HandleProgramColourChange()
{
	UpdateBankImageList();
	colourP->SetBackgroundColour(RobVisFrame::theVisFrame->GetBotPainter().GetProgramColour(info.botInfo.program));
	colourP->Refresh();
	UpdateTaskData(info, true); // to refresh the task viewer
}

void DebugDialog::SetupColumns()
{
	int x, y;

	bankLC->InsertColumn(0, _("Num"), wxLIST_FORMAT_RIGHT, -1);
	bankLC->GetTextExtent(wxT("99999"), &x, &y); bankLC->SetColumnWidth(0, x);
	bankLC->InsertColumn(1, _("Cnt"), wxLIST_FORMAT_CENTER, -1);
	bankLC->GetTextExtent(wxT("99999"), &x, &y); bankLC->SetColumnWidth(1, x);
	bankLC->InsertColumn(2, _("Name"), wxLIST_FORMAT_LEFT, -1);
	bankLC->SetGrowableCol(2);

	unnamedVarLC->InsertColumn(0, _("ID"), wxLIST_FORMAT_LEFT, -1);
	unnamedVarLC->GetTextExtent(wxT("#9999"), &x, &y); unnamedVarLC->SetColumnWidth(0, x);
	unnamedVarLC->InsertColumn(1, _("Value"), wxLIST_FORMAT_LEFT, -1);
	unnamedVarLC->SetGrowableCol(1);
}

void DebugDialog::FindWindows()
{
	followCB    = dynamic_cast<wxCheckBox*      >(FindWindow(DD_FOLLOW   ));
	taskSL      = dynamic_cast<wxSlider*        >(FindWindow(DD_TASK     ));
	bankLC      = dynamic_cast<ImprovedListCtrl*>(FindWindow(DD_BANKS    ));
	instrLB     = dynamic_cast<wxListBox*       >(FindWindow(DD_INSTR    ));
	unnamedVarLC= dynamic_cast<ImprovedListCtrl*>(FindWindow(DD_VARIABLES));
	for(int v = 0; v < rt::botVarLength; v++)
		namedVarT[v] = 0;
	namedVarT[rt::botActive]     = dynamic_cast<wxStaticText*>(FindWindow(DD_ACTIVE));
	namedVarT[rt::botMobile]     = dynamic_cast<wxStaticText*>(FindWindow(DD_MOBILE));
	namedVarT[rt::botGeneration] = dynamic_cast<wxStaticText*>(FindWindow(DD_GENERATION));
	namedVarT[rt::botPosX]       = dynamic_cast<wxStaticText*>(FindWindow(DD_XPOS));
	namedVarT[rt::botPosY]       = dynamic_cast<wxStaticText*>(FindWindow(DD_YPOS));
	namedVarT[rt::botInstrSet]   = dynamic_cast<wxStaticText*>(FindWindow(DD_INSTRSET));

	colourP     = dynamic_cast<wxPanel*     >(FindWindow(DD_COLOUR   ));
	taskViewP   = dynamic_cast<wxPanel*     >(FindWindow(DD_BOTPAINT ));
	programT    = dynamic_cast<wxStaticText*>(FindWindow(DD_PROGRAM  ));
	taskNumT    = dynamic_cast<wxStaticText*>(FindWindow(DD_TASKNUM  ));
	curBankT    = dynamic_cast<wxStaticText*>(FindWindow(DD_CURBANK  ));
	curInstrT   = dynamic_cast<wxStaticText*>(FindWindow(DD_CURINSTR ));
	curInstrTextT= dynamic_cast<wxStaticText*>(FindWindow(DD_CURINSTRTEXT));
}

void DebugDialog::SetLCLength(wxListCtrl* lc, long numEntries) // makes sure that exactly numEntries exist in lc
{
	long prevEntries = lc->GetItemCount();
	if(prevEntries < numEntries)
		for(long e = prevEntries; e < numEntries; e++)
			lc->InsertItem(e, wxT(""));
	else if(prevEntries > numEntries)
		for(long e = prevEntries-1; e >= numEntries; e--)
			lc->DeleteItem(e);
}

void DebugDialog::StoreSettings()
{
	wxConfigBase* cfg = wxConfig::Get();
	cfg->Write(wxT("/GUI/DbgFollow"), follow);
	Wx::StoreRect(this, wxT("/GUI/DbgWindowRect"));
}

void DebugDialog::LoadSettings()
{
	wxConfigBase* cfg = wxConfig::Get();
	Wx::RestoreRect(this, wxT("/GUI/DbgWindowRect"));
	follow = (bool) cfg->Read(wxT("/GUI/DbgFollow"), (long)false);
	followCB->SetValue(follow);
}


DebugDialog::~DebugDialog()
{
	StoreSettings();
}

void DebugDialog::HandleUpdate(DebugInfo& newInfo, bool first)
{
	if(first)
	{
		colourP->SetBackgroundColour(RobVisFrame::theVisFrame->GetBotPainter().GetProgramColour(newInfo.botInfo.program));
		programT->SetLabel(Wx::Conv(newInfo.programName));
	}
	// for all components: check if newInfo is changed vs. info 
	// and then update them. 
	bool forceInstrRefresh = first;
	if(curBank >= newInfo.banks.length()) {
		curBank = 0;
		forceInstrRefresh = true;
	}
	SetLCLength(bankLC, newInfo.banks.length());
	for(int b = 0; b < newInfo.banks.length(); b++)
	{
		if(newInfo.banks[b].hasChanged || first)
		{
			bankLC->SetItem(b, 0, wxString::Format(wxT("%d"), b+1), (int) newInfo.banks[b].owner);
			bankLC->SetItem(b, 1, wxString::Format(wxT("%d"), (int) newInfo.banks[b].instructions.length()));
			bankLC->SetItem(b, 2, Wx::Conv(newInfo.banks[b].name));
		}
	}
	if(first)
		bankLC->SetItemState(0, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
	if(newInfo.banks[curBank].hasChanged || forceInstrRefresh)
		UpdateInstrList(newInfo);

	// update variables
	if(newInfo.namedVarsHasChanged || first)
	{
		for(int v = 0; v < rt::botVarLength; v++)
		{
			if(newInfo.namedVars[v] != info.namedVars[v])
			{
				wxStaticText* displayT = namedVarT[v];
				if(displayT)
					displayT->SetLabel(wxString::Format(wxT("%d"), (int) newInfo.namedVars[v]));
			}
		}
	}
	if(newInfo.unnamedVarsHasChanged || first)
	{
		SetLCLength(unnamedVarLC, newInfo.unnamedVars.length());
		for(int v = 0; v < newInfo.unnamedVars.length(); v++)
		{
			unnamedVarLC->SetItem(v, 0, wxString::Format(wxT("#%d"), (int) (v+1)));
			unnamedVarLC->SetItem(v, 1, wxString::Format(wxT("%d"), (int) newInfo.unnamedVars[v]));
		}
	}

	UpdateTaskData(newInfo, first);

	if(newInfo.alive == false)
		programT->SetLabel(_("The bot was destroyed."));

	// finally, the 'newInfo' is now the current one. 
	info = newInfo;
}

void DebugDialog::UpdateTaskData(DebugInfo& newInfo, bool force)
{
	if(newInfo.tasks.length() <= curTask)
	{
		curTask = 0;
		taskSL->SetValue(curTask);
		force = true;
	}
	if((newInfo.tasks.length() != info.tasks.length()) || force)
	{
		taskSL->SetRange(0, newInfo.tasks.length() - 1);
		taskNumT->SetLabel(wxString::Format(wxT("%d/%d"), (int) (curTask+1), (int) newInfo.tasks.length()));
	}

	DebugInfo::TaskInfo& myTaskInfo = newInfo.tasks[curTask];
	if(force || myTaskInfo.hasChanged)
	{
		// labels
		curInstrT->SetLabel(wxString::Format(wxT("%d"), (int) (myTaskInfo.instrPos+1)));
		curInstrTextT->SetLabel(Wx::Conv(myTaskInfo.instrText));
		curBankT->SetLabel(wxString::Format(wxT("%d"), (int) (myTaskInfo.runBank+1)));
		// viewer
		{
			wxClientDC viewDC(taskViewP);
			BotInfo botInfo = newInfo.botInfo;
			for(int i = 0; i < 4; i++)
				botInfo.head[i] = -1;
			botInfo.head[myTaskInfo.dir] = newInfo.banks[myTaskInfo.runBank].owner;
			RobVisFrame::theVisFrame->GetBotPainter().DrawBot(botInfo, viewDC, wxPoint(0,0), 16);
		}
		// follow managing (bank list, instr list)
		if(follow) {
			if(myTaskInfo.runBank != curBank)
			{
				curBank = myTaskInfo.runBank;
				bankLC->SetItemState(curBank, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
				bankLC->EnsureVisible(curBank);
				UpdateInstrList(newInfo);
			}
			else if(myTaskInfo.instrPos >= 0 && myTaskInfo.instrPos < instrLB->GetCount())
				instrLB->SetSelection(myTaskInfo.instrPos);
		}
	}
}

void DebugDialog::UpdateInstrList(DebugInfo& newInfo)
{
	// update instruction list
	lrt::Vector<lrt::String>& instr = newInfo.banks[curBank].instructions;
	instrLB->Clear();
	for(int i = 0; i < instr.length(); i++)
		instrLB->Append(Wx::Conv(instr[i]));
	if(follow)
	{
		int instrPos = newInfo.tasks[curTask].instrPos;
		if(instrPos >= 0 && instrPos < instr.length())
			instrLB->SetSelection(instrPos);
	}
}

void DebugDialog::UpdateBankImageList()
{
	BotPainter& bp = RobVisFrame::theVisFrame->GetBotPainter();
	wxImageList* il = new wxImageList(8, 16);
	wxMemoryDC dc;
	wxBrush brush(*wxBLACK, wxSOLID);
	wxPen pen(*wxBLACK, 1, wxTRANSPARENT);
	dc.SetPen(pen);
	for(int p = 0; p < bp.GetProgramColourCount(); p++)
	{
		wxBitmap bmp(8, 16);
		dc.SelectObject(bmp);
		brush.SetColour(bp.GetProgramColour(p));
		dc.SetBrush(brush);
		dc.DrawRectangle(0,0, 8,16); // Clear() doesn't seem to work
		dc.SelectObject(wxNullBitmap);
		il->Add(bmp);
	}
	bankLC->AssignImageList(il, wxIMAGE_LIST_SMALL);
}

void DebugDialog::OnFollow(wxCommandEvent&)
{
	follow = followCB->GetValue();
}

void DebugDialog::OnTask(wxScrollEvent&)
{
	int newTask = taskSL->GetValue();
	if(newTask != curTask)
	{
		curTask = newTask;
		UpdateTaskData(info, true);
	}
}

void DebugDialog::OnBankList(wxListEvent& event)
{
	long row = event.GetIndex();
	if(row >= 0 && row < info.banks.length())
	{
		curBank = row;
		UpdateInstrList(info);
	}
}

void DebugDialog::OnCloseWindow(wxCloseEvent&)
{ 
	if(!RobVisFrame::guiInfo.isDestroyed)
	{
		if(!RobVisFrame::theVisFrame->HandleDebugDialogClose(info.debugID))
			Destroy(); // he couldn't find us
	}
	else
		Destroy();
}

