/*
 * robwxgl.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

// Implements the OpenGL 3D field view 

#ifdef __RT_OPENGL__ // You need to enable this to compile OpenGL related stuff

#define NOMINMAX // Windows headers need this, otherwise they will #define min to something!
#include <wx/wx.h>
#include <wx/image.h>
#include <math.h>

#include "robwxgl.h"
#include <GL/gl.h>
#include <GL/glu.h>
#include "robwxutil.h"

// textures
#include "bitmaps/tex_checkers.xpm"
#include "bitmaps/tex_wood.xpm"
#include "bitmaps/tex_steel.xpm"

///////////////
// OPENGL COORDINATE SYSTEM
// 
//          ^ y
//          |
//          |
//          |
//          x------> x
//         /
//      z /
//       /
//      v
//
//////////////////////
// common angles in radians
//const float DEG180 = 3.1415;
//const float DEG360 = 6.2830;
//const float DEG90  = 1.5708;
const float DEG_TO_RAD = 0.0174533f;
const float RAD_TO_DEG = 57.29577f;
// increases for the camera positions (on keypress)
const float incPosDist = 0.1f;
const float incAngUp = 1.f;
const float incAngSide = 1.f;

// increases for the camera positions (on dragging, per pixel)
const float pixIncPosDist = 0.01f;
const float pixIncAngUp = 0.1f;
const float pixIncAngSide = 0.3f;

// increases for the camera positions (on mouse wheel, per 'line')
const float wheelIncPosDist = 0.5f;

// event table
BEGIN_EVENT_TABLE(GLFieldPanel, wxGLCanvas)
  EVT_PAINT(GLFieldPanel::OnPaint)
  EVT_SIZE(GLFieldPanel::OnSize)
  EVT_ERASE_BACKGROUND(GLFieldPanel::OnEraseBackground)
  EVT_KEY_DOWN(GLFieldPanel::OnKeyDown)
  EVT_ENTER_WINDOW(GLFieldPanel::OnEnterWindow)
  EVT_LEFT_DOWN(GLFieldPanel::OnLeftDown)
  EVT_RIGHT_DOWN(GLFieldPanel::OnRightDown)
  EVT_LEFT_UP(GLFieldPanel::OnLeftUp)
  EVT_RIGHT_UP(GLFieldPanel::OnRightUp)
  EVT_MOTION(GLFieldPanel::OnMouseMotion)
  EVT_MOUSEWHEEL(GLFieldPanel::OnMouseWheel)
  EVT_LEAVE_WINDOW(GLFieldPanel::OnLeaveWindow)
END_EVENT_TABLE()

// Set OpenGL window parameters so that it renders most quickly
// (we do not need accumulation, auxiliary nor alpha buffers)
static int glParams[] = { WX_GL_RGBA, WX_GL_DEPTH_SIZE, 16, WX_GL_AUX_BUFFERS, 0, WX_GL_DOUBLEBUFFER,  
                          WX_GL_MIN_RED, 5, WX_GL_MIN_BLUE, 5, WX_GL_MIN_GREEN, 5, WX_GL_MIN_ALPHA, 0,
                          WX_GL_MIN_ACCUM_RED, 0, WX_GL_MIN_ACCUM_GREEN, 0, WX_GL_MIN_ACCUM_BLUE, 0, 
				          WX_GL_MIN_ACCUM_ALPHA, 0,
                          0/*End*/ };

GLFieldPanel::GLFieldPanel(wxWindow* parent, RobVisFrame* vframe, wxWindowID id)
	: wxGLCanvas(parent, id, wxDefaultPosition, wxSize(640, 480), 0/*style*/, wxT("wxGLCanvas"), glParams), FieldDisplay(vframe),
	  glInitDone(false), useHQ(false), posDist(10), angUp(45), angSide(0), quadric(0)
{
	// pre-initialize simInfo, so we can draw the empty field 
	simInfo.height = 16; simInfo.width = 16; 
	simInfo.cycle = 0; simInfo.maxBanks = 50;
	simInfo.timeout = 100000; simInfo.simNum = 0; 

	// texture identifiers: they first have to be loaded...
	texField = 0; 
	texIset[0] = 0; texIset[1] = 0; texIset[2] = 0; 

	// list identifiers: they first have to be created...
	listBot = 0; listBotHQ = 0; listHead = 0;
	listBigField = 0; lbfX = 0; lbfY = 0; 
}

GLFieldPanel::~GLFieldPanel()
{
	gluDeleteQuadric((GLUquadric*) quadric);
}

void GLFieldPanel::EnterSim(const SimInfo& info)
{
	simInfo = info;
}

void GLFieldPanel::ExitSim()
{
}

void GLFieldPanel::StepSim(const lrt::Vector<BotInfo>& bots)
{
	status = bots;
	Update();
}

// This method draws the field (and robots) in OpenGL. 
void GLFieldPanel::Update(bool force)
{
	wxSize sz = GetSize();
	if(sz.x <= 0 || sz.y <= 0)
		return;

	SetCurrent();
	//************* BEGIN scene **************************//

    // init OpenGL once, but after SetCurrent 
    if (!glInitDone)
    {
        InitGL();
        glInitDone = true;
    }
	// field size might have changed; ensure that the field list contains correct size
	UpdateBigFieldIfNeeded();

	////////////////////////// setup camera /////////
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	// setup camera perspective : field-of-view 60 degrees, aspect ratio as determined by the window, front & back planes
	gluPerspective(60.0, (float)sz.x / (float)sz.y, 0.4, 100.0);

	// setup look at: look from positive z at the origin
	gluLookAt(0,0,posDist, 0,0,0, 0,1,0);

	// now rotate upwards
	glRotatef(angUp, 1,0,0);

	// average board "radius"
	float halfBoardSize = (simInfo.width + simInfo.height) / 4.f; 	
	// Move the camera a bit backwards (off the semi-sphere it normally uses) so that the bots on the front
	// are displayed as well as those on back. 
	//
	// mvBack                      
	//  ^    /\                  If (upAngle > FOV/2 (30)):   lookAtMove =   
	//  |   /  \                       linear interpolation (0 when vertical, boardRadius/3 when at 30)
	//  |  /    \               
	//  | /      \               If (upAngle < FOV/2 (30)):   lookAtMove =
	//  |/        \                    linear interpolation (boardRadius/3 when at 30, 0 when horizontal)
	//  +----+-----+--> angUp
	//  0  FOV/2   90
	float lookAtMove = (angUp > 30) ? ((90.f - angUp) / 90.f * (halfBoardSize / 2.f)) : ((angUp / 30.f) * (halfBoardSize / 3.f));
	glTranslatef(0,0,-lookAtMove);

	// rotate around the center
	glRotatef(angSide, 0,1,0);
	// store this projection matrix for click point computation
	glGetDoublev(GL_PROJECTION_MATRIX, projMatrix);

	/////////////////////////// setup scene /////////
    glMatrixMode(GL_MODELVIEW);
	// clear color and depth buffers
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();
	// set light
	GLfloat lightpos[] = {-1, 1, 1, 0};
	glLightfv(GL_LIGHT0, GL_POSITION, lightpos);


	// paint scene
	// move field & bots so that its center is at the origin
	// (we paint them at their (x,z) coordinates, that would be only the front right quadrant)
	glTranslatef(-simInfo.width / 2 + 0.5f, 0, -simInfo.height / 2 + 0.5f);
	// store this modelview matrix for click point computation
	glGetDoublev(GL_MODELVIEW_MATRIX, fieldModelMatrix);

	while(glGetError()); // clear previous errors

	// draw the whole field in one polygon
	glCallList(listBigField);
	if(glGetError()) wxLogError(wxT("glCallList(field)"));

	GLfloat white[] = {1, 1, 1, 1};
	glMaterialfv(GL_FRONT, GL_SPECULAR, white); // set highlights to white
	glMateriali(GL_FRONT, GL_SHININESS, 100); // make bots shiny
	// draw bots (each field)
	for(int x = 0; x < simInfo.width; x++)
		for(int y = 0; y < simInfo.height; y++)
			DrawField(x, y, status[x + y * simInfo.width]);
	glMateriali(GL_FRONT, GL_SHININESS, 0); // make other things not shiny

	//*************** END scene ************************************//
	glFlush();     // commit commands
	SwapBuffers(); // make them visible
}

void GLFieldPanel::HandleProgramColourChange()
{
}

void GLFieldPanel::OnPaint(wxPaintEvent& event)
{
	wxPaintDC dc(this);
	Update(true);
}

void GLFieldPanel::OnSize(wxSizeEvent& event)
{
    // this is also necessary to update the context on some platforms
    wxGLCanvas::OnSize(event);

    // set GL viewport (not called by wxGLCanvas::OnSize on all platforms...)
    int w, h;
    GetClientSize(&w, &h);
#ifndef __WXMOTIF__
    if (GetContext())
#endif
    {
        SetCurrent();
        glViewport(0, 0, (GLint) w, (GLint) h);
		glGetIntegerv(GL_VIEWPORT, viewport);
    }
}

void GLFieldPanel::OnEraseBackground(wxEraseEvent& event)
{
  // Do nothing, to avoid flashing.
}

void GLFieldPanel::OnLeftDown(wxMouseEvent& event)
{
	dragPoint = event.GetPosition();
	isClickEvent = true; // there was no dragging yet
}

void GLFieldPanel::OnRightDown(wxMouseEvent& event)
{
	dragPoint = event.GetPosition();
}

void GLFieldPanel::OnLeftUp(wxMouseEvent& event)
{
	if(!isClickEvent)
		return; // user dragged, so don't interprete clicks
	// it was a click: so handle it: compute field position
	SetCurrent();
	int pixX = dragPoint.x, pixY = viewport[3] - dragPoint.y;
//	lrt::System::println(Wx::Conv(wxString::Format("pix coord: <%d,%d> vp<%d,%d,%d,%d>", pixX, pixY, viewport[0], viewport[1], viewport[2], viewport[3])));
	double objX, objY, objZ, yTmp;
	if(!gluUnProject(pixX, pixY, 0.0, fieldModelMatrix, projMatrix, viewport, &objX, &objY, &objZ))
		wxLogError(wxT("gluUnProject(x,y,0)"));
	yTmp = objY;
//	lrt::System::println(Wx::Conv(wxString::Format("front coord: <%f,%f,%f>", objX, objY, objZ)));
	if(!gluUnProject(pixX, pixY, 1.0, fieldModelMatrix, projMatrix, viewport, &objX, &objY, &objZ))
		wxLogError(wxT("gluUnProject(x,y,1)"));
//	lrt::System::println(Wx::Conv(wxString::Format("back coord: <%f,%f,%f>", objX, objY, objZ)));
	// interpolate so that Y=0 (field level)
	if((objY > 0 && yTmp > 0) || yTmp < objY) {
		lrt::System::println("cannot find click point (view line does not cross Y=0 plane, or comes from the floor)");
		return; 
	}
	double t = 0.5; // perform binary search on winZ parameter;
	double inc = t;
	while(true) {
		inc /= 2;
		gluUnProject(pixX, pixY, t, fieldModelMatrix, projMatrix, viewport, &objX, &objY, &objZ);
//		lrt::System::println(Wx::Conv(wxString::Format("coord<%f>: <%f,%f,%f>", t, objX, objY, objZ)));
		if(objY - 0.0 > 1e-5)
			t += inc;
		else if(objY - 0.0 < -1e-5)
			t -= inc; 
		else
			break; 
	}
	int fieldX = floor(objX + 0.5);
	int fieldY = floor(objZ + 0.5);
	if(fieldX < 0 || fieldX >= simInfo.width || fieldY < 0 || fieldY >= simInfo.height)
	{
		lrt::System::println("clicked outside of field!");
		return; 
	}
	lrt::System::println(Wx::Conv(wxString::Format(wxT("click point: <%f,%f, [%f]> => <%d,%d>"), objX, objZ, objY, fieldX, fieldY)));
	vframe->HandleFieldClick(fieldX, fieldY);
}
void GLFieldPanel::OnRightUp(wxMouseEvent& event)
{
}

void GLFieldPanel::OnMouseWheel(wxMouseEvent& event)
{
	float distInc = -wheelIncPosDist * event.GetWheelRotation() / event.GetWheelDelta();
	posDist += distInc;
	if(posDist < 0.1) posDist = 0.1f;
	if(posDist > 100) posDist = 100;
	Update();

//	if(event.GetWheelRotation())
//		lrt::System::println(lrt::String("wheel rotation: ") + event.GetWheelRotation() + ", Delta=" + event.GetWheelDelta());
}

void GLFieldPanel::OnMouseMotion(wxMouseEvent& event)
{
	if(lrt::Math::abs(event.GetX() - dragPoint.x) + lrt::Math::abs(event.GetY() - dragPoint.y) <= 2)
		return; // user has not really started dragging; mouse was not moved more than 2px
	// ok, dragging started
	isClickEvent = false; 

	if(event.LeftIsDown() || event.RightIsDown())
	{
		int dx = event.GetX() - dragPoint.x, dy = event.GetY() - dragPoint.y; 
		if(event.LeftIsDown()) { // dragging left button
			float sideInc = pixIncAngSide * dx, upInc = pixIncAngUp * dy;
			angSide += sideInc; 
			if(angSide < 0) angSide += 360;
			if(angSide > 360) angSide -= 360;
			angUp += upInc;
			if(angUp < 1) angUp = 1;
			if(angUp > 89.9f) angUp = 89.9f;
		}
		/*else*/ if(event.RightIsDown()) { // dragging right button
			float distInc = pixIncPosDist * dy;
			posDist += distInc;
			if(posDist < 0.1) posDist = 0.1f;
			if(posDist > 100) posDist = 100;
		}
		Update();
		WarpPointer(dragPoint.x, dragPoint.y);
	}
} 

void GLFieldPanel::OnLeaveWindow( wxMouseEvent& event )
{
}

void GLFieldPanel::OnEnterWindow( wxMouseEvent& event )
{
    SetFocus();
}

void GLFieldPanel::OnKeyDown(wxKeyEvent& event)
{
	switch(event.GetKeyCode())
	{
	  case WXK_PRIOR: // Page Up
		posDist -= incPosDist;
		if(posDist <= 0)
			posDist = 0;
		break;
	  case WXK_NEXT: // Page Down
		posDist += incPosDist;
		if(posDist >= 100)
			posDist = 100;
		break;
	  case WXK_UP: // Up key
	  case WXK_NUMPAD8:
		angUp += incAngUp;
		if(angUp >= 90)
			angUp = 90;
		break;
	  case WXK_DOWN: // Down key
	  case WXK_NUMPAD2:
		angUp -= incAngUp;
		if(angUp <= -10)
			angUp = -10;
		break;
	  case WXK_LEFT: // Side key
	  case WXK_NUMPAD4:
		angSide += incAngSide;
		if(angSide >= 360)
			angSide -= 360;
		break;
	  case WXK_RIGHT: // Down key
	  case WXK_NUMPAD6:
		angSide -= incAngSide;
		if(angSide <= 0)
			angSide += 360;
		break;
	  default:
		return;
	}

	Update();
}


bool GLFieldPanel::Show(bool doShow)
{
	return wxGLCanvas::Show(doShow);
}

wxWindow* GLFieldPanel::GetAsWindow()
{
	return (wxWindow*) this; 
}

void GLFieldPanel::DrawField(int x, int z, const BotInfo& info)
{
	int fieldIndex = (x+z) & 1;
	float factor = (info.active ? 1.f : 0.5f);

	glPushMatrix();
	glTranslatef(x, 0, z);

	if(info.program >= 0) // there IS a bot here...
	{
		float scale = (simInfo.maxBanks - info.banks);
		scale /= simInfo.maxBanks;
		scale *= scale; 
		scale *= scale; 
		scale *= 0.9f;
		scale = 1.0f - scale; 
		glScalef(1, scale, 1);


		GLfloat col[4];
		MakeGLColour(vframe->GetBotPainter().GetProgramColour(info.program), factor, col);
		if(info.type >= 0 && info.type <= 2 && texIset[info.type])
		{
			glEnable(GL_TEXTURE_2D);
			glBindTexture(GL_TEXTURE_2D, texIset[info.type]);
			for(int i = 0; i < 3; i++) col[i] *= 1.2f; 
		}
		glMaterialfv(GL_FRONT, GL_DIFFUSE, col);
		for(int i = 0; i < 3; i++) col[i] *= 0.5f;
		glMaterialfv(GL_FRONT, GL_AMBIENT, col);
		if(useHQ) // high quality bots
			glCallList(listBotHQ);
		else // low [?] quality bots
			glCallList(listBot);
		glDisable(GL_TEXTURE_2D);

		for(int h = 0; h < 4; h++)
		{
			if(info.head[h] >= 0)
			{
				if((info.head[h] == info.program) || !info.active)
					factor = 0.5f;
				else 
					factor = 1.f;
				MakeGLColour(vframe->GetBotPainter().GetProgramColour(info.head[h]), factor, col);
				glMaterialfv(GL_FRONT, GL_DIFFUSE, col);
				for(int i = 0; i < 3; i++) col[i] *= 0.5;
				glMaterialfv(GL_FRONT, GL_AMBIENT, col);
				glCallList(listHead);
			}
			if(h != 3) 
				glRotatef(90, 0,1,0);
		}
	}

	glPopMatrix();
}

void GLFieldPanel::UpdateBigFieldIfNeeded()
{
	if(!texField || !glIsTexture(texField))
	{
		wxLogError(wxT("update big field called, but texField not loaded!"));
	}

	if(simInfo.width == lbfX && simInfo.height == lbfY)
		return; // nothing to do

	// otherwise gen big field list. 
	if(!listBigField)
		listBigField = glGenLists(1);

	float w = simInfo.width; float h = simInfo.height; 
	float halfWidth = simInfo.width / 2.f;
	float halfHeight = simInfo.height / 2.f;

	while(glGetError());

	GLfloat white[4] = {1, 1, 1, 1};
	GLfloat dark[4] = {0.1f, 0.1f, 0.1f, 1};

	glNewList(listBigField, GL_COMPILE);
	if(glGetError()) wxLogError(wxT("glNewList()"));
	  glEnable(GL_TEXTURE_2D);
	  glMaterialfv(GL_FRONT, GL_DIFFUSE, white);
	  glMaterialfv(GL_FRONT, GL_AMBIENT, dark);
	  glBindTexture(GL_TEXTURE_2D, texField);
	  glBegin(GL_QUADS);
	    glNormal3f(0,1,0);
	    glTexCoord2f(0, 0);
		glVertex3f(-0.5f, 0.f, -0.5f);
	    glTexCoord2f(0, halfHeight);
		glVertex3f(-0.5f, 0.f, h-0.5f);
	    glTexCoord2f(halfWidth, halfHeight);
		glVertex3f(w-0.5f, 0.f, h-0.5f);
	    glTexCoord2f(halfWidth, 0);
		glVertex3f(w-0.5f, 0.f, -0.5f);
	  glEnd();
	  glDisable(GL_TEXTURE_2D);
	glEndList();

	lbfX = simInfo.width; lbfY = simInfo.height;
}

int GLFieldPanel::LoadTexture(const wxBitmap& bitmap)
{
	while(glGetError());

	wxImage img = bitmap.ConvertToImage();
	unsigned int texNum;
	glEnable(GL_TEXTURE_2D);
	glGenTextures(1, &texNum);
	if(glGetError()) wxLogError(wxT("glGenTextures()"));
	glBindTexture(GL_TEXTURE_2D, texNum);
	if(glGetError()) wxLogError(wxT("glBindTexture()"));
	gluBuild2DMipmaps(GL_TEXTURE_2D, 3, img.GetWidth(), img.GetHeight(), GL_RGB, GL_UNSIGNED_BYTE, img.GetData());
	if(glGetError()) wxLogError(wxT("gluBuild2DMipmaps()"));
//	glDisable(GL_TEXTURE_2D);
	return texNum;
}

void GLFieldPanel::MakeGLColour(const wxColour& wc, float factor, float* dest)
{
	factor /= 255.;
	dest[0] = wc.Red() * factor;
	dest[1] = wc.Green() * factor;
	dest[2] = wc.Blue() * factor;
	dest[3] = 1; 
}

// Initializes OpenGL once at the start.
void GLFieldPanel::InitGL()
{
    SetCurrent();

	// create & setup the GLU quadric object used to draw the cylinders and disks
	quadric = gluNewQuadric();
	gluQuadricTexture((GLUquadric*) quadric, GL_TRUE);

	// setup background colour (window's background colour)
	GLfloat cc[4];
	MakeGLColour(GetBackgroundColour(), 1, cc);
	glClearColor(cc[0], cc[1], cc[2], 0);

	// setup various opengl parameters
	glShadeModel(GL_SMOOTH);	// gouraud shading
	glEnable(GL_NORMALIZE);		// make normal vectors unit length automatically
	glEnable(GL_CULL_FACE);		// remove the faces
	glCullFace(GL_BACK);		// ... which point away from the user
    glEnable(GL_DEPTH_TEST);	// use z-buffer
    glEnable(GL_LIGHTING);		// use lighting
    glEnable(GL_LIGHT0);		// use one light
	glLightModeli(GL_LIGHT_MODEL_TWO_SIDE,GL_FALSE); // light only front side (back side is culled anyway)
	glEnable(GL_TEXTURE_2D);	// use texturing
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_NEAREST); // use interpolation & mipmaps for texturing
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR); // use interpolation for texturing (when enlarging)

	// load texture images
	LoadTextures();
	// create display lists
	GenLists();


}

// Generates display lists, i.e. stored OpenGL commands ("opengl user functions") that draw large things at once
void GLFieldPanel::GenLists()
{
	// make list identifiers
	int listStart = glGenLists(3);
	listBot = listStart;
	listBotHQ = listStart+1;
	listHead = listStart+2;

	// the robot base - is just a cylinder + disk on top, but rotated
	glNewList(listBot, GL_COMPILE);
	  glPushMatrix();
	  glRotatef(-90, 1,0,0);
	  gluCylinder((GLUquadric*) quadric, 0.45, 0.45, 1, 7, 1);
	  glTranslatef(0,0,1);
	  gluDisk((GLUquadric*) quadric, 0, 0.451, 7, 1);
	  glPopMatrix();
	glEndList();

	// the robot base in high quality - same as above, but more triangles
	glNewList(listBotHQ, GL_COMPILE);
	  glPushMatrix();
	  glRotatef(-90, 1,0,0);
	  gluCylinder((GLUquadric*) quadric, 0.45, 0.45, 1, 15, 1);
	  glTranslatef(0,0,1);
	  gluDisk((GLUquadric*) quadric, 0, 0.451, 15, 1);
	  glPopMatrix();
	glEndList();

	// a robot's head (symbolizes a task) - a triangular extrusion
	glNewList(listHead, GL_COMPILE);
	  glBegin(GL_QUAD_STRIP); 	  // sides
	    glNormal3f(0.2f, 0, -0.98f);
		glVertex3f(0.45f, 1.15f, 0.0f);
		glVertex3f(0.45f, 1.0f,  0.0f);
		glVertex3f(0,     1.15f, -0.1f);
		glVertex3f(0,     1.0f,  -0.1f);
	    glNormal3f(-1, 0, 0);
		glVertex3f(0,     1.15f, 0.1f);
		glVertex3f(0,     1.0f,  0.1f);
	    glNormal3f(0.2f, 0, 0.98f);
		glVertex3f(0.45f, 1.15f, 0.0f);
		glVertex3f(0.45f, 1.0f,  0.0f);
	  glEnd();
	  glBegin(GL_TRIANGLES);	// top
	    glNormal3f(0,1,0);
		glVertex3f(0.45f,  1.15f, 0.0f);
		glVertex3f(0,      1.15f, -0.1f);
		glVertex3f(0,      1.15f, 0.1f);
	  glEnd();
    glEndList();
}

void GLFieldPanel::LoadTextures()
{
	wxBitmap texFieldBmp(tex_checkers_xpm);
	texField = LoadTexture(texFieldBmp);

	wxBitmap texWoodBmp(tex_wood_xpm);
	texIset[1] = LoadTexture(texWoodBmp);
	wxBitmap texSteelBmp(tex_steel_xpm);
	texIset[2] = LoadTexture(texSteelBmp);
}

#endif // __RT_OPENGL__ enabled
