#include "seaview.h"
#include <ctype.h>
#include <FL/Fl_Check_Button.H>

extern "C" { 
#include "raa_acnuc.h"
}

typedef struct {
	raa_db_access *raa; 
	SEA_VIEW *view;
	FILE *infile; 
	int listrank;
	int elt;
	int count, interrupted, use_spec_name, max_seq_length, truncated;
	Fl_Box *runb; 
} runningw_struct;


/* local functions */
int racnuc_fetch(const char *db_name, const char *name, FILE *list, int use_spec_name, SEA_VIEW *view);
int racnuc_fetch_file_or_list(FILE *list, int listrank, 
							   int use_spec_name, raa_db_access *raa, SEA_VIEW *view);
static void idle_callback(void *data);
void racnuc_dialog(SEA_VIEW *view);
void ok_callback(Fl_Widget *ob, void *data);
void close_callback(Fl_Widget *ob, void *data);
void fsel_callback(Fl_Widget *ob, void *data);
void more_callback(Fl_Widget *ob, void *data);
static void interrupt_callback(Fl_Widget *ob, void *data);
char *import_one_seq(raa_db_access *raa, const char *id, int use_spec_name, int *plength, int init_max_seq_length, 
	int *ptruncated, SEA_VIEW *view, int try_as_keyword);
int import_one_keyword(raa_db_access *raa, const char *keyword, int use_spec_name,  
						 int *ptruncated, SEA_VIEW *view);


extern int add_seq_to_align(SEA_VIEW *view, char *newname, char *newseq, int lenseq);
extern int compute_size_params(SEA_VIEW *view, int force_recompute);
extern void update_comment_gc(int seqnum, int ncbigc, SEA_VIEW *view);
extern "C" int get_ncbi_gc_number(int gc);


int racnuc_fetch(const char *db_name, const char *name, FILE *listfile, int use_spec_name, 
				 SEA_VIEW *view)
/*
return value: 0 iff OK, 1 if name does not exist, or socket error code + 1
*/
{
raa_db_access *raa; 
int err, num, length, truncated = FALSE;
char *url, *server_ip;
int port;

#ifndef WIN32
url = getenv("racnuc");
if(url == NULL) 
#endif
url = (char *)"pbil.univ-lyon1.fr:5558";
raa_decode_address(url, &server_ip, &port, NULL);
err = raa_acnucopen_alt(server_ip, port , (char *)db_name, (char *)"seaview", &raa); 
if(err) {
	if(listfile != NULL) fclose(listfile);
	if(err == 4) fl_message("DataBase %s is offline. Please try again later.", db_name);
	else fl_message("DataBase connection failed.\nNo network connection?\n"
		"Firewall blocks outbound connections to port %d?", port);
	return err + 1;
	}

if(view->masename == NULL) view->masename = strdup("newseqs");
if(view->tot_seqs == 0) view->max_seq_length = FL_max(view->max_seq_length, 20000);
err = 0;
my_watch_cursor(view->dnawin);

if(listfile == NULL) {
	if(import_one_seq(raa, name, use_spec_name, &length, view->max_seq_length, 
			&truncated, view, TRUE) == NULL) {
		fl_reset_cursor(view->dnawin);
		fl_message("Sequence/Keyword %s not found in %s", name, db_name);
		}
	}
else {
	truncated = racnuc_fetch_file_or_list(listfile, 0, use_spec_name, raa, view);
	}
raa_acnucclose(raa);
length = 0;
for(num = 0; num < view->tot_seqs; num++) length = FL_max(length, strlen(view->seqname[num]));
view->wid_names = FL_min(length, 20);
compute_size_params(view, TRUE);
view->DNA_obj->damage(FL_DAMAGE_ALL);
view->horsli->damage(FL_DAMAGE_ALL);
view->vertsli->damage(FL_DAMAGE_ALL);
fl_reset_cursor(view->dnawin);
if(truncated)
	fl_message("Some seqs have been truncated to the length of %d", view->max_seq_length);
return err;
}


int racnuc_fetch_file_or_list(FILE *listfile, int listrank, int use_spec_name, raa_db_access *raa, 
							   SEA_VIEW *view)
{
	runningw_struct runningw_data;
	Fl_Window *runw = new Fl_Window(330,60);
	runw->label("importing sequences");
	runningw_data.runb = new Fl_Box(FL_UP_FRAME, 5, 5, 320, 20, "");
	Fl_Button *runi = new Fl_Button(130,30, 60, 20, "Interrupt");
	runi->callback(interrupt_callback, &(runningw_data.interrupted));
	runw->end();
    runw->position((Fl::w() - runw->w())/2, (Fl::h() - runw->h())/2);
	runw->show();
	runningw_data.raa=raa;
	runningw_data.max_seq_length = view->max_seq_length;
	if(listfile != NULL) runningw_data.infile=listfile;
	else  {
		runningw_data.listrank = listrank;
		runningw_data.infile = NULL;
		runningw_data.elt = 1;
		}
	runningw_data.interrupted = FALSE;
	runningw_data.truncated = FALSE;
	runningw_data.use_spec_name = use_spec_name;
	runningw_data.view = view;
	runningw_data.count = 0;
	Fl::add_idle(idle_callback, &runningw_data);
	if(listfile != NULL) {
		while(runningw_data.infile != NULL) Fl::wait();
		}
	else  {
		while(runningw_data.listrank != 0) Fl::wait();
		}
	delete runw;
	return runningw_data.truncated;
}


static void idle_callback(void *data)
{
static char aux[150];
char line[100], *p, *mnemo;
int length, doit, next;
runningw_struct *runningw_data = (runningw_struct *)data;

doit = FALSE;
if(!runningw_data->interrupted) {
	if(runningw_data->infile != NULL) {
		if( fgets(line, sizeof(line), runningw_data->infile) != NULL ) doit = TRUE;
		}
	else if((next = raa_nexteltinlist(runningw_data->raa, runningw_data->elt, runningw_data->listrank, 
								 &mnemo, NULL)) != 0) {
		strcpy(line, mnemo);
		runningw_data->elt = next;
		doit = TRUE;
		}
	}
if(doit) {
	p = line + strlen(line) - 1;
	if(p > line && *p == '\n') *p = 0;
	mnemo = import_one_seq(runningw_data->raa, line, runningw_data->use_spec_name, &length, 
		runningw_data->max_seq_length, &runningw_data->truncated, runningw_data->view, FALSE);
	if(mnemo != NULL) 
		sprintf(aux, "Name: %s Count: %d", mnemo, ++runningw_data->count);
	else
		sprintf(aux, "Name: %s not found", line);
	runningw_data->runb->label(aux);
	runningw_data->runb->window()->redraw();
	}
else {
	Fl::remove_idle(idle_callback, data);
	if(runningw_data->infile != NULL){
		fclose(runningw_data->infile);
		runningw_data->infile = NULL;
		}
	runningw_data->listrank = 0;
	}
}


typedef struct {
	SEA_VIEW *view;
	Fl_Input *seq_name_field;
	Fl_Choice *dbchoice;
	Fl_Choice *namingchoice;
	} raa_dialog_struct;


void racnuc_dialog(SEA_VIEW *view)
{
static int first = TRUE;
static Fl_Window *load_form;
static raa_dialog_struct raa_fields;

if(first) {
	first = FALSE;
	load_form = new Fl_Window(433,150);
	load_form->box(FL_FLAT_BOX);
	load_form->label("Database Sequence Import");
	
	raa_fields.dbchoice = new Fl_Choice(80, 5, 105, 20, "Database:");
	raa_fields.dbchoice->add("embl|genbank|swissprot");
	raa_fields.dbchoice->value(0);
	
	raa_fields.namingchoice = new Fl_Choice(280, 5, 145, 20, "Name using:");
	raa_fields.namingchoice->add("ID-LOCUS name|Species name");
	raa_fields.namingchoice->value(0);
	
	new Fl_Box(FL_UP_FRAME, 5, 30, 420, 90, "");


	raa_fields.seq_name_field = new Fl_Input(190, 35, 150, 20, "Name/Acc. no./Keyword:");
	Fl_Return_Button *ok = new Fl_Return_Button(345, 35, 50, 20 , "OK");
	ok->callback(ok_callback, &raa_fields);
	
	new Fl_Box(FL_DOWN_FRAME, 10, 65, 20, 20, "or");
	
	new Fl_Box(FL_NO_BOX, 35, 90, 150, 20, "File of names/acc. nos.:");
	Fl_Button *fsel = new Fl_Button(190, 90, 100, 20 , "Browse for file");
	fsel->callback(fsel_callback, &raa_fields);

	Fl_Button *close = new Fl_Button(370, 125, 55, 20 , "Close");
	close->callback(close_callback, NULL);
	
	Fl_Check_Button *more = new Fl_Check_Button(5, 125, 135, 20 , "More Databases");
	more->callback(more_callback, &raa_fields);
	
	load_form->end();
    load_form->position((Fl::w() - load_form->w())/2, 
		(Fl::h() - load_form->h())/2);
	}
raa_fields.seq_name_field->value("");
raa_fields.seq_name_field->take_focus();
raa_fields.view = view;
load_form->show();
}


void ok_callback(Fl_Widget *ob, void *data)
{
raa_dialog_struct *raa_fields = (raa_dialog_struct *)data;
if(strlen(raa_fields->seq_name_field->value()) == 0) return;
//ob->window()->hide();
int use_spec_name = raa_fields->namingchoice->value() == 1;
racnuc_fetch(raa_fields->dbchoice->text(), raa_fields->seq_name_field->value(), NULL, 
	use_spec_name, raa_fields->view);
}


void close_callback(Fl_Widget *ob, void *data)
{
ob->window()->hide();
}

#include "FL/Fl_Native_File_Chooser.H"

void fsel_callback(Fl_Widget *ob, void *data)
{
#define MESSAGE "Select file of list of seq. names"
raa_dialog_struct *raa_fields = (raa_dialog_struct *)data;

ob->window()->hide();
Fl_Native_File_Chooser *chooser = new Fl_Native_File_Chooser();
chooser->type(Fl_Native_File_Chooser::BROWSE_FILE);   // let user browse a single file
chooser->title(MESSAGE);                        
if ( chooser->show() != 0 ) return;
FILE *list = fopen(chooser->filename(), "r");
int use_spec_name = raa_fields->namingchoice->value() == 1;
if(list != NULL) {
	racnuc_fetch(raa_fields->dbchoice->text(), NULL, list, use_spec_name, raa_fields->view);
	}
}


void more_callback(Fl_Widget *ob, void *data)
{
raa_dialog_struct *raa_fields = (raa_dialog_struct *)data;
raa_db_access *raa;
char **names, **descriptions, *url, *server_ip;
int num, tot, err, port;
Fl_Check_Button *b = (Fl_Check_Button *)ob;

if(b->value()) {
	b->deactivate();
	#ifndef WIN32
	url = getenv("racnuc");
	if(url == NULL) 
	#endif
	url = (char *)"pbil.univ-lyon1.fr:5558";
	raa_decode_address(url, &server_ip, &port, NULL);
	err = raa_open_socket(server_ip, port , (char *)"seaview", &raa); 
	if(err) return;
	tot = raa_knowndbs(raa, &names, &descriptions); 
	for(num = 0; num < tot; num++) {
		raa_fields->dbchoice->add(names[num], 0, NULL, 0, 0);
		}
	raa_acnucclose(raa);
	}
}


static void interrupt_callback(Fl_Widget *ob, void *data)
{
*(int *)data = TRUE;
}


char *import_one_seq(raa_db_access *raa, const char *id, int use_spec_name, int *plength, 
	int init_max_seq_length, int *ptruncated, SEA_VIEW *view, int try_as_keyword)
{
char *mnemo, *acc, *specname, *descr, *p, *seq;
int length, frame, gencode;
static char line[500];

mnemo = raa_getattributes(raa, id, NULL, &length, &frame, &gencode, &acc, &descr, &specname, &seq);
if(mnemo == NULL) {
	if(try_as_keyword) {
		return (import_one_keyword(raa, id, use_spec_name, ptruncated, view) ? NULL : (char *)id);
		}
	else return NULL;
	}

gencode = get_ncbi_gc_number(gencode);
if(length > init_max_seq_length) {
	*ptruncated = TRUE;
	length = init_max_seq_length;
	seq[length] = 0;
	}
if(frame == 1) {
	seq -= 2; /* this terrible thing is possible after raa_getattributes */
	length += 2;
	*seq = '-'; *(seq + 1) = '-';
	}
else if(frame == 2) {
	seq -= 1; length += 1;
	*seq = '-'; 
	}
add_seq_to_align(view, use_spec_name ? specname : mnemo, seq, length);
*plength = length;
if(view->comments == NULL) view->comments = (char **)calloc(view->tot_seqs, sizeof(char *));
p = view->comments[view->tot_seqs-1];
*line = 0;
if(p != NULL && strcmp(p, ";\n") != 0) strcpy(line, p);
if(p != NULL) free(p);
p = strchr(descr, ' '); /* skip 1st word (name) from descr */
if(p != NULL) p++; else p = descr;
sprintf(line + strlen(line), ";%s\n", p);
p = line + strlen(line);
if(use_spec_name) 
	sprintf(p, ";Name=%s\n;Acc=%s\n", mnemo, acc);
else 
	sprintf(p, ";/organism=\"%s\"\n;Acc=%s\n", specname, acc);
if(frame != 0 && strstr(line, "/codon_start=") == NULL) sprintf(line + strlen(line), 
	";/codon_start=%d\n", frame + 1);
int l = strlen(line);
p = (char *)malloc(l + 1);
if(p != NULL) strcpy(p, line);
view->comments[view->tot_seqs-1] = p;
if(view->comments != NULL && gencode != 1) update_comment_gc(view->tot_seqs - 1, gencode, view);
return mnemo;
}


int import_one_keyword(raa_db_access *raa, const char *keyword, int use_spec_name, 
						 int *ptruncated, SEA_VIEW *view)
/* returns 0 iff keyword exists */
{
	int numlist, err;
	char query[WIDTH_MAX + 20];
	
	sprintf(query, "k=%s", keyword);
	err = raa_proc_query(raa, query, NULL, "bykeyword", &numlist, NULL, NULL, NULL);
	if(err) return 1;
	*ptruncated = racnuc_fetch_file_or_list(NULL, numlist, use_spec_name, raa, view);
	return 0;
}
