/*
 *  Part of the shrinkta program, a dvd backup tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/
#include <inttypes.h>
#include <config.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <dvdread/ifo_read.h>
#include <string.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "dvd.h"
#include "dvd-time.h"
#include "dvd-marshal.h"


static GObjectClass *dvd_chapter_parent_class = NULL;
static void     dvd_chapter_class_init	 (DvdChapterClass *class);
static void     dvd_chapter_instance_init(GTypeInstance	  *instance,
					  gpointer	   g_class);
static void     dvd_chapter_dispose	 (GObject	  *object);

static void
dvd_chapter_class_init	(DvdChapterClass *class)
{
	GObjectClass *object_class = (GObjectClass *) class;

	dvd_chapter_parent_class = g_type_class_ref (G_TYPE_OBJECT);
	object_class->dispose = dvd_chapter_dispose;
}

static void
dvd_chapter_instance_init(GTypeInstance	*instance,
			 gpointer	 g_class)
{
	DvdChapter *chapter;
	
	chapter = DVD_CHAPTER (instance);
	
	chapter->video_pts = 0xffffffff;
	chapter->audio_pts = 0xffffffff;
}

static void
dvd_chapter_dispose	(GObject	*object)
{
	DvdChapter *chapter;
	
	chapter = DVD_CHAPTER (object);
	
	g_free (chapter->time);
	g_free (chapter->start_time);
	if (chapter->next_chapter != NULL) {
		g_object_unref (G_OBJECT (chapter->next_chapter));
	}
	
	G_OBJECT_CLASS (dvd_chapter_parent_class)->dispose (G_OBJECT (chapter));
}

/**
 * dvd_title_get_type
 * @return The GType for the DvdTitle class.
 */
GType
dvd_chapter_get_type	(void)
{
	static GType chapter_type = 0;

	if (chapter_type == 0) {
		GTypeInfo chapter_info = {
			sizeof (DvdChapterClass),
			NULL,
			NULL,
			(GClassInitFunc) dvd_chapter_class_init,
			NULL,
			NULL, /* class_data */
			sizeof (DvdChapter),
			0, /* n_preallocs */
			(GInstanceInitFunc) dvd_chapter_instance_init
	    	};
		chapter_type = g_type_register_static (G_TYPE_OBJECT,
						       "DvdChapter",
						       &chapter_info, 0);
	}
	return chapter_type;
}

/**
 * Sets the title position id of a chapter.
 * This would normally be set automatically when the chapter is added to a #DvdTitle.
 * @returns void.
 */
void
dvd_chapter_set_id	(DvdChapter	*chapter,
			 guint8		 chapter_id)
{
	g_assert (chapter != NULL);
	
	chapter->chapter_id = chapter_id;
}


/**
 * Gets the total title play time.
 * @param chapter A #DvdChapter.
 * @returns A constant #DvdTime.
 */
G_CONST_RETURN
DvdTime
*dvd_chapter_get_time	   (DvdChapter	*chapter)
{
	g_assert (chapter != NULL);
	
	return chapter->time;
}

/**
 * Gets the chapter's start time position in reference to
 * the title that it belongs to.
 * @param chapter A #DvdChapter.
 * @returns #DvdTime.
 */
G_CONST_RETURN
DvdTime
*dvd_chapter_get_start_time(DvdChapter	*chapter)
{
	g_assert (chapter != NULL);
	
	return chapter->start_time;
}

/**
 * Gets the estimated (vob) kilobytes of the chapter.
 * @note This is only an estimate of the original
 * chapter disk size, including all audio tracks.
 * @param chapter A #DvdChapter.
 * @returns Number of kilobytes.
 */
guint32
dvd_chapter_get_kbs	(DvdChapter	*chapter)
{
	g_assert (chapter != NULL);
	
	return chapter->kbs;
}

/**
 * Gets the estimated (vob) size of a chapter.
 * @note This is only an estimate of the original
 * chapter disk size, including all audio tracks.
 * @param chapter A #DvdChapter.
 * @param gbytes Returned number of gigabytes.
 * @param mbytes Returned number of megabytes.
 * @param kbytes Returned number of kilobytes.
 * @note No parameter may be null.
 * @returns Number of kilobytes.
 */
void
dvd_chapter_get_size	(DvdChapter	*chapter,
			 guint8		*gbytes,
			 guint16	*mbytes,
			 guint16	*kbytes)
{
	guint32 kbs;
	
	g_assert (chapter != NULL);
	g_assert (gbytes != NULL);
	g_assert (mbytes != NULL);
	g_assert (kbytes != NULL);
	
	kbs = dvd_chapter_get_kbs (chapter);
	*gbytes = kbs / 0x100000;
	kbs = kbs - *gbytes * 0x100000;
	*mbytes = kbs / 0x400;
	*kbytes = kbs - *mbytes * 0x400;
}

