/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance;

import java.awt.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.*;
import javax.swing.border.Border;
import javax.swing.border.EmptyBorder;
import javax.swing.plaf.BorderUIResource;
import javax.swing.plaf.ComponentUI;
import javax.swing.plaf.basic.BasicButtonListener;
import javax.swing.plaf.basic.BasicButtonUI;

import org.jvnet.lafwidget.animation.*;
import org.jvnet.lafwidget.layout.TransitionLayout;
import org.jvnet.substance.button.SubstanceButtonShaper;
import org.jvnet.substance.combo.SubstanceComboBoxButton;
import org.jvnet.substance.scroll.SubstanceScrollBarButton;
import org.jvnet.substance.utils.*;
import org.jvnet.substance.utils.icon.GlowingIcon;

/**
 * UI for buttons in <b>Substance</b> look and feel.
 * 
 * @author Kirill Grouchnikov
 */
public class SubstanceButtonUI extends BasicButtonUI {
	/**
	 * Property used during the button shaper switch.
	 */
	public static final String BORDER_COMPUTED = "substancelaf.buttonbordercomputed";

	/**
	 * Property used during the button shaper switch.
	 */
	public static final String BORDER_COMPUTING = "substancelaf.buttonbordercomputing";

	/**
	 * Property used to store the original (pre-<b>Substance</b>) button
	 * border.
	 */
	public static final String BORDER_ORIGINAL = "substancelaf.buttonborderoriginal";

	/**
	 * Property used to store the original button icon.
	 */
	public static final String ICON_ORIGINAL = "substancelaf.buttoniconoriginal";

	/**
	 * Property used to store the original (pre-<b>Substance</b>) button
	 * opacity.
	 */
	public static final String OPACITY_ORIGINAL = "substancelaf.buttonopacityoriginal";

	/**
	 * Property used to lock the original (pre-<b>Substance</b>) button
	 * opacity.
	 */
	public static final String LOCK_OPACITY = "substancelaf.lockopacity";

	/**
	 * Internal property used to mark close buttons on title panes.
	 */
	public static final String IS_TITLE_CLOSE_BUTTON = "substancelaf.internal.isTitleCloseButton";

	/**
	 * Painting delegate.
	 */
	private ButtonBackgroundDelegate delegate;

	/**
	 * The rollover button listener.
	 */
	private RolloverButtonListener substanceButtonListener;

	/**
	 * The matching glowing icon. Is used only when
	 * {@link FadeConfigurationManager#fadeAllowed(FadeKind, Component)} returns
	 * true on {@link FadeKind#ICON_GLOW}.
	 */
	protected GlowingIcon glowingIcon;

	// private ChangeListener substanceIconGlowListener;

	// /**
	// * Model change listener for ghost image effects.
	// */
	// private GhostingListener substanceModelChangeListener;

	/**
	 * Property change listener. Listens on changes to the
	 * {@link SubstanceLookAndFeel#BUTTON_SHAPER_PROPERTY} property and
	 * {@link AbstractButton#MODEL_CHANGED_PROPERTY} property.
	 */
	protected PropertyChangeListener substancePropertyListener;

	/**
	 * Listener for fade animations.
	 */
	protected FadeStateListener substanceFadeStateListener;

	//
	// protected long focusLoopFadeInstanceId;
	//
	// protected FocusListener substanceFocusListener;

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.ComponentUI#createUI(javax.swing.JComponent)
	 */
	public static ComponentUI createUI(JComponent b) {
		AbstractButton button = (AbstractButton) b;
		button.setRolloverEnabled(true);
		button.setOpaque(false);
		return new SubstanceButtonUI();
	}

	/**
	 * Simple constructor.
	 */
	public SubstanceButtonUI() {
		this.delegate = new ButtonBackgroundDelegate();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicButtonUI#installDefaults(javax.swing.AbstractButton)
	 */
	@Override
	public void installDefaults(final AbstractButton b) {
		super.installDefaults(b);

		if (b.getClientProperty(SubstanceButtonUI.BORDER_ORIGINAL) == null)
			b.putClientProperty(SubstanceButtonUI.BORDER_ORIGINAL, b
					.getBorder());

		trackGlowingIcon(b);

		SubstanceButtonShaper shaper = SubstanceCoreUtilities
				.getButtonShaper(b);

		if (b.getClientProperty(SubstanceButtonUI.BORDER_COMPUTED) == null) {
			b.setBorder(shaper.getButtonBorder(b));
		} else {
			Border currBorder = b.getBorder();
			if (!(currBorder instanceof SubstanceButtonBorder)) {
				b.setBorder(shaper.getButtonBorder(b));
			} else {
				SubstanceButtonBorder sbCurrBorder = (SubstanceButtonBorder) currBorder;
				if (shaper.getClass() != sbCurrBorder.getButtonShaperClass())
					b.setBorder(shaper.getButtonBorder(b));
			}
		}
		b.putClientProperty(SubstanceButtonUI.OPACITY_ORIGINAL, b.isOpaque());
		if (b instanceof SubstanceComboBoxButton) {
			b.setBorder(new BorderUIResource.CompoundBorderUIResource(
					new EmptyBorder(1, 1, 1, 1), b.getBorder()));
		}

		// Color fg = b.getForeground();
		// if ((fg == null) || (fg instanceof UIResource)) {
		// b.setForeground(new ColorUIResource(SubstanceCoreUtilities
		// .getDefaultScheme(b).getForegroundColor()));
		// }
		//
		// Color bg = b.getBackground();
		// if (bg instanceof UIResource) {
		// b.setBackground(new ButtonColorDelegate(b, false));
		// }

		// Color fg = b.getForeground();
		// if (fg instanceof UIResource) {
		// b.setForeground(new ButtonColorDelegate(b));
		// }
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicButtonUI#uninstallDefaults(javax.swing.AbstractButton)
	 */
	@Override
	public void uninstallDefaults(AbstractButton b) {
		super.uninstallDefaults(b);

		b.setBorder((Border) b
				.getClientProperty(SubstanceButtonUI.BORDER_ORIGINAL));
		b.setOpaque((Boolean) b
				.getClientProperty(SubstanceButtonUI.OPACITY_ORIGINAL));
		Icon origIcon = (Icon) b
				.getClientProperty(SubstanceButtonUI.ICON_ORIGINAL);
		if (origIcon != null)
			b.setIcon(origIcon);
		b.putClientProperty(SubstanceButtonUI.OPACITY_ORIGINAL, null);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicButtonUI#createButtonListener(javax.swing.AbstractButton)
	 */
	@Override
	protected BasicButtonListener createButtonListener(AbstractButton b) {
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicButtonUI#installListeners(javax.swing.AbstractButton)
	 */
	@Override
	protected void installListeners(final AbstractButton b) {
		super.installListeners(b);

		this.substanceButtonListener = new RolloverButtonListener(b);
		b.addMouseListener(this.substanceButtonListener);
		b.addMouseMotionListener(this.substanceButtonListener);
		b.addFocusListener(this.substanceButtonListener);
		b.addPropertyChangeListener(this.substanceButtonListener);
		b.addChangeListener(this.substanceButtonListener);

		this.substancePropertyListener = new PropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent evt) {
				if (SubstanceLookAndFeel.BUTTON_SHAPER_PROPERTY.equals(evt
						.getPropertyName())) {
					SwingUtilities.invokeLater(new Runnable() {
						public void run() {
							SwingUtilities.updateComponentTreeUI(b);
						}
					});
				}
				if (AbstractButton.MODEL_CHANGED_PROPERTY.equals(evt
						.getPropertyName())) {
					if (substanceFadeStateListener != null)
						substanceFadeStateListener.unregisterListeners();
					substanceFadeStateListener = new FadeStateListener(b, b
							.getModel(), SubstanceCoreUtilities
							.getFadeCallback(b));
					substanceFadeStateListener
							.registerListeners(b instanceof SubstanceScrollBarButton);

					//
					// if (substanceModelChangeListener != null)
					// substanceModelChangeListener.unregisterListeners();
					// substanceModelChangeListener = new GhostingListener(b, b
					// .getModel());
					// substanceModelChangeListener.registerListeners();
				}
				if (AbstractButton.ICON_CHANGED_PROPERTY.equals(evt
						.getPropertyName())) {
					trackGlowingIcon(b);
				}
			}
		};
		b.addPropertyChangeListener(this.substancePropertyListener);

		this.substanceFadeStateListener = new FadeStateListener(b,
				b.getModel(), SubstanceCoreUtilities.getFadeCallback(b));
		this.substanceFadeStateListener
				.registerListeners(b instanceof SubstanceScrollBarButton);
		//
		// this.substanceModelChangeListener = new GhostingListener(b, b
		// .getModel());
		// this.substanceModelChangeListener.registerListeners();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicButtonUI#uninstallListeners(javax.swing.AbstractButton)
	 */
	@Override
	protected void uninstallListeners(AbstractButton b) {
		b.removeMouseListener(this.substanceButtonListener);
		b.removeMouseMotionListener(this.substanceButtonListener);
		b.removeFocusListener(this.substanceButtonListener);
		b.removePropertyChangeListener(this.substanceButtonListener);
		b.removeChangeListener(this.substanceButtonListener);
		this.substanceButtonListener = null;

		b.removePropertyChangeListener(this.substancePropertyListener);
		this.substancePropertyListener = null;

		this.substanceFadeStateListener.unregisterListeners();
		this.substanceFadeStateListener = null;
		//
		// this.substanceModelChangeListener.unregisterListeners();
		// this.substanceModelChangeListener = null;

		super.uninstallListeners(b);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.JComponent#paintComponent(java.awt.Graphics)
	 */
	@Override
	public void update(Graphics g, JComponent c) {
		AbstractButton button = (AbstractButton) c;
		// trackGlowingIcon(button);
		if (button instanceof JButton) {
			JButton jb = (JButton) button;
			if (PulseTracker.isPulsating(jb)) {
				PulseTracker.update(jb);
			} else {
				// System.out.println(System.currentTimeMillis() + ":"
				// + button.getText());
			}
		}
		// if ("2".equals(button.getText()))
		// System.out.println(((AlphaComposite) TransitionLayout
		// .getAlphaComposite(button)).getAlpha()
		// + ":" + button.getClientProperty(TransitionLayout.ALPHA));
		this.delegate.updateBackground(g, button);
		this.paint(g, c);

		// Some ugly hack to allow fade-out of focus ring. The code
		// in BasicButtonUI doesn't call paintFocus() at all
		// when the component is not focus owner.
		AbstractButton b = (AbstractButton) c;
		FontMetrics fm = c.getFontMetrics(c.getFont());

		Insets i = c.getInsets();

		// Dimension size = new Dimension();
		Rectangle viewRect = new Rectangle();
		Rectangle iconRect = new Rectangle();
		Rectangle textRect = new Rectangle();
		viewRect.x = i.left;
		viewRect.y = i.top;
		viewRect.width = b.getWidth() - (i.right + viewRect.x);
		viewRect.height = b.getHeight() - (i.bottom + viewRect.y);

		textRect.x = textRect.y = textRect.width = textRect.height = 0;
		iconRect.x = iconRect.y = iconRect.width = iconRect.height = 0;

		Font f = c.getFont();
		g.setFont(f);

		// layout the text and icon
		@SuppressWarnings("unused")
		String text = SwingUtilities.layoutCompoundLabel(c, fm, b.getText(), b
				.getIcon(), b.getVerticalAlignment(), b
				.getHorizontalAlignment(), b.getVerticalTextPosition(), b
				.getHorizontalTextPosition(), viewRect, iconRect, textRect, b
				.getText() == null ? 0 : b.getIconTextGap());

		// b.putClientProperty("icon.bounds", iconRect);

		if (!(b.hasFocus() && b.isFocusPainted())) {
			if (FadeTracker.getInstance().isTracked(c, FadeKind.FOCUS))
				this.paintFocus(g, b, viewRect, textRect, iconRect);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.ComponentUI#getPreferredSize(javax.swing.JComponent)
	 */
	@Override
	public Dimension getPreferredSize(JComponent c) {
		AbstractButton button = (AbstractButton) c;
		SubstanceButtonShaper shaper = SubstanceCoreUtilities
				.getButtonShaper(button);

		// fix for defect 263
		Dimension superPref = super.getPreferredSize(button);
		if (superPref == null)
			return null;

		return shaper.getPreferredSize(button, superPref);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicButtonUI#paintFocus(java.awt.Graphics,
	 *      javax.swing.AbstractButton, java.awt.Rectangle, java.awt.Rectangle,
	 *      java.awt.Rectangle)
	 */
	@Override
	protected void paintFocus(Graphics g, AbstractButton b, Rectangle viewRect,
			Rectangle textRect, Rectangle iconRect) {
		if (!b.isFocusPainted())
			return;

		SubstanceCoreUtilities.paintFocus(g, b, b, null, textRect, 1.0f,
				2 + SubstanceSizeUtils.getExtraPadding(b.getFont().getSize()));
		//
		// FadeTracker fadeTracker = FadeTracker.getInstance();
		// FocusKind focusKind = SubstanceCoreUtilities.getFocusKind(b);
		// if ((focusKind == FocusKind.NONE)
		// && (!fadeTracker.isTracked(b, FadeKind.FOCUS)))
		// return;
		//
		// Graphics2D graphics = (Graphics2D) g.create();
		// graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
		// RenderingHints.VALUE_ANTIALIAS_ON);
		//
		// float alpha = 1.0f;
		// if (fadeTracker.isTracked(b, FadeKind.FOCUS)) {
		// alpha = fadeTracker.getFade10(b, FadeKind.FOCUS) / 10.f;
		// }
		// graphics.setComposite(TransitionLayout.getAlphaComposite(b, alpha));
		//
		// Color color = SubstanceColorUtilities.getFocusColor(b);
		// graphics.setColor(color);
		// focusKind.paintFocus(b, graphics, textRect);
		// graphics.dispose();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicButtonUI#paintButtonPressed(java.awt.Graphics,
	 *      javax.swing.AbstractButton)
	 */
	@Override
	protected void paintButtonPressed(Graphics g, AbstractButton b) {
		// overriden to remove default metal effect
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.ComponentUI#contains(javax.swing.JComponent, int,
	 *      int)
	 */
	@Override
	public boolean contains(JComponent c, int x, int y) {
		return ButtonBackgroundDelegate.contains((JButton) c, x, y);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicButtonUI#paintIcon(java.awt.Graphics,
	 *      javax.swing.JComponent, java.awt.Rectangle)
	 */
	@Override
	protected void paintIcon(Graphics g, JComponent c, Rectangle iconRect) {
		Graphics2D graphics = (Graphics2D) g.create();
		FadeTracker fadeTracker = FadeTracker.getInstance();
		AbstractButton b = (AbstractButton) c;
		Icon icon = SubstanceCoreUtilities.getIcon(b, this.glowingIcon, false);

		graphics.setComposite(TransitionLayout.getAlphaComposite(b, g));
		// super.paintIcon(graphics, c, iconRect);
		if (fadeTracker.isTracked(b, FadeKind.ROLLOVER)) {
			ComponentState state = ComponentState.getState(b.getModel(), b);
			// System.out.println(state.name() + ":" + state.isRollover());
			if (state.isKindActive(FadeKind.ROLLOVER)) {// ==
				// ComponentState.ROLLOVER_UNSELECTED) {
				// Came from default state
				SubstanceCoreUtilities.getIcon(b, this.glowingIcon, true)
						.paintIcon(b, graphics, iconRect.x, iconRect.y);
				graphics
						.setComposite(TransitionLayout
								.getAlphaComposite(b, fadeTracker.getFade10(b,
										FadeKind.ROLLOVER) / 10.0f, g));
				icon.paintIcon(b, graphics, iconRect.x, iconRect.y);
			} else {
				// if (state == ComponentState.DEFAULT) {
				// Came from rollover state
				icon.paintIcon(b, graphics, iconRect.x, iconRect.y);
				graphics
						.setComposite(TransitionLayout
								.getAlphaComposite(b, fadeTracker.getFade10(b,
										FadeKind.ROLLOVER) / 10.0f, g));
				b.getIcon().paintIcon(b, graphics, iconRect.x, iconRect.y);
			}
		} else {
			icon.paintIcon(b, graphics, iconRect.x, iconRect.y);
		}
		graphics.dispose();
	}

	// @Override
	// protected void paintText(Graphics g, AbstractButton b, Rectangle
	// textRect,
	// String text) {
	// // blur the text shadow
	// int width = b.getWidth();
	// int height = b.getHeight();
	// BufferedImage blurred = SubstanceCoreUtilities.getBlankImage(width,
	// height);
	// Graphics2D gBlurred = (Graphics2D) blurred.getGraphics();
	// gBlurred.setFont(g.getFont());
	// gBlurred.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING,
	// RenderingHints.VALUE_TEXT_ANTIALIAS_OFF);
	// Color negative = SubstanceColorUtilities.getNegativeColor(b
	// .getForeground());
	// int nr = negative.getRed();
	// int ng = negative.getGreen();
	// int nb = negative.getBlue();
	// gBlurred.setColor(negative);
	// float[] kernel = new float[36];
	// Arrays.fill(kernel, 0.1f);
	// ConvolveOp convolve = new ConvolveOp(new Kernel(6, 6, kernel),
	// ConvolveOp.EDGE_NO_OP, null);
	//
	// BasicGraphicsUtils.drawStringUnderlineCharAt(gBlurred, text, b
	// .getDisplayedMnemonicIndex(),
	// textRect.x + getTextShiftOffset(), textRect.y
	// + b.getFontMetrics(g.getFont()).getAscent()
	// + getTextShiftOffset());
	//
	// // super.paintText(gBlurred, b, textRect, text);
	// blurred = convolve.filter(blurred, null);
	// int[] dstBuffer = ((DataBufferInt) blurred.getRaster().getDataBuffer())
	// .getData();
	// int count = dstBuffer.length;
	// for (int i = 0; i < count; i++) {
	// int val = dstBuffer[i];
	// int alpha = (val >>> 24) & 0xFF;
	// if (alpha > 0) {
	// // alpha = 255 - (255-alpha)/2;
	// dstBuffer[i] = (alpha << 24) | (nr << 16) | (ng << 8) | nb;
	// }
	// // else
	// // dstBuffer[i] = (255<<24) | (255<<16);
	// }
	//
	// Graphics2D graphics = (Graphics2D) g.create();
	// // graphics.setComposite(AlphaComposite.getInstance(
	// // AlphaComposite.SRC_OUT, 0.5f));
	// graphics.drawImage(blurred, 0, 0, null);
	// // graphics.fillRect(5, 5, 5, 5);
	// graphics.dispose();
	// super.paintText(g, b, textRect, text);
	// }

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicButtonUI#paintText(java.awt.Graphics,
	 *      javax.swing.JComponent, java.awt.Rectangle, java.lang.String)
	 */
	@Override
	protected void paintText(Graphics g, JComponent c, Rectangle textRect,
			String text) {
		SubstanceCoreUtilities.paintText(g, (AbstractButton) c, textRect, text,
				this.getTextShiftOffset(), this.getPropertyPrefix());
	}

	/**
	 * Tracks possible usage of glowing icon.
	 * 
	 * @param b
	 *            Button.
	 */
	protected void trackGlowingIcon(AbstractButton b) {
		Icon currIcon = b.getIcon();
		if (currIcon instanceof GlowingIcon)
			return;
		if (currIcon == null)
			return;
		this.glowingIcon = new GlowingIcon(currIcon, b);
	}
}
