/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance;

import java.awt.*;

import javax.swing.*;

import org.jvnet.lafwidget.LafWidgetUtilities;
import org.jvnet.lafwidget.layout.TransitionLayout;
import org.jvnet.substance.theme.SubstanceTheme;
import org.jvnet.substance.title.SubstanceHeaderPainter;
import org.jvnet.substance.title.SubstanceTitlePainter;
import org.jvnet.substance.utils.SubstanceColorUtilities;
import org.jvnet.substance.utils.SubstanceCoreUtilities;

/**
 * Delegate for painting filled backgrounds.
 * 
 * @author Kirill Grouchnikov
 */
public class SubstanceFillBackgroundDelegate {
	/**
	 * Alpha composite of <code>this</code> delegate. The default value is 1.0
	 * which results in completely opaque background. However, in some cases, we
	 * need to draw partially translucent background, as in menus.
	 */
	private float alphaComposite;

	/**
	 * Creates a new opaque fill background delegate.
	 */
	public SubstanceFillBackgroundDelegate() {
		this(1.0f);
	}

	/**
	 * Creates a new translucent fill background delegate.
	 * 
	 * @param alphaComposite
	 *            Alpha composite of <code>this</code> delegate. The default
	 *            value is 1.0 which results in completely opaque background.
	 *            However, in some cases, we need to draw partially translucent
	 *            background, as in menus.
	 */
	public SubstanceFillBackgroundDelegate(float alphaComposite) {
		this.alphaComposite = alphaComposite;
	}

	/**
	 * Updates the background of the specified component on the specified
	 * graphic context.
	 * 
	 * @param g
	 *            Graphic context.
	 * @param c
	 *            Component.
	 */
	public void update(Graphics g, Component c) {
		this.update(g, c, SubstanceCoreUtilities.isInHeader(c));
	}

	// public void _update(Graphics g, Component c) {
	// this.update(g, c);
	// }
	//
	/**
	 * Updates the background of the specified component on the specified
	 * graphic context. The background is updated only if the component is
	 * opaque.
	 * 
	 * @param g
	 *            Graphic context.
	 * @param c
	 *            Component.
	 */
	public void updateIfOpaque(Graphics g, Component c) {
		if (TransitionLayout.isOpaque(c))
			this.update(g, c);
	}

	/**
	 * Updates the background of the specified component on the specified
	 * graphic context.
	 * 
	 * @param g
	 *            Graphic context.
	 * @param c
	 *            Component.
	 * @param useHeaderPainter
	 *            If <code>true</code>, the currently installed title painter
	 *            will be used. The {@link SubstanceHeaderPainter} that has been
	 *            added in version 3.3 handles background paintings on controls
	 *            in menu bars and tool bars (the latter depends on the value of
	 *            {@link SubstanceHeaderPainter#isPaintingContainer(Container)}.
	 */
	public void update(Graphics g, Component c, boolean useHeaderPainter) {
		// failsafe for LAF change
		if (!(UIManager.getLookAndFeel() instanceof SubstanceLookAndFeel)) {
			return;
		}

		boolean isInCellRenderer = (c.getParent() instanceof CellRendererPane);
		boolean isPreviewMode = false;
		if (c instanceof JComponent) {
			isPreviewMode = (Boolean.TRUE.equals(((JComponent) c)
					.getClientProperty(LafWidgetUtilities.PREVIEW_MODE)));
		}
		if ((!isPreviewMode) && (!c.isShowing()) && (!isInCellRenderer)) {
			return;
		}

		Graphics2D graphics = (Graphics2D) g.create();
		synchronized (c) {
			// if (TransitionLayout.isOpaque(c)) {
			// fill background
			graphics.setComposite(TransitionLayout.getAlphaComposite(c, g));
			// this.alphaComposite));
			JRootPane rootPane = SwingUtilities.getRootPane(c);
			SubstanceTitlePainter titlePainter = SubstanceCoreUtilities
					.getTitlePainter(rootPane);
			boolean shouldUseHeaderPainter = useHeaderPainter
					&& (titlePainter instanceof SubstanceHeaderPainter);
			boolean paintDropShadow = false;
			if (shouldUseHeaderPainter
					&& ((c instanceof JToolBar) || (SwingUtilities
							.getAncestorOfClass(JToolBar.class, c) != null))) {
				SubstanceHeaderPainter headerPainter = (SubstanceHeaderPainter) titlePainter;
				shouldUseHeaderPainter = headerPainter
						.isPaintingContainer((Container) c);
				paintDropShadow = shouldUseHeaderPainter ? false
						: headerPainter.isPaintingToolbarDropShadows();
			}
			Container headerParent = SubstanceCoreUtilities.getHeaderParent(c);
			if (shouldUseHeaderPainter
					&& (SwingUtilities.getAncestorOfClass(JToolBar.class, c) != null)
			/* (headerParent instanceof JToolBar) */) {
				shouldUseHeaderPainter = ((SubstanceHeaderPainter) titlePainter)
						.isPaintingContainer((Container) c);
			}

			Window window = SwingUtilities.windowForComponent(c);
			boolean isSelected = (window == null) ? true : window.isActive();
			SubstanceTheme theme = isSelected ? SubstanceLookAndFeel.getTheme()
					.getActiveTitlePaneTheme() : SubstanceLookAndFeel
					.getTheme().getDefaultTitlePaneTheme();
			if (shouldUseHeaderPainter) {

				// if (c instanceof JMenuItem) {
				// System.out.println("Using " + theme.getDisplayName()
				// + " to paint " + ((JMenuItem) c).getText());
				// }

				SubstanceHeaderPainter headerPainter = (SubstanceHeaderPainter) titlePainter;
				// if (!(c instanceof JMenuItem)) {
				headerPainter.paintExtraBackground(graphics, headerParent, c, c
						.getWidth(), c.getHeight(), theme, false);
				// }
			} else {
				if (c instanceof JMenu) {
					Color backgroundColor = SubstanceColorUtilities
							.getBackgroundColor(SubstanceLookAndFeel.getTheme());
					Color headerBackgroundColor = SubstanceColorUtilities
							.getBackgroundColor(SubstanceLookAndFeel.getTheme()
									.getActiveTitlePaneTheme());
					if (shouldUseHeaderPainter
							&& (c.getParent() instanceof JMenuBar))
						graphics.setColor(headerBackgroundColor);
					else
						graphics.setColor(backgroundColor);
				} else {
					graphics.setColor(c.getBackground());
				}
				if (c instanceof JMenuItem) {
					// System.out.println("Filling "
					// + ((JMenuItem) c).getText() + " with "
					// + graphics.getColor().getRed() + ":"
					// + graphics.getColor().getGreen() + ":"
					// + graphics.getColor().getBlue());
					// graphics.setColor(Color.red);
				}
				graphics.fillRect(0, 0, c.getWidth(), c.getHeight());
			}
			if (paintDropShadow) {
				Color headerColor = SubstanceColorUtilities.getBackgroundColor(
						SubstanceLookAndFeel.getTheme()).darker();
				// theme.getDefaultColorScheme().getMidColor();

				// Color neg = SubstanceColorUtilities.getNegativeColor(c
				// .getBackground());

				// need to handle toolbars "embedded" in other toolbars
				int dy = 0;
				int totalOffsetY = 0;
				Component comp = c;
				while (comp.getParent() != null) {
					Component parent = comp.getParent();
					dy += comp.getY();
					if (parent instanceof JToolBar) {
						totalOffsetY += dy;
						dy = 0;
					}
					comp = parent;
				}

				graphics.translate(0, -totalOffsetY);
				graphics
						.setPaint(new GradientPaint(0, 0,
								SubstanceColorUtilities.getAlphaColor(
										headerColor, 160), 0, 4,
								SubstanceColorUtilities.getAlphaColor(
										headerColor, 16)));
				graphics.fillRect(0, 0, c.getWidth(), 4);
				graphics.translate(0, totalOffsetY);
			}
			// if (c.getClass().getName().contains("DesktopPanel"))
			// System.out.println("Painting DesktopPanel with color "
			// + c.getBackground()
			// + "["
			// + c.isOpaque()
			// + "]"
			// + " on "
			// + ((AlphaComposite) graphics.getComposite())
			// .getAlpha());
			// if (c instanceof JDesktopPane)
			// System.out.println("Painting JDesktopPane with color "
			// + c.getBackground()
			// + "["
			// + c.isOpaque()
			// + "]"
			// + " on "
			// + ((AlphaComposite) graphics.getComposite())
			// .getAlpha());
			// if (c.getClass().getName().contains("TextFieldsPanel"))
			// System.out.println("Painting TextFieldsPanel with color "
			// + c.getBackground()
			// + "["
			// + c.isOpaque()
			// + "]"
			// + " on "
			// + ((AlphaComposite) graphics.getComposite())
			// .getAlpha());

			graphics.setComposite(TransitionLayout.getAlphaComposite(c,
					this.alphaComposite, g));
			// graphics.setComposite(AlphaComposite.getInstance(
			// AlphaComposite.SRC_OVER, this.alphaComposite));
			// stamp watermark
			if (!isPreviewMode && !isInCellRenderer && c.isShowing()
					&& SubstanceCoreUtilities.toDrawWatermark(c)) {
				SubstanceLookAndFeel.getCurrentWatermark().drawWatermarkImage(
						graphics, c, 0, 0, c.getWidth(), c.getHeight());

				// paint the background second time with 50% translucency,
				// making the watermark 'bleed' through.
				if (shouldUseHeaderPainter) {
					// Window window = SwingUtilities.windowForComponent(c);
					// boolean isSelected = (window == null) ? true : window
					// .isActive();
					// SubstanceTheme theme = isSelected ? SubstanceLookAndFeel
					// .getTheme().getActiveTitlePaneTheme()
					// : SubstanceLookAndFeel.getTheme()
					// .getDefaultTitlePaneTheme();

					SubstanceHeaderPainter headerPainter = (SubstanceHeaderPainter) titlePainter;
					graphics.setComposite(TransitionLayout.getAlphaComposite(c,
							0.5f * this.alphaComposite, g));
					headerPainter.paintExtraBackground(graphics, headerParent,
							c, c.getWidth(), c.getHeight(), theme, false);
				} else {
					// graphics.setColor(c.getBackground());
					// if (c instanceof JMenuItem) {
					// System.out.println("Filling "
					// + ((JMenuItem) c).getText() + " with "
					// + graphics.getColor().getRed() + ":"
					// + graphics.getColor().getGreen() + ":"
					// + graphics.getColor().getBlue());
					// }
					// graphics.fillRect(0, 0, c.getWidth(), c.getHeight());
				}
				// }
			}
		}
		graphics.dispose();
	}

	/**
	 * Updates the background of the specified component on the specified
	 * graphic context in the specified rectangle.
	 * 
	 * @param g
	 *            Graphic context.
	 * @param c
	 *            Component.
	 * @param fillColor
	 *            Fill color.
	 * @param rect
	 *            The rectangle to fill.
	 */
	public void fillAndWatermark(Graphics g, JComponent c, Color fillColor,
			Rectangle rect) {
		// failsafe for LAF change
		if (!(UIManager.getLookAndFeel() instanceof SubstanceLookAndFeel)) {
			return;
		}

		boolean isInCellRenderer = (c.getParent() instanceof CellRendererPane);
		if ((!c.isShowing()) && (!isInCellRenderer)) {
			return;
		}

		Graphics2D graphics = (Graphics2D) g.create();
		synchronized (c) {
			if (TransitionLayout.isOpaque(c)) {
				// fill background
				graphics.setComposite(TransitionLayout.getAlphaComposite(c, g));
				graphics.setColor(fillColor);
				graphics.fillRect(rect.x, rect.y, rect.width, rect.height);
				graphics.setComposite(TransitionLayout.getAlphaComposite(c,
						alphaComposite, g));
				// stamp watermark
				if (!isInCellRenderer && c.isShowing()
						&& SubstanceCoreUtilities.toDrawWatermark(c))
					SubstanceLookAndFeel.getCurrentWatermark()
							.drawWatermarkImage(graphics, c, rect.x, rect.y,
									rect.width, rect.height);
			}
		}
		graphics.dispose();
	}

	/**
	 * Sets the alpha (translucency) attribute for this delegate.
	 * 
	 * @param alphaComposite
	 *            Alpha (translucency) attribute for this delegate.
	 */
	public void setAlphaComposite(float alphaComposite) {
		this.alphaComposite = alphaComposite;
	}
}
