/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.painter;

import java.awt.Shape;
import java.awt.image.BufferedImage;

import org.jvnet.substance.border.SubstanceBorderPainter;
import org.jvnet.substance.color.ColorScheme;
import org.jvnet.substance.color.MixColorScheme;
import org.jvnet.substance.utils.SubstanceCoreUtilities;

/**
 * Base class for gradient painters. Handles mixed themes. This class is part of
 * officially supported API.
 * 
 * @author Kirill Grouchnikov
 */
public abstract class BaseGradientPainter implements SubstanceGradientPainter {
	/**
	 * The matching border painter.
	 */
	protected SubstanceBorderPainter borderPainter;

	/**
	 * Returns the matching border painter.
	 * 
	 * @return The matching border painter.
	 */
	public SubstanceBorderPainter getBorderPainter() {
		return borderPainter;
	}

	/**
	 * Returns the background (including border) that matches the specified
	 * parameters.
	 * 
	 * @param width
	 *            Width of a UI component.
	 * @param height
	 *            Height of a UI component.
	 * @param contour
	 *            Contour of a UI component.
	 * @param isFocused
	 *            Indication whether component owns the focus.
	 * @param colorScheme1
	 *            The first color scheme.
	 * @param colorScheme2
	 *            The second color scheme.
	 * @param cyclePos
	 *            Cycle position. Is used for rollover and pulsation effects.
	 *            Must be in 0..10 range.
	 * @param hasShine
	 *            Indication whether the returned image should have a 3D shine
	 *            spot in its top half.
	 * @param useCyclePosAsInterpolation
	 *            Indicates the algorithm to use for computing various colors.
	 *            If <code>true</code>, the <code>cyclePos</code> is used
	 *            to interpolate colors between different color components of
	 *            both color schemes. If <code>false</code>, the
	 *            <code>cyclePos</code> is used to interpolate colors between
	 *            different color components of the first color scheme.
	 * @return The background (including border) that matches the specified
	 *         parameters if one of the color schemes is {@link MixColorScheme},
	 *         <code>null</code> otherwise.
	 */
	protected BufferedImage getMixContourBackground(int width, int height,
			Shape contour, boolean isFocused, ColorScheme colorScheme1,
			ColorScheme colorScheme2, float cyclePos, boolean hasShine,
			boolean useCyclePosAsInterpolation) {
		if (colorScheme1 instanceof MixColorScheme) {
			MixColorScheme mixColorScheme = (MixColorScheme) colorScheme1;
			ColorScheme[] origSchemes = mixColorScheme.getOrigSchemes();

			BufferedImage[] components = new BufferedImage[origSchemes.length];
			for (int i = 0; i < origSchemes.length; i++) {
				components[i] = this.getContourBackground(width, height,
						contour, isFocused, origSchemes[i], colorScheme2,
						cyclePos, hasShine, useCyclePosAsInterpolation);
			}

			// Let the blending begin
			BufferedImage current = components[0];
			for (int i = 1; i < components.length; i++) {
				double start = (double) (i - 0.3) / (double) components.length;
				double end = (double) (i + 0.3) / (double) components.length;
				current = SubstanceCoreUtilities.blendImagesHorizontal(current,
						components[i], start, end);
			}
			return current;
		}

		if (colorScheme2 instanceof MixColorScheme) {
			MixColorScheme mixColorScheme = (MixColorScheme) colorScheme2;
			ColorScheme[] origSchemes = mixColorScheme.getOrigSchemes();

			BufferedImage[] components = new BufferedImage[origSchemes.length];
			for (int i = 0; i < origSchemes.length; i++) {
				components[i] = this.getContourBackground(width, height,
						contour, isFocused, colorScheme1, origSchemes[i],
						cyclePos, hasShine, useCyclePosAsInterpolation);
			}

			// Let the blending begin
			BufferedImage current = components[0];
			for (int i = 1; i < components.length; i++) {
				double start = (double) (i - 0.3) / (double) components.length;
				double end = (double) (i + 0.3) / (double) components.length;
				current = SubstanceCoreUtilities.blendImagesHorizontal(current,
						components[i], start, end);
			}
			return current;
		}
		return null;
	}
}
