/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.theme;

import org.jvnet.substance.color.SaturatedColorScheme;
import org.jvnet.substance.color.ShiftColorScheme;

/**
 * Saturated theme. This class is part of officially supported API.
 * 
 * @author Kirill Grouchnikov
 * @see ShiftColorScheme
 */
public class SubstanceSaturatedTheme extends SubstanceWrapperTheme {
	/**
	 * The saturation factor of <code>this</code> theme.
	 */
	private double saturationFactor;

	/**
	 * If <code>true</code>, all visual components of the original theme will
	 * be saturated. If <code>false</code>, only the
	 * {@link #getActiveTheme()} will be saturated.
	 */
	private boolean toSaturateEverything;

	/**
	 * Creates a new saturated theme. <b>Do not</b> use this constructor
	 * directly, use {@link SubstanceTheme#saturate(double)} instead.
	 * 
	 * @param substanceTheme
	 *            The original theme.
	 * @param saturationFactor
	 *            Saturation factor.
	 * @param toSaturateEverything
	 *            If <code>true</code>, all visual components of
	 *            <code>this</code> theme will be saturated. If
	 *            <code>false</code>, only the {@link #getActiveTheme()} will
	 *            be saturated.
	 */
	SubstanceSaturatedTheme(SubstanceTheme substanceTheme,
			double saturationFactor, boolean toSaturateEverything) {
		super(substanceTheme, new SaturatedColorScheme(substanceTheme
				.getColorScheme(), saturationFactor), "Saturated "
				+ substanceTheme.getDisplayName() + " "
				+ (int) (100 * saturationFactor) + "% [" + toSaturateEverything
				+ "]", substanceTheme.getKind());
		this.saturationFactor = saturationFactor;
		this.toSaturateEverything = toSaturateEverything;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getBorderTheme()
	 */
	@Override
	public SubstanceTheme getBorderTheme() {
		if (this.borderTheme == null)
			this.borderTheme = this.originalTheme.getBorderTheme().saturate(
					this.saturationFactor);
		return this.borderTheme;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getDefaultTheme()
	 */
	@Override
	public SubstanceTheme getDefaultTheme() {
		if (this.toSaturateEverything) {
			if (this.defaultTheme == null)
				this.defaultTheme = this.originalTheme.getDefaultTheme()
						.saturate(this.saturationFactor);
			return this.defaultTheme;
		} else {
			return this.originalTheme.getDefaultTheme();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getDisabledTheme()
	 */
	@Override
	public SubstanceTheme getDisabledTheme() {
		if (this.toSaturateEverything) {
			if (this.disabledTheme == null)
				this.disabledTheme = this.originalTheme.getDisabledTheme()
						.saturate(this.saturationFactor);
			return this.disabledTheme;
		} else {
			return this.originalTheme.getDisabledTheme();
		}
	}
}
