/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.title;

import java.awt.Container;

import javax.swing.JToolBar;
import javax.swing.SwingUtilities;

/**
 * 
 * @author Kirill Grouchnikov
 */
public abstract class BaseHeaderPainter implements SubstanceHeaderPainter {
	/**
	 * Indicates whether this painter is used to paint the toolbar background.
	 */
	protected boolean isPaintingToolbars;

	/**
	 * Indicates whether this painter is used to paint the background of general
	 * decoration areas.
	 */
	protected boolean isPaintingGeneralDecorationAreas;

	/**
	 * Indicates whether this painter will cause drop shadows to appear on the
	 * top portion of the toolbars that are <b>not</b> painted by this painter.
	 */
	private boolean isPaintingToolbarDropShadows;

	/**
	 * Creates a new header painter that is used to paint title panes, menubars
	 * <b>and</b> toolbars.
	 */
	public BaseHeaderPainter() {
		this(true, true);
	}

	/**
	 * Creates a new header painter
	 * 
	 * @param isPaintingToolbars
	 *            If <code>true</code>, this painter will be used to paint
	 *            toolbars in addition to painting title panes and menubars.
	 * @param isPaintingGeneralDecorationAreas
	 *            If <code>true</code>, this painter will be used to paint
	 *            additional general decoration areas.
	 */
	public BaseHeaderPainter(boolean isPaintingToolbars,
			boolean isPaintingGeneralDecorationAreas) {
		super();
		this.isPaintingToolbars = isPaintingToolbars;
		this.isPaintingGeneralDecorationAreas = isPaintingGeneralDecorationAreas;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.title.SubstanceHeaderPainter#isPaintingContainer(java.awt.Container)
	 */
	public boolean isPaintingContainer(Container cont) {
		if (cont instanceof JToolBar)
			return isPaintingToolbars;
		if (SwingUtilities.getAncestorOfClass(JToolBar.class, cont) != null)
			return isPaintingToolbars;
		return isPaintingGeneralDecorationAreas;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.title.SubstanceHeaderPainter#isPaintingToolbarDropShadows()
	 */
	public boolean isPaintingToolbarDropShadows() {
		return isPaintingToolbarDropShadows;
	}

	/**
	 * Sets indication whether this painter will cause drop shadows to appear on
	 * the top portion of the toolbars that are <b>not</b> painted by this
	 * painter.
	 * 
	 * @param isPaintingToolbarDropShadows
	 *            Indication whether this painter will cause drop shadows to
	 *            appear on the top portion of the toolbars that are <b>not</b>
	 *            painted by this painter.
	 */
	public void setPaintingToolbarDropShadows(
			boolean isPaintingToolbarDropShadows) {
		this.isPaintingToolbarDropShadows = isPaintingToolbarDropShadows;
	}

	/**
	 * Sets indication whether this painter is used to paint the toolbar
	 * background.
	 * 
	 * @param isPaintingToolbars
	 *            Indication whether this painter is used to paint the toolbar
	 *            background.
	 */
	public void setPaintingToolbars(boolean isPaintingToolbars) {
		this.isPaintingToolbars = isPaintingToolbars;
	}

	/**
	 * Sets indication whether this painter is used to paint the general
	 * decoration areas.
	 * 
	 * @param isPaintingGeneralDecorationAreas
	 *            Indication whether this painter is used to paint the general
	 *            decoration areas.
	 */
	public void setPaintingGeneralDecorationAreas(
			boolean isPaintingGeneralDecorationAreas) {
		this.isPaintingGeneralDecorationAreas = isPaintingGeneralDecorationAreas;
	}
}
