/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.title;

import java.awt.*;
import java.awt.image.BufferedImage;
import java.util.LinkedHashMap;
import java.util.Map.Entry;

import javax.swing.*;

import org.jvnet.lafwidget.layout.TransitionLayout;
import org.jvnet.substance.SubstanceImageCreator;
import org.jvnet.substance.theme.SubstanceTheme;
import org.jvnet.substance.utils.SubstanceCoreUtilities;

/**
 * Implementation of {@link SubstanceHeaderPainter} that uses brushed metal
 * painting on title panes.
 * 
 * @author Kirill Grouchnikov
 * @since version 4.0
 */
public abstract class ImageWrapperHeaderPainter extends BaseHeaderPainter {
	protected Image singleTile = null;

	protected SubstanceTitlePainter baseTitlePainter;

	protected ImageWrapperHeaderPainter secondaryTitlePainter;

	protected LinkedHashMap<String, Image> colorizedTileMap;

	protected float textureAlpha;

	protected boolean isPaintingSeparators;

	/**
	 * Creates a new brushed metal header painter that is used to paint title
	 * panes, menubars <b>and</b> toolbars.
	 */
	public ImageWrapperHeaderPainter() {
		this(true, true);
	}

	/**
	 * Creates a new brushed metal header painter
	 * 
	 * @param isPaintingToolbars
	 *            If <code>true</code>, this painter will be used to paint
	 *            toolbars in addition to painting title panes and menubars.
	 * @param isPaintingGeneralDecorationAreas
	 *            If <code>true</code>, this painter will be used to paint
	 *            additional general decoration areas.
	 */
	public ImageWrapperHeaderPainter(boolean isPaintingToolbars,
			boolean isPaintingGeneralDecorationAreas) {
		super(isPaintingToolbars, isPaintingGeneralDecorationAreas);
		this.isPaintingSeparators = false;
		// this.baseTitlePainter = new FlatTitlePainter();
		this.textureAlpha = 0.3f;

		this.colorizedTileMap = new LinkedHashMap<String, Image>() {
			@Override
			protected boolean removeEldestEntry(Entry<String, Image> eldest) {
				return this.size() > 10;
			}
		};
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.title.SubstanceTitlePainter#paintTitleBackground(java.awt.Graphics2D,
	 *      java.awt.Component, int, int, int, int,
	 *      org.jvnet.substance.theme.SubstanceTheme, float)
	 */
	public void paintTitleBackground(Graphics2D graphics, Component comp,
			int width, int height, int leftTitleX, int rightTitleX,
			SubstanceTheme theme, float borderAlpha) {

		SubstanceTheme tileTheme = theme;
		if (this.baseTitlePainter == null) {
			tileTheme = theme.getActiveTitlePaneTheme();
			graphics.setColor(tileTheme.getColorScheme().getMidColor());
			graphics.fillRect(0, 0, width, height);
		} else {
			this.baseTitlePainter.paintTitleBackground(graphics, comp, width,
					height, leftTitleX, rightTitleX, theme, borderAlpha);
		}

		Graphics2D temp = (Graphics2D) graphics.create();
		this.tileArea(temp, comp, tileTheme, 0, 0, 0, 0, width, height);

		if (this.isPaintingSeparators) {
			temp.translate(0, height - 1);
			SubstanceCoreUtilities.paintSeparator(comp, temp, theme
					.getActiveTitlePaneTheme().getColorScheme(),
					SubstanceCoreUtilities.isThemeDark(theme
							.getActiveTitlePaneTheme()), width, 1,
					JSeparator.HORIZONTAL, false, 0);
			temp.dispose();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.title.SubstanceHeaderPainter#paintExtraBackground(java.awt.Graphics2D,
	 *      java.awt.Container, java.awt.Component, int, int,
	 *      org.jvnet.substance.theme.SubstanceTheme, boolean)
	 */
	public void paintExtraBackground(Graphics2D graphics, Container parent,
			Component comp, int width, int height, SubstanceTheme theme,
			boolean hasDarkBorder) {

		JRootPane rootPane = SwingUtilities.getRootPane(parent);
		// fix for bug 234 - Window doesn't have a root pane.
		int dx = 0;
		int dy = 0;
		JComponent titlePane = null;
		if (rootPane != null) {
			titlePane = SubstanceCoreUtilities.getTitlePane(rootPane);

			if (titlePane != null) {
				if (comp.isShowing() && titlePane.isShowing()) {
					dx += (comp.getLocationOnScreen().x - titlePane
							.getLocationOnScreen().x);
					dy += (comp.getLocationOnScreen().y - titlePane
							.getLocationOnScreen().y);
				} else {
					// have to traverse the hierarchy
					Component c = comp;
					dx = 0;
					dy = 0;
					while (c != rootPane) {
						dx += c.getX();
						dy += c.getY();
						c = c.getParent();
					}
					c = titlePane;
					while (c != rootPane) {
						dx -= c.getX();
						dy -= c.getY();
						c = c.getParent();
					}
				}
			}
		}

		SubstanceTheme tileTheme = theme;
		if (this.baseTitlePainter instanceof ArcHeaderPainter) {
			ArcHeaderPainter baseArc = (ArcHeaderPainter) this.baseTitlePainter;
			baseArc.paintExtraBackground(graphics, parent, comp, width, height,
					theme, hasDarkBorder);
		} else {
			tileTheme = theme.getActiveTitlePaneTheme();
			graphics.setColor(tileTheme.getColorScheme().getMidColor());
			graphics.fillRect(0, 0, width, height);
		}
		Graphics2D temp = (Graphics2D) graphics.create();
		this.tileArea(temp, comp, tileTheme, dx, dy, 0, 0, width, height);

		// paint separators only on top-level toolbars (not on toolbars
		// embedded in other toolbars).
		if (this.isPaintingSeparators
				&& (SwingUtilities.getAncestorOfClass(JToolBar.class, comp) == null)) {
			int pWidth = (titlePane == null) ? parent.getWidth() : titlePane
					.getWidth();
			temp.translate(-dx, height - 1);
			SubstanceCoreUtilities.paintSeparator(comp, temp, theme
					.getActiveTitlePaneTheme().getColorScheme(),
					SubstanceCoreUtilities.isThemeDark(theme
							.getActiveTitlePaneTheme()), pWidth, 1,
					JSeparator.HORIZONTAL, false, 0);
		}
		temp.dispose();
	}

	protected void tileArea(Graphics2D g, Component comp,
			SubstanceTheme tileTheme, int offsetTextureX, int offsetTextureY,
			int x, int y, int width, int height) {

		Graphics2D graphics = (Graphics2D) g.create();
		graphics.setComposite(TransitionLayout.getAlphaComposite(comp,
				this.textureAlpha, g));

		Image colorizedTile = this.getColorizedTile(tileTheme);
		int tileWidth = colorizedTile.getWidth(null);
		int tileHeight = colorizedTile.getHeight(null);

		offsetTextureX = offsetTextureX % tileWidth;
		offsetTextureY = offsetTextureY % tileHeight;
		int currTileTop = -offsetTextureY;
		do {
			int currTileLeft = -offsetTextureX;
			do {
				graphics.drawImage(colorizedTile, currTileLeft, currTileTop,
						null);
				currTileLeft += tileWidth;
			} while (currTileLeft < width);
			currTileTop += tileHeight;
		} while (currTileTop < height);

		graphics.dispose();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.title.SubstanceHeaderPainter#getSecondaryPainter()
	 */
	public SubstanceTitlePainter getSecondaryPainter() {
		if (this.baseTitlePainter instanceof ArcHeaderPainter) {
			return ((ArcHeaderPainter) this.baseTitlePainter)
					.getSecondaryPainter();
		}
		return this;
	}

	public void setBaseTitlePainter(SubstanceTitlePainter baseTitlePainter) {
		this.baseTitlePainter = baseTitlePainter;
	}

	public void setTextureAlpha(float textureAlpha) {
		this.textureAlpha = textureAlpha;
	}

	/**
	 * Sets indication whether this painter paints separators.
	 * 
	 * @param isPaintingSeparators
	 *            Indication whether this painter paints separators.
	 */
	public void setPaintingSeparators(boolean isPaintingSeparators) {
		this.isPaintingSeparators = isPaintingSeparators;
	}

	protected synchronized Image getColorizedTile(SubstanceTheme theme) {
		Image result = this.colorizedTileMap.get(theme.getDisplayName());
		if (result == null) {
			BufferedImage tileBi = new BufferedImage(this.singleTile
					.getWidth(null), this.singleTile.getHeight(null),
					BufferedImage.TYPE_INT_ARGB);
			tileBi.getGraphics().drawImage(this.singleTile, 0, 0, null);
			result = SubstanceImageCreator.getThemeImage(tileBi, theme, false);
			this.colorizedTileMap.put(theme.getDisplayName(), result);
		}
		return result;
	}
}
