/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.title;

import java.awt.*;
import java.awt.image.BufferedImage;
import java.util.*;

import org.jvnet.substance.SubstanceImageCreator;
import org.jvnet.substance.theme.SubstanceTheme;
import org.jvnet.substance.utils.SubstanceColorUtilities;
import org.jvnet.substance.utils.SubstanceCoreUtilities;

/**
 * Title painter that paints transitions with random cubes. This class is part
 * of officially supported API.
 * 
 * @author Kirill Grouchnikov
 */
public class RandomCubesTitlePainter implements SubstanceTitlePainter {
	/**
	 * The display name for the title painters of this class.
	 */
	public static final String DISPLAY_NAME = "Random Cubes";

	/**
	 * Cube dimension.
	 */
	public static final int CUBE_DIMENSION = 5;

	/**
	 * Cube color matrix.
	 * 
	 * @author Kirill Grouchnikov
	 */
	private static class CubeColors {
		/**
		 * Left transition matrix.
		 */
		public double[][] leftTransitionMatrix;

		/**
		 * Right transition matrix.
		 */
		public double[][] rightTransitionMatrix;
	}

	/**
	 * Cache for cube color maps.
	 */
	private static WeakHashMap<Component, CubeColors> cubeColorMap = new WeakHashMap<Component, CubeColors>();

	/**
	 * Returns a new random cube color matrix.
	 * 
	 * @param width
	 *            Width.
	 * @param height
	 *            Height.
	 * @return New random cube color matrix.
	 */
	private static double[][] getCubeColorMatrix(int width, int height) {
		int cubeCountY = (height / CUBE_DIMENSION);
		int cubeCountX = 1 + width / CUBE_DIMENSION;

		double[][] result = new double[cubeCountX][cubeCountY];
		for (int col = 0; col < cubeCountX; col++) {
			for (int row = 0; row < cubeCountY; row++) {
				// decide if we should put a cube
				if (Math.random() < 0.45) {
					result[col][row] = -1.0;
					continue;
				}
				// Make semi-random choice of color. It should lie
				// close to the interpolated color, but still appear
				// random
				double coef = 1.0 - (((double) col / (double) cubeCountX) + 0.9 * (Math
						.random() - 0.5));
				coef = Math.max(0.0, coef);
				coef = Math.min(1.0, coef);
				// fill cube
				result[col][row] = coef;
			}
		}
		return result;
	}

	/**
	 * Returns cube color image.
	 * 
	 * @param width
	 *            Image width.
	 * @param height
	 *            Image height.
	 * @param isDark
	 *            Indication whether the color theme is dark.
	 * @param leftTransColor
	 *            Left transition color.
	 * @param rightTransColor
	 *            Right transition color.
	 * @param colorMatrix
	 *            Color matrix to apply.
	 * @return Cube color image.
	 */
	private static BufferedImage getCubes(int width, int height,
			boolean isDark, Color leftTransColor, Color rightTransColor,
			double[][] colorMatrix) {
		BufferedImage image = SubstanceCoreUtilities.getBlankImage(width,
				height);
		Graphics2D graphics = (Graphics2D) image.getGraphics();

		int cubeCountY = Math.min(height / CUBE_DIMENSION,
				colorMatrix[0].length);
		int cubeCountX = Math.min(1 + width / CUBE_DIMENSION,
				colorMatrix.length);
		int cubeStartY = (height % CUBE_DIMENSION) / 2;
		int cubeStartX = 0;// -(SubstanceImageCreator.CUBE_DIMENSION - (width %
		// SubstanceImageCreator.CUBE_DIMENSION));

		for (int col = 0; col < cubeCountX; col++) {
			for (int row = 0; row < cubeCountY; row++) {
				// decide if we should put a cube
				double coef = colorMatrix[col][row];
				if (coef < 0.0) {
					continue;
				}

				// Compute RGB components
				int r = (int) (coef * leftTransColor.getRed() + (1.0 - coef)
						* rightTransColor.getRed());
				int g = (int) (coef * leftTransColor.getGreen() + (1.0 - coef)
						* rightTransColor.getGreen());
				int b = (int) (coef * leftTransColor.getBlue() + (1.0 - coef)
						* rightTransColor.getBlue());
				// fill cube
				// graphics.setColor(new Color((int) (0.9 * r), (int) (0.9 * g),
				// (int) (0.9 * b)));
				Color cubeColor = new Color(r, g, b);
				graphics.setColor(cubeColor);
				graphics.fillRect(cubeStartX + col * CUBE_DIMENSION, cubeStartY
						+ row * CUBE_DIMENSION, CUBE_DIMENSION, CUBE_DIMENSION);
				// draw cube's border in slightly different color
				if (isDark)
					graphics.setColor(SubstanceColorUtilities.getDarkerColor(
							cubeColor, 0.2));
				else
					graphics.setColor(SubstanceColorUtilities.getLighterColor(
							cubeColor, 0.2));
				graphics.drawRect(cubeStartX + col * CUBE_DIMENSION, cubeStartY
						+ row * CUBE_DIMENSION, CUBE_DIMENSION, CUBE_DIMENSION);
			}
		}
		return image;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.painter.SubstanceGradientPainter#getDisplayName()
	 */
	public String getDisplayName() {
		return DISPLAY_NAME;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.title.SubstanceTitlePainter#paintTitleBackground(java.awt.Graphics2D,
	 *      java.awt.Component, int, int, int, int,
	 *      org.jvnet.substance.theme.SubstanceTheme, float)
	 */
	public void paintTitleBackground(Graphics2D graphics, Component comp,
			int width, int height, int leftTitleX, int rightTitleX,
			SubstanceTheme theme, float borderAlpha) {
		int leftTransitionStart = leftTitleX - 10;
		int leftTransitionEnd = leftTitleX + 10;
		int rightTransitionStart = rightTitleX - 10;
		int rightTransitionEnd = rightTitleX + 10;

		Map<Integer, Color> waypoints = new HashMap<Integer, Color>();
		Color colorEnd = theme.getColorScheme().getMidColor();
		Color colorTrans = theme.getColorScheme().getLightColor();
		Color colorMid = theme.getColorScheme().getExtraLightColor();
		waypoints.put(leftTransitionStart, colorTrans);
		waypoints.put(leftTransitionEnd, colorMid);
		waypoints.put(rightTransitionStart, colorMid);
		waypoints.put(rightTransitionEnd, colorTrans);
		BufferedImage lineEven = SubstanceImageCreator.getOneLineGradient(
				width + 1, colorEnd, colorEnd, waypoints);

		for (int row = 0; row < height; row++) {
			graphics.drawImage(lineEven, 0, row, null);
		}

		CubeColors cubeColors = cubeColorMap.get(comp);
		if (cubeColors == null) {
			double[][] cubesLeft = null;
			if (leftTransitionStart < leftTransitionEnd) {
				cubesLeft = getCubeColorMatrix(leftTransitionEnd
						- leftTransitionStart, height);
			}
			double[][] cubesRight = null;
			if (rightTransitionStart < rightTransitionEnd) {
				cubesRight = getCubeColorMatrix(rightTransitionEnd
						- rightTransitionStart, height);
			}
			cubeColors = new CubeColors();
			cubeColors.leftTransitionMatrix = cubesLeft;
			cubeColors.rightTransitionMatrix = cubesRight;
			cubeColorMap.put(comp, cubeColors);
		}

		if (cubeColors.leftTransitionMatrix != null) {
			BufferedImage cubesLeft = getCubes(leftTransitionEnd
					- leftTransitionStart, height, SubstanceCoreUtilities
					.isThemeDark(theme), colorEnd, colorMid,
					cubeColors.leftTransitionMatrix);
			graphics.drawImage(cubesLeft, leftTransitionStart, 0, null);
		}
		if (cubeColors.rightTransitionMatrix != null) {
			BufferedImage cubesRight = getCubes(rightTransitionEnd
					- rightTransitionStart, height, SubstanceCoreUtilities
					.isThemeDark(theme), colorMid, colorEnd,
					cubeColors.rightTransitionMatrix);
			graphics.drawImage(cubesRight, rightTransitionStart, 0, null);
		}
	}
}
