/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance.utils;

import java.awt.*;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.image.*;
import java.net.URL;
import java.util.*;

import javax.swing.*;
import javax.swing.border.Border;
import javax.swing.border.CompoundBorder;
import javax.swing.plaf.*;
import javax.swing.plaf.basic.BasicGraphicsUtils;

import org.jvnet.lafwidget.*;
import org.jvnet.lafwidget.animation.*;
import org.jvnet.lafwidget.layout.TransitionLayout;
import org.jvnet.lafwidget.utils.RenderingUtils;
import org.jvnet.substance.*;
import org.jvnet.substance.border.SubstanceBorderPainter;
import org.jvnet.substance.button.SubstanceButtonShaper;
import org.jvnet.substance.color.*;
import org.jvnet.substance.combo.ComboPopupPrototypeCallback;
import org.jvnet.substance.combo.SubstanceComboBoxButton;
import org.jvnet.substance.grip.GripPainter;
import org.jvnet.substance.painter.*;
import org.jvnet.substance.scroll.SubstanceScrollButton;
import org.jvnet.substance.skin.AutumnSkin;
import org.jvnet.substance.skin.MagmaSkin;
import org.jvnet.substance.tabbed.TabCloseCallback;
import org.jvnet.substance.theme.*;
import org.jvnet.substance.theme.SubstanceTheme.ThemeKind;
import org.jvnet.substance.title.*;
import org.jvnet.substance.utils.ComponentState.ColorSchemeKind;
import org.jvnet.substance.utils.SubstanceConstants.*;
import org.jvnet.substance.utils.icon.SubstanceIconFactory;
import org.jvnet.substance.utils.icon.TransitionAwareIcon;

/**
 * Various utility functions. This class is <b>for internal use only</b>.
 * 
 * @author Kirill Grouchnikov
 * @author Romain Guy
 */
public class SubstanceCoreUtilities {
	/**
	 * Client property name for the previous component state. The client
	 * property value should be an instance of {@link ComponentState}.
	 */
	private static final String PREV_COMPONENT_STATE = "substancelaf.internal.prevComponentState";

	/**
	 * Client property name for the previous component state. The client
	 * property value should be an instance of {@link ComponentState}.
	 */
	private static final String PREV_SEL_COMPONENT_STATE = "substancelaf.internal.prevSelComponentState";

	/**
	 * Client property name for the next component state. The client property
	 * value should be an instance of {@link ComponentState}.
	 */
	private static final String NEXT_COMPONENT_STATE = "substancelaf.internal.nextComponentState";

	/**
	 * Client property name for the next component state. The client property
	 * value should be an instance of {@link ComponentState}.
	 */
	private static final String NEXT_SEL_COMPONENT_STATE = "substancelaf.internal.nextSelComponentState";

	/**
	 * Private constructor. Is here to enforce using static methods only.
	 */
	private SubstanceCoreUtilities() {
	}

	/**
	 * Clips string based on specified font metrics and available width (in
	 * pixels). Returns the clipped string, which contains the beginning and the
	 * end of the input string separated by ellipses (...) in case the string is
	 * too long to fit into the specified width, and the origianl string
	 * otherwise.
	 * 
	 * @param metrics
	 *            Font metrics.
	 * @param availableWidth
	 *            Available width in pixels.
	 * @param fullText
	 *            String to clip.
	 * @return The clipped string, which contains the beginning and the end of
	 *         the input string separated by ellipses (...) in case the string
	 *         is too long to fit into the specified width, and the origianl
	 *         string otherwise.
	 */
	public static String clipString(FontMetrics metrics, int availableWidth,
			String fullText) {

		if (metrics.stringWidth(fullText) <= availableWidth)
			return fullText;

		String ellipses = "...";
		int ellipsesWidth = metrics.stringWidth(ellipses);
		if (ellipsesWidth > availableWidth)
			return "";

		String starter = "";
		String ender = "";

		int w = fullText.length();
		int w2 = (w / 2) + (w % 2);
		String prevTitle = "";
		for (int i = 0; i < w2; i++) {
			String newStarter = starter + fullText.charAt(i);
			String newEnder = ender;
			if ((w - i) > w2)
				newEnder = fullText.charAt(w - i - 1) + newEnder;
			String newTitle = newStarter + ellipses + newEnder;
			if (metrics.stringWidth(newTitle) <= availableWidth) {
				starter = newStarter;
				ender = newEnder;
				prevTitle = newTitle;
				continue;
			}
			return prevTitle;
		}
		return fullText;
	}

	/**
	 * Checks whether the specified button has associated icon.
	 * 
	 * @param button
	 *            Button.
	 * @return If the button has associated icon, <code>true</code> is
	 *         returned, otherwise <code>false</code>.
	 */
	public static boolean hasIcon(AbstractButton button) {
		return (button.getIcon() != null);
	}

	/**
	 * Checks whether the specified button has associated text.
	 * 
	 * @param button
	 *            Button.
	 * @return If the button has associated text, <code>true</code> is
	 *         returned, otherwise <code>false</code>.
	 */
	public static boolean hasText(AbstractButton button) {
		String text = button.getText();
		if ((text != null) && (text.length() > 0)) {
			return true;
		}
		return false;
	}

	/**
	 * Checks and answers if the specified button is in a combo box.
	 * 
	 * @param button
	 *            the button to check
	 * @return <code>true</code> if in tool bar, <code>false</code>
	 *         otherwise
	 */
	public static boolean isComboBoxButton(AbstractButton button) {
		Container parent = button.getParent();
		return (parent != null)
				&& ((parent instanceof JComboBox) || (parent.getParent() instanceof JComboBox));
	}

	/**
	 * Checks and answers if the specified button is in a combo box.
	 * 
	 * @param button
	 *            the button to check
	 * @return <code>true</code> if in tool bar, <code>false</code>
	 *         otherwise
	 */
	public static boolean isScrollBarButton(AbstractButton button) {
		Container parent = button.getParent();
		return (parent != null)
				&& ((parent instanceof JScrollBar) || (parent.getParent() instanceof JScrollBar));
	}

	/**
	 * Checks and answers if the specified button is in a spinner.
	 * 
	 * @param button
	 *            the button to check
	 * @return <code>true</code> if in spinner, <code>false</code> otherwise
	 */
	public static boolean isSpinnerButton(AbstractButton button) {
		Container parent = button.getParent();
		if (!(button instanceof SubstanceSpinnerButton))
			return false;
		return (parent != null)
				&& ((parent instanceof JSpinner) || (parent.getParent() instanceof JSpinner));
	}

	/**
	 * Checks and answers if the specified button is in a toolbar.
	 * 
	 * @param button
	 *            the button to check
	 * @return <code>true</code> if in toolbar, <code>false</code> otherwise
	 */
	public static boolean isToolBarButton(AbstractButton button) {
		if (button instanceof SubstanceComboBoxButton)
			return false;
		if (button instanceof SubstanceSpinnerButton)
			return false;
		Container parent = button.getParent();
		return (parent != null)
				&& ((parent instanceof JToolBar) || (parent.getParent() instanceof JToolBar));
	}

	/**
	 * Checks answers if the specified component is a button in a scroll
	 * control, such as scroll bar or tabbed pane (as tab scroller).
	 * 
	 * @param comp
	 *            The component to check
	 * @return <code>true</code> if the specified component is a button in a
	 *         scroll control, <code>false</code> otherwise
	 */
	public static boolean isScrollButton(JComponent comp) {
		return (comp instanceof SubstanceScrollButton);
	}

	/**
	 * Returns the theme of the specified component.
	 * 
	 * @param component
	 *            The component.
	 * @param toReturnCurrent
	 *            if <code>true</code>, the currently set <b>Substance</b>
	 *            theme will be returned if the
	 *            {@link SubstanceLookAndFeel#THEME_PROPERTY} is not set on the
	 *            component and any of its ancestors.
	 * @return The theme of the specified component.
	 */
	public static SubstanceTheme getTheme(Component component,
			boolean toReturnCurrent) {
		if (component != null) {
			Component comp = component;
			// while (comp != null) {
			if (comp instanceof JComponent) {
				JComponent jcomp = (JComponent) comp;
				Object controlThemeObj = jcomp
						.getClientProperty(SubstanceLookAndFeel.THEME_PROPERTY);
				if (controlThemeObj != null) {
					if (controlThemeObj instanceof String)
						return SubstanceTheme
								.getTheme((String) controlThemeObj);
					if (controlThemeObj instanceof ThemeInfo)
						return SubstanceTheme
								.createInstance((ThemeInfo) controlThemeObj);
					if (controlThemeObj instanceof SubstanceTheme)
						return ((SubstanceTheme) controlThemeObj);
				}
			}
			// comp = comp.getParent();
			// }
		}
		if (toReturnCurrent) {
			// if (component == null)
			// return SubstanceLookAndFeel.getTheme();
			// if ((component instanceof JMenuItem)
			// || (component instanceof JPopupMenu)
			// || (component.getParent() instanceof JPopupMenu)) {
			// return SubstanceLookAndFeel.getTheme()
			// .getActiveTitlePaneTheme();
			// } else {
			// if (component instanceof SubstanceTitleButton)
			// return SubstanceLookAndFeel.getTheme()
			// .getActiveTitlePaneTheme();
			return SubstanceLookAndFeel.getTheme();
			// }
		}
		return null;
	}

	/**
	 * Returns the theme of the specified component in the specified parent.
	 * 
	 * @param component
	 *            The component.
	 * @param parent
	 *            The component parent of the component.
	 * @param toReturnCurrent
	 *            if <code>true</code>, the currently set <b>Substance</b>
	 *            theme will be returned if the
	 *            {@link SubstanceLookAndFeel#THEME_PROPERTY} is not set on the
	 *            component and any of its ancestors.
	 * @return The theme of the specified component.
	 */
	public static SubstanceTheme getTheme(Component component,
			Component parent, boolean toReturnCurrent) {
		SubstanceTheme compTheme = getTheme(component, toReturnCurrent);
		if (compTheme != null)
			return compTheme;

		return getTheme(parent, toReturnCurrent);
	}

	/**
	 * Returns the active theme of the specified component.
	 * 
	 * @param component
	 *            The component.
	 * @param toReturnCurrent
	 *            if <code>true</code>, the currently set <b>Substance</b>
	 *            theme will be returned if the
	 *            {@link SubstanceLookAndFeel#THEME_PROPERTY} is not set on the
	 *            component and any of its ancestors.
	 * @return The active theme of the specified component.
	 */
	public static SubstanceTheme getActiveTheme(Component component,
			boolean toReturnCurrent) {
		SubstanceTheme componentTheme = getTheme(component, toReturnCurrent);
		if (componentTheme == null)
			return null;
		return componentTheme.getActiveTheme();
	}

	// /**
	// * Returns the highlight background theme of the specified component.
	// *
	// * @param component
	// * The component.
	// * @param toReturnCurrent
	// * if <code>true</code>, the currently set <b>Substance</b>
	// * theme will be returned if the
	// * {@link SubstanceLookAndFeel#THEME_PROPERTY} is not set on the
	// * component and any of its ancestors.
	// * @return The highlight background theme of the specified component.
	// */
	// public static SubstanceTheme getHighlightBackgroundTheme(
	// Component component, boolean toReturnCurrent) {
	// SubstanceTheme componentTheme = getTheme(component, toReturnCurrent);
	// if (componentTheme == null)
	// return null;
	// return componentTheme.getHighlightBackgroundTheme();
	// }
	//
	/**
	 * Returns the default theme of the specified component.
	 * 
	 * @param component
	 *            The component.
	 * @param toReturnCurrent
	 *            if <code>true</code>, the currently set <b>Substance</b>
	 *            theme will be returned if the
	 *            {@link SubstanceLookAndFeel#THEME_PROPERTY} is not set on the
	 *            component and any of its ancestors.
	 * @return The default theme of the specified component.
	 */
	public static SubstanceTheme getDefaultTheme(Component component,
			boolean toReturnCurrent) {
		return getDefaultTheme(component, toReturnCurrent, false);
	}

	/**
	 * Returns the default theme of the specified component.
	 * 
	 * @param component
	 *            The component.
	 * @param toReturnCurrent
	 *            if <code>true</code>, the currently set <b>Substance</b>
	 *            theme will be returned if the
	 *            {@link SubstanceLookAndFeel#THEME_PROPERTY} is not set on the
	 *            component and any of its ancestors.
	 * @param checkHierarchy
	 *            If <code>true</code>, the entire component hierarchy will
	 *            be scanned for the
	 *            {@link SubstanceLookAndFeel#PAINT_ACTIVE_PROPERTY}.
	 * @return The default theme of the specified component.
	 */
	public static SubstanceTheme getDefaultTheme(Component component,
			boolean toReturnCurrent, boolean checkHierarchy) {
		if ((component != null)
				&& SubstanceCoreUtilities.isControlAlwaysPaintedActive(
						component, checkHierarchy)) {
			return getActiveTheme(component, toReturnCurrent);
		} else {
			SubstanceTheme theme = getTheme(component, toReturnCurrent);
			if (theme != null)
				return theme.getDefaultTheme();
			return null;
		}
	}

	/**
	 * Returns the theme of the component.
	 * 
	 * @param component
	 *            Component.
	 * @param parent
	 *            Component parent.
	 * @param componentState
	 *            Component state.
	 * @param toReturnCurrent
	 *            if <code>true</code>, the currently set <b>Substance</b>
	 *            theme will be returned if the
	 *            {@link SubstanceLookAndFeel#THEME_PROPERTY} is not set on the
	 *            component and any of its ancestors.
	 * @return Component theme.
	 */
	public static SubstanceTheme getTheme(Component component,
			Component parent, ComponentState componentState,
			boolean toReturnCurrent) {
		SubstanceTheme compTheme = getTheme(component, componentState, false,
				false);
		if (compTheme != null)
			return compTheme;

		return getTheme(parent, componentState, toReturnCurrent, false);
	}

	/**
	 * Returns the highlight theme of the component.
	 * 
	 * @param component
	 *            Component.
	 * @param parent
	 *            Component parent.
	 * @param componentState
	 *            Component state.
	 * @param toReturnCurrent
	 *            if <code>true</code>, the currently set <b>Substance</b>
	 *            theme will be returned if the
	 *            {@link SubstanceLookAndFeel#THEME_PROPERTY} is not set on the
	 *            component and any of its ancestors.
	 * @return Component highlight theme.
	 */
	public static SubstanceTheme getHighlightTheme(Component component,
			Component parent, ComponentState componentState,
			boolean toReturnCurrent) {
		SubstanceTheme compTheme = getHighlightTheme(component, componentState,
				false, false);
		if (compTheme != null)
			return compTheme;

		return getHighlightTheme(parent, componentState, toReturnCurrent, false);
	}

	/**
	 * Returns the theme of the component.
	 * 
	 * @param component
	 *            Component.
	 * @param componentState
	 *            Component state.
	 * @param toReturnCurrent
	 *            if <code>true</code>, the currently set <b>Substance</b>
	 *            theme will be returned if the
	 *            {@link SubstanceLookAndFeel#THEME_PROPERTY} is not set on the
	 *            component and any of its ancestors.
	 * @param checkHierarchy
	 *            If <code>true</code>, the entire component hierarchy will
	 *            be scanned for the
	 *            {@link SubstanceLookAndFeel#PAINT_ACTIVE_PROPERTY}.
	 * @return Component theme.
	 */
	public static SubstanceTheme getTheme(Component component,
			ComponentState componentState, boolean toReturnCurrent,
			boolean checkHierarchy) {
		if ((component != null)
				&& SubstanceCoreUtilities.isControlAlwaysPaintedActive(
						component, checkHierarchy)
				&& !componentState.isKindActive(FadeKind.PRESS)) {
			return getActiveTheme(component, toReturnCurrent);
		} else {
			SubstanceTheme theme = getTheme(component, toReturnCurrent);
			if (theme != null)
				return theme.getTheme(component, componentState);
			return null;
		}
	}

	/**
	 * Returns the highlight theme of the component.
	 * 
	 * @param component
	 *            Component.
	 * @param componentState
	 *            Component state.
	 * @param toReturnCurrent
	 *            if <code>true</code>, the currently set <b>Substance</b>
	 *            theme will be returned if the
	 *            {@link SubstanceLookAndFeel#THEME_PROPERTY} is not set on the
	 *            component and any of its ancestors.
	 * @param checkHierarchy
	 *            If <code>true</code>, the entire component hierarchy will
	 *            be scanned for the
	 *            {@link SubstanceLookAndFeel#PAINT_ACTIVE_PROPERTY}.
	 * @return Component highlight theme.
	 */
	public static SubstanceTheme getHighlightTheme(Component component,
			ComponentState componentState, boolean toReturnCurrent,
			boolean checkHierarchy) {
		SubstanceTheme theme = getTheme(component, toReturnCurrent);
		if (theme != null)
			return theme.getHighlightTheme(component, componentState);
		return null;
	}

	/**
	 * Returns the alpha channel of the highlight theme of the component.
	 * 
	 * @param component
	 *            Component.
	 * @param componentState
	 *            Component state.
	 * @param toReturnCurrent
	 *            if <code>true</code>, the currently set <b>Substance</b>
	 *            theme will be returned if the
	 *            {@link SubstanceLookAndFeel#THEME_PROPERTY} is not set on the
	 *            component and any of its ancestors.
	 * @return Highlight theme alpha channel.
	 */
	public static float getHighlightAlpha(Component component,
			ComponentState componentState, boolean toReturnCurrent) {
		SubstanceTheme theme = getTheme(component, toReturnCurrent);
		if (theme != null)
			return theme.getHighlightThemeAlpha(component, componentState);
		return 0.0f;
	}

	/**
	 * Returns the disabled theme of the specified component.
	 * 
	 * @param component
	 *            The component.
	 * @param toReturnCurrent
	 *            if <code>true</code>, the currently set <b>Substance</b>
	 *            theme will be returned if the
	 *            {@link SubstanceLookAndFeel#THEME_PROPERTY} is not set on the
	 *            component and any of its ancestors.
	 * @return The disabled theme of the specified component.
	 */
	public static SubstanceTheme getDisabledTheme(Component component,
			boolean toReturnCurrent) {
		SubstanceTheme theme = getTheme(component, toReturnCurrent);
		if (theme != null)
			return theme.getDisabledTheme();
		return null;
	}

	/**
	 * Returns the color scheme of the component.
	 * 
	 * @param component
	 *            Component.
	 * @param componentState
	 *            Component state.
	 * @return Component color scheme.
	 */
	public static ColorScheme getScheme(Component component,
			ComponentState componentState) {
		SubstanceTheme theme = getTheme(component, componentState, true, false);
		if (theme != null)
			return theme.getColorScheme();
		return null;
	}

	/**
	 * Returns the active scheme of the specified component.
	 * 
	 * @param component
	 *            The component.
	 * @return The active scheme of the specified component.
	 */
	public static ColorScheme getActiveScheme(Component component) {
		SubstanceTheme theme = getActiveTheme(component, true);
		if (theme != null)
			return theme.getColorScheme();
		return null;
	}

	/**
	 * Returns the default scheme of the specified component.
	 * 
	 * @param component
	 *            The component.
	 * @param toReturnCurrent
	 *            if <code>true</code>, the currently set <b>Substance</b>
	 *            theme will be returned if the
	 *            {@link SubstanceLookAndFeel#THEME_PROPERTY} is not set on the
	 *            component and any of its ancestors.
	 * @return The default scheme of the specified component.
	 */
	public static ColorScheme getDefaultScheme(Component component,
			boolean toReturnCurrent) {
		SubstanceTheme theme = getDefaultTheme(component, toReturnCurrent);
		if (theme != null)
			return theme.getColorScheme();
		return null;
	}

	/**
	 * Returns the default scheme of the specified component.
	 * 
	 * @param component
	 *            The component.
	 * @return The default scheme of the specified component.
	 */
	public static ColorScheme getDefaultScheme(Component component) {
		return getDefaultScheme(component, true);
	}

	/**
	 * Returns the disabled scheme of the specified component.
	 * 
	 * @param component
	 *            The component.
	 * @param toReturnCurrent
	 *            if <code>true</code>, the currently set <b>Substance</b>
	 *            theme will be returned if the
	 *            {@link SubstanceLookAndFeel#THEME_PROPERTY} is not set on the
	 *            component and any of its ancestors.
	 * @return The disabled scheme of the specified component.
	 */
	public static ColorScheme getDisabledScheme(Component component,
			boolean toReturnCurrent) {
		SubstanceTheme theme = getDisabledTheme(component, toReturnCurrent);
		if (theme != null)
			return theme.getColorScheme();
		return null;
	}

	/**
	 * Returns the disabled scheme of the specified component.
	 * 
	 * @param component
	 *            The component.
	 * @return The disabled scheme of the specified component.
	 */
	public static ColorScheme getDisabledScheme(Component component) {
		return getDisabledScheme(component, true);
	}

	/**
	 * Returns the active scheme of the specified component in the specified
	 * parent.
	 * 
	 * @param component
	 *            The component.
	 * @param parent
	 *            The component parent of the component.
	 * @return The active scheme of the specified component.
	 */
	public static ColorScheme getActiveScheme(Component component,
			Component parent) {
		SubstanceTheme theme = getTheme(component, parent, false);
		if (theme != null)
			return theme.getActiveTheme().getColorScheme();
		return SubstanceLookAndFeel.getActiveColorScheme();
	}

	/**
	 * Returns the default scheme of the specified component in the specified
	 * parent.
	 * 
	 * @param component
	 *            The component.
	 * @param parent
	 *            The component parent of the component.
	 * @return The default scheme of the specified component.
	 */
	public static ColorScheme getDefaultScheme(Component component,
			Component parent) {

		boolean isComponentAlwaysActive = isControlAlwaysPaintedActive(component);
		boolean isParentAlwaysActive = isControlAlwaysPaintedActive(parent);

		if (isComponentAlwaysActive || isParentAlwaysActive) {
			return getActiveScheme(component, parent);
		}

		SubstanceTheme theme = getTheme(component, parent, false);
		if (theme != null)
			return theme.getDefaultTheme().getColorScheme();
		return SubstanceLookAndFeel.getDefaultColorScheme();
	}

	/**
	 * Returns the disabled scheme of the specified component in the specified
	 * parent.
	 * 
	 * @param component
	 *            The component.
	 * @param parent
	 *            The component parent of the component.
	 * @return The disabled scheme of the specified component.
	 */
	public static ColorScheme getDisabledScheme(Component component,
			Component parent) {
		SubstanceTheme theme = getTheme(component, parent, false);
		if (theme != null)
			return theme.getDisabledTheme().getColorScheme();
		return SubstanceLookAndFeel.getDisabledColorScheme();
	}

	/**
	 * Checks whether the specified control is always painted in currently
	 * active color (ignoring the transition states that normally result in
	 * default appearance).
	 * 
	 * @param comp
	 *            Control.
	 * @return <code>true</code> if the specified control is always painted in
	 *         currently active color (ignoring the transition states that
	 *         normally result in default appearance), <code>false</code>
	 *         otherwise.
	 */
	public static boolean isControlAlwaysPaintedActive(Component comp) {
		return isControlAlwaysPaintedActive(comp, false);
	}

	/**
	 * Checks whether the specified control is always painted in currently
	 * active color (ignoring the transition states that normally result in
	 * default appearance).
	 * 
	 * @param comp
	 *            Control.
	 * @param checkHierarchy
	 *            If <code>true</code>, the entire component hierarchy will
	 *            be scanned for the
	 *            {@link SubstanceLookAndFeel#PAINT_ACTIVE_PROPERTY}.
	 * @return <code>true</code> if the specified control is always painted in
	 *         currently active color (ignoring the transition states that
	 *         normally result in default appearance), <code>false</code>
	 *         otherwise.
	 */
	public static boolean isControlAlwaysPaintedActive(Component comp,
			boolean checkHierarchy) {
		if (!checkHierarchy) {
			if (comp instanceof JComponent) {
				JComponent jcomp = (JComponent) comp;
				Object componentProp = jcomp
						.getClientProperty(SubstanceLookAndFeel.PAINT_ACTIVE_PROPERTY);
				if (componentProp != null) {
					if (Boolean.TRUE.equals(componentProp))
						return true;
					if (Boolean.FALSE.equals(componentProp))
						return false;
				}
			}
		} else {
			Component c = comp;
			while (c != null) {
				if (c instanceof JComponent) {
					JComponent jcomp = (JComponent) c;
					Object componentProp = jcomp
							.getClientProperty(SubstanceLookAndFeel.PAINT_ACTIVE_PROPERTY);
					if (componentProp != null) {
						if (Boolean.TRUE.equals(componentProp))
							return true;
						if (Boolean.FALSE.equals(componentProp))
							return false;
					}
				}
				c = c.getParent();
			}
		}
		return Boolean.TRUE.equals(UIManager
				.get(SubstanceLookAndFeel.PAINT_ACTIVE_PROPERTY));
	}

	/**
	 * Checks whether the specified button never paints its background.
	 * 
	 * @param button
	 *            Button.
	 * @return <code>true</code> if the specified button never paints its
	 *         background, <code>false</code> otherwise.
	 */
	public static boolean isButtonNeverPainted(AbstractButton button) {
		Component c = button;
		while (c != null) {
			if (c instanceof JComponent) {
				JComponent jc = (JComponent) c;
				Object prop = jc
						.getClientProperty(SubstanceLookAndFeel.BUTTON_PAINT_NEVER_PROPERTY);
				if (prop != null) {
					if (Boolean.TRUE.equals(prop))
						return true;
					if (Boolean.FALSE.equals(prop))
						return false;
				}
			}
			c = c.getParent();
		}

		return Boolean.TRUE.equals(UIManager
				.get(SubstanceLookAndFeel.BUTTON_PAINT_NEVER_PROPERTY));
	}

	/**
	 * Returns the focus ring kind of the specified component.
	 * 
	 * @param component
	 *            Component.
	 * @return The focus ring kind of the specified component.
	 */
	public static FocusKind getFocusKind(Component component) {
		while (component != null) {
			if (component instanceof JComponent) {
				JComponent jcomp = (JComponent) component;
				Object jcompFocusKind = jcomp
						.getClientProperty(SubstanceLookAndFeel.FOCUS_KIND);
				if (jcompFocusKind instanceof FocusKind)
					return (FocusKind) jcompFocusKind;
			}
			component = component.getParent();
		}
		Object globalFocusKind = UIManager.get(SubstanceLookAndFeel.FOCUS_KIND);
		if (globalFocusKind instanceof FocusKind)
			return (FocusKind) globalFocusKind;
		return FocusKind.ALL_INNER;
	}

	/**
	 * Returns the text alignment kind of the specified tabbed pane.
	 * 
	 * @param tabPane
	 *            Tabbed pane.
	 * @return The text alignment kind of the specified tabbed pane.
	 */
	public static TabTextAlignmentKind getTabTextAlignmentKind(
			JTabbedPane tabPane) {
		Object jcompAlignmentKind = tabPane
				.getClientProperty(SubstanceLookAndFeel.TABBED_PANE_TEXT_ALIGNMENT_KIND);
		if (jcompAlignmentKind instanceof TabTextAlignmentKind)
			return (TabTextAlignmentKind) jcompAlignmentKind;

		Object globalAlignmentKind = UIManager
				.get(SubstanceLookAndFeel.TABBED_PANE_TEXT_ALIGNMENT_KIND);
		if (globalAlignmentKind instanceof TabTextAlignmentKind)
			return (TabTextAlignmentKind) globalAlignmentKind;
		return TabTextAlignmentKind.DEFAULT;
	}

	/**
	 * Returns indication whether the watermark should be drawn on the specified
	 * component.
	 * 
	 * @param component
	 *            Component.
	 * @return <code>true</code> if the watermark should be drawn on the
	 *         specified component, <code>false</code> otherwise.
	 */
	public static boolean toDrawWatermark(Component component) {
		while (component != null) {
			if (component instanceof JComponent) {
				JComponent jcomp = (JComponent) component;
				Object obj = jcomp
						.getClientProperty(SubstanceLookAndFeel.WATERMARK_IGNORE);
				if (Boolean.TRUE.equals(obj))
					return false;
				if (Boolean.FALSE.equals(obj))
					return true;
			}
			component = component.getParent();
		}
		Object obj = UIManager.get(SubstanceLookAndFeel.WATERMARK_IGNORE);
		if (Boolean.TRUE.equals(obj))
			return false;
		return true;
	}

	/**
	 * Returns indication whether the watermark should "bleed" through the
	 * specified component.
	 * 
	 * @param component
	 *            Component.
	 * @return <code>true</code> if the watermark should "bleed" through the
	 *         specified component, <code>false</code> otherwise.
	 */
	public static boolean toBleedWatermark(Component component) {
		if (component instanceof JComponent) {
			JComponent jcomp = (JComponent) component;
			Object obj = jcomp
					.getClientProperty(SubstanceLookAndFeel.WATERMARK_TO_BLEED);
			if (Boolean.TRUE.equals(obj))
				return true;
			if (Boolean.FALSE.equals(obj))
				return false;
		}
		return SubstanceLookAndFeel.toBleedWatermark();
	}

	/**
	 * Returns the button shaper of the specified button.
	 * 
	 * @param button
	 *            The button.
	 * @return The button shaper of the specified button.
	 */
	public static SubstanceButtonShaper getButtonShaper(AbstractButton button) {
		SubstanceButtonShaper shaper = null;
		Component c = button;
		while (c != null) {
			if (c instanceof JComponent) {
				JComponent jcomp = (JComponent) c;
				Object customShaper = jcomp
						.getClientProperty(SubstanceLookAndFeel.BUTTON_SHAPER_PROPERTY);
				if (customShaper != null) {
					if (customShaper instanceof SubstanceButtonShaper)
						return (SubstanceButtonShaper) customShaper;
					if (customShaper instanceof String) {
						try {
							shaper = (SubstanceButtonShaper) Class.forName(
									(String) customShaper).newInstance();
						} catch (Exception exc) {
							shaper = null;
						}

						if (shaper != null)
							return shaper;
					}
				}
			}
			c = c.getParent();
		}
		if (shaper == null) {
			shaper = SubstanceLookAndFeel.getCurrentButtonShaper();
		}
		return shaper;
	}

	/**
	 * Returns the gradient painter of the specified component.
	 * 
	 * @param comp
	 *            Component.
	 * @return The gradient painter of the specified component.
	 */
	public static SubstanceGradientPainter getGradientPainter(JComponent comp) {
		SubstanceGradientPainter painter = SubstanceLookAndFeel
				.getCurrentGradientPainter();
		Object customPainter = comp
				.getClientProperty(SubstanceLookAndFeel.GRADIENT_PAINTER_PROPERTY);
		if (customPainter != null) {
			if (customPainter instanceof SubstanceGradientPainter)
				return (SubstanceGradientPainter) customPainter;
			try {
				painter = (SubstanceGradientPainter) Class.forName(
						(String) customPainter).newInstance();
			} catch (Exception exc) {
				painter = SubstanceLookAndFeel.getCurrentGradientPainter();
			}
		}
		return painter;
	}

	/**
	 * Retrieves the <code>modified</code> state for the specified component
	 * in a tabbed pane.
	 * 
	 * @param tabComponent
	 *            The associated tab component.
	 * @return <code>true</code> if the specified component in a tabbed pane
	 *         is marked as modified, <code>false</code> otherwise.
	 * @see SubstanceLookAndFeel#WINDOW_MODIFIED
	 */
	public static boolean isTabModified(Component tabComponent) {
		boolean isWindowModified = false;
		// boolean toEnd = (tabComponent == null);
		Component comp = tabComponent;
		// while (!toEnd) {
		if (comp instanceof JComponent) {
			JComponent jc = (JComponent) comp;

			isWindowModified = Boolean.TRUE.equals(jc
					.getClientProperty(SubstanceLookAndFeel.WINDOW_MODIFIED));
			// if (isWindowModified)
			// break;
		}
		// if (comp instanceof JTabbedPane)
		// toEnd = true;
		// comp = comp.getParent();
		// if (comp == null)
		// toEnd = true;
		// }
		return isWindowModified;
	}

	/**
	 * Retrieves the <code>modified</code> state for the specified root pane.
	 * 
	 * @param rootPane
	 *            The root pane.
	 * @return <code>true</code> if the specified root pane is marked as
	 *         modified, <code>false</code> otherwise.
	 * @see SubstanceLookAndFeel#WINDOW_MODIFIED
	 */
	public static boolean isRootPaneModified(JRootPane rootPane) {
		return Boolean.TRUE.equals(rootPane
				.getClientProperty(SubstanceLookAndFeel.WINDOW_MODIFIED));
	}

	/**
	 * Retrieves the <code>modified</code> state for the specified internal
	 * frame.
	 * 
	 * @param internalFrame
	 *            The internal frame.
	 * @return <code>true</code> if the specified internal frame is marked as
	 *         modified, <code>false</code> otherwise.
	 * @see SubstanceLookAndFeel#WINDOW_MODIFIED
	 */
	public static boolean isInternalFrameModified(JInternalFrame internalFrame) {
		Object frameProp = internalFrame
				.getClientProperty(SubstanceLookAndFeel.WINDOW_MODIFIED);
		if (Boolean.TRUE.equals(frameProp))
			return true;
		if (Boolean.FALSE.equals(frameProp))
			return false;
		return Boolean.TRUE.equals(internalFrame.getRootPane()
				.getClientProperty(SubstanceLookAndFeel.WINDOW_MODIFIED));
	}

	// /**
	// * Returns the tab animation kind of the specified tab component.
	// *
	// * @param tabComponent
	// * Tab component.
	// * @return Tab animation kind of the specified tab component.
	// */
	// public static TabAnimationKind getTabAnimationKind(Component
	// tabComponent) {
	// Component comp = tabComponent;
	// // while (!toEnd) {
	// if (comp instanceof JComponent) {
	// JComponent jc = (JComponent) comp;
	//
	// Object animObj = jc
	// .getClientProperty(SubstanceLookAndFeel.TABBED_PANE_TAB_ANIMATION_KIND);
	// if (animObj instanceof TabAnimationKind)
	// return (TabAnimationKind) animObj;
	// }
	// return null;
	// }
	//
	/**
	 * Returns the tab background composite of the specified tabbed pane.
	 * 
	 * @param component
	 *            Tabbed pane.
	 * @return Tab background composite of the specified tabbed pane.
	 */
	public static ControlBackgroundComposite getControlBackgroundComposite(
			Component component) {
		Component c = component;
		while (c != null) {
			if (c instanceof JComponent) {
				JComponent jc = (JComponent) c;
				Object objProp = jc
						.getClientProperty(SubstanceLookAndFeel.BACKGROUND_COMPOSITE);
				if (objProp instanceof ControlBackgroundComposite)
					return (ControlBackgroundComposite) objProp;
			}
			c = c.getParent();
		}
		Object globalProp = UIManager
				.get(SubstanceLookAndFeel.BACKGROUND_COMPOSITE);
		if (globalProp instanceof ControlBackgroundComposite)
			return (ControlBackgroundComposite) globalProp;

		return SubstanceLookAndFeel.getBackgroundComposite(component);
	}

	/**
	 * Checks whether the specified tab has a close button.
	 * 
	 * @param tabbedPane
	 *            Tabbed pane.
	 * @param tabIndex
	 *            Tab index.
	 * @return <code>true</code> if the specified tab has a close button,
	 *         <code>false</code> otherwise.
	 */
	public static boolean hasCloseButton(JTabbedPane tabbedPane, int tabIndex) {
		int tabCount = tabbedPane.getTabCount();
		if ((tabIndex < 0) || (tabIndex >= tabCount))
			return false;

		// if tab is disabled, it doesn't have a close button
		if (!tabbedPane.isEnabledAt(tabIndex))
			return false;

		// check property on tab component
		Component tabComponent = tabbedPane.getComponentAt(tabIndex);
		if (tabComponent instanceof JComponent) {
			Object compProp = ((JComponent) tabComponent)
					.getClientProperty(SubstanceLookAndFeel.TABBED_PANE_CLOSE_BUTTONS_PROPERTY);
			if (Boolean.TRUE.equals(compProp))
				return true;
			if (Boolean.FALSE.equals(compProp))
				return false;
		}
		// check property on tabbed pane
		Object tabProp = tabbedPane
				.getClientProperty(SubstanceLookAndFeel.TABBED_PANE_CLOSE_BUTTONS_PROPERTY);
		if (Boolean.TRUE.equals(tabProp))
			return true;
		if (Boolean.FALSE.equals(tabProp))
			return false;
		// check property in UIManager
		return UIManager
				.getBoolean(SubstanceLookAndFeel.TABBED_PANE_CLOSE_BUTTONS_PROPERTY);
	}

	/**
	 * Returns the size of the close button for a tab in the specified tabbed
	 * pane.
	 * 
	 * @param tabbedPane
	 *            Tabbed pane.
	 * @param tabIndex
	 *            Tab index.
	 * @return The size of the close button for a tab in the specified tabbed
	 *         pane.
	 */
	public static int getCloseButtonSize(JTabbedPane tabbedPane, int tabIndex) {
		if (!SubstanceCoreUtilities.hasCloseButton(tabbedPane, tabIndex))
			return 0;
		return SubstanceSizeUtils.getTabCloseIconSize(SubstanceSizeUtils
				.getComponentFontSize(tabbedPane));
	}

	/**
	 * Checks whether the specified tab should show vertically-aligned (rotated)
	 * components.
	 * 
	 * @param tabbedPane
	 *            Tabbed pane.
	 * @return <code>true</code> if the specified tab should show
	 *         vertically-aligned (rotated) components, <code>false</code>
	 *         otherwise.
	 */
	public static boolean toLayoutVertically(JTabbedPane tabbedPane) {
		int tabPlacement = tabbedPane.getTabPlacement();
		if ((tabPlacement == SwingConstants.TOP)
				|| (tabPlacement == SwingConstants.BOTTOM))
			return false;

		// check property on tabbed pane
		Object tabProp = tabbedPane
				.getClientProperty(SubstanceLookAndFeel.TABBED_PANE_VERTICAL_ORIENTATION);
		if (Boolean.TRUE.equals(tabProp))
			return true;
		if (Boolean.FALSE.equals(tabProp))
			return false;
		// check property in UIManager
		return UIManager
				.getBoolean(SubstanceLookAndFeel.TABBED_PANE_VERTICAL_ORIENTATION);
	}

	/**
	 * Checks whether the specified tab should show unrotated icon when the tab
	 * itself is layed-out vertically.
	 * 
	 * @param tabbedPane
	 *            Tabbed pane.
	 * @param tabIndex
	 *            Tab index.
	 * @return <code>true</code> if the specified tab should show unrotated
	 *         icon when the tab itself is layed-out vertically,
	 *         <code>false</code> otherwise.
	 */
	public static boolean toShowIconUnrotated(JTabbedPane tabbedPane,
			int tabIndex) {
		int tabCount = tabbedPane.getTabCount();
		if ((tabIndex < 0) || (tabIndex >= tabCount))
			return false;
		// check property on tab component
		Component tabComponent = tabbedPane.getComponentAt(tabIndex);
		if (tabComponent instanceof JComponent) {
			Object compProp = ((JComponent) tabComponent)
					.getClientProperty(SubstanceLookAndFeel.TABBED_PANE_VERTICAL_ORIENTATION_ROTATE_ICONS);
			if (Boolean.TRUE.equals(compProp))
				return true;
			if (Boolean.FALSE.equals(compProp))
				return false;
		}
		// check property on tabbed pane
		Object tabProp = tabbedPane
				.getClientProperty(SubstanceLookAndFeel.TABBED_PANE_VERTICAL_ORIENTATION_ROTATE_ICONS);
		if (Boolean.TRUE.equals(tabProp))
			return true;
		if (Boolean.FALSE.equals(tabProp))
			return false;
		// check property in UIManager
		return UIManager
				.getBoolean(SubstanceLookAndFeel.TABBED_PANE_VERTICAL_ORIENTATION_ROTATE_ICONS);
	}

	/**
	 * Returns the content border kind of the specified tabbed pane.
	 * 
	 * @param tabbedPane
	 *            Tabbed pane.
	 * @return Content border kind of the specified tabbed pane.
	 */
	public static TabContentPaneBorderKind getContentBorderKind(
			JTabbedPane tabbedPane) {
		// check property on tabbed pane
		Object tabProp = tabbedPane
				.getClientProperty(SubstanceLookAndFeel.TABBED_PANE_CONTENT_BORDER_KIND);
		if (tabProp instanceof TabContentPaneBorderKind)
			return (TabContentPaneBorderKind) tabProp;
		// check property in UIManager
		Object globalProp = UIManager
				.get(SubstanceLookAndFeel.TABBED_PANE_CONTENT_BORDER_KIND);
		if (globalProp instanceof TabContentPaneBorderKind)
			return (TabContentPaneBorderKind) globalProp;
		return TabContentPaneBorderKind.DOUBLE_FULL;
	}

	/**
	 * Checks whether the specified tab should show modified animation only on
	 * its close button.
	 * 
	 * @param tabbedPane
	 *            Tabbed pane.
	 * @param tabIndex
	 *            Tab index.
	 * @return <code>true</code> if the specified tab should show modified
	 *         animation only on its close button, <code>false</code>
	 *         otherwise.
	 */
	public static boolean toAnimateCloseIconOfModifiedTab(
			JTabbedPane tabbedPane, int tabIndex) {
		int tabCount = tabbedPane.getTabCount();
		if ((tabIndex < 0) || (tabIndex >= tabCount))
			return false;

		if (!hasCloseButton(tabbedPane, tabIndex))
			return false;

		// check property on tab component
		Component tabComponent = tabbedPane.getComponentAt(tabIndex);
		if (tabComponent instanceof JComponent) {
			Object compProp = ((JComponent) tabComponent)
					.getClientProperty(SubstanceLookAndFeel.TABBED_PANE_CLOSE_BUTTONS_MODIFIED_ANIMATION);
			if (Boolean.TRUE.equals(compProp))
				return true;
			if (Boolean.FALSE.equals(compProp))
				return false;
		}
		// check property on tabbed pane
		Object tabProp = tabbedPane
				.getClientProperty(SubstanceLookAndFeel.TABBED_PANE_CLOSE_BUTTONS_MODIFIED_ANIMATION);
		if (Boolean.TRUE.equals(tabProp))
			return true;
		if (Boolean.FALSE.equals(tabProp))
			return false;
		// check property in UIManager
		return UIManager
				.getBoolean(SubstanceLookAndFeel.TABBED_PANE_CLOSE_BUTTONS_MODIFIED_ANIMATION);
	}

	/**
	 * Retrieves transparent image of specified dimension.
	 * 
	 * @param width
	 *            Image width.
	 * @param height
	 *            Image height.
	 * @return Transparent image of specified dimension.
	 */
	public static BufferedImage getBlankImage(int width, int height) {
		if (MemoryAnalyzer.isRunning()) {
			// see if the request is unusual
			if ((width >= 100) || (height >= 100)) {
				// throw an Exception and then analyze it
				try {
					throw new Exception();
				} catch (Exception exc) {
					StringBuffer sb = new StringBuffer();
					StackTraceElement[] stack = exc.getStackTrace();
					int count = 0;
					for (StackTraceElement stackEntry : stack) {
						if (count++ > 8)
							break;
						sb.append(stackEntry.getClassName() + ".");
						sb.append(stackEntry.getMethodName() + " [");
						sb.append(stackEntry.getLineNumber() + "]");
						sb.append("\n");
					}
					MemoryAnalyzer.enqueueUsage("Blank " + width + "*" + height
							+ "\n" + sb.toString());

				}
			}
		}

		BufferedImage image = new BufferedImage(width, height,
				BufferedImage.TYPE_INT_ARGB);

		// get graphics and set hints
		Graphics2D graphics = (Graphics2D) image.getGraphics().create();
		graphics.setColor(new Color(0, 0, 0, 0));
		graphics.setComposite(AlphaComposite.Src);
		graphics.fillRect(0, 0, width, height);
		graphics.dispose();

		return image;
	}

	/**
	 * Checks whether the specified button should have minimal size.
	 * 
	 * @param button
	 *            Button.
	 * @return <code>false</code> if the specified button should have minimal
	 *         size, <code>true</code> otherwise.
	 */
	public static boolean hasNoMinSizeProperty(AbstractButton button) {
		Object noMinSizeProperty = button
				.getClientProperty(SubstanceLookAndFeel.BUTTON_NO_MIN_SIZE_PROPERTY);
		if (Boolean.TRUE.equals(noMinSizeProperty))
			return true;
		if (Boolean.FALSE.equals(noMinSizeProperty))
			return false;
		Container parent = button.getParent();
		if (parent instanceof JComponent) {
			noMinSizeProperty = ((JComponent) parent)
					.getClientProperty(SubstanceLookAndFeel.BUTTON_NO_MIN_SIZE_PROPERTY);
			if (Boolean.TRUE.equals(noMinSizeProperty))
				return true;
			if (Boolean.FALSE.equals(noMinSizeProperty))
				return false;
		}
		return (Boolean.TRUE.equals(UIManager
				.get(SubstanceLookAndFeel.BUTTON_NO_MIN_SIZE_PROPERTY)));
	}

	/**
	 * Checks whether the specified component is flat.
	 * 
	 * @param comp
	 *            Component.
	 * @param defaultValue
	 *            The value to return if there is no
	 *            {@link SubstanceLookAndFeel#FLAT_PROPERTY} defined on button
	 *            hierarchy or {@link UIManager}.
	 * @return <code>false</code> if the specified button is flat,
	 *         <code>true</code> otherwise.
	 */
	public static boolean hasFlatAppearance(Component comp, boolean defaultValue) {
		Component c = comp;
		while (c != null) {
			if (c instanceof JComponent) {
				JComponent jcomp = (JComponent) c;
				Object flatProperty = jcomp
						.getClientProperty(SubstanceLookAndFeel.FLAT_PROPERTY);
				if (Boolean.TRUE.equals(flatProperty))
					return true;
				if (Boolean.FALSE.equals(flatProperty))
					return false;
			}
			c = c.getParent();
		}
		Object flatProperty = UIManager.get(SubstanceLookAndFeel.FLAT_PROPERTY);
		if (Boolean.TRUE.equals(flatProperty))
			return true;
		if (Boolean.FALSE.equals(flatProperty))
			return false;
		return defaultValue;
	}

	/**
	 * Computes whether the specified button has flat appearance.
	 * 
	 * @param button
	 *            Button.
	 * @return <code>true</code> if the button has flat appearance,
	 *         <code>false</code> otherwise.
	 */
	public static boolean hasFlatAppearance(AbstractButton button) {
		return ((SubstanceCoreUtilities.isToolBarButton(button) && SubstanceCoreUtilities
				.hasFlatAppearance(button, true)) || SubstanceCoreUtilities
				.hasFlatAppearance(button, false));
	}

	/**
	 * Returns the popup flyout orientation for the specified combobox.
	 * 
	 * @param combobox
	 *            Combobox.
	 * @return The popup flyout orientation for the specified combobox.
	 */
	public static int getPopupFlyoutOrientation(JComboBox combobox) {
		Object comboProperty = combobox
				.getClientProperty(SubstanceLookAndFeel.COMBO_BOX_POPUP_FLYOUT_ORIENTATION);
		if (comboProperty instanceof Integer)
			return (Integer) comboProperty;
		Object globalProperty = UIManager
				.get(SubstanceLookAndFeel.COMBO_BOX_POPUP_FLYOUT_ORIENTATION);
		if (globalProperty instanceof Integer)
			return (Integer) globalProperty;
		return SwingConstants.SOUTH;
	}

	/**
	 * Makes the specified component and all its descendants non-opaque.
	 * 
	 * @param comp
	 *            Component.
	 * @param opaquenessSnapshot
	 *            The "snapshot" map that will contain the original opacity
	 *            status of the specified component and all its descendants.
	 */
	public static void makeNonOpaque(Component comp,
			Map<Component, Boolean> opaquenessSnapshot) {
		if (comp instanceof JComponent) {
			JComponent jcomp = (JComponent) comp;
			opaquenessSnapshot.put(comp, jcomp.isOpaque());
			jcomp.setOpaque(false);
		}
		if (comp instanceof Container) {
			Container cont = (Container) comp;
			for (int i = 0; i < cont.getComponentCount(); i++)
				makeNonOpaque(cont.getComponent(i), opaquenessSnapshot);
		}
	}

	/**
	 * Restores the opacity of the specified component and all its descendants.
	 * 
	 * @param comp
	 *            Component.
	 * @param opaquenessSnapshot
	 *            The "snapshot" map that contains the original opacity status
	 *            of the specified component and all its descendants.
	 */
	public static void restoreOpaque(Component comp,
			Map<Component, Boolean> opaquenessSnapshot) {
		if (comp instanceof JComponent) {
			JComponent jcomp = (JComponent) comp;
			// fix for defect 159 - snapshot may not contain
			// opacity for table header of a table when it's used
			// inside tree cell renderer (wrapper in a scroll pane).
			if (opaquenessSnapshot.containsKey(comp))
				jcomp.setOpaque(opaquenessSnapshot.get(comp));
			else
				jcomp.setOpaque(true);
		}
		if (comp instanceof Container) {
			Container cont = (Container) comp;
			for (int i = 0; i < cont.getComponentCount(); i++)
				restoreOpaque(cont.getComponent(i), opaquenessSnapshot);
		}
	}

	/**
	 * Makes the specified component and all its descendants non-double
	 * buffered.
	 * 
	 * @param comp
	 *            Component.
	 * @param dbSnapshot
	 *            The "snapshot" map that will contain the original double
	 *            buffer status of the specified component and all its
	 *            descendants.
	 */
	public static void makeNonDoubleBuffered(Component comp,
			Map<Component, Boolean> dbSnapshot) {
		if (comp instanceof JComponent) {
			JComponent jcomp = (JComponent) comp;
			dbSnapshot.put(comp, jcomp.isDoubleBuffered());
			jcomp.setDoubleBuffered(false);
		}
		if (comp instanceof Container) {
			Container cont = (Container) comp;
			for (int i = 0; i < cont.getComponentCount(); i++)
				makeNonDoubleBuffered(cont.getComponent(i), dbSnapshot);
		}
	}

	/**
	 * Restores the double buffer of the specified component and all its
	 * descendants.
	 * 
	 * @param comp
	 *            Component.
	 * @param dbSnapshot
	 *            The "snapshot" map that contains the original double buffer
	 *            status of the specified component and all its descendants.
	 */
	public static void restoreDoubleBuffered(Component comp,
			Map<Component, Boolean> dbSnapshot) {
		if (comp instanceof JComponent) {
			JComponent jcomp = (JComponent) comp;
			// fix for defect 159 - snapshot may not contain
			// opacity for table header of a table when it's used
			// inside tree cell renderer (wrapper in a scroll pane).
			if (dbSnapshot.containsKey(comp))
				jcomp.setDoubleBuffered(dbSnapshot.get(comp));
			else
				jcomp.setOpaque(true);
		}
		if (comp instanceof Container) {
			Container cont = (Container) comp;
			for (int i = 0; i < cont.getComponentCount(); i++)
				restoreDoubleBuffered(cont.getComponent(i), dbSnapshot);
		}
	}

	/**
	 * Creates a compatible image (for efficient processing and drawing).
	 * 
	 * @param image
	 *            The original image.
	 * @return Compatible version of the original image.
	 * @author Romain Guy
	 */
	public static BufferedImage createCompatibleImage(BufferedImage image) {
		GraphicsEnvironment e = GraphicsEnvironment
				.getLocalGraphicsEnvironment();
		GraphicsDevice d = e.getDefaultScreenDevice();
		GraphicsConfiguration c = d.getDefaultConfiguration();
		BufferedImage compatibleImage = c.createCompatibleImage(image
				.getWidth(), image.getHeight());
		Graphics g = compatibleImage.getGraphics();
		g.drawImage(image, 0, 0, null);
		g.dispose();
		return compatibleImage;
	}

	// /**
	// * Checks whether the specified button in a toolbar is flat.
	// *
	// * @param button
	// * Button.
	// * @return <code>true</code> if the specified button is flat in a toolbar,
	// * <code>false</code> otherwise.
	// */
	// public static boolean isToolbarButtonFlat(AbstractButton button) {
	// Component comp = button;
	// JToolBar toolbar = null;
	// while (comp != null) {
	// if (comp instanceof JToolBar) {
	// toolbar = (JToolBar) comp;
	// break;
	// }
	// comp = comp.getParent();
	// }
	// if (toolbar == null)
	// return false;
	//
	// return isToolbarButtonFlat(button, toolbar);
	// }
	//
	// /**
	// * Checks whether the specified button in the specified toolbar is flat.
	// *
	// * @param button
	// * Button.
	// * @param toolbar
	// * Toolbar that contains the specified button.
	// * @return <code>true</code> if the specified button is flat in the
	// * specified toolbar, <code>false</code> otherwise.
	// */
	// public static boolean isToolbarButtonFlat(AbstractButton button,
	// JToolBar toolbar) {
	// Object isFlatProperty = button
	// .getClientProperty(SubstanceLookAndFeel.FLAT_PROPERTY);
	// if (Boolean.TRUE.equals(isFlatProperty))
	// return true;
	// if (Boolean.FALSE.equals(isFlatProperty))
	// return false;
	// isFlatProperty = toolbar
	// .getClientProperty(SubstanceLookAndFeel.FLAT_PROPERTY);
	// if (Boolean.TRUE.equals(isFlatProperty))
	// return true;
	// if (Boolean.FALSE.equals(isFlatProperty))
	// return false;
	// if (Boolean.FALSE.equals(UIManager
	// .get(SubstanceLookAndFeel.FLAT_PROPERTY)))
	// return false;
	// return true;
	// }

	/**
	 * Checks whether the specified component will show theme-colorized icon in
	 * the default state.
	 * 
	 * @param comp
	 *            Component.
	 * @return <code>true</code> if the specified component will show
	 *         theme-colorized icon in the default state, <code>false</code>
	 *         otherwise.
	 */
	public static boolean useThemedDefaultIcon(JComponent comp) {
		if (comp != null) {
			Object useThemedDefaultIconProperty = comp
					.getClientProperty(SubstanceLookAndFeel.USE_THEMED_DEFAULT_ICONS);
			if (Boolean.TRUE.equals(useThemedDefaultIconProperty))
				return true;
			if (Boolean.FALSE.equals(useThemedDefaultIconProperty))
				return false;
		}
		if (Boolean.TRUE.equals(UIManager
				.get(SubstanceLookAndFeel.USE_THEMED_DEFAULT_ICONS)))
			return true;
		return false;
	}

	/**
	 * Returns the callback to be called upon tab closing (using the tab close
	 * button).
	 * 
	 * @param me
	 *            Mouse event.
	 * @param tabbedPane
	 *            Tabbed pane.
	 * @param tabIndex
	 *            Tab index.
	 * @return Callback to be called upon tab closing (using the tab close
	 *         button).
	 */
	public static TabCloseCallback getTabCloseCallback(MouseEvent me,
			JTabbedPane tabbedPane, int tabIndex) {
		int tabCount = tabbedPane.getTabCount();
		if ((tabIndex < 0) || (tabIndex >= tabCount))
			return null;

		// check property on tab component
		Component tabComponent = tabbedPane.getComponentAt(tabIndex);
		if (tabComponent instanceof JComponent) {
			Object compProp = ((JComponent) tabComponent)
					.getClientProperty(SubstanceLookAndFeel.TABBED_PANE_CLOSE_CALLBACK);
			if (compProp instanceof TabCloseCallback)
				return (TabCloseCallback) compProp;
		}

		// check property on tabbed pane
		Object tabProp = tabbedPane
				.getClientProperty(SubstanceLookAndFeel.TABBED_PANE_CLOSE_CALLBACK);
		if (tabProp instanceof TabCloseCallback)
			return (TabCloseCallback) tabProp;

		Object globProp = UIManager
				.get(SubstanceLookAndFeel.TABBED_PANE_CLOSE_CALLBACK);
		if (globProp instanceof TabCloseCallback)
			return (TabCloseCallback) globProp;

		return null;
	}

	/**
	 * Blends two images along X-axis.
	 * 
	 * @param imageLeft
	 *            The left image.
	 * @param imageRight
	 *            The right image.
	 * @param start
	 *            Relative start of the blend area (in 0.0-1.0 range).
	 * @param end
	 *            Relative end of the blend area (in 0.0-1.0 range).
	 * @return Blended image.
	 */
	public static BufferedImage blendImagesHorizontal(BufferedImage imageLeft,
			BufferedImage imageRight, double start, double end) {
		int width = imageLeft.getWidth();
		if (width != imageRight.getWidth())
			throw new IllegalArgumentException("Widths are not the same: "
					+ imageLeft.getWidth() + " and " + imageRight.getWidth());
		int height = imageLeft.getHeight();
		if (height != imageRight.getHeight())
			throw new IllegalArgumentException("Heights are not the same: "
					+ imageLeft.getHeight() + " and " + imageRight.getHeight());

		BufferedImage result = getBlankImage(width, height);
		Graphics2D graphics = (Graphics2D) result.getGraphics().create();

		int startX = (int) (start * width);
		int endX = (int) (end * width);
		int rampWidth = endX - startX;

		if (rampWidth == 0) {
			graphics.drawImage(imageLeft, 0, 0, startX, height, 0, 0, startX,
					height, null);
			graphics.drawImage(imageRight, startX, 0, width, height, startX, 0,
					width, height, null);
		} else {
			BufferedImage rampRight = getBlankImage(rampWidth, height);
			Graphics2D rampRightG = (Graphics2D) rampRight.getGraphics();
			rampRightG
					.setPaint(new GradientPaint(new Point(0, 0), new Color(0,
							0, 0, 255), new Point(rampWidth, 0), new Color(0,
							0, 0, 0)));
			rampRightG.fillRect(0, 0, rampWidth, height);

			BufferedImage tempRight = getBlankImage(width - startX, height);
			Graphics2D tempRightG = (Graphics2D) tempRight.getGraphics();
			tempRightG.drawImage(imageRight, 0, 0, width - startX, height,
					startX, 0, width, height, null);
			tempRightG.setComposite(AlphaComposite.DstOut);
			tempRightG.drawImage(rampRight, 0, 0, null);

			tempRightG.setComposite(AlphaComposite.SrcOver);
			graphics.drawImage(imageLeft, 0, 0, null);
			graphics.drawImage(tempRight, startX, 0, null);
		}
		graphics.dispose();
		return result;
	}

	/**
	 * Blends two images along Y-axis.
	 * 
	 * @param imageTop
	 *            The left image.
	 * @param imageBottom
	 *            The right image.
	 * @param start
	 *            Relative start of the blend area (in 0.0-1.0 range).
	 * @param end
	 *            Relative end of the blend area (in 0.0-1.0 range).
	 * @return Blended image.
	 */
	public static BufferedImage blendImagesVertical(BufferedImage imageTop,
			BufferedImage imageBottom, double start, double end) {
		int width = imageTop.getWidth();
		if (width != imageBottom.getWidth())
			throw new IllegalArgumentException("Widths are not the same: "
					+ imageTop.getWidth() + " and " + imageBottom.getWidth());
		int height = imageTop.getHeight();
		if (height != imageBottom.getHeight())
			throw new IllegalArgumentException("Heights are not the same: "
					+ imageTop.getHeight() + " and " + imageBottom.getHeight());

		BufferedImage result = getBlankImage(width, height);
		Graphics2D graphics = (Graphics2D) result.getGraphics().create();

		int startY = (int) (start * height);
		int endY = (int) (end * height);
		int rampHeight = endY - startY;

		if (rampHeight == 0) {
			graphics.drawImage(imageTop, 0, 0, width, startY, 0, 0, width,
					startY, null);
			graphics.drawImage(imageBottom, 0, startY, width, height, 0,
					startY, width, height, null);
		} else {
			BufferedImage rampBottom = getBlankImage(width, rampHeight);
			Graphics2D rampBottomG = (Graphics2D) rampBottom.getGraphics();
			rampBottomG.setPaint(new GradientPaint(new Point(0, 0), new Color(
					0, 0, 0, 255), new Point(0, rampHeight), new Color(0, 0, 0,
					0)));
			rampBottomG.fillRect(0, 0, width, rampHeight);

			BufferedImage tempBottom = getBlankImage(width, height - startY);
			Graphics2D tempBottomG = (Graphics2D) tempBottom.getGraphics();
			tempBottomG.drawImage(imageBottom, 0, 0, width, height - startY, 0,
					startY, width, height, null);
			tempBottomG.setComposite(AlphaComposite.DstOut);
			tempBottomG.drawImage(rampBottom, 0, 0, null);

			tempBottomG.setComposite(AlphaComposite.SrcOver);
			graphics.drawImage(imageTop, 0, 0, null);
			graphics.drawImage(tempBottom, 0, startY, null);
		}
		graphics.dispose();
		return result;
	}

	/**
	 * Retruns the unique ID for the specified color scheme.
	 * 
	 * @param colorScheme
	 *            Color scheme.
	 * @return Unique ID for the specified color scheme.
	 */
	public static String getSchemeId(ColorScheme colorScheme) {
		if (colorScheme instanceof BaseColorScheme) {
			BaseColorScheme base = (BaseColorScheme) colorScheme;
			return base.getId();
		}

		return colorScheme.getClass().getName();
	}

	/**
	 * Returns the color scheme for the icon of option panes with the specified
	 * message type.
	 * 
	 * @param messageType
	 *            Option pane message type.
	 * @param mainScheme
	 *            Main color scheme.
	 * @return Color scheme for the icon of option panes with the specified
	 *         message type.
	 */
	public static ColorScheme getOptionPaneColorScheme(int messageType,
			ColorScheme mainScheme) {
		if (!SubstanceLookAndFeel.isToUseConstantThemesOnDialogs())
			return mainScheme;

		switch (messageType) {
		case JOptionPane.INFORMATION_MESSAGE:
			return new BottleGreenColorScheme();
		case JOptionPane.QUESTION_MESSAGE:
			return new LightAquaColorScheme();
		case JOptionPane.WARNING_MESSAGE:
			return new SunsetColorScheme();
		case JOptionPane.ERROR_MESSAGE:
			return new SunfireRedColorScheme();
		}
		return null;
	}

	/**
	 * Checks whether the specified theme is dark.
	 * 
	 * @param theme
	 *            Theme.
	 * @return <code>true</code> if the specified theme is dark,
	 *         <code>false</code> otherwise.
	 */
	public static boolean isThemeDark(SubstanceTheme theme) {
		if (theme instanceof SubstanceInvertedTheme) {
			return !isThemeDark(((SubstanceInvertedTheme) theme)
					.getOriginalTheme());
		}
		if (theme instanceof SubstanceNegatedTheme) {
			return isThemeDark(((SubstanceNegatedTheme) theme)
					.getOriginalTheme());
		}
		if (theme instanceof SubstanceMixTheme) {
			SubstanceMixTheme mix = (SubstanceMixTheme) theme;
			for (SubstanceTheme mixCompTheme : mix.getOriginalThemes())
				if (isThemeDark(mixCompTheme))
					return true;
			return false;
		}
		if (theme instanceof SubstanceBlendBiTheme) {
			SubstanceBlendBiTheme blend = (SubstanceBlendBiTheme) theme;
			return isThemeDark(blend.getOriginalFirstTheme())
					|| isThemeDark(blend.getOriginalSecondTheme());
		}
		return (theme.getKind() == ThemeKind.DARK);
	}

	/**
	 * Returns the component theme.
	 * 
	 * @param component
	 *            Component.
	 * @param colorSchemeKind
	 *            Color scheme kind.
	 * @return Component theme.
	 */
	public static SubstanceTheme getComponentTheme(JComponent component,
			ComponentState.ColorSchemeKind colorSchemeKind) {
		return getComponentTheme(component, colorSchemeKind, false);

	}

	/**
	 * Returns the component theme.
	 * 
	 * @param component
	 *            Component.
	 * @param colorSchemeKind
	 *            Color scheme kind.
	 * @param checkHierarchy
	 *            If <code>true</code>, the entire component hierarchy will
	 *            be scanned for the
	 *            {@link SubstanceLookAndFeel#PAINT_ACTIVE_PROPERTY}.
	 * @return Component theme.
	 */
	public static SubstanceTheme getComponentTheme(JComponent component,
			ComponentState.ColorSchemeKind colorSchemeKind,
			boolean checkHierarchy) {
		if (colorSchemeKind == ColorSchemeKind.DISABLED) {
			return SubstanceCoreUtilities.getDisabledTheme(component, true);
		} else {
			if (colorSchemeKind == ColorSchemeKind.CURRENT) {
				return SubstanceCoreUtilities.getActiveTheme(component, true);
			} else {
				return SubstanceCoreUtilities.getDefaultTheme(component, true,
						checkHierarchy);
			}
		}
	}

	/**
	 * Returns the popup prototype display value for the specified combo box.
	 * This value is used to compute the width of the combo popup.
	 * 
	 * @param combo
	 *            Combo box.
	 * @return The popup prototype display value for the specified combo box.
	 */
	public static Object getComboPopupPrototypeDisplayValue(JComboBox combo) {
		Object objProp = combo
				.getClientProperty(SubstanceLookAndFeel.COMBO_POPUP_PROTOTYPE);
		if (objProp == null)
			objProp = UIManager.get(SubstanceLookAndFeel.COMBO_POPUP_PROTOTYPE);
		if (objProp == null)
			return null;

		if (objProp instanceof ComboPopupPrototypeCallback) {
			ComboPopupPrototypeCallback callback = (ComboPopupPrototypeCallback) objProp;
			return callback.getPopupPrototypeDisplayValue(combo);
		}

		// check if this object is in the model ???
		return objProp;
	}

	/**
	 * Returns the title painter of the specified root pane.
	 * 
	 * @param rp
	 *            Root pane.
	 * @return The title painter of the specified root pane.
	 */
	public static SubstanceTitlePainter getTitlePainter(JRootPane rp) {
		if (rp != null) {
			Object custObj = rp
					.getClientProperty(SubstanceLookAndFeel.TITLE_PAINTER_PROPERTY);
			if (custObj instanceof String) {
				String custStr = (String) custObj;
				try {
					Class<?> cl = Class.forName(custStr);
					Object custInst = cl.newInstance();
					if (custInst instanceof SubstanceTitlePainter)
						return (SubstanceTitlePainter) custInst;
				} catch (Exception exc) {
				}
			}
			if (custObj instanceof SubstanceTitlePainter) {
				return (SubstanceTitlePainter) custObj;
			}
		}
		return SubstanceLookAndFeel.getCurrentTitlePainter();
	}

	/**
	 * Returns the title painter of the specified internal frame.
	 * 
	 * @param jif
	 *            Internal frame.
	 * @return The title painter of the specified internal frame.
	 */
	public static SubstanceTitlePainter getTitlePainter(JInternalFrame jif) {
		Object custObj = jif
				.getClientProperty(SubstanceLookAndFeel.TITLE_PAINTER_PROPERTY);
		if (custObj instanceof String) {
			String custStr = (String) custObj;
			try {
				Class<?> cl = Class.forName(custStr);
				Object custInst = cl.newInstance();
				if (custInst instanceof SubstanceTitlePainter)
					return (SubstanceTitlePainter) custInst;
			} catch (Exception exc) {
			}
		}
		if (custObj instanceof SubstanceTitlePainter) {
			return (SubstanceTitlePainter) custObj;
		}
		return SubstanceLookAndFeel.getCurrentTitlePainter();
	}

	/**
	 * Returns the scroll bar buttons kind of the specified scroll bar.
	 * 
	 * @param scrollBar
	 *            Scroll bar.
	 * @return The scroll bar buttons kind of the specified scroll bar.
	 */
	public static ScrollPaneButtonPolicyKind getScrollPaneButtonsPolicyKind(
			JScrollBar scrollBar) {
		Component parent = scrollBar.getParent();
		if (parent instanceof JScrollPane) {
			Object jspKind = ((JScrollPane) parent)
					.getClientProperty(SubstanceLookAndFeel.SCROLL_PANE_BUTTONS_POLICY);
			if (jspKind instanceof ScrollPaneButtonPolicyKind)
				return (ScrollPaneButtonPolicyKind) jspKind;
		}
		Object globalJspKind = UIManager
				.get(SubstanceLookAndFeel.SCROLL_PANE_BUTTONS_POLICY);
		if (globalJspKind instanceof ScrollPaneButtonPolicyKind)
			return (ScrollPaneButtonPolicyKind) globalJspKind;
		return ScrollPaneButtonPolicyKind.OPPOSITE;
	}

	/**
	 * Returns the set of sides registered on the specified button.
	 * 
	 * @param button
	 *            Button.
	 * @param propertyName
	 *            Client property name for retrieving the registered sides.
	 * @return Set of sides registered on the specified button.
	 */
	@SuppressWarnings("unchecked")
	public static Set<Side> getSides(AbstractButton button, String propertyName) {
		Object prop = button.getClientProperty(propertyName);

		if (prop instanceof Set) {
			return (Set<Side>) prop;
		}

		Set<Side> result = new HashSet<Side>();
		if (prop != null) {
			if (prop instanceof String) {
				result.add(SubstanceConstants.Side.valueOf((String) prop));
			}
			if (prop instanceof String[]) {
				String[] clientSides = (String[]) prop;
				for (String side : clientSides) {
					result.add(SubstanceConstants.Side.valueOf(side));
				}
			}
			if (prop instanceof Side) {
				result.add((Side) prop);
			}
			if (prop instanceof Side[]) {
				Side[] clientSides = (Side[]) prop;
				for (Side side : clientSides) {
					result.add(side);
				}
			}
		}
		return result;
	}

	/**
	 * Returns the corner radius of the specified toolbar button.
	 * 
	 * @param button
	 *            Toolbar button.
	 * @param insets
	 *            Button insets.
	 * @return Corner radius of the specified toolbar button.
	 */
	public static float getToolbarButtonCornerRadius(AbstractButton button,
			Insets insets) {

		JToolBar toolbar = null;
		Component c = button.getParent();
		while (c != null) {
			if (c instanceof JToolBar) {
				toolbar = (JToolBar) c;
				break;
			}
			c = c.getParent();
		}
		if (toolbar == null)
			return 2.0f;

		int width = button.getWidth();
		int height = button.getHeight();

		if (insets != null) {
			width -= (insets.left + insets.right);
			height -= (insets.top + insets.bottom);
		}
		float maxRadius = (width > height) ? (height) / 2.0f : (width) / 2.0f;

		Object buttonProp = button
				.getClientProperty(SubstanceLookAndFeel.CORNER_RADIUS);
		if (buttonProp instanceof Float)
			return Math.min(maxRadius, ((Float) buttonProp).floatValue());

		Object toolbarProp = toolbar
				.getClientProperty(SubstanceLookAndFeel.CORNER_RADIUS);
		if (toolbarProp instanceof Float)
			return Math.min(maxRadius, ((Float) toolbarProp).floatValue());

		Object globalProp = UIManager.get(SubstanceLookAndFeel.CORNER_RADIUS);
		if (globalProp instanceof Float)
			return Math.min(maxRadius, ((Float) globalProp).floatValue());

		return 2.0f;
	}

	/**
	 * Returns the number of echo characters per each password chanaracter.
	 * 
	 * @param jpf
	 *            Password field.
	 * @return The number of echo characters per each password chanaracter.
	 */
	public static int getEchoPerChar(JPasswordField jpf) {
		Object obj = jpf
				.getClientProperty(SubstanceLookAndFeel.PASSWORD_ECHO_PER_CHAR);
		if ((obj != null) && (obj instanceof Integer)) {
			int result = (Integer) obj;
			if (result >= 1)
				return result;
		}

		obj = UIManager.get(SubstanceLookAndFeel.PASSWORD_ECHO_PER_CHAR);
		if ((obj != null) && (obj instanceof Integer)) {
			int result = (Integer) obj;
			if (result >= 1)
				return result;
		}
		return 1;
	}

	/**
	 * Creates a clip image for soft-clipping. Code taken from <a
	 * href="http://weblogs.java.net/blog/campbell/archive/2006/07/java_2d_tricker_2.html">here</a>.
	 * 
	 * @param s
	 *            Clip shape.
	 * @param width
	 *            Image width.
	 * @param height
	 *            Image height.
	 * @return Clip image.
	 * @author Chris Campbell.
	 */
	public static BufferedImage createClipImage(Shape s, int width, int height) {
		// Create a translucent intermediate image in which we can perform
		// the soft clipping
		GraphicsEnvironment e = GraphicsEnvironment
				.getLocalGraphicsEnvironment();
		GraphicsDevice d = e.getDefaultScreenDevice();
		GraphicsConfiguration c = d.getDefaultConfiguration();

		BufferedImage img = c.createCompatibleImage(width, height,
				Transparency.TRANSLUCENT);
		Graphics2D g2 = img.createGraphics();

		// Clear the image so all pixels have zero alpha
		g2.setComposite(AlphaComposite.Clear);
		g2.fillRect(0, 0, width, height);

		// Render our clip shape into the image. Note that we enable
		// antialiasing to achieve the soft clipping effect. Try
		// commenting out the line that enables antialiasing, and
		// you will see that you end up with the usual hard clipping.
		g2.setComposite(AlphaComposite.Src);
		g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);
		g2.setColor(Color.WHITE);
		g2.fill(s);
		g2.dispose();

		return img;
	}

	/**
	 * Returns the color of mark icons (checkbox, radio button, scrollbar
	 * arrows, combo arrows, menu arrows etc) for the specified theme.
	 * 
	 * @param theme
	 *            Theme.
	 * @param isEnabled
	 *            If <code>true</code>, the mark should be painted in enabled
	 *            state.
	 * @return Color of mark icons.
	 */
	public static Color getMarkColor(SubstanceTheme theme, boolean isEnabled) {
		ColorScheme scheme = theme.getColorScheme();
		if (SubstanceCoreUtilities.isThemeDark(theme)) {
			return SubstanceColorUtilities.getInterpolatedColor(scheme
					.getForegroundColor(), scheme.getUltraLightColor(), 0.9);
		} else {
			Color color1 = isEnabled ? scheme.getUltraDarkColor() : scheme
					.getUltraDarkColor();
			Color color2 = isEnabled ? scheme.getDarkColor() : scheme
					.getLightColor();
			return SubstanceColorUtilities.getInterpolatedColor(color1, color2,
					0.9);
		}
	}

	/**
	 * Checks whether the specified component has overlay enabled.
	 * 
	 * @param component
	 *            Component.
	 * @return <code>true</code> if the specified component has overlay
	 *         enabled, <code>false</code> otherwise.
	 */
	public static boolean hasOverlayProperty(Component component) {
		if (component instanceof JComponent) {
			Object flatProperty = ((JComponent) component)
					.getClientProperty(SubstanceLookAndFeel.OVERLAY_PROPERTY);
			if (Boolean.TRUE.equals(flatProperty))
				return true;
			if (Boolean.FALSE.equals(flatProperty))
				return false;
		}

		return Boolean.TRUE.equals(UIManager
				.get(SubstanceLookAndFeel.OVERLAY_PROPERTY));
	}

	/**
	 * Checks whether the specified component has extra Substance-specific UI
	 * elements.
	 * 
	 * @param component
	 *            Component.
	 * @return <code>true</code> if the specified component has extra
	 *         Substance-specific UI elements, <code>false</code> otherwise.
	 */
	public static boolean toShowExtraElements(Component component) {
		Component c = component;
		while (c != null) {
			if (c instanceof JComponent) {
				JComponent jcomp = (JComponent) c;
				Object componentProp = jcomp
						.getClientProperty(SubstanceLookAndFeel.NO_EXTRA_ELEMENTS);
				if (componentProp != null) {
					if (Boolean.TRUE.equals(componentProp))
						return false;
					if (Boolean.FALSE.equals(componentProp))
						return true;
				}
			}
			c = c.getParent();
		}
		return SubstanceLookAndFeel.toShowExtraElements();
	}

	/**
	 * Returns the grip painter for the specified component.
	 * 
	 * @param component
	 *            Component.
	 * @param defaultPainter
	 *            Default painter to use if no custom grip painter is specified.
	 * @return The grip painter for the specified component.
	 */
	public static GripPainter getGripPainter(JComponent component,
			GripPainter defaultPainter) {
		Component c = component;
		while (c != null) {
			if (c instanceof JComponent) {
				JComponent jcomp = (JComponent) c;
				Object jcompSetting = jcomp
						.getClientProperty(SubstanceLookAndFeel.GRIP_PAINTER);
				if (jcompSetting instanceof GripPainter)
					return (GripPainter) jcompSetting;
			}
			c = c.getParent();
		}
		Object globalSetting = UIManager.get(SubstanceLookAndFeel.GRIP_PAINTER);
		if (globalSetting instanceof GripPainter)
			return (GripPainter) globalSetting;
		return defaultPainter;
	}

	/**
	 * Returns indication whether the specified component's border is a
	 * Substance-specific border.
	 * 
	 * @param c
	 *            Component.
	 * @return <code>true</code> if the specified component's border is a
	 *         Substance-specific border, <code>false</code> otherwise.
	 */
	public static boolean hasSubstanceBorder(JComponent c) {
		if (c == null)
			return false;

		Border border = c.getBorder();
		if (border instanceof SubstanceBorder)
			return true;

		if (border instanceof CompoundBorder) {
			CompoundBorder cb = (CompoundBorder) border;
			if (cb.getOutsideBorder() instanceof SubstanceBorder)
				return true;
		}

		return false;
	}

	public static Icon getActiveIcon(Icon origIcon, JComponent comp,
			ButtonModel model, Icon glowingIcon, boolean ignoreRolloverSetting) {
		if (origIcon == null)
			return null;
		Icon result = origIcon;
		if (SubstanceCoreUtilities.useThemedDefaultIcon(comp)) {
			boolean useThemedVersion = !model.isArmed() && !model.isPressed()
					&& !model.isSelected();
			if (!ignoreRolloverSetting)
				useThemedVersion = useThemedVersion && !model.isRollover();
			if (useThemedVersion) {
				result = new ImageIcon(SubstanceImageCreator.getThemeImage(
						comp, result, SubstanceCoreUtilities.getDefaultTheme(
								comp, true), false));
			}
		}
		if (FadeConfigurationManager.getInstance().fadeAllowed(
				FadeKind.ICON_GLOW, comp)
				&& model.isRollover() && (glowingIcon != null)) {
			result = glowingIcon;
		}
		return result;
	}

	/**
	 * Returns the current icon for the specified button. This method is <b>for
	 * internal use only</b>.
	 * 
	 * @param b
	 *            Button.
	 * @param glowingIcon
	 *            The glowing icon.
	 * @param ignoreRolloverSetting
	 *            If <code>true</code>, the rollover status of the specified
	 *            button is ignored.
	 * @return Icon for the specified button.
	 */
	public static Icon getIcon(AbstractButton b, Icon glowingIcon,
			boolean ignoreRolloverSetting) {
		ButtonModel model = b.getModel();
		Icon icon = getActiveIcon(b.getIcon(), b, model, glowingIcon,
				ignoreRolloverSetting);
		if (icon == null)
			return null;

		Icon tmpIcon = null;

		if (icon != null) {
			if (!model.isEnabled()) {
				if (model.isSelected()) {
					tmpIcon = b.getDisabledSelectedIcon();
				} else {
					tmpIcon = b.getDisabledIcon();
				}
			} else if (model.isPressed() && model.isArmed()) {
				tmpIcon = b.getPressedIcon();
			} else if (b.isRolloverEnabled() && model.isRollover()) {
				if (model.isSelected()) {
					tmpIcon = b.getRolloverSelectedIcon();
				} else {
					tmpIcon = b.getRolloverIcon();
				}
			} else if (model.isSelected()) {
				tmpIcon = b.getSelectedIcon();
			}

			if (tmpIcon != null) {
				icon = tmpIcon;
			}
		}
		return icon;
	}

	/**
	 * Returns the global menu gutter fill kind.
	 * 
	 * @return The global menu gutter fill kind.
	 */
	public static MenuGutterFillKind getMenuGutterFillKind() {
		Object globalSetting = UIManager
				.get(SubstanceLookAndFeel.MENU_GUTTER_FILL_KIND);
		if (globalSetting instanceof MenuGutterFillKind)
			return (MenuGutterFillKind) globalSetting;
		return MenuGutterFillKind.HARD;
	}

	/**
	 * Given a component, returns the parent for computing the
	 * {@link SubstanceHeaderPainter}.
	 * 
	 * @param c
	 *            Component.
	 * @return The parent for computing the {@link SubstanceHeaderPainter}.
	 */
	public static Container getHeaderParent(Component c) {
		Component comp = c.getParent();
		Container result = null;
		while (comp != null) {
			// the second part fixes the incorrect alignments on
			// internal frames.
			if ((comp instanceof JLayeredPane) && (result == null))
				result = (Container) comp;
			if ((result == null) && (comp instanceof Window))
				result = (Container) comp;
			comp = comp.getParent();
		}
		return result;
		//		
		// while (comp != null) {
		// if (comp instanceof JToolBar)
		// return (JToolBar) comp;
		// if (comp instanceof JMenuBar)
		// return (JMenuBar) comp;
		// comp = comp.getParent();
		// }
		// return c.getParent();
	}

	/**
	 * Checks whether the specified component is inside a container painted by
	 * {@link SubstanceHeaderPainter}.
	 * 
	 * @param c
	 *            Component.
	 * @return <code>true</code>, if the specified component is inside a
	 *         container painted by {@link SubstanceHeaderPainter},
	 *         <code>false</code> otherwise.
	 */
	public static boolean isInHeader(Component c) {
		Component comp = c.getParent();
		while (comp != null) {
			if (comp instanceof JToolBar)
				return true;
			if (comp instanceof JMenuBar)
				return true;
			comp = comp.getParent();
		}

		return false;
	}

	/**
	 * Paints the text with the specified parameters.
	 * 
	 * @param g
	 *            Graphics context.
	 * @param button
	 *            Button.
	 * @param textRect
	 *            Text rectangle.
	 * @param text
	 *            Text to paint.
	 * @param textShiftOffset
	 *            Text shift offset.
	 * @param propertyPrefix
	 *            Property prefix - used for computing the color of text on
	 *            disabled buttons.
	 * 
	 */
	public static void paintText(Graphics g, AbstractButton button,
			Rectangle textRect, String text, int textShiftOffset,
			String propertyPrefix) {
		ButtonModel model = button.getModel();
		FontMetrics fm = g.getFontMetrics();
		int mnemonicIndex = button.getDisplayedMnemonicIndex();

		ComponentState state = ComponentState.getState(model, button);
		ComponentState prevState = SubstanceCoreUtilities
				.getPrevComponentState(button);

		// special case for enabled buttons with no background -
		// always use the theme for the default state.
		if (SubstanceCoreUtilities.isButtonNeverPainted(button)
				|| !button.isContentAreaFilled()
				|| (button instanceof JRadioButton)
				|| (button instanceof JCheckBox)) {
			if (state.isKindActive(FadeKind.ENABLE)) {
				state = ComponentState.DEFAULT;
				prevState = ComponentState.DEFAULT;
			}
		}

		// compute color scheme
		SubstanceTheme theme = SubstanceCoreUtilities.getTheme(button, state,
				true, true);

		Color fgColor = button.getForeground();
		if (fgColor instanceof UIResource) {
			fgColor = SubstanceCoreUtilities.getInterpolatedForegroundColor(
					button, null, theme, state, prevState, FadeKind.ROLLOVER,
					FadeKind.SELECTION, FadeKind.PRESS);
		}

		Graphics2D graphics = (Graphics2D) g.create();
		graphics.setComposite(TransitionLayout.getAlphaComposite(button,
				SubstanceCoreUtilities.getTheme(button, true).getThemeAlpha(
						button, state), g));
		graphics.setColor(fgColor);

		SubstanceCoreUtilities.workaroundBug6576507(graphics);

		BasicGraphicsUtils.drawStringUnderlineCharAt(graphics, text,
				mnemonicIndex, textRect.x + textShiftOffset, textRect.y
						+ fm.getAscent() + textShiftOffset);
		graphics.dispose();
	}

	/**
	 * Paints menu text.
	 * 
	 * @param menuItem
	 *            Menu item.
	 * @param g
	 *            Graphics context.
	 * @param text
	 *            Text to paint.
	 * @param textRect
	 *            Text rectangle.
	 */
	public static void paintMenuText(JMenuItem menuItem, Graphics g,
			String text, Rectangle textRect) {
		// ButtonModel model = menuItem.getModel();
		FontMetrics fm = g.getFontMetrics();
		int mnemIndex = menuItem.getDisplayedMnemonicIndex();

		ComponentState state = ComponentState.getState(menuItem.getModel(),
				menuItem, true);
		Graphics2D graphics = (Graphics2D) g.create();
		graphics.setComposite(TransitionLayout.getAlphaComposite(menuItem,
				SubstanceCoreUtilities.getTheme(menuItem, true).getThemeAlpha(
						menuItem, state), g));

		SubstanceCoreUtilities.workaroundBug6576507(graphics);

		// System.out.println(menuItem.getText() + "->" + graphics.getFont());
		if (!state.isKindActive(FadeKind.ENABLE)) {
			graphics
					.setColor(UIManager.getColor("MenuItem.disabledForeground"));
			BasicGraphicsUtils.drawStringUnderlineCharAt(graphics, text,
					mnemIndex, textRect.x, textRect.y + fm.getAscent());
		} else {
			// Ignore the selection state of the menu item. This is especially
			// relevant for dark themes.
			ComponentState prevState = SubstanceCoreUtilities
					.getPrevComponentState(menuItem);
			// fix for defect 232 - respect the application menu color
			Color fg = menuItem.getForeground();
			if (fg instanceof UIResource) {
				fg = SubstanceCoreUtilities.getInterpolatedForegroundColor(
						menuItem, null, SubstanceCoreUtilities
								.getHighlightTheme(menuItem, menuItem
										.getParent(), state, true), state,
						prevState, FadeKind.ROLLOVER, FadeKind.PRESS,
						FadeKind.ARM, FadeKind.SELECTION);
			}

			// System.out.println(menuItem.getText() + ":" + prevState + "-"
			// + state + " : " + fg);

			// if (menuItem.getParent() instanceof JMenuBar) {
			// fg = UIManager.getColor("MenuBar.foreground");
			// }
			graphics.setColor(fg);
			BasicGraphicsUtils.drawStringUnderlineCharAt(graphics, text,
					mnemIndex, textRect.x, textRect.y + fm.getAscent());
		}
		graphics.dispose();
	}

	/**
	 * Paints the focus ring on the specified component.
	 * 
	 * @param g
	 *            Graphics context.
	 * @param mainComp
	 *            The main component for the focus painting.
	 * @param focusedComp
	 *            The actual component that has the focus. For example, the main
	 *            component can be a {@link JSpinner}, while the focused
	 *            component is a text field inside the the spinner editor.
	 * @param focusRect
	 *            Focus rectangle. May be <code>null</code> - in this case,
	 *            the bounds of <code>mainComp</code> will be used.
	 * @param textRect
	 *            Text rectangle (if relevant).
	 * @param maxAlphaCoef
	 *            Maximum alhpa coefficient for painting the focus. Values lower
	 *            than 1.0 will result in a translucent focus ring (can be used
	 *            to paint a focus ring that doesn't draw too much attention
	 *            away from the content, for example on text components).
	 * @param extraPadding
	 *            Extra padding between the component bounds and the focus ring
	 *            painting.
	 */
	public static void paintFocus(Graphics g, Component mainComp,
			Component focusedComp, Rectangle focusRect, Rectangle textRect,
			float maxAlphaCoef, int extraPadding) {
		FadeTracker fadeTracker = FadeTracker.getInstance();
		FocusKind focusKind = SubstanceCoreUtilities.getFocusKind(mainComp);
		if ((focusKind == FocusKind.NONE)
				&& (!fadeTracker.isTracked(focusedComp, FadeKind.FOCUS)))
			return;

		Graphics2D graphics = (Graphics2D) g.create();
		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		float alpha = 1.0f;
		if (fadeTracker.isTracked(focusedComp, FadeKind.FOCUS)) {
			alpha = fadeTracker.getFade10(focusedComp, FadeKind.FOCUS) / 10.f;
		}
		alpha *= maxAlphaCoef;
		graphics.setComposite(TransitionLayout.getAlphaComposite(mainComp,
				alpha, g));

		Color color = SubstanceColorUtilities.getFocusColor(mainComp);
		graphics.setColor(color);
		focusKind.paintFocus(mainComp, focusedComp, graphics, focusRect,
				textRect, extraPadding);
		graphics.dispose();
	}

	/**
	 * Paints the focus ring on the specified component.
	 * 
	 * @param g
	 *            Graphics context.
	 * @param mainComp
	 *            The main component for the focus painting.
	 * @param focusedComp
	 *            The actual component that has the focus. For example, the main
	 *            component can be a {@link JSpinner}, while the focused
	 *            component is a text field inside the the spinner editor.
	 * @param textRect
	 *            Text rectangle (if relevant).
	 */
	public static void paintFocus(Graphics g, Component mainComp,
			Component focusedComp, Rectangle textRect) {
		paintFocus(g, mainComp, focusedComp, null, textRect, 1.0f, 0);
	}

	/**
	 * Paints a separator.
	 * 
	 * @param c
	 *            Component.
	 * @param graphics
	 *            Graphics context.
	 * @param colorScheme
	 *            Color scheme.
	 * @param isDark
	 *            Indication whether the color scheme is dark.
	 * @param width
	 *            Separator width.
	 * @param height
	 *            Separator height.
	 * @param orientation
	 *            Separator orientation.
	 */
	public static void paintSeparator(Component c, Graphics2D graphics,
			ColorScheme colorScheme, boolean isDark, int width, int height,
			int orientation) {
		SubstanceCoreUtilities.paintSeparator(c, graphics, colorScheme, isDark,
				width, height, orientation, true, 10);
	}

	/**
	 * Paints a separator.
	 * 
	 * @param c
	 *            Component.
	 * @param graphics
	 *            Graphics context.
	 * @param colorScheme
	 *            Color scheme.
	 * @param isDark
	 *            Indication whether the color scheme is dark.
	 * @param width
	 *            Separator width.
	 * @param height
	 *            Separator height.
	 * @param orientation
	 *            Separator orientation.
	 * @param hasShadow
	 *            If <code>true</code>, the separator painting will have
	 *            shadow.
	 * @param maxGradLength
	 *            Specifies the maximum pixel length of "ramp" portions of the
	 *            separator. The ramp portions are located on separator ends and
	 *            allow providing a faded appearance on those ends.
	 */
	public static void paintSeparator(Component c, Graphics2D graphics,
			ColorScheme colorScheme, boolean isDark, int width, int height,
			int orientation, boolean hasShadow, int maxGradLength) {
		paintSeparator(c, graphics, colorScheme, isDark, width, height,
				orientation, hasShadow, maxGradLength, maxGradLength);
	}

	/**
	 * Paints a separator.
	 * 
	 * @param c
	 *            Component.
	 * @param graphics
	 *            Graphics context.
	 * @param colorScheme
	 *            Color scheme.
	 * @param isDark
	 *            Indication whether the color scheme is dark.
	 * @param width
	 *            Separator width.
	 * @param height
	 *            Separator height.
	 * @param orientation
	 *            Separator orientation.
	 * @param hasShadow
	 *            If <code>true</code>, the separator painting will have
	 *            shadow.
	 * @param maxGradLengthStart
	 *            Specifies the maximum pixel length of the starting "ramp"
	 *            portion of the separator. The starting ramp portion is located
	 *            on top / left separator end and allows providing a faded
	 *            appearance on that end.
	 * @param maxGradLengthEnd
	 *            Specifies the maximum pixel length of the ending "ramp"
	 *            portion of the separator. The ending ramp portion is located
	 *            on bottom / right separator end and allows providing a faded
	 *            appearance on that end.
	 */
	public static void paintSeparator(Component c, Graphics2D graphics,
			ColorScheme colorScheme, boolean isDark, int width, int height,
			int orientation, boolean hasShadow, int maxGradLengthStart,
			int maxGradLengthEnd) {
		Color foreLight = isDark ? colorScheme.getLightColor()
				: SubstanceColorUtilities.getInterpolatedColor(colorScheme
						.getLightColor(), colorScheme.getDarkColor(), 0.8);
		Color foreDark = isDark ? colorScheme.getExtraLightColor()
				: SubstanceColorUtilities.getInterpolatedColor(colorScheme
						.getMidColor(), colorScheme.getDarkColor(), 0.4);
		Color back = isDark ? colorScheme.getDarkColor() : colorScheme
				.getUltraLightColor();

		float borderStrokeWidth = SubstanceSizeUtils
				.getBorderStrokeWidth(SubstanceSizeUtils
						.getComponentFontSize(c));
		graphics.setStroke(new BasicStroke(borderStrokeWidth,
				BasicStroke.CAP_BUTT, BasicStroke.JOIN_ROUND));
		if (orientation == JSeparator.VERTICAL) {
			int gradStart = Math.min(maxGradLengthStart, height / 2);
			int gradEnd = Math.min(maxGradLengthEnd, height / 2);
			graphics.translate(Math.max(0, width / 2 - 1), 0);
			graphics.setPaint(new GradientPaint(0, 0, SubstanceColorUtilities
					.getAlphaColor(foreLight, 32), 0, gradStart,
					SubstanceColorUtilities.getAlphaColor(foreDark, 240)));
			graphics.drawLine(0, 0, 0, gradStart);
			if (height / 2 > gradStart) {
				graphics.setColor(SubstanceColorUtilities.getAlphaColor(
						foreDark, 240));
				graphics.drawLine(0, gradStart, 0, height - gradEnd);
			}
			graphics.setPaint(new GradientPaint(0, height - gradEnd,
					SubstanceColorUtilities.getAlphaColor(foreDark, 240), 0,
					height, SubstanceColorUtilities
							.getAlphaColor(foreLight, 32)));
			graphics.drawLine(0, height - gradEnd, 0, height);

			if (hasShadow) {
				int offset = (int) borderStrokeWidth;
				graphics.setPaint(new GradientPaint(offset, 0,
						SubstanceColorUtilities.getAlphaColor(back, 32),
						offset, gradStart, SubstanceColorUtilities
								.getAlphaColor(back, 196)));
				graphics.drawLine(offset, 0, offset, gradStart);
				if (height / 2 > gradStart) {
					graphics.setColor(SubstanceColorUtilities.getAlphaColor(
							back, 196));
					graphics.drawLine(offset, gradStart, offset, height
							- gradEnd);
				}
				graphics.setPaint(new GradientPaint(offset, height - gradEnd,
						SubstanceColorUtilities.getAlphaColor(back, 196),
						offset, height, SubstanceColorUtilities.getAlphaColor(
								back, 32)));
				graphics.drawLine(offset, height - gradEnd, offset, height);
			}
		} else {
			// HORIZONTAL
			int gradStart = Math.min(maxGradLengthStart, width / 2);
			int gradEnd = Math.min(maxGradLengthEnd, width / 2);
			graphics.translate(0, Math.max(0, height / 2 - 1));
			graphics.setPaint(new GradientPaint(0, 0, SubstanceColorUtilities
					.getAlphaColor(foreLight, 32), gradStart, 0,
					SubstanceColorUtilities.getAlphaColor(foreDark, 240)));
			graphics.drawLine(0, 0, gradStart, 0);
			// if (width / 2 > gradStart) {
			graphics.setColor(SubstanceColorUtilities.getAlphaColor(foreDark,
					240));
			graphics.drawLine(gradStart, 0, width - gradEnd, 0);
			// }
			graphics.setPaint(new GradientPaint(width - gradEnd, 0,
					SubstanceColorUtilities.getAlphaColor(foreDark, 240),
					width, 0, SubstanceColorUtilities.getAlphaColor(foreLight,
							32)));
			graphics.drawLine(width - gradEnd, 0, width, 0);

			if (hasShadow) {
				int offset = (int) borderStrokeWidth;
				graphics.setPaint(new GradientPaint(0, offset,
						SubstanceColorUtilities.getAlphaColor(back, 32),
						gradStart, offset, SubstanceColorUtilities
								.getAlphaColor(back, 196)));
				graphics.drawLine(0, offset, gradStart, offset);
				// if (width / 2 > gradStart) {
				graphics.setColor(SubstanceColorUtilities.getAlphaColor(back,
						196));
				graphics.drawLine(gradStart, offset, width - gradEnd, offset);
				// }
				graphics.setPaint(new GradientPaint(width - gradEnd, offset,
						SubstanceColorUtilities.getAlphaColor(back, 196),
						width, offset, SubstanceColorUtilities.getAlphaColor(
								back, 32)));
				graphics.drawLine(width - gradEnd, offset, width, offset);
			}
		}

	}

	/**
	 * Blur the provided source image into a new image. <i>Note that the
	 * resulting image will be bigger than the original one.</i>
	 * 
	 * @param src
	 *            source image
	 * @param shadowSize
	 *            blur size
	 * @return an image containing the blurred source image
	 */
	public static BufferedImage blur(BufferedImage src, int shadowSize) {
		int srcWidth = src.getWidth();
		int srcHeight = src.getHeight();

		int dstWidth = srcWidth + shadowSize;
		int dstHeight = srcHeight + shadowSize;

		BufferedImage dst = new BufferedImage(dstWidth, dstHeight,
				BufferedImage.TYPE_INT_ARGB);

		int left = (shadowSize - 1) >> 1;
		int right = shadowSize - left;
		int xStop = dstWidth - right;
		int yStop = dstHeight - right;

		Graphics2D g = dst.createGraphics();
		g.drawImage(src, left, left, null);
		g.dispose();

		int[] aHistory = new int[shadowSize];
		int[] rHistory = new int[shadowSize];
		int[] vHistory = new int[shadowSize];
		int[] bHistory = new int[shadowSize];
		int historyIdx;

		int aSum;
		int rSum;
		int vSum;
		int bSum;

		float hSumDivider = 1.0f / shadowSize;
		float vSumDivider = 1.0f / shadowSize;

		// horizontal pass
		for (int y = left; y < yStop; y++) {

			aSum = 0;
			rSum = 0;
			vSum = 0;
			bSum = 0;

			for (historyIdx = 0; historyIdx < left; historyIdx++) {
				aHistory[historyIdx] = 0;
				rHistory[historyIdx] = 0;
				vHistory[historyIdx] = 0;
				bHistory[historyIdx] = 0;
			}

			for (int x = 0; x < right; x++, historyIdx++) {
				int argb = dst.getRGB(x, y);
				int a = argb >>> 24;
				int r = (argb & 0x00FF0000) >>> 16;
				int v = (argb & 0x0000FF00) >>> 8;
				int b = (argb & 0x000000FF);
				aHistory[historyIdx] = a;
				rHistory[historyIdx] = r;
				vHistory[historyIdx] = v;
				bHistory[historyIdx] = b;
				aSum += a;
				rSum += r;
				vSum += v;
				bSum += b;
			}

			historyIdx = 0;

			for (int x = 0; x < xStop; x++) {
				int a = (int) (aSum * hSumDivider);
				int r = (int) (rSum * hSumDivider);
				int v = (int) (vSum * hSumDivider);
				int b = (int) (bSum * hSumDivider);
				dst.setRGB(x, y, a << 24 | r << 16 | v << 8 | b);

				// substract the oldest pixel from the sum
				aSum -= aHistory[historyIdx];
				rSum -= rHistory[historyIdx];
				vSum -= vHistory[historyIdx];
				bSum -= bHistory[historyIdx];

				// get the lastest pixel
				int argb = dst.getRGB(x + right, y);
				a = argb >>> 24;
				r = (argb & 0x00FF0000) >>> 16;
				v = (argb & 0x0000FF00) >>> 8;
				b = (argb & 0x000000FF);
				aHistory[historyIdx] = a;
				rHistory[historyIdx] = r;
				vHistory[historyIdx] = v;
				bHistory[historyIdx] = b;
				aSum += a;
				rSum += r;
				vSum += v;
				bSum += b;

				if (++historyIdx >= shadowSize)
					historyIdx -= shadowSize;
			}

			for (int x = xStop; x < dstWidth; x++) {
				int a = (int) (aSum * hSumDivider);
				int r = (int) (rSum * hSumDivider);
				int v = (int) (vSum * hSumDivider);
				int b = (int) (bSum * hSumDivider);
				dst.setRGB(x, y, a << 24 | r << 16 | v << 8 | b);

				// substract the oldest pixel from the sum
				aSum -= aHistory[historyIdx];
				rSum -= rHistory[historyIdx];
				vSum -= vHistory[historyIdx];
				bSum -= bHistory[historyIdx];

				if (++historyIdx >= shadowSize)
					historyIdx -= shadowSize;
			}

		}

		// vertical pass
		for (int x = 0; x < dstWidth; x++) {

			aSum = 0;
			rSum = 0;
			vSum = 0;
			bSum = 0;

			for (historyIdx = 0; historyIdx < left; historyIdx++) {
				aHistory[historyIdx] = 0;
				rHistory[historyIdx] = 0;
				vHistory[historyIdx] = 0;
				bHistory[historyIdx] = 0;
			}

			for (int y = 0; y < right; y++, historyIdx++) {
				int argb = dst.getRGB(x, y);
				int a = argb >>> 24;
				int r = (argb & 0x00FF0000) >>> 16;
				int v = (argb & 0x0000FF00) >>> 8;
				int b = (argb & 0x000000FF);
				aHistory[historyIdx] = a;
				rHistory[historyIdx] = r;
				vHistory[historyIdx] = v;
				bHistory[historyIdx] = b;
				aSum += a;
				rSum += r;
				vSum += v;
				bSum += b;
			}

			historyIdx = 0;

			for (int y = 0; y < yStop; y++) {
				int a = (int) (aSum * vSumDivider);
				int r = (int) (rSum * vSumDivider);
				int v = (int) (vSum * vSumDivider);
				int b = (int) (bSum * vSumDivider);
				dst.setRGB(x, y, a << 24 | r << 16 | v << 8 | b);

				// substract the oldest pixel from the sum
				aSum -= aHistory[historyIdx];
				rSum -= rHistory[historyIdx];
				vSum -= vHistory[historyIdx];
				bSum -= bHistory[historyIdx];

				// get the lastest pixel
				int argb = dst.getRGB(x, y + right);
				a = argb >>> 24;
				r = (argb & 0x00FF0000) >>> 16;
				v = (argb & 0x0000FF00) >>> 8;
				b = (argb & 0x000000FF);
				aHistory[historyIdx] = a;
				rHistory[historyIdx] = r;
				vHistory[historyIdx] = v;
				bHistory[historyIdx] = b;
				aSum += a;
				rSum += r;
				vSum += v;
				bSum += b;

				if (++historyIdx >= shadowSize)
					historyIdx -= shadowSize;
			}

			for (int y = yStop; y < dstHeight; y++) {
				int a = (int) (aSum * vSumDivider);
				int r = (int) (rSum * vSumDivider);
				int v = (int) (vSum * vSumDivider);
				int b = (int) (bSum * vSumDivider);
				dst.setRGB(x, y, a << 24 | r << 16 | v << 8 | b);

				// substract the oldest pixel from the sum
				aSum -= aHistory[historyIdx];
				rSum -= rHistory[historyIdx];
				vSum -= vHistory[historyIdx];
				bSum -= bHistory[historyIdx];

				if (++historyIdx >= shadowSize)
					historyIdx -= shadowSize;
			}
		}

		return dst;
	}

	/**
	 * Returns indication whether the specified button is a close button on some
	 * title pane.
	 * 
	 * @param ab
	 *            Button.
	 * @return <code>true</code> if the specified button is a close button on
	 *         some title pane, <code>false</code> otherwise.
	 */
	public static boolean isTitleCloseButton(AbstractButton ab) {
		if ((ab instanceof SubstanceTitleButton)
				&& Boolean.TRUE
						.equals(ab
								.getClientProperty(SubstanceButtonUI.IS_TITLE_CLOSE_BUTTON)))
			return true;
		return false;
	}

	/**
	 * Uninstalls the specified menu item.
	 * 
	 * @param menuItem
	 *            Menu item.
	 */
	public static void uninstallMenu(JMenuItem menuItem) {
		if (menuItem instanceof JMenu) {
			JMenu menu = (JMenu) menuItem;
			for (Component comp : menu.getMenuComponents())
				if (comp instanceof JMenuItem)
					SubstanceCoreUtilities.uninstallMenu((JMenuItem) comp);
		}

		ButtonUI menuItemUI = menuItem.getUI();
		if (menuItemUI instanceof SubstanceMenu) {
			SubstanceMenu sMenu = (SubstanceMenu) menuItemUI;
			if (sMenu.getAssociatedMenuItem() != null) {
				menuItemUI.uninstallUI(menuItem);
			}
		}

		for (ActionListener actionListener : menuItem.getActionListeners())
			menuItem.removeActionListener(actionListener);

		menuItem.removeAll();
	}

	/**
	 * Returns an icon pointed to by the specified string.
	 * 
	 * @param iconResource
	 *            Resource location string.
	 * @return Icon.
	 */
	public static Icon getIcon(String iconResource) {
		ClassLoader cl = getClassLoaderForResources();
		URL iconUrl = cl.getResource(iconResource);
		if (iconUrl == null)
			return null;
		return new IconUIResource(new ImageIcon(iconUrl));
	}

	public static ClassLoader getClassLoaderForResources() {
		// the following is fix by Dag Joar and Christian Schlichtherle
		// for application running with -Xbootclasspath VM flag. In this case,
		// the using MyClass.class.getClassLoader() would return null,
		// but the context class loader will function properly that classes will
		// be properly loaded regardless of whether the lib is added to the
		// system class path, the extension class path and regardless of the
		// class loader architecture set up by some frameworks.
		ClassLoader cl = (ClassLoader) UIManager.get("ClassLoader");
		if (cl == null)
			cl = Thread.currentThread().getContextClassLoader();
		return cl;
	}

	/**
	 * Returns the fade callback for the specified button.
	 * 
	 * @param button
	 *            Button.
	 * @return Fade callback for the specified button.
	 */
	public static FadeTrackerCallback getFadeCallback(
			final AbstractButton button) {
		return getFadeCallback(button, button.getModel(), false);
	}

	/**
	 * Returns the fade callback for the specified component.
	 * 
	 * @param component
	 *            Component.
	 * @param model
	 *            Model for tracking the transitions. For button components,
	 *            pass the {@link AbstractButton#getModel()}, for other
	 *            controls pass a dummy (synthesized) model.
	 * @param toIgnoreSelection
	 *            If <code>true</code>, the {@link ButtonModel#isSelected()}
	 *            will not be checked. This can be used for tracking transitions
	 *            on menu items that use <code>armed</code> state instead,
	 *            when we don't want to use different rollover themes for
	 *            selected and unselected checkbox and radio button menu items
	 *            (to preserve consistent visual appearence of highlights).
	 * @return Fade callback for the specified component.
	 */
	public static FadeTrackerCallback getFadeCallback(
			final JComponent component, final ButtonModel model,
			final boolean toIgnoreSelection) {
		if (isScrollButton(component))
			return null;
		FadeTrackerCallback callback = new FadeTrackerAdapter() {
			@Override
			public void fadeReversed(FadeKind fadeKind, boolean isFadingIn,
					float fadeCycle10) {
				component
						.putClientProperty(
								SubstanceCoreUtilities.PREV_COMPONENT_STATE,
								component
										.getClientProperty(SubstanceCoreUtilities.NEXT_COMPONENT_STATE));
				component
						.putClientProperty(
								SubstanceCoreUtilities.PREV_SEL_COMPONENT_STATE,
								component
										.getClientProperty(SubstanceCoreUtilities.NEXT_SEL_COMPONENT_STATE));
				// String text = (component instanceof AbstractButton) ?
				// ((AbstractButton)
				// component)
				// .getText()
				// : "";
				// System.out.println(component.getClass().getSimpleName()
				// + "["
				// + text
				// + "]"
				// + " : "
				// + fadeKind.toString()
				// + " - tracking prev as "
				// + ((ComponentState) component
				// .getClientProperty(PREV_COMPONENT_STATE))
				// .name());
				component.repaint();
			}

			/*
			 * (non-Javadoc)
			 * 
			 * @see org.jvnet.lafwidget.animation.FadeTrackerAdapter#fadeEnded(org.jvnet.lafwidget.animation.FadeKind)
			 */
			@Override
			public void fadeEnded(FadeKind fadeKind) {
				// System.out.println(toIgnoreSelection);
				component.putClientProperty(
						SubstanceCoreUtilities.PREV_COMPONENT_STATE,
						ComponentState.getState(model, component,
								toIgnoreSelection));
				component.putClientProperty(
						SubstanceCoreUtilities.PREV_SEL_COMPONENT_STATE,
						ComponentState.getState(model, component, false));
				component.putClientProperty(
						SubstanceCoreUtilities.NEXT_COMPONENT_STATE, null);
				component.putClientProperty(
						SubstanceCoreUtilities.NEXT_SEL_COMPONENT_STATE, null);
				// String text = (component instanceof AbstractButton) ?
				// ((AbstractButton)
				// component)
				// .getText()
				// : "";
				// ComponentState prevState = (ComponentState) component
				// .getClientProperty(PREV_COMPONENT_STATE);
				// System.out.println(component.getClass().getSimpleName() + "["
				// + text + "]" + " : " + fadeKind.toString()
				// + " - tracking prev as " + prevState.name());
				component.repaint();
			}

			@Override
			public void fadePerformed(FadeKind fadeKind, float fadeCycle10) {
				component.putClientProperty(
						SubstanceCoreUtilities.NEXT_COMPONENT_STATE,
						ComponentState.getState(model, component,
								toIgnoreSelection));
				component.putClientProperty(
						SubstanceCoreUtilities.NEXT_SEL_COMPONENT_STATE,
						ComponentState.getState(model, component, false));
				component.repaint();
			}
		};
		return callback;
	}

	/**
	 * Returns the previous state of the specified component.
	 * 
	 * @param comp
	 *            Component.
	 * @return The previous state of the specified component.
	 */
	public static ComponentState getPrevComponentState(JComponent comp) {
		ComponentState result = (ComponentState) comp
				.getClientProperty(PREV_COMPONENT_STATE);
		if (result == null) {
			result = ComponentState.DEFAULT;
		}
		return result;
	}

	/**
	 * Returns the previous state of the specified component.
	 * 
	 * @param comp
	 *            Component.
	 * @return The previous state of the specified component.
	 */
	public static ComponentState getPrevSelComponentState(JComponent comp) {
		ComponentState result = (ComponentState) comp
				.getClientProperty(PREV_SEL_COMPONENT_STATE);
		if (result == null) {
			result = ComponentState.DEFAULT;
		}
		return result;
	}

	/**
	 * Checks whether a component has the specified client property set to the
	 * specified value.
	 * 
	 * @param comp
	 *            Component.
	 * @param propertyName
	 *            Client property name.
	 * @param expectedValue
	 *            Expected value.
	 * @param checkHierarchy
	 *            if <code>true</code>, the entire component hierarchy is
	 *            traversed.
	 * @return <code>true</code> if the component has the specified client
	 *         property set to the specified value, <code>false</code>
	 *         otherwise.
	 */
	public static boolean hasPropertySetTo(Component comp, String propertyName,
			boolean expectedValue, boolean checkHierarchy) {
		if (!checkHierarchy) {
			if (comp instanceof JComponent) {
				JComponent jcomp = (JComponent) comp;
				Object componentProp = jcomp.getClientProperty(propertyName);
				if (componentProp != null) {
					if (componentProp.equals(expectedValue))
						return true;
				}
			}
		} else {
			Component c = comp;
			while (c != null) {
				if (c instanceof JComponent) {
					JComponent jcomp = (JComponent) c;
					Object componentProp = jcomp
							.getClientProperty(propertyName);
					if (componentProp != null) {
						if (componentProp.equals(expectedValue))
							return true;
					}
				}
				c = c.getParent();
			}
		}
		Object globalProp = UIManager.get(propertyName);
		if (globalProp != null) {
			return globalProp.equals(expectedValue);
		}
		return false;
	}

	/**
	 * Returns the resource bundle for the specified component.
	 * 
	 * @param jcomp
	 *            Component.
	 * @return Resource bundle for the specified component.
	 */
	public static ResourceBundle getResourceBundle(JComponent jcomp) {
		if (LafWidgetUtilities.toIgnoreGlobalLocale(jcomp)) {
			return SubstanceLookAndFeel.getLabelBundle(jcomp.getLocale());
		} else {
			return SubstanceLookAndFeel.getLabelBundle();
		}
	}

	/**
	 * Returns the border painter for the specified component.
	 * 
	 * @param comp
	 *            Component.
	 * @param gradientPainter
	 *            Gradient painter for the specified component.
	 * @return Border painter for the specified component.
	 */
	public static SubstanceBorderPainter getBorderPainter(Component comp,
			SubstanceGradientPainter gradientPainter) {
		// check the client property on component hierarchy
		while (comp != null) {
			if (comp instanceof JComponent) {
				Object clientProp = ((JComponent) comp)
						.getClientProperty(SubstanceLookAndFeel.BORDER_PAINTER_PROPERTY);
				if (clientProp instanceof SubstanceBorderPainter)
					return (SubstanceBorderPainter) clientProp;
				if (clientProp instanceof String) {
					try {
						return (SubstanceBorderPainter) Class.forName(
								(String) clientProp).newInstance();
					} catch (Exception exc) {
					}
				}
			}
			comp = comp.getParent();
		}
		// check property on UIManager
		Object globalProp = UIManager
				.get(SubstanceLookAndFeel.BORDER_PAINTER_PROPERTY);
		if (globalProp instanceof SubstanceBorderPainter)
			return (SubstanceBorderPainter) globalProp;
		if (globalProp instanceof String) {
			try {
				return (SubstanceBorderPainter) Class.forName(
						(String) globalProp).newInstance();
			} catch (Exception exc) {
			}
		}
		// check gradient painter for specific border painter
		if (gradientPainter instanceof BaseGradientPainter) {
			BaseGradientPainter base = (BaseGradientPainter) gradientPainter;
			SubstanceBorderPainter bPainter = base.getBorderPainter();
			if (bPainter != null)
				return bPainter;
		}
		// return currently installed global border painter
		return SubstanceLookAndFeel.getCurrentBorderPainter();
	}

	/**
	 * Resets the menu bars on the specified component.
	 * 
	 * @param component
	 *            Component.
	 */
	public static void resetMenuBars(Component component) {
		if (component instanceof JRootPane) {
			JRootPane jrp = (JRootPane) component;
			JMenuBar jmb = jrp.getJMenuBar();
			if (jmb != null) {
				MenuBarUI ui = jmb.getUI();
				if (ui instanceof SubstanceMenuBarUI) {
					Set<?> lafWidgets = ((SubstanceMenuBarUI) ui)
							.getLafWidgets();
					for (Iterator<?> it = lafWidgets.iterator(); it.hasNext();) {
						LafWidget lw = (LafWidget) it.next();
						if (lw instanceof Resettable) {
							((Resettable) lw).reset();
						}
					}
				}
			}
		}
		if (component instanceof Container) {
			Container cont = (Container) component;
			for (int i = 0; i < cont.getComponentCount(); i++) {
				Component child = cont.getComponent(i);
				resetMenuBars(child);
			}
		}
	}

	/**
	 * Returns the foreground color for the specified component.
	 * 
	 * @param comp
	 *            Component.
	 * @param componentId
	 *            Optional component ID. Can be used to differentiate sub-parts
	 *            of the component, such as tabs in tabbed pane, cells in list
	 *            etc.
	 * @param theme
	 *            Component theme.
	 * @param state
	 *            Component current state.
	 * @param prevState
	 *            Component previous state.
	 * @param kinds
	 *            Animation kinds to consult for computing the foreground color.
	 * @return Foreground color.
	 */
	public static Color getInterpolatedForegroundColor(Component comp,
			Comparable<?> componentId, SubstanceTheme theme,
			ComponentState state, ComponentState prevState, FadeKind... kinds) {
		ColorScheme colorScheme = theme.getColorScheme();
		ColorScheme colorScheme2 = colorScheme;
		float cyclePos = state.getCycleCount();

		FadeState fadeState = SubstanceFadeUtilities.getFadeState(comp,
				componentId, kinds);
		if (fadeState != null) {
			colorScheme = SubstanceCoreUtilities.getScheme(comp, state);
			colorScheme2 = SubstanceCoreUtilities.getScheme(comp, prevState);
			cyclePos = fadeState.getFadePosition();
			if (!fadeState.isFadingIn())
				cyclePos = 10 - cyclePos;
		}

		Color c1 = colorScheme.getForegroundColor();
		Color c2 = colorScheme2.getForegroundColor();

		// special case for components in default state on menu bar
		if (comp.getParent() instanceof JMenuBar) {
			if (state == ComponentState.DEFAULT)
				c1 = UIManager.getColor("MenuBar.foreground");
			if (prevState == ComponentState.DEFAULT)
				c2 = UIManager.getColor("MenuBar.foreground");
		}

		return SubstanceColorUtilities.getInterpolatedColor(c1, c2,
				cyclePos / 10.);
	}

	/**
	 * Paints text with drop shadow.
	 * 
	 * @param c
	 *            Component.
	 * @param g
	 *            Graphics context.
	 * @param foregroundColor
	 *            Foreground color.
	 * @param text
	 *            Text to paint.
	 * @param width
	 *            Text rectangle width.
	 * @param height
	 *            Text rectangle height.
	 * @param xOffset
	 *            Text rectangle X offset.
	 * @param yOffset
	 *            Text rectangle Y offset.
	 */
	public static void paintTextWithDropShadow(Component c, Graphics g,
			Color foregroundColor, String text, int width, int height,
			int xOffset, int yOffset) {
		Graphics2D graphics = (Graphics2D) g.create();
		// graphics.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING,
		// RenderingHints.VALUE_TEXT_ANTIALIAS_OFF);

		// blur the text shadow
		BufferedImage blurred = SubstanceCoreUtilities.getBlankImage(width,
				height);
		Graphics2D gBlurred = (Graphics2D) blurred.getGraphics();
		gBlurred.setFont(graphics.getFont());
		gBlurred.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING,
				RenderingHints.VALUE_TEXT_ANTIALIAS_OFF);
		// Color neg =
		// SubstanceColorUtilities.getNegativeColor(foregroundColor);
		float luminFactor = SubstanceColorUtilities
				.getColorStrength(foregroundColor);
		gBlurred.setColor(SubstanceColorUtilities
				.getNegativeColor(foregroundColor));
		ConvolveOp convolve = new ConvolveOp(new Kernel(3, 3, new float[] {
				.0f, .05f, .1f, .05f, .0f, .1f, .1f, .1f, .1f }),
				ConvolveOp.EDGE_NO_OP, null);
		gBlurred.drawString(text, xOffset + 1, yOffset + 1);
		blurred = convolve.filter(blurred, null);

		graphics.setComposite(TransitionLayout.getAlphaComposite(c,
				luminFactor, g));
		graphics.drawImage(blurred, 0, 0, null);
		graphics.setComposite(TransitionLayout.getAlphaComposite(c, g));

		graphics.setColor(foregroundColor);
		RenderingUtils.installDesktopHints(graphics);
		graphics.drawString(text, xOffset, yOffset);

		graphics.dispose();
	}

	/**
	 * Provides workaround for <a
	 * href="http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=6576507">bug
	 * 6576507</a>. This is especially relevant for skins that use translucent
	 * themes, such as {@link AutumnSkin} and {@link MagmaSkin}.
	 * 
	 * @param graphics
	 *            Graphics context.
	 */
	public static void workaroundBug6576507(Graphics graphics) {
		Font font = graphics.getFont();
		font = font.deriveFont(font.getStyle(), font.getSize2D());
		graphics.setFont(font);
	}

	/**
	 * Returns the component hierarchy.
	 * 
	 * @param comp
	 *            Component.
	 * @return Component hierarchy string.
	 */
	public static String getHierarchy(Component comp) {
		StringBuffer buffer = new StringBuffer();
		getHierarchy(comp, buffer, 0);
		while (true) {
			if (comp instanceof Window) {
				Window w = (Window) comp;
				comp = w.getOwner();
				if (comp != null) {
					buffer.append("Owner --->\n");
					getHierarchy(comp, buffer, 0);
				}
			} else {
				break;
			}
		}
		return buffer.toString();
	}

	/**
	 * Computes the component hierarchy.
	 * 
	 * @param comp
	 *            Component.
	 * @param buffer
	 *            Hierarchy representation buffer.
	 * @param level
	 *            Hierarchy level.
	 */
	public static void getHierarchy(Component comp, StringBuffer buffer,
			int level) {
		for (int i = 0; i < level; i++)
			buffer.append("   ");
		String name = comp.getName();
		if (comp instanceof Dialog)
			name = ((Dialog) comp).getTitle();
		if (comp instanceof Frame)
			name = ((Frame) comp).getTitle();
		buffer.append(comp.getClass().getName() + "[" + name + "]\n");
		if (comp instanceof Container) {
			Container cont = (Container) comp;
			for (int i = 0; i < cont.getComponentCount(); i++)
				getHierarchy(cont.getComponent(i), buffer, level + 1);
		}
	}

	/**
	 * Returns the title pane of the specified root pane.
	 * 
	 * @param rootPane
	 *            Root pane.
	 * @return The title pane of the specified root pane.
	 */
	public static JComponent getTitlePane(JRootPane rootPane) {
		JInternalFrame jif = (JInternalFrame) SwingUtilities
				.getAncestorOfClass(JInternalFrame.class, rootPane);
		if (jif != null) {
			SubstanceInternalFrameUI ui = (SubstanceInternalFrameUI) jif
					.getUI();
			return ui.getTitlePane();
		}
		SubstanceRootPaneUI ui = (SubstanceRootPaneUI) rootPane.getUI();
		return ui.getTitlePane();
	}

	public static void resetCaches() {
		SubstanceIconFactory.reset();
		ButtonBackgroundDelegate.reset();
		SubstanceCheckBoxUI.reset();
		SubstanceProgressBarUI.reset();
		SubstanceRadioButtonUI.reset();
		SubstanceScrollBarUI.reset();
		SubstanceTabbedPaneUI.reset();
		// SubstanceComboBoxUI.reset();
		SubstanceScrollBarUI.reset();
		ClassicTitlePainter.reset();
	}

	/**
	 * Returns the arrow icon.
	 * 
	 * @param comp
	 *            Component.
	 * @param button
	 *            Button.
	 * @param orientation
	 *            Arrow orientation.
	 * @return Arrow icon.
	 */
	public static Icon getArrowIcon(final Component comp,
			final AbstractButton button, final int orientation) {
		Icon result = new TransitionAwareIcon(button,
				new TransitionAwareIcon.Delegate() {
					public Icon getThemeIcon(SubstanceTheme theme) {
						int fontSize = SubstanceSizeUtils
								.getComponentFontSize(comp);
						return SubstanceImageCreator
								.getArrowIcon(
										SubstanceSizeUtils
												.getArrowIconWidth(fontSize),
										(orientation == SwingConstants.CENTER) ? 2 * SubstanceSizeUtils
												.getArrowIconHeight(fontSize)
												: SubstanceSizeUtils
														.getArrowIconHeight(fontSize),
										SubstanceSizeUtils
												.getArrowStrokeWidth(fontSize),
										orientation, theme);
					}
				});
		return result;
	}

	/**
	 * Returns the arrow icon.
	 * 
	 * @param comp
	 *            Component.
	 * @param button
	 *            Button.
	 * @param orientation
	 *            Arrow orientation.
	 * @return Arrow icon.
	 */
	public static Icon getDoubleArrowIcon(final Component comp,
			final AbstractButton button, final int orientation) {
		Icon result = new TransitionAwareIcon(button,
				new TransitionAwareIcon.Delegate() {
					public Icon getThemeIcon(SubstanceTheme theme) {
						int fontSize = SubstanceSizeUtils
								.getComponentFontSize(comp);
						return SubstanceImageCreator
								.getDoubleArrowIcon(
										SubstanceSizeUtils
												.getArrowIconWidth(fontSize),
										SubstanceSizeUtils
												.getArrowIconHeight(fontSize) + 2,
										SubstanceSizeUtils
												.getDoubleArrowStrokeWidth(fontSize),
										orientation, theme);
					}
				});
		return result;
	}
}
