/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package test.check;

import java.awt.Event;
import java.awt.Frame;
import java.awt.event.*;
import java.util.LinkedList;
import java.util.Map;

import javax.swing.*;

import org.jvnet.substance.SubstanceImageCreator;
import org.jvnet.substance.SubstanceLookAndFeel;
import org.jvnet.substance.skin.SkinInfo;
import org.jvnet.substance.skin.SubstanceSkin;
import org.jvnet.substance.theme.*;
import org.jvnet.substance.theme.SubstanceTheme.ThemeKind;
import org.jvnet.substance.utils.menu.TraitMenuHandler;

import test.Check;

/**
 * Factory that creates menus for the test applications.
 * 
 * @author Kirill Grouchnikov
 */
public class SampleMenuFactory {
	/**
	 * Returns the menu for changing the themes.
	 * 
	 * @return The menu for changing the themes.
	 */
	public static JMenu getThemeMenu() {
		Map<String, ThemeInfo> allThemes = SubstanceLookAndFeel.getAllThemes();

		final TraitMenuHandler themeMenuHandler = new TraitMenuHandler();
		JMenu themeMenu = new JMenu("Theme");
		JMenu brightThemes = new JMenu(SubstanceLookAndFeel.getLabelBundle()
				.getString("SystemMenu.themesBright"));
		JMenu coldThemes = new JMenu(SubstanceLookAndFeel.getLabelBundle()
				.getString("SystemMenu.themesCold"));
		JMenu darkThemes = new JMenu(SubstanceLookAndFeel.getLabelBundle()
				.getString("SystemMenu.themesDark"));
		JMenu invertedThemes = SubstanceLookAndFeel.toEnableInvertedThemes() ? new JMenu(
				SubstanceLookAndFeel.getLabelBundle().getString(
						"SystemMenu.themesInverted"))
				: null;
		JMenu negatedThemes = SubstanceLookAndFeel.toEnableNegatedThemes() ? new JMenu(
				SubstanceLookAndFeel.getLabelBundle().getString(
						"SystemMenu.themesNegated"))
				: null;
		JMenu mixedThemes = SubstanceLookAndFeel.hasMixedThemes() ? new JMenu(
				SubstanceLookAndFeel.getLabelBundle().getString(
						"SystemMenu.themesMixed")) : null;
		JMenu jmShiftTheme = new JMenu("Shift themes");

		themeMenu.add(brightThemes);
		themeMenu.add(coldThemes);
		themeMenu.add(darkThemes);
		if (invertedThemes != null)
			themeMenu.add(invertedThemes);
		if (negatedThemes != null)
			themeMenu.add(negatedThemes);
		if (mixedThemes != null)
			themeMenu.add(mixedThemes);

		for (Map.Entry<String, ThemeInfo> themeEntry : allThemes.entrySet()) {
			final ThemeInfo themeInfo = themeEntry.getValue();
			final String themeClassName = themeInfo.getClassName();
			JRadioButtonMenuItem jmiTheme = new JRadioButtonMenuItem(themeEntry
					.getKey());
			themeMenuHandler.addTraitButton(themeInfo.getDisplayName(),
					jmiTheme);
			jmiTheme.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					setTheme(themeInfo);
				}
			});
			try {
				switch (themeInfo.getThemeKind()) {
				case MIXED:
					MixedThemeInfo mixedThemeInfo = (MixedThemeInfo) themeInfo;
					String[] themeClassNames = mixedThemeInfo
							.getThemeClassNames();
					SubstanceTheme[] themeInstances = new SubstanceTheme[themeClassNames.length];
					for (int i = 0; i < themeClassNames.length; i++) {
						Class<?> themeClass = Class.forName(themeClassNames[i]);
						themeInstances[i] = (SubstanceTheme) themeClass
								.newInstance();
					}

					SubstanceTheme mixTheme = new SubstanceMixTheme(
							themeInstances);
					jmiTheme.setIcon(SubstanceImageCreator
							.getThemeIcon(mixTheme));
					break;
				default:
					Class<?> themeClass = Class.forName(themeClassName);
					SubstanceTheme theme = (SubstanceTheme) themeClass
							.newInstance();
					if (themeInfo.getThemeKind() == ThemeKind.INVERTED)
						theme = new SubstanceInvertedTheme(theme);
					if (themeInfo.getThemeKind() == ThemeKind.NEGATED)
						theme = new SubstanceNegatedTheme(theme);
					jmiTheme.setIcon(SubstanceImageCreator.getThemeIcon(theme));
				}
			} catch (Exception exc) {
				continue;
			}
			if (themeEntry.getKey().equals(
					SubstanceLookAndFeel.getCurrentThemeName())) {
				jmiTheme.setSelected(true);
			}
			switch (themeEntry.getValue().getThemeKind()) {
			case BRIGHT:
				brightThemes.add(jmiTheme);
				break;
			case COLD:
				coldThemes.add(jmiTheme);
				break;
			case DARK:
				darkThemes.add(jmiTheme);
				break;
			case INVERTED:
				invertedThemes.add(jmiTheme);
				break;
			case NEGATED:
				negatedThemes.add(jmiTheme);
				break;
			case MIXED:
				mixedThemes.add(jmiTheme);
				break;
			// case DARK_MIXED:
			// mixedDarkThemes.add(jmiTheme);
			// break;
			}
		}

		ThemeChangeListener themeChangeListener = new ThemeChangeListener() {
			public void themeChanged() {
				themeMenuHandler.selectTraitButton(SubstanceLookAndFeel
						.getTheme());
			}
		};
		SubstanceLookAndFeel.registerThemeChangeListener(themeChangeListener);

		JMenuItem jmiAquaTint10Theme = new JMenuItem("Tint Aqua 10%");
		jmiAquaTint10Theme.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(new SubstanceAquaTheme().tint(0.1));
			}
		});
		jmShiftTheme.add(jmiAquaTint10Theme);

		JMenuItem jmiAquaTint25Theme = new JMenuItem("Tint Aqua 25%");
		jmiAquaTint25Theme.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(new SubstanceAquaTheme().tint(0.25));
			}
		});
		jmShiftTheme.add(jmiAquaTint25Theme);

		JMenuItem jmiAquaTint40Theme = new JMenuItem("Tint Aqua 40%");
		jmiAquaTint40Theme.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(new SubstanceAquaTheme().tint(0.4));
			}
		});
		jmShiftTheme.add(jmiAquaTint40Theme);

		JMenuItem jmiAquaTone10Theme = new JMenuItem("Tone Aqua 10%");
		jmiAquaTone10Theme.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(new SubstanceAquaTheme().tone(0.1));
			}
		});
		jmShiftTheme.add(jmiAquaTone10Theme);

		JMenuItem jmiAquaTone25Theme = new JMenuItem("Tone Aqua 25%");
		jmiAquaTone25Theme.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(new SubstanceAquaTheme().tone(0.25));
			}
		});
		jmShiftTheme.add(jmiAquaTone25Theme);

		JMenuItem jmiAquaTone40Theme = new JMenuItem("Tone Aqua 40%");
		jmiAquaTone40Theme.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(new SubstanceAquaTheme().tone(0.4));
			}
		});
		jmShiftTheme.add(jmiAquaTone40Theme);

		JMenuItem jmiAquaShade10Theme = new JMenuItem("Shade Aqua 10%");
		jmiAquaShade10Theme.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(new SubstanceAquaTheme().shade(0.1));
			}
		});
		jmShiftTheme.add(jmiAquaShade10Theme);

		JMenuItem jmiAquaShade25Theme = new JMenuItem("Shade Aqua 25%");
		jmiAquaShade25Theme.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(new SubstanceAquaTheme().shade(0.25));
			}
		});
		jmShiftTheme.add(jmiAquaShade25Theme);

		JMenuItem jmiAquaShade40Theme = new JMenuItem("Shade Aqua 40%");
		jmiAquaShade40Theme.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(new SubstanceAquaTheme().shade(0.4));
			}
		});
		jmShiftTheme.add(jmiAquaShade40Theme);

		JMenuItem jmiSteelBlueTone25Theme = new JMenuItem("Tone Steel Blue 25%");
		jmiSteelBlueTone25Theme.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(new SubstanceSteelBlueTheme().tone(0.25));
			}
		});
		jmShiftTheme.add(jmiSteelBlueTone25Theme);

		JMenuItem jmiLimeGreenShade25Theme = new JMenuItem(
				"Shade Lime Green 25%");
		jmiLimeGreenShade25Theme.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(new SubstanceLimeGreenTheme().shade(0.25));
			}
		});
		jmShiftTheme.add(jmiLimeGreenShade25Theme);

		JMenuItem jmiBarbyPinkTint25Theme = new JMenuItem("Tint Barby Pink 25%");
		jmiBarbyPinkTint25Theme.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(new SubstanceBarbyPinkTheme().tint(0.25));
			}
		});
		jmShiftTheme.add(jmiBarbyPinkTint25Theme);

		JMenuItem jmiAquaBottleGreenMixTint25Theme = new JMenuItem(
				"Tint Aqua Bottle Green Mix 25%");
		jmiAquaBottleGreenMixTint25Theme
				.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent e) {
						setTheme(new SubstanceMixTheme(
								new SubstanceAquaTheme(),
								new SubstanceBottleGreenTheme()).tint(0.25));
					}
				});
		jmShiftTheme.add(jmiAquaBottleGreenMixTint25Theme);

		JMenuItem jmiAquaDesaturate50Theme = new JMenuItem(
				"Aqua Desaturate 50%");
		jmiAquaDesaturate50Theme.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(new SubstanceAquaTheme().saturate(-0.5));
			}
		});
		jmShiftTheme.add(jmiAquaDesaturate50Theme);

		JMenuItem jmiAquaSaturate50Theme = new JMenuItem("Aqua Saturate 50%");
		jmiAquaSaturate50Theme.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(new SubstanceAquaTheme().saturate(0.5));
			}
		});
		jmShiftTheme.add(jmiAquaSaturate50Theme);

		jmShiftTheme.addSeparator();

		JMenuItem jmiGreenBlendTheme = new JMenuItem("Green aqua blend");
		jmiGreenBlendTheme.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(new SubstanceMixTheme(new SubstanceBlendBiTheme(
						new SubstanceBottleGreenTheme(),
						new SubstanceLimeGreenTheme(), 0.6),
						new SubstanceBlendBiTheme(new SubstancePurpleTheme(),
								new SubstanceAquaTheme(), 0.4)));
			}
		});
		jmShiftTheme.add(jmiGreenBlendTheme);

		themeMenu.add(jmShiftTheme);

		themeMenu.addSeparator();

		JMenu changeTheme = new JMenu("Change current");

		JMenuItem invertCurrent = new JMenuItem("Inverted current");
		invertCurrent.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(SubstanceLookAndFeel.getTheme().invert());
			}
		});
		changeTheme.add(invertCurrent);

		JMenuItem negateCurrent = new JMenuItem("Negated current");
		negateCurrent.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(SubstanceLookAndFeel.getTheme().negate());
			}
		});
		changeTheme.add(negateCurrent);

		changeTheme.addSeparator();

		JMenuItem tintCurrent = new JMenuItem("Tinted current");
		tintCurrent.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(SubstanceLookAndFeel.getTheme().tint(0.3));
			}
		});
		changeTheme.add(tintCurrent);

		JMenuItem toneCurrent = new JMenuItem("Toned current");
		toneCurrent.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(SubstanceLookAndFeel.getTheme().tone(0.3));
			}
		});
		changeTheme.add(toneCurrent);

		JMenuItem shadeCurrent = new JMenuItem("Shaded current");
		shadeCurrent.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(SubstanceLookAndFeel.getTheme().shade(0.3));
			}
		});
		changeTheme.add(shadeCurrent);

		changeTheme.addSeparator();

		JMenuItem saturateCurrent = new JMenuItem("Saturated current");
		saturateCurrent.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(SubstanceLookAndFeel.getTheme().saturate(0.3));
			}
		});
		changeTheme.add(saturateCurrent);

		JMenuItem saturateActiveCurrent = new JMenuItem(
				"Saturated active current");
		saturateActiveCurrent.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(SubstanceLookAndFeel.getTheme().saturate(0.3, false));
			}
		});
		changeTheme.add(saturateActiveCurrent);

		JMenuItem desaturateCurrent = new JMenuItem("Desaturated current");
		desaturateCurrent.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(SubstanceLookAndFeel.getTheme().saturate(-0.3));
			}
		});
		changeTheme.add(desaturateCurrent);

		JMenuItem desaturateActiveCurrent = new JMenuItem(
				"Desaturated active current");
		desaturateActiveCurrent.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(SubstanceLookAndFeel.getTheme().saturate(-0.3, false));
			}
		});
		changeTheme.add(desaturateActiveCurrent);

		changeTheme.addSeparator();

		JMenuItem hueShiftCurrent_p002 = new JMenuItem("Hue-shift current 0.02");
		hueShiftCurrent_p002.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(SubstanceLookAndFeel.getTheme().hueShift(0.02));
			}
		});
		changeTheme.add(hueShiftCurrent_p002);

		JMenuItem hueShiftCurrent_m002 = new JMenuItem(
				"Hue-shift current -0.02");
		hueShiftCurrent_m002.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(SubstanceLookAndFeel.getTheme().hueShift(-0.02));
			}
		});
		changeTheme.add(hueShiftCurrent_m002);

		JMenuItem hueShiftCurrent_p01 = new JMenuItem("Hue-shift current 0.1");
		hueShiftCurrent_p01.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(SubstanceLookAndFeel.getTheme().hueShift(0.1));
			}
		});
		changeTheme.add(hueShiftCurrent_p01);

		JMenuItem hueShiftCurrent_m01 = new JMenuItem("Hue-shift current -0.1");
		hueShiftCurrent_m01.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(SubstanceLookAndFeel.getTheme().hueShift(-0.1));
			}
		});
		changeTheme.add(hueShiftCurrent_m01);

		JMenuItem hueShiftCurrent_p05 = new JMenuItem("Hue-shift current 0.5");
		hueShiftCurrent_p05.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(SubstanceLookAndFeel.getTheme().hueShift(0.5));
			}
		});
		changeTheme.add(hueShiftCurrent_p05);

		JMenuItem hueShiftCurrent_m05 = new JMenuItem("Hue-shift current -0.5");
		hueShiftCurrent_m05.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(SubstanceLookAndFeel.getTheme().hueShift(-0.5));
			}
		});
		changeTheme.add(hueShiftCurrent_m05);

		changeTheme.addSeparator();

		final JMenuItem restoreOrig = new JMenuItem("Restore original");
		restoreOrig
				.setEnabled(SubstanceLookAndFeel.getTheme() instanceof SubstanceWrapperTheme);
		restoreOrig.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(((SubstanceWrapperTheme) SubstanceLookAndFeel
						.getTheme()).getOriginalTheme());
			}
		});
		SubstanceLookAndFeel
				.registerThemeChangeListener(new ThemeChangeListener() {
					public void themeChanged() {
						restoreOrig
								.setEnabled(SubstanceLookAndFeel.getTheme() instanceof SubstanceWrapperTheme);
					}
				});
		changeTheme.add(restoreOrig);

		themeMenu.add(changeTheme);

		JMenu cbMenu = new JMenu("Color blindness");
		JMenuItem protanopiaCurrent = new JMenuItem("Protanopia current");
		protanopiaCurrent.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(SubstanceLookAndFeel.getTheme().protanopia());
			}
		});
		cbMenu.add(protanopiaCurrent);
		JMenuItem deuteranopiaCurrent = new JMenuItem("Deuteranopia current");
		deuteranopiaCurrent.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(SubstanceLookAndFeel.getTheme().deuteranopia());
			}
		});
		cbMenu.add(deuteranopiaCurrent);
		JMenuItem tritanopiaCurrent = new JMenuItem("Tritanopia current");
		tritanopiaCurrent.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(SubstanceLookAndFeel.getTheme().tritanopia());
			}
		});
		cbMenu.add(tritanopiaCurrent);
		themeMenu.add(cbMenu);

		themeMenu.addSeparator();

		JMenuItem jmiCaribbeanBlueTheme = new JMenuItem("User - Caribbean blue");
		jmiCaribbeanBlueTheme.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(new SubstanceTheme(new CaribbeanBlueColorScheme(),
						"Caribbean blue", SubstanceTheme.ThemeKind.BRIGHT));
			}
		});
		themeMenu.add(jmiCaribbeanBlueTheme);

		JMenuItem jmiCrimsonTheme = new JMenuItem("User - Crimson");
		jmiCrimsonTheme.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setTheme(new SubstanceTheme(new CrimsonColorScheme(),
						"Crimson", SubstanceTheme.ThemeKind.BRIGHT));
			}
		});
		themeMenu.add(jmiCrimsonTheme);

		return themeMenu;
	}

	/**
	 * Sets the specified theme.
	 * 
	 * @param theme
	 *            Theme to set.
	 */
	public static void setTheme(final SubstanceTheme theme) {
		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				try {
					SubstanceLookAndFeel.setCurrentTheme(theme);
				} catch (Exception exc) {
					exc.printStackTrace();
				}
				for (Frame frame : Frame.getFrames())
					SwingUtilities.updateComponentTreeUI(frame);
			}
		});
	}

	/**
	 * Sets the specified theme.
	 * 
	 * @param themeInfo
	 *            Theme information object.
	 */
	public static void setTheme(final ThemeInfo themeInfo) {
		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				try {
					SubstanceLookAndFeel.setCurrentTheme(themeInfo);
				} catch (Exception exc) {
					exc.printStackTrace();
				}
				for (Frame frame : Frame.getFrames())
					SwingUtilities.updateComponentTreeUI(frame);
			}
		});
	}

	/**
	 * Returns a sample test menu.
	 * 
	 * @return Sample test menu.
	 */
	public static JMenu getTestMenu() {
		JMenu testMenu = new JMenu("Test");
		testMenu.setMnemonic('1');
		int mcount = 0;
		for (LinkedList<JMenuItem> miList : getTestMenuItems()) {
			if (mcount > 0) {
				if (mcount % 2 == 0)
					testMenu.addSeparator();
				else
					testMenu.add(new JSeparator());
			}
			for (JMenuItem menuItem : miList) {
				testMenu.add(menuItem);
			}
			mcount++;
		}

		return testMenu;
	}

	/**
	 * Returns menu items for a sample test menu.
	 * 
	 * @return Menu items for a sample test menu.
	 */
	public static LinkedList<LinkedList<JMenuItem>> getTestMenuItems() {
		LinkedList<LinkedList<JMenuItem>> result = new LinkedList<LinkedList<JMenuItem>>();
		LinkedList<JMenuItem> list1 = new LinkedList<JMenuItem>();
		final JMenuItem jmi1 = new JMenuItem("Menu item enabled", Check
				.getIcon("flag_sweden"));
		jmi1.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_A,
				Event.CTRL_MASK));
		JMenuItem jmi2 = new JMenuItem("Menu item disabled");
		jmi2.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_B,
				Event.CTRL_MASK | Event.ALT_MASK));
		jmi2.setEnabled(false);

		list1.add(jmi1);
		list1.add(jmi2);
		result.add(list1);

		LinkedList<JMenuItem> list2 = new LinkedList<JMenuItem>();
		final JCheckBoxMenuItem jcbmi1 = new JCheckBoxMenuItem(
				"Check enabled selected", SubstanceImageCreator.getHexaMarker(
						2, SubstanceLookAndFeel.getTheme()));
		jcbmi1.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_C,
				Event.CTRL_MASK));
		jcbmi1.setSelected(true);
		JCheckBoxMenuItem jcbmi2 = new JCheckBoxMenuItem(
				"Check enabled unselected");
		jcbmi2.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_D,
				Event.CTRL_MASK));
		jcbmi2.setSelected(false);
		JCheckBoxMenuItem jcbmi3 = new JCheckBoxMenuItem(
				"Check disabled selected");
		jcbmi3.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_E,
				Event.CTRL_MASK));
		jcbmi3.setSelected(true);
		jcbmi3.setEnabled(false);
		final JCheckBoxMenuItem jcbmi4 = new JCheckBoxMenuItem(
				"Check disabled unselected", SubstanceImageCreator
						.getHexaMarker(3, SubstanceLookAndFeel.getTheme()));
		jcbmi4.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F,
				Event.CTRL_MASK));
		jcbmi4.setSelected(false);
		jcbmi4.setEnabled(false);

		list2.add(jcbmi1);
		list2.add(jcbmi2);
		list2.add(jcbmi3);
		list2.add(jcbmi4);
		result.add(list2);

		LinkedList<JMenuItem> list3 = new LinkedList<JMenuItem>();
		final JRadioButtonMenuItem jrbmi1 = new JRadioButtonMenuItem(
				"Radio enabled selected", SubstanceImageCreator.getHexaMarker(
						4, SubstanceLookAndFeel.getTheme()));
		jrbmi1.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_G,
				Event.CTRL_MASK));
		jrbmi1.setSelected(true);
		JRadioButtonMenuItem jrbmi2 = new JRadioButtonMenuItem(
				"Radio enabled unselected");
		jrbmi2.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_H,
				Event.CTRL_MASK));
		jrbmi2.setSelected(false);
		ButtonGroup bgRadioMenu1 = new ButtonGroup();
		bgRadioMenu1.add(jrbmi1);
		bgRadioMenu1.add(jrbmi2);
		JRadioButtonMenuItem jrbmi3 = new JRadioButtonMenuItem(
				"Radio disabled selected");
		jrbmi3.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_I,
				Event.CTRL_MASK));
		jrbmi3.setSelected(true);
		jrbmi3.setEnabled(false);
		JRadioButtonMenuItem jrbmi4 = new JRadioButtonMenuItem(
				"Radio disabled unselected");
		jrbmi4.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_J,
				Event.CTRL_MASK));
		jrbmi4.setSelected(false);
		jrbmi4.setEnabled(false);
		ButtonGroup bgRadioMenu2 = new ButtonGroup();
		bgRadioMenu2.add(jrbmi3);
		bgRadioMenu2.add(jrbmi4);

		list3.add(jrbmi1);
		list3.add(jrbmi2);
		list3.add(jrbmi3);
		list3.add(jrbmi4);
		result.add(list3);

		LinkedList<JMenuItem> list4 = new LinkedList<JMenuItem>();
		JMenu submenu1 = new JMenu("submenu1");
		submenu1.add(new JMenuItem("submenu item1"));
		submenu1.add(new JMenuItem("submenu item2"));
		submenu1.add(new JMenuItem("submenu item3"));
		final JMenu submenu11 = new JMenu("submenu1-1");
		submenu11.setIcon(SubstanceImageCreator.getHexaMarker(5,
				SubstanceLookAndFeel.getTheme()));
		submenu11.add(new JMenuItem("submenu item111"));
		submenu11.add(new JMenuItem("submenu item112"));
		submenu11.add(new JMenuItem("submenu item113"));
		submenu11.add(new JMenuItem("submenu item114"));
		submenu1.add(submenu11);
		final JMenu submenu12 = new JMenu("submenu1-2");
		submenu12.setIcon(SubstanceImageCreator.getHexaMarker(5,
				SubstanceLookAndFeel.getTheme()));
		submenu12.add(new JMenuItem("submenu item121"));
		submenu12.add(new JMenuItem("submenu item122"));
		submenu12.add(new JMenuItem("submenu item123"));
		submenu12.add(new JMenuItem("submenu item124"));
		submenu1.add(submenu12);
		final JMenu submenu13 = new JMenu("submenu1-3");
		submenu13.setIcon(SubstanceImageCreator.getHexaMarker(5,
				SubstanceLookAndFeel.getTheme()));
		submenu13.add(new JMenuItem("submenu item131"));
		submenu13.add(new JMenuItem("submenu item132"));
		submenu13.add(new JMenuItem("submenu item133"));
		submenu13.add(new JMenuItem("submenu item134"));
		submenu13.add(new JMenuItem("submenu item135"));
		submenu1.add(submenu13);
		list4.add(submenu1);

		JMenu submenu2 = new JMenu("submenu2");
		submenu2.add(new JMenuItem("submenu item1"));
		submenu2.add(new JMenuItem("submenu item2"));
		submenu2.add(new JMenuItem("submenu item3"));
		final JMenu submenu21 = new JMenu("submenu2-1");
		submenu21.setIcon(SubstanceImageCreator.getHexaMarker(5,
				SubstanceLookAndFeel.getTheme()));
		submenu21.add(new JMenuItem("submenu item211"));
		submenu21.add(new JMenuItem("submenu item212"));
		submenu21.add(new JMenuItem("submenu item213"));
		submenu21.add(new JMenuItem("submenu item214"));
		submenu2.add(submenu21);
		final JMenu submenu22 = new JMenu("submenu2-2");
		submenu22.setIcon(SubstanceImageCreator.getHexaMarker(5,
				SubstanceLookAndFeel.getTheme()));
		submenu22.add(new JMenuItem("submenu item221"));
		submenu22.add(new JMenuItem("submenu item222"));
		submenu22.add(new JMenuItem("submenu item223"));
		submenu22.add(new JMenuItem("submenu item224"));
		submenu2.add(submenu22);
		final JMenu submenu23 = new JMenu("submenu2-3");
		submenu23.setIcon(SubstanceImageCreator.getHexaMarker(5,
				SubstanceLookAndFeel.getTheme()));
		submenu23.add(new JMenuItem("submenu item231"));
		submenu23.add(new JMenuItem("submenu item232"));
		submenu23.add(new JMenuItem("submenu item233"));
		submenu23.add(new JMenuItem("submenu item234"));
		submenu2.add(submenu23);
		list4.add(submenu2);

		JMenu submenu3 = new JMenu("submenu3 (disabled)");
		submenu3.setEnabled(false);
		list4.add(submenu3);

		result.add(list4);

		SubstanceLookAndFeel
				.registerThemeChangeListener(new ThemeChangeListener() {
					public void themeChanged() {
						// jmi1.setIcon(SubstanceImageCreator.getHexaMarker(1,
						// SubstanceLookAndFeel.getTheme()));
						jcbmi1.setIcon(SubstanceImageCreator.getHexaMarker(2,
								SubstanceLookAndFeel.getTheme()));
						jcbmi4.setIcon(SubstanceImageCreator.getHexaMarker(3,
								SubstanceLookAndFeel.getTheme()));
						jrbmi1.setIcon(SubstanceImageCreator.getHexaMarker(4,
								SubstanceLookAndFeel.getTheme()));
						submenu11.setIcon(SubstanceImageCreator.getHexaMarker(
								5, SubstanceLookAndFeel.getTheme()));
						submenu12.setIcon(SubstanceImageCreator.getHexaMarker(
								5, SubstanceLookAndFeel.getTheme()));
						submenu13.setIcon(SubstanceImageCreator.getHexaMarker(
								5, SubstanceLookAndFeel.getTheme()));
						submenu21.setIcon(SubstanceImageCreator.getHexaMarker(
								5, SubstanceLookAndFeel.getTheme()));
						submenu22.setIcon(SubstanceImageCreator.getHexaMarker(
								5, SubstanceLookAndFeel.getTheme()));
						submenu23.setIcon(SubstanceImageCreator.getHexaMarker(
								5, SubstanceLookAndFeel.getTheme()));
					}
				});

		return result;
	}

	/**
	 * Returns menu for setting skins.
	 * 
	 * @return Menu for setting skins.
	 */
	public static JMenu getSkinMenu() {
		JMenu jmSkin = new JMenu("Skins");
		Map<String, SkinInfo> skinMap = SubstanceLookAndFeel.getAllSkins();
		for (final Map.Entry<String, SkinInfo> entry : skinMap.entrySet()) {
			JMenuItem jmiSkin = new JMenuItem(entry.getValue().getDisplayName());
			jmiSkin.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					SubstanceLookAndFeel.setSkin(entry.getValue()
							.getClassName());
				}
			});

			try {
				SubstanceSkin skin = ((SubstanceSkin) Class.forName(
						entry.getValue().getClassName()).newInstance());
				jmiSkin
						.setIcon(new ImageIcon(
								SubstanceImageCreator
										.getThemeImage(
												jmiSkin,
												new ImageIcon(
														SampleMenuFactory.class
																.getClassLoader()
																.getResource(
																		"test/resource/image-x-generic.png")),
												skin.getTheme(), false)));
			} catch (Exception exc) {
			}

			jmSkin.add(jmiSkin);
		}
		return jmSkin;
	}

	public static JMenu getLookAndFeelMenu(JFrame frame) {
		JMenu lafMenu = new JMenu("Look & feel");
		JMenu substanceMenus = new JMenu("Substance family");
		substanceMenus.add(SubstanceLafChanger.getMenuItem(frame, "Substance",
				"org.jvnet.substance.SubstanceLookAndFeel"));
		substanceMenus.add(SubstanceLafChanger.getMenuItem(frame, "Default",
				"org.jvnet.substance.SubstanceDefaultLookAndFeel"));
		substanceMenus.addSeparator();
		substanceMenus.add(SubstanceLafChanger.getMenuItem(frame, "Business",
				"org.jvnet.substance.skin.SubstanceBusinessLookAndFeel"));
		substanceMenus
				.add(SubstanceLafChanger
						.getMenuItem(frame, "Business Black Steel",
								"org.jvnet.substance.skin.SubstanceBusinessBlackSteelLookAndFeel"));
		substanceMenus
				.add(SubstanceLafChanger
						.getMenuItem(frame, "Business Blue Steel",
								"org.jvnet.substance.skin.SubstanceBusinessBlueSteelLookAndFeel"));
		substanceMenus.add(SubstanceLafChanger.getMenuItem(frame, "Creme",
				"org.jvnet.substance.skin.SubstanceCremeLookAndFeel"));
		substanceMenus.add(SubstanceLafChanger.getMenuItem(frame, "Moderate",
				"org.jvnet.substance.skin.SubstanceModerateLookAndFeel"));
		substanceMenus.add(SubstanceLafChanger.getMenuItem(frame, "Nebula",
				"org.jvnet.substance.skin.SubstanceNebulaLookAndFeel"));
		substanceMenus
				.add(SubstanceLafChanger
						.getMenuItem(frame, "Nebula Brick Wall",
								"org.jvnet.substance.skin.SubstanceNebulaBrickWallLookAndFeel"));
		substanceMenus
				.add(SubstanceLafChanger
						.getMenuItem(frame, "Office Silver 2007",
								"org.jvnet.substance.skin.SubstanceOfficeSilver2007LookAndFeel"));
		substanceMenus.add(SubstanceLafChanger.getMenuItem(frame, "Sahara",
				"org.jvnet.substance.skin.SubstanceSaharaLookAndFeel"));
		substanceMenus.addSeparator();
		substanceMenus.add(SubstanceLafChanger.getMenuItem(frame,
				"Field of Wheat",
				"org.jvnet.substance.skin.SubstanceFieldOfWheatLookAndFeel"));
		substanceMenus.add(SubstanceLafChanger.getMenuItem(frame,
				"Green Magic",
				"org.jvnet.substance.skin.SubstanceGreenMagicLookAndFeel"));
		substanceMenus.add(SubstanceLafChanger.getMenuItem(frame, "Mango",
				"org.jvnet.substance.skin.SubstanceMangoLookAndFeel"));
		substanceMenus.add(SubstanceLafChanger.getMenuItem(frame,
				"Office Blue 2007",
				"org.jvnet.substance.skin.SubstanceOfficeBlue2007LookAndFeel"));
		substanceMenus.addSeparator();
		substanceMenus.add(SubstanceLafChanger.getMenuItem(frame,
				"Challenger Deep",
				"org.jvnet.substance.skin.SubstanceChallengerDeepLookAndFeel"));
		substanceMenus.add(SubstanceLafChanger.getMenuItem(frame,
				"Emerald Dusk",
				"org.jvnet.substance.skin.SubstanceEmeraldDuskLookAndFeel"));
		substanceMenus.add(SubstanceLafChanger.getMenuItem(frame, "Magma",
				"org.jvnet.substance.skin.SubstanceMagmaLookAndFeel"));
		substanceMenus.add(SubstanceLafChanger.getMenuItem(frame, "Raven",
				"org.jvnet.substance.skin.SubstanceRavenLookAndFeel"));
		substanceMenus.add(SubstanceLafChanger.getMenuItem(frame,
				"Raven Graphite",
				"org.jvnet.substance.skin.SubstanceRavenGraphiteLookAndFeel"));
		substanceMenus
				.add(SubstanceLafChanger
						.getMenuItem(frame, "Raven Graphite Glass",
								"org.jvnet.substance.skin.SubstanceRavenGraphiteGlassLookAndFeel"));
		lafMenu.add(substanceMenus);
		lafMenu.addSeparator();
		JMenu coreLafMenus = new JMenu("Core LAFs");
		lafMenu.add(coreLafMenus);
		coreLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "System",
				UIManager.getSystemLookAndFeelClassName()));
		coreLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "Metal",
				"javax.swing.plaf.metal.MetalLookAndFeel"));
		coreLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "Windows",
				"com.sun.java.swing.plaf.windows.WindowsLookAndFeel"));
		coreLafMenus.add(SubstanceLafChanger.getMenuItem(frame,
				"Windows Classic",
				"com.sun.java.swing.plaf.windows.WindowsClassicLookAndFeel"));
		coreLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "Motif",
				"com.sun.java.swing.plaf.motif.MotifLookAndFeel"));
		coreLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "Nimbus",
				"sun.swing.plaf.nimbus.NimbusLookAndFeel"));

		JMenu customLafMenus = new JMenu("Custom LAFs");
		lafMenu.add(customLafMenus);
		JMenu jgoodiesMenu = new JMenu("JGoodies family");
		customLafMenus.add(jgoodiesMenu);
		jgoodiesMenu.add(SubstanceLafChanger.getMenuItem(frame,
				"JGoodies Plastic",
				"com.jgoodies.looks.plastic.PlasticLookAndFeel"));
		jgoodiesMenu.add(SubstanceLafChanger.getMenuItem(frame,
				"JGoodies PlasticXP",
				"com.jgoodies.looks.plastic.PlasticXPLookAndFeel"));
		jgoodiesMenu.add(SubstanceLafChanger.getMenuItem(frame,
				"JGoodies Plastic3D",
				"com.jgoodies.looks.plastic.Plastic3DLookAndFeel"));
		jgoodiesMenu.add(SubstanceLafChanger.getMenuItem(frame,
				"JGoodies Windows",
				"com.jgoodies.looks.windows.WindowsLookAndFeel"));

		JMenu jtattooMenu = new JMenu("JTattoo family");
		customLafMenus.add(jtattooMenu);
		jtattooMenu.add(SubstanceLafChanger.getMenuItem(frame, "JTattoo Acryl",
				"com.jtattoo.plaf.acryl.AcrylLookAndFeel"));
		jtattooMenu.add(SubstanceLafChanger.getMenuItem(frame, "JTattoo Aero",
				"com.jtattoo.plaf.aero.AeroLookAndFeel"));
		jtattooMenu.add(SubstanceLafChanger.getMenuItem(frame,
				"JTattoo Aluminium",
				"com.jtattoo.plaf.aluminium.AluminiumLookAndFeel"));
		jtattooMenu.add(SubstanceLafChanger.getMenuItem(frame,
				"JTattoo Bernstein",
				"com.jtattoo.plaf.bernstein.BernsteinLookAndFeel"));
		jtattooMenu.add(SubstanceLafChanger.getMenuItem(frame, "JTattoo Fast",
				"com.jtattoo.plaf.fast.FastLookAndFeel"));
		jtattooMenu.add(SubstanceLafChanger.getMenuItem(frame, "JTattoo HiFi",
				"com.jtattoo.plaf.hifi.HiFiLookAndFeel"));
		jtattooMenu.add(SubstanceLafChanger.getMenuItem(frame, "JTattoo Luna",
				"com.jtattoo.plaf.luna.LunaLookAndFeel"));
		jtattooMenu.add(SubstanceLafChanger.getMenuItem(frame, "JTattoo McWin",
				"com.jtattoo.plaf.mcwin.McWinLookAndFeel"));
		jtattooMenu.add(SubstanceLafChanger.getMenuItem(frame, "JTattoo Mint",
				"com.jtattoo.plaf.mint.MintLookAndFeel"));
		jtattooMenu.add(SubstanceLafChanger.getMenuItem(frame, "JTattoo Smart",
				"com.jtattoo.plaf.smart.SmartLookAndFeel"));

		JMenu syntheticaMenu = new JMenu("Synthetica family");
		customLafMenus.add(syntheticaMenu);
		syntheticaMenu.add(SubstanceLafChanger.getMenuItem(frame,
				"Synthetica base",
				"de.javasoft.plaf.synthetica.SyntheticaStandardLookAndFeel"));
		syntheticaMenu.add(SubstanceLafChanger.getMenuItem(frame,
				"Synthetica BlackMoon",
				"de.javasoft.plaf.synthetica.SyntheticaBlackMoonLookAndFeel"));
		syntheticaMenu.add(SubstanceLafChanger.getMenuItem(frame,
				"Synthetica BlackStar",
				"de.javasoft.plaf.synthetica.SyntheticaBlackStarLookAndFeel"));
		syntheticaMenu.add(SubstanceLafChanger.getMenuItem(frame,
				"Synthetica BlueIce",
				"de.javasoft.plaf.synthetica.SyntheticaBlueIceLookAndFeel"));
		syntheticaMenu.add(SubstanceLafChanger.getMenuItem(frame,
				"Synthetica BlueMoon",
				"de.javasoft.plaf.synthetica.SyntheticaBlueMoonLookAndFeel"));
		syntheticaMenu.add(SubstanceLafChanger.getMenuItem(frame,
				"Synthetica BlueSteel",
				"de.javasoft.plaf.synthetica.SyntheticaBlueSteelLookAndFeel"));
		syntheticaMenu.add(SubstanceLafChanger.getMenuItem(frame,
				"Synthetica GreenDream",
				"de.javasoft.plaf.synthetica.SyntheticaGreenDreamLookAndFeel"));
		syntheticaMenu
				.add(SubstanceLafChanger
						.getMenuItem(frame, "Synthetica MauveMetallic",
								"de.javasoft.plaf.synthetica.SyntheticaMauveMetallicLookAndFeel"));
		syntheticaMenu
				.add(SubstanceLafChanger
						.getMenuItem(frame, "Synthetica OrangeMetallic",
								"de.javasoft.plaf.synthetica.SyntheticaOrangeMetallicLookAndFeel"));
		syntheticaMenu
				.add(SubstanceLafChanger
						.getMenuItem(frame, "Synthetica SkyMetallic",
								"de.javasoft.plaf.synthetica.SyntheticaSkyMetallicLookAndFeel"));
		syntheticaMenu.add(SubstanceLafChanger.getMenuItem(frame,
				"Synthetica SilverMoon",
				"de.javasoft.plaf.synthetica.SyntheticaSilverMoonLookAndFeel"));
		syntheticaMenu
				.add(SubstanceLafChanger
						.getMenuItem(frame, "Synthetica WhiteVision",
								"de.javasoft.plaf.synthetica.SyntheticaWhiteVisionLookAndFeel"));

		JMenu officeMenu = new JMenu("Office family");
		customLafMenus.add(officeMenu);
		officeMenu.add(SubstanceLafChanger.getMenuItem(frame, "Office 2003",
				"org.fife.plaf.Office2003.Office2003LookAndFeel"));
		officeMenu.add(SubstanceLafChanger.getMenuItem(frame, "Office XP",
				"org.fife.plaf.OfficeXP.OfficeXPLookAndFeel"));
		officeMenu.add(SubstanceLafChanger.getMenuItem(frame,
				"Visual Studio 2005",
				"org.fife.plaf.VisualStudio2005.VisualStudio2005LookAndFeel"));

		customLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "A03",
				"a03.swing.plaf.A03LookAndFeel"));
		customLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "Alloy",
				"com.incors.plaf.alloy.AlloyLookAndFeel"));
		customLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "FH",
				"com.shfarr.ui.plaf.fh.FhLookAndFeel"));
		customLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "Hippo",
				"se.diod.hippo.plaf.HippoLookAndFeel"));
		customLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "InfoNode",
				"net.infonode.gui.laf.InfoNodeLookAndFeel"));
		customLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "Kuntstoff",
				"com.incors.plaf.kunststoff.KunststoffLookAndFeel"));
		customLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "Liquid",
				"com.birosoft.liquid.LiquidLookAndFeel"));
		customLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "Lipstik",
				"com.lipstikLF.LipstikLookAndFeel"));
		customLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "Metouia",
				"net.sourceforge.mlf.metouia.MetouiaLookAndFeel"));
		customLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "Napkin",
				"net.sourceforge.napkinlaf.NapkinLookAndFeel"));
		customLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "Nimbus",
				"org.jdesktop.swingx.plaf.nimbus.NimbusLookAndFeel"));
		customLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "NimROD",
				"com.nilo.plaf.nimrod.NimRODLookAndFeel"));
		customLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "Oyoaha",
				"com.oyoaha.swing.plaf.oyoaha.OyoahaLookAndFeel"));
		customLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "Pagosoft",
				"com.pagosoft.plaf.PgsLookAndFeel"));
		customLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "Simple",
				"com.memoire.slaf.SlafLookAndFeel"));
		customLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "Skin",
				"com.l2fprod.gui.plaf.skin.SkinLookAndFeel"));
		customLafMenus.add(SubstanceLafChanger.getMenuItem(frame,
				"Smooth Metal", "smooth.metal.SmoothLookAndFeel"));
		customLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "Squareness",
				"net.beeger.squareness.SquarenessLookAndFeel"));
		customLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "Tiny",
				"de.muntjak.tinylookandfeel.TinyLookAndFeel"));
		customLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "Tonic",
				"com.digitprop.tonic.TonicLookAndFeel"));
		customLafMenus.add(SubstanceLafChanger.getMenuItem(frame, "Trendy",
				"com.Trendy.swing.plaf.TrendyLookAndFeel"));

		lafMenu.addSeparator();
		JMenu localeMenus = new JMenu("Change locale");
		lafMenu.add(localeMenus);
		// Locale changing
		JMenuItem localeArabic = new JMenuItem("Arabic Locale", Check
				.getIcon("flag_saudi_arabia"));
		localeArabic.addActionListener(new MyLocaleChangeListener("ar", "AR",
				frame));

		JMenuItem localeBulgarian = new JMenuItem("Bulgarian Locale", Check
				.getIcon("flag_bulgaria"));
		localeBulgarian.addActionListener(new MyLocaleChangeListener("bg",
				"BG", frame));

		JMenuItem localeChinese = new JMenuItem("Chinese (Simplified) Locale",
				Check.getIcon("flag_china"));
		localeChinese.addActionListener(new MyLocaleChangeListener("zh", "CN",
				frame));

		JMenuItem localeChineseHK = new JMenuItem("Chinese (Hong Kong) Locale",
				Check.getIcon("flag_hong_kong"));
		localeChineseHK.addActionListener(new MyLocaleChangeListener("zh",
				"HK", frame));

		JMenuItem localeChineseTW = new JMenuItem("Chinese (Taiwan) Locale",
				Check.getIcon("flag_taiwan"));
		localeChineseTW.addActionListener(new MyLocaleChangeListener("zh",
				"TW", frame));

		JMenuItem localeCzech = new JMenuItem("Czech Locale", Check
				.getIcon("flag_czech_republic"));
		localeCzech.addActionListener(new MyLocaleChangeListener("cs", "CZ",
				frame));

		JMenuItem localeDanish = new JMenuItem("Danish Locale", Check
				.getIcon("flag_denmark"));
		localeDanish.addActionListener(new MyLocaleChangeListener("da", "DK",
				frame));

		JMenuItem localeDutch = new JMenuItem("Dutch Locale", Check
				.getIcon("flag_netherlands"));
		localeDutch.addActionListener(new MyLocaleChangeListener("nl", "NL",
				frame));

		JMenuItem localeEnglish = new JMenuItem("English Locale", Check
				.getIcon("flag_united_states"));
		localeEnglish.addActionListener(new MyLocaleChangeListener("en", "US",
				frame));

		JMenuItem localeFinnish = new JMenuItem("Finnish Locale", Check
				.getIcon("flag_finland"));
		localeFinnish.addActionListener(new MyLocaleChangeListener("fi", "FI",
				frame));

		JMenuItem localeFrench = new JMenuItem("French Locale", Check
				.getIcon("flag_france"));
		localeFrench.addActionListener(new MyLocaleChangeListener("fr", "FR",
				frame));

		JMenuItem localeFrenchCA = new JMenuItem("French (Canada) Locale",
				Check.getIcon("flag_canada"));
		localeFrenchCA.addActionListener(new MyLocaleChangeListener("fr", "CA",
				frame));

		JMenuItem localeGerman = new JMenuItem("German Locale", Check
				.getIcon("flag_germany"));
		localeGerman.addActionListener(new MyLocaleChangeListener("de", "DE",
				frame));

		JMenuItem localeGreek = new JMenuItem("Greek Locale", Check
				.getIcon("flag_greece"));
		localeGreek.addActionListener(new MyLocaleChangeListener("el", "GR",
				frame));

		JMenuItem localeHebrew = new JMenuItem("Hebrew Locale", Check
				.getIcon("flag_israel"));
		localeHebrew.addActionListener(new MyLocaleChangeListener("iw", "IL",
				frame));

		JMenuItem localeHungarian = new JMenuItem("Hungarian Locale", Check
				.getIcon("flag_hungary"));
		localeHungarian.addActionListener(new MyLocaleChangeListener("hu",
				"HU", frame));

		JMenuItem localeItalian = new JMenuItem("Italian Locale", Check
				.getIcon("flag_italy"));
		localeItalian.addActionListener(new MyLocaleChangeListener("it", "IT",
				frame));

		JMenuItem localeJapanese = new JMenuItem("Japanese Locale", Check
				.getIcon("flag_japan"));
		localeJapanese.addActionListener(new MyLocaleChangeListener("ja", "JP",
				frame));

		JMenuItem localeNorwegian = new JMenuItem("Norwegian Locale", Check
				.getIcon("flag_norway"));
		localeNorwegian.addActionListener(new MyLocaleChangeListener("no",
				"NO", frame));

		JMenuItem localePolish = new JMenuItem("Polish Locale", Check
				.getIcon("flag_poland"));
		localePolish.addActionListener(new MyLocaleChangeListener("pl", "PL",
				frame));

		JMenuItem localePortuguese = new JMenuItem("Portuguese Locale", Check
				.getIcon("flag_portugal"));
		localePortuguese.addActionListener(new MyLocaleChangeListener("pt",
				"PT", frame));

		JMenuItem localePortugueseBR = new JMenuItem(
				"Portuguese (Brazil) Locale", Check.getIcon("flag_brazil"));
		localePortugueseBR.addActionListener(new MyLocaleChangeListener("pt",
				"BR", frame));

		JMenuItem localeRomanian = new JMenuItem("Romanian Locale", Check
				.getIcon("flag_romania"));
		localeRomanian.addActionListener(new MyLocaleChangeListener("ro", "RO",
				frame));

		JMenuItem localeRussian = new JMenuItem("Russian Locale", Check
				.getIcon("flag_russia"));
		localeRussian.addActionListener(new MyLocaleChangeListener("ru", "RU",
				frame));

		JMenuItem localeSpanish = new JMenuItem("Spanish Locale", Check
				.getIcon("flag_spain"));
		localeSpanish.addActionListener(new MyLocaleChangeListener("es", "ES",
				frame));

		JMenuItem localeSpanishMX = new JMenuItem("Spanish (Mexico) Locale",
				Check.getIcon("flag_mexico"));
		localeSpanishMX.addActionListener(new MyLocaleChangeListener("es",
				"MX", frame));

		JMenuItem localeSwedish = new JMenuItem("Swedish Locale", Check
				.getIcon("flag_sweden"));
		localeSwedish.addActionListener(new MyLocaleChangeListener("sv", "SE",
				frame));

		JMenuItem localeThai = new JMenuItem("Thai Locale", Check
				.getIcon("flag_thailand"));
		localeThai.addActionListener(new MyLocaleChangeListener("th", "TH",
				frame));

		JMenuItem localeTurkish = new JMenuItem("Turkish Locale", Check
				.getIcon("flag_turkey"));
		localeTurkish.addActionListener(new MyLocaleChangeListener("tr", "TR",
				frame));

		localeMenus.add(localeEnglish);
		localeMenus.add(localeArabic);
		localeMenus.add(localeBulgarian);
		localeMenus.add(localeChinese);
		localeMenus.add(localeChineseHK);
		localeMenus.add(localeChineseTW);
		localeMenus.add(localeCzech);
		localeMenus.add(localeDanish);
		localeMenus.add(localeDutch);
		localeMenus.add(localeFinnish);
		localeMenus.add(localeFrench);
		localeMenus.add(localeFrenchCA);
		localeMenus.add(localeGerman);
		localeMenus.add(localeGreek);
		localeMenus.add(localeHebrew);
		localeMenus.add(localeHungarian);
		localeMenus.add(localeItalian);
		localeMenus.add(localeJapanese);
		localeMenus.add(localeNorwegian);
		localeMenus.add(localePolish);
		localeMenus.add(localePortuguese);
		localeMenus.add(localePortugueseBR);
		localeMenus.add(localeRomanian);
		localeMenus.add(localeRussian);
		localeMenus.add(localeSpanish);
		localeMenus.add(localeSpanishMX);
		localeMenus.add(localeSwedish);
		localeMenus.add(localeThai);
		localeMenus.add(localeTurkish);

		return lafMenu;
	}
}
