package org.herac.tuxguitar.gui.editors.piano;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.herac.tuxguitar.gui.SystemImages;
import org.herac.tuxguitar.gui.TuxGuitar;
import org.herac.tuxguitar.gui.actions.caret.GoLeftAction;
import org.herac.tuxguitar.gui.actions.caret.GoRightAction;
import org.herac.tuxguitar.gui.actions.duration.DecrementDurationAction;
import org.herac.tuxguitar.gui.actions.duration.IncrementDurationAction;
import org.herac.tuxguitar.gui.actions.tools.SelectScaleAction;
import org.herac.tuxguitar.gui.editors.tab.Caret;
import org.herac.tuxguitar.gui.editors.tab.MeasureComponent;
import org.herac.tuxguitar.gui.editors.tab.NoteCoords;
import org.herac.tuxguitar.gui.undo.undoables.measure.UndoableMeasureGeneric;
import org.herac.tuxguitar.song.managers.SongManager;
import org.herac.tuxguitar.song.models.Duration;
import org.herac.tuxguitar.song.models.InstrumentString;
import org.herac.tuxguitar.song.models.Note;
import org.herac.tuxguitar.song.models.NoteEffect;

public class Piano extends Composite{

	private static final boolean TYPE_NOTES[] = new boolean[]{true,false,true,false,true,true,false,true,false,true,false,true};
	
	private static final int NATURAL_NOTES = 7;
	
	private static final int MAX_OCTAVES = 8;
	
	private static final int NATURAL_WIDTH = 15;
	
	private static final int SHARP_WIDTH = 8;
	
	private static final int NATURAL_HEIGHT = 60;
	
	private static final int SHARP_HEIGHT = 40;				
	
	private static final int MOUSE_MOVE_DELAY = 10;

	private PianoListener listener;
	
	private Image image;		
	
	private Composite pianoComposite;
	
	private Composite toolComposite;

	private Label durationLabel;	
	
	private Button scale;
	
	private List components;
	
	private long lastMouseMoveTime;	
	
	private boolean scaleChanges;
	
	public Piano(Composite parent, int style) {
		super(parent, style);
		this.setLayout(new GridLayout());
		this.setLayoutData(new GridData(SWT.FILL,SWT.FILL,true,true));
		this.listener =  new PianoListener();
		this.components = new ArrayList();
		
		initToolBar();
		makePiano();
	}

	

	private void initToolBar() {
		GridLayout layout = new GridLayout();
		layout.makeColumnsEqualWidth = false;
		layout.numColumns = 0;
		
	    toolComposite = new Composite(this, SWT.NONE);	    
	    	    
		// position
	    layout.numColumns ++;
		Button goLeft = new Button(toolComposite, SWT.ARROW | SWT.LEFT);
		goLeft.addSelectionListener(TuxGuitar.instance().getAction(GoLeftAction.NAME));
		
		layout.numColumns ++;
		Button goRight = new Button(toolComposite, SWT.ARROW | SWT.RIGHT);
		goRight.addSelectionListener(TuxGuitar.instance().getAction(GoRightAction.NAME));
		
		// separator		
		layout.numColumns ++;
		makeToolSeparator(toolComposite);
		
		// duration 
		layout.numColumns ++;
		Button decrement = new Button(toolComposite, SWT.ARROW | SWT.MIN);
		decrement.addSelectionListener(TuxGuitar.instance().getAction(DecrementDurationAction.NAME));

		layout.numColumns ++;
		durationLabel = new Label(toolComposite, SWT.BORDER);
		durationLabel.setImage(getDurationImage());
		
		layout.numColumns ++;
		Button increment = new Button(toolComposite, SWT.ARROW | SWT.MAX);
		increment.addSelectionListener(TuxGuitar.instance().getAction(IncrementDurationAction.NAME));
		
		// separator
		layout.numColumns ++;
		makeToolSeparator(toolComposite);
		
		// escala
		layout.numColumns ++;
		scale = new Button(toolComposite, SWT.PUSH);
		scale.setText(TuxGuitar.getProperty("scale"));
		scale.addSelectionListener(TuxGuitar.instance().getAction(SelectScaleAction.NAME));		

				
		toolComposite.setLayout(layout);	    
		toolComposite.setLayoutData(new GridData(SWT.FILL,SWT.TOP,true,true));
	}	
	

	private void makeToolSeparator(Composite parent){
		Label separator = new Label(parent,SWT.SEPARATOR);
		separator.setLayoutData(new GridData(20,20));
	}
	
	private Image getDurationImage() {
		Duration duration = (Duration) TuxGuitar.instance().getTablatureEditor().getTablature().getCaret().getDuration().clone();
		return SystemImages.getDuration(duration.getValue());		
	}

	
	private void makePiano(){
		final Point position = new Point(0,0);		
		this.image = makePianoImage();		
		this.pianoComposite = new Composite(this,SWT.BORDER | SWT.DOUBLE_BUFFERED);		
		this.pianoComposite.setLayout(new GridLayout());
		this.pianoComposite.setLayoutData(new GridData((NATURAL_WIDTH * (MAX_OCTAVES * NATURAL_NOTES) ),NATURAL_HEIGHT));
		this.pianoComposite.addPaintListener(this.listener);		
		this.pianoComposite.addMouseListener(this.listener);
	}

	/**
	 * Crea la imagen del piano
	 *
	 * @return
	 */
	private Image makePianoImage(){
		Image image = new Image(getDisplay(),(NATURAL_WIDTH * (MAX_OCTAVES * NATURAL_NOTES) ),NATURAL_HEIGHT);
		GC gc = new GC(image);
				
		int x = 0;
		int y = 0;		
		gc.setBackground(getDisplay().getSystemColor(SWT.COLOR_WHITE));
		gc.fillRectangle(x,y,(NATURAL_WIDTH * (MAX_OCTAVES * NATURAL_NOTES) ),NATURAL_HEIGHT);
		for(int i = 0; i < (MAX_OCTAVES * TYPE_NOTES.length); i ++){
			
			if(TYPE_NOTES[i % TYPE_NOTES.length]){
				gc.setForeground(getDisplay().getSystemColor(SWT.COLOR_BLACK));
				gc.drawRectangle(x,y,NATURAL_WIDTH,NATURAL_HEIGHT);
				x += NATURAL_WIDTH;
			}else{
				gc.setBackground(getDisplay().getSystemColor(SWT.COLOR_BLACK));							
				gc.fillRectangle(x - (SHARP_WIDTH / 2),y,SHARP_WIDTH,SHARP_HEIGHT);		
			}			
		}
		paintScale(gc);
		
		gc.dispose();
		return image;
	}
	
	/**
	 * Pinta la nota a partir del indice
	 * 	 
	 * @param gc
	 * @param value
	 */
	private void paintScale(GC gc){
		
		gc.setBackground(getDisplay().getSystemColor(SWT.COLOR_RED));
		gc.setForeground(getDisplay().getSystemColor(SWT.COLOR_RED));	
		int posX = 0;
		int y = 0;
		
		for(int i = 0; i < (MAX_OCTAVES * TYPE_NOTES.length); i ++){			
			int width = 0;

			if(TYPE_NOTES[i % TYPE_NOTES.length]){
				width = NATURAL_WIDTH;
				if(i > 0 && !TYPE_NOTES[(i - 1)  % TYPE_NOTES.length]){
					width -= ((SHARP_WIDTH / 2));
				}
				if(!TYPE_NOTES[(i + 1)  % TYPE_NOTES.length]){
					width -= ((SHARP_WIDTH / 2));
				}			
			}else{
				width = SHARP_WIDTH;
			}
						
			if(TuxGuitar.instance().getScaleManager().getScale().getNote(i)){			
				if(TYPE_NOTES[i % TYPE_NOTES.length] ){					
					int x = posX;
					if(i > 0 && !TYPE_NOTES[(i - 1)  % TYPE_NOTES.length]){
						x -= ((SHARP_WIDTH / 2));
					}
					
					int size = SHARP_WIDTH;
					gc.fillRectangle( (x + 1 + (((NATURAL_WIDTH - size) / 2))) ,(NATURAL_HEIGHT - size - (((NATURAL_WIDTH - size) / 2))),size,size);					
				}else{
					int size = width - 2;
					gc.fillRectangle(posX + 1, SHARP_HEIGHT - SHARP_WIDTH + 1,SHARP_WIDTH - 2,SHARP_WIDTH - 2);					
				}									
			}
						
			posX += width;
		}
			
	}	
		
	
	/**
	 * Pinta la nota a partir del indice
	 * 	 
	 * @param gc
	 * @param value
	 */
	private void paintNote(GC gc,int value){
		gc.setBackground(getDisplay().getSystemColor(SWT.COLOR_GRAY));		
		int posX = 0;
		int y = 0;
		
		for(int i = 0; i < (MAX_OCTAVES * TYPE_NOTES.length); i ++){			
			int width = 0;

			if(TYPE_NOTES[i % TYPE_NOTES.length]){
				width = NATURAL_WIDTH;
				if(i > 0 && !TYPE_NOTES[(i - 1)  % TYPE_NOTES.length]){
					width -= ((SHARP_WIDTH / 2));
				}
				if(!TYPE_NOTES[(i + 1)  % TYPE_NOTES.length]){
					width -= ((SHARP_WIDTH / 2));
				}			
			}else{
				width = SHARP_WIDTH;
			}
						
			if(i == value){			
				if(TYPE_NOTES[i % TYPE_NOTES.length]){					
					gc.fillRectangle(posX + 1,y + 1,width - 1,SHARP_HEIGHT);
					
					int x = posX;
					if(i > 0 && !TYPE_NOTES[(i - 1)  % TYPE_NOTES.length]){
						x -= ((SHARP_WIDTH / 2));
					}
					gc.fillRectangle(x + 1,(y + SHARP_HEIGHT) + 1,NATURAL_WIDTH - 1,(NATURAL_HEIGHT - SHARP_HEIGHT) - 1);					
				}else{
					gc.fillRectangle(posX + 1,y + 1,width - 1,SHARP_HEIGHT - 1);
				}									
			}
						
			posX += width;
		}
			
	}	
	
	/**
	 * Retorna el indice de la nota seleccionada
	 * 
	 * @param point
	 * @return
	 */
	private int getSelection(Point point){		
		int posX = 0;

		for(int i = 0; i < (MAX_OCTAVES * TYPE_NOTES.length); i ++){			
			int width = 0;

			if(TYPE_NOTES[i % TYPE_NOTES.length]){
				width = NATURAL_WIDTH;
				if(i > 0 && !TYPE_NOTES[(i - 1)  % TYPE_NOTES.length]){
					width -= ((SHARP_WIDTH / 2));
				}
				if(!TYPE_NOTES[(i + 1)  % TYPE_NOTES.length]){
					width -= ((SHARP_WIDTH / 2));
				}			
			}else{
				width = SHARP_WIDTH;
			}
			
			if(point.x >= posX && point.x < (posX + width)  ){			
				return i;								
			}
						
			posX += width;
		}
		return -1;	
	}		

	
	private void hit(int x, int y) {
		int value = getSelection(new Point(x,y));
		
		if (!removeNote(value)) {
			addNote(value);					
		}
	}

	private boolean removeNote(int value) {
		Iterator it = this.components.iterator();
		while (it.hasNext()) {
			MeasureComponent component = (MeasureComponent) it.next();
			if (component instanceof NoteCoords) {
				NoteCoords note = (NoteCoords) component;
				if (note.getRealValue() == value) {
			        //comienza el undoable
			        UndoableMeasureGeneric undoable = UndoableMeasureGeneric.startUndo();     
					
					Caret caret = TuxGuitar.instance().getTablatureEditor().getTablature().getCaret();
					SongManager manager = TuxGuitar.instance().getSongManager(); 
					manager.getMeasureManager().removeNote(caret.getMeasureCoords().getMeasure(),note.getNote());
					
			        //termia el undoable
					TuxGuitar.instance().getTablatureEditor().getUndoManager().addEdit(undoable.endUndo());   					
					
					return true;
				}
			}
		}
		return false;
	}	
	
	private boolean addNote(int value) {
		Caret caret = TuxGuitar.instance().getTablatureEditor().getTablature().getCaret();
		
		List strings = caret.getSongTrackCoords().getTrack().getStrings();
		for(int i = 0;i < strings.size();i ++){
			InstrumentString string = (InstrumentString)strings.get(i);
			if(value >= string.getValue()){
				boolean emptyString = true;
				
				Iterator it = this.components.iterator();
				while (it.hasNext()) {
					MeasureComponent component = (MeasureComponent) it.next();
					if (component instanceof NoteCoords && ((NoteCoords) component).getNote().getString() == string.getNumber()) {
						emptyString = false;
						break;
					}
				}
				if(emptyString){
			        //comienza el undoable
					UndoableMeasureGeneric undoable = UndoableMeasureGeneric.startUndo();
										
					Duration duration = (Duration) caret.getDuration().clone();
					Note note = new Note((value - string.getValue()), caret.getPosition(), duration, caret.getVelocity(), string.getNumber(), false, new NoteEffect());
					
					SongManager manager = TuxGuitar.instance().getSongManager();
					manager.getMeasureManager().addNote(caret.getMeasureCoords().getMeasure(),note);					
					
			        //termia el undoable
					TuxGuitar.instance().getTablatureEditor().getUndoManager().addEdit(undoable.endUndo());
					
			        //reprodusco las notas en el pulso
			        caret.getMeasureCoords().playBeat(note.getStart());
					
					return true;
				}
			}
		}
		
		
		return false;
	}	
	
	private void afterAction() {
		TuxGuitar.instance().getTablatureEditor().getTablature().getViewLayout().fireUpdate(TuxGuitar.instance().getTablatureEditor().getTablature().getCaret().getMeasureCoords().getMeasure().getNumber(), false);		
		TuxGuitar.instance().redraw();
		TuxGuitar.instance().updateCache(true);
	}


	public boolean isScaleChanges(){
		return this.scaleChanges;
	}
	
	public void setScaleChanges(boolean changes){
		this.scaleChanges = changes;
	}
	
	private void updateEditor(){
		if(isVisible()){
			if(isScaleChanges()){
				this.image.dispose();
				this.image = makePianoImage();
			}
			SongManager manager = TuxGuitar.instance().getSongManager();
			if(TuxGuitar.instance().getPlayer().isRunning()){
				this.components = TuxGuitar.instance().getEditorCache().getPlayingComponents();
			}else{
				this.components = TuxGuitar.instance().getEditorCache().getCaretComponents();
			}
		}
	}
	
	public void redraw() {
	    if(!super.isDisposed()){	    	
	        super.redraw();
	        this.pianoComposite.redraw();
	        this.durationLabel.setImage(getDurationImage());
	    }
	}

	public void redrawPlayingMode(){     
		if(!super.isDisposed()){
	        this.pianoComposite.redraw();
		}
	 }
	
	public void dispose(){
		super.dispose();
		this.image.dispose();
	}
	
    public void loadProperties(){
	    this.scale.setText(TuxGuitar.getProperty("scale"));
	    this.layout();
    }	
	
	private class PianoListener implements PaintListener,MouseListener {

		public void paintControl(PaintEvent e) {
			updateEditor();			
			e.gc.drawImage(image,0,0);			
			//pinto notas
			Iterator it = components.iterator();
			while(it.hasNext()){
				MeasureComponent component = (MeasureComponent)it.next();
				if (component instanceof NoteCoords) {
					NoteCoords note = (NoteCoords) component;
					paintNote(e.gc,note.getRealValue());
				}
			}
			e.gc.dispose();
		}
		
		public void mouseUp(MouseEvent e) {
			if(e.button == 1){				
				hit(e.x, e.y);
				afterAction();
			}else{
				TuxGuitar.instance().getAction(GoRightAction.NAME).process(e);
			}
		}			
		
		public void mouseDoubleClick(MouseEvent e) {		
		}

		public void mouseDown(MouseEvent e) {			
		}	
	}	
}
