#!/bin/bash

# Copyright (c) 2002-2008 Userful Corporation. All rights reserved.
# http://www.userful.com/

# install-license.sh - get some info from the system, submit it via URL, and 
# if we get a valid license: install it, if not: display a meaningful
# error message

function usage() {
    echo "
This script attempts to install a license key from keys.userful.com.

Options:
   -a code      supply an Activation code
   -f file      supply a File that contains an activation code
   -d           Delete the activation file after using it (happens even
                if there is an error)
   -D           only Delete the file if a license was successfully installed
                if there is an error)
   -n           do Not use the graphical display 
   -h           display this Help message and exit
   -s #         specify the number of Stations to request 
   -l file      install a local file - can be a zip file or the actual license file
"
}

if [ -f /opt/userful/debug ] ; then
    DEBUG=1
fi

TEXTDOMAINDIR=/usr/share/locale
TEXTDOMAIN=userful-control-panel

ACTIVATION_CODE=""
ACTIVATION_FILE=""
ACTIVATION_CODE_FROM_FILE=""
DELETE_ACTIVATION_FILE=false
SUPPLIED_ACTIVATION_FILE=false
NON_GRAPHICAL=false
DELETE_ACTIVATION_FILE_ON_SUCCESS=false
NUMBER_OF_STATIONS=0

PROVIDED_LICENSE=""

# parse arguments:
while getopts "a:df:hns:l:" flag ; do
    case "$flag" in
        a)
            ACTIVATION_CODE="$OPTARG"
            ;;
        d)
            DELETE_ACTIVATION_FILE=true
            ;;
        D)
            DELETE_ACTIVATION_FILE_ON_SUCCESS=true
            ;;
        f)
            SUPPLIED_ACTIVATION_FILE=true
            ACTIVATION_FILE="$OPTARG"
            ;;
        h)
            usage
            exit 0
            ;;
        n)
            NON_GRAPHICAL=true
            unset DISPLAY
            ;;
        s)
            NUMBER_OF_STATIONS="$OPTARG"
            ;;
        l)
            PROVIDED_LICENSE="$OPTARG"
            ;;
        *)
            usage
            exit 1
            ;;
    esac
done

###########################################################################
#
# DEFINES 
#
# define some basic stuff that will be used throughout the file
#
###########################################################################

# MAC ADDRESS (in various forms) --- #### THIS WILL BE REPLACED BY A UTILITY TO LOOK AT ALL INTERFACES #####
MACS=`LANG=C /sbin/ifconfig -a | grep "HWaddr" | awk '{ print $5 }'`
for MAC in $MACS ; do
    if [ -z "$MAC" ] || [ ${#MAC} != 17 ] ; then 
        # MAC is no good
        MAC=nomac
        MAC_COLONS=$MAC
        MAC_DASHES=${MAC//:/-}
        MAC_UNDERS=${MAC//:/_}
    else
        MAC_COLONS=$MAC
        MAC_DASHES=${MAC//:/-}
        MAC_UNDERS=${MAC//:/_}
		break
    fi
done

[ -n "$DEBUG" ] && echo "got MAC: $MAC"

# UMX Utils' SYS-ID (not to be confused with DiscoverStation's systemid)
SYS_ID=unknown
if [ -f /etc/userful/sys-id ] ; then
    SYS_ID_TMP=`cat /etc/userful/sys-id`
    if [ -n "$SYS_ID_TMP" ] ; then
        SYS_ID=$SYS_ID_TMP
    fi
fi
SYS_ID_PART=${SYS_ID#*-
}
# various helper programs

# check to see is local versions of the programs exists, 
# if they do, use them, if not, use the default install locations
# if those don't exists use 'echo'
ERR_DLG=/usr/share/umx-utils/error-dialog.py
if [ ! -x $ERR_DLG ] ; then
    ERR_DLG=./error-dialog.py
fi 
if [ ! -x $ERR_DLG ] ; then
    ERR_DLG=echo
fi

ACTIVATION_DLG=./activation-dialog.py
if [ ! -x $ACTIVATION_DLG ] ; then
    ACTIVATION_DLG=/usr/share/umx-utils/activation-dialog.py
fi
if [ ! -x $ACTIVATION_DLG ] ; then
    ACTIVATION_DLG=$ERR_DLG
fi

PROGRESS_DLG=./progress-dialog.py
if [ ! -x $PROGRESS_DLG ] ; then
    PROGRESS_DLG=/usr/share/umx-utils/progress-dialog.py
fi
if [ ! -x $PROGRESS_DLG ] ; then
    PROGRESS_DLG=$ERR_DLG
fi

if [ -z "$DISPLAY" ] ; then
    ERR_DLG=echo
    ACTIVATION_DLG=echo
    PROGRESS_DLG=echo
fi

if [ $NON_GRAPHICAL == true ] ; then
    unset DISPLAY
fi

################
#
# END DEFINES
#
################

################
#
# Function: install_file
#
# takes the provided file and puts it in the right place
# - makes a backup if one already exists
# - attempts to get root privileges if not already
#
# TODO: check the priority of the file and warn the user if this file probably will not be used due to another file in a higher priority
#
################
function install_file () {

    SRC_THING=$1

    STAT=`which stat`
    RET=$?
    if [ $RET != 0 ] ; then
        STAT=false
    fi

    SIZE=`$STAT -c %s $SRC_THING`
    RET=$?
    if [ $RET == 0 ] ; then
        if [ "$SIZE" != 256 ] ; then
            # the size is *NOT* the correct size to be a license file
            MSG=`gettext "The file is not the correct size to be a license file."`
            exec $ERR_DLG $MSG
        fi
    fi

    INSTALL_DIR=/etc/X11
    if [ ! -w $INSTALL_DIR ] ; then
        mv -f $SRC_THING /tmp/1Box.info &> /dev/null && SRC_THING=/tmp/1Box.info
       MSG=`gettext "License key retrieved but not installed. You must be root for automatic installation. The file has been saved to \$SRC_THING . Ask your system administrator to copy this file to \$INSTALL_DIR or re-run this application as root."`
        exec $ERR_DLG "$MSG"
    fi

    BACKUP_DIR=$INSTALL_DIR/1Box-info-backup
    mkdir -p $BACKUP_DIR 
    
    if ls $INSTALL_DIR/*1Box.info* &> /dev/null ; then
        mv -f --backup=numbered $INSTALL_DIR/*1Box.info* $BACKUP_DIR
    fi
    
    mv $SRC_THING $INSTALL_DIR/1Box.info 
    
    if [ "$DELETE_ACTIVATION_FILE_ON_SUCCESS" == true ] ; then
        if [ -n "$ACTIVATION_FILE" ] && [ -f $ACTIVATION_FILE ] ; then
            rm -f $ACTIVATION_FILE || exit 1
        fi
    fi

	MSG=`gettext "Successfully installed"`
    MESSAGE="$MSG ${SRC_THING##*/}."
    # "sucess" spelt wrong intentionally
    $ERR_DLG type=sucess "text=$MESSAGE"
}

##############################
#
# LOCAL LICENSE Directory INSTALL
#
##############################
if [ -n "$PROVIDED_LICENSE" ] ; then
    
    TMP=`mktemp -d -t install-license.sh.XXXXXX`
    trap "rm $TMP* 2>/dev/null" EXIT
    mkdir -p $TMP
    
    CONFIG_FILE=$TMP/config
    
    SAVE_FILE=$TMP-sav
    mkdir -p $SAVE_FILE
    
    echo "MAC_COLONS=$MAC_COLONS" >> $CONFIG_FILE
    echo "MAC_DASHES=$MAC_DASHES" >> $CONFIG_FILE
    echo "MAC_UNDERS=$MAC_UNDERS" >> $CONFIG_FILE
    if [ -f /etc/userful/sys-id ] ; then
	    echo -n "SYS_ID=" >> $CONFIG_FILE
	    cat /etc/userful/sys-id >> $CONFIG_FILE
    fi
    
    echo "${0##*/} -d \"$PROVIDED_LICENSE\" -c \"$CONFIG_FILE\" -s \"$SAVE_FILE\" -t \"$TMP\""
    
    MSG=`gettext "Looking for file"`
    /usr/share/umx-utils/find-license.sh -d "$PROVIDED_LICENSE" -c "$CONFIG_FILE" -s "$SAVE_FILE" -t "$TMP" |  zenity --progress --pulsate --text="$MSG" --auto-close

    #ARCHIVE=no
    
    #UNZIP_TMP=`mktemp -d -t install-license.sh.XXXXXX`
    #trap "rm -rf $UNZIP_TMP 2>/dev/null" EXIT
    
    #mkdir -p $UNZIP_TMP

    # this file would be passed from the activation dialog - so it should exist

    # file does not appear to be a zip file, check the size and maybe install it
    
    # Bug #666 - look for files that might work in the directory.

    #CONTAINING_DIR=${PROVIDED_LICENSE%/*}

    # THIS PATH should not have any quotes

    FOUND=`find "$SAVE_FILE" -type f`
    if [ -n "$FOUND" ] ; then
        for FILE in $FOUND ; do
            
	        echo "${0##*/} found $FILE"
            SIZE=256
            if [ `which stat` ] ; then
                SIZE=`stat -c %s $FILE`
            fi
            if [ "$SIZE" == 256 ] ; then
        # the size is the correct size to be a license file
                
                install_file $FILE
                
                RET=$?
                if [ $RET != 0 ] ; then
                    $ERR_DLG "text=Error: $OUT"
                fi
                exit $RET
            else 
                echo "File size is not 256: $SIZE"
            fi
        done
    fi
    
    #MSG=`gettext "The directory $PROVIDED_LICENSE does not contain a valid license file. Please choose another directory."`
    MSG=`gettext "The directory does not contain a valid license file. Please choose another directory."`
    $ERR_DLG "text=$MSG"
    exec $ACTIVATION_DLG
    
    exit 0
fi

###############################################
#
# Deal with the activation code file
#
###############################################

if [ $DELETE_ACTIVATION_FILE == true ] && [ $SUPPLIED_ACTIVATION_FILE == false ] ; then
    echo "ERROR: specified activation code file deletion, but did not specify an activation code file"
    usage
    exit 1
fi

if [ $SUPPLIED_ACTIVATION_FILE == true ] ; then
    if [ -n "$ACTIVATION_FILE" ] && [ -f "$ACTIVATION_FILE" ] ; then
        ACTIVATION_CODE_FROM_FILE=`cat "$ACTIVATION_FILE"` || exit 1
        if [ -z "$ACTIVATION_CODE_FROM_FILE" ] ; then
            echo "ERROR: activation file \"$ACTIVATION_FILE\" is empty"
            usage
            exit 1
        fi
    else
        echo "ERROR: \"$ACTIVATION_FILE\" does not exist"
        usage
        exit 1
    fi

    if [ "$DELETE_ACTIVATION_FILE" == true ] ; then
        if [ -n "$ACTIVATION_FILE" ] && [ -f $ACTIVATION_FILE ] ; then
            rm -f $ACTIVATION_FILE || exit 1
        fi
    fi
fi

# at this point the non-getopt style arguments should remain - this is backward compatible with the way it used to work

#######################################
#
# Show a progress dialog
#
#######################################

$PROGRESS_DLG &
PROGRESS_DLG_PID=$!
KILL_PROGRESS="kill -SIGKILL $PROGRESS_DLG_PID"

###################################################
#
# Build the URL to get the license from the key server
#
###################################################

URL="http://keys.userful.com/license/control/GetLicense?"

SYSID=none
if [ -n "$SYS_ID" ] ; then
    SYSID=$SYS_ID
fi
URL="${URL}sysid=$SYSID"

# We always include an activation code even if it is empty because the 
# server requires it
if [ -n "$ACTIVATION_CODE_FROM_FILE" ] ; then
    if [ -n "$ACTIVATION_CODE" ] ; then
        echo "WARNING: both an activation code and activation code file were supplied, using activation code from file."
    fi
    ACTIVATION_CODE="$ACTIVATION_CODE_FROM_FILE"
fi
if [ -n "$ACTIVATION_CODE" ] ; then
    URL="${URL}&"
    URL="${URL}activation=$ACTIVATION_CODE"
fi

# license file order:
# http://wiki.userful.ca/wiki/index.php?title=License_File
# order of preferred MAC addr formats: : - _

FILENAME=
if [ -e /etc/X11/$MAC_COLONS.1Box.info ]; then
    FILENAME="/etc/X11/$MAC_COLONS.1Box.info"
elif [ -e /etc/X11/$MAC_DASHES.1Box.info ]; then
    FILENAME=/etc/X11/$MAC_DASHES.1Box.info
elif [ -e /etc/X11/$MAC_UNDERS.1Box.info ]; then
    FILENAME=/etc/X11/$MAC_UNDERS.1Box.info
elif [ -e /etc/X11/1Box.info ] ; then
    FILENAME="/etc/X11/1Box.info"
elif [ -e /etc/X11/1Box.info.trial ]; then
    FILENAME="/etc/X11/1Box.info.trial"
fi

LICENSE_KEY=
if [ -n "$FILENAME" ] ; then
    echo "Found existing license: $FILENAME"
    LICENSE_KEY="`/usr/share/umx-utils/b64encode.py $FILENAME`"
    URL="${URL}&"
    URL="${URL}lickey=$LICENSE_KEY"
else
    URL="${URL}&lickey="
fi

if [ $SYSID != none ] ; then
    MAC=000000000000
    # why not? 00000000 - Marvin says so
fi

if [ $MAC == nomac ] ; then
    MESSAGE=`gettext "No valid MAC address found. Please contact Userful: http://keys.userful.com/ or 1-(866)-USERFUL."`
    exec $ERR_DLG "text=$MESSAGE"
else
    URL="${URL}&"
    URL="${URL}mac=${MAC//:/}"
fi

#if [ "$NUMBER_OF_STATIONS" == 0 ] ; then
#
#    X11_STATIONS=0
#
#    if [ -f /etc/X11/stations ] ; then
#        X11_STATIONS=`cat /etc/X11/stations`
#    fi
#
#    if [ "$X11_STATIONS" -gt $NUMBER_OF_STATIONS ] ; then
#        # this will also help detect if x11-stations is not a number
#        NUMBER_OF_STATIONS=$X11_STATIONS
#    fi
#fi

URL="${URL}&"
URL="${URL}stations=$NUMBER_OF_STATIONS"

# if an oem file exists, get the hash and send that too
OEMFILE=/etc/userful/oem
if [ -f $OEMFILE ] ; then
    which sha1sum &> /dev/null || OEMHASH=none
    if [ "$OEMHASH" != none ] ; then
        SHA=`sha1sum $OEMFILE`
        OEMHASH=${SHA// *}
        URL="${URL}&oem=$OEMHASH"
    fi
fi

# add the language to the url
if [ -n "$LANG" ] ; then
    URL="${URL}&"
	URL="${URL}lang=$LANG"
else
    # if it's empty, send it anyway
    URL="${URL}&"
	URL="${URL}lang="
fi

# add the version to the url
FILE=/etc/userful/dm-release
VERSION=unknown
if [ -f $FILE ] ; then
    OS_VERSION=`/usr/share/umx-utils/b64encode.py $FILE`
fi
URL="${URL}&"
URL="${URL}version=$OS_VERSION"

# add the OS version to the URL
FILE=/etc/userful/os-info
OS_VERSION="unknown=="
if [ -f $FILE ] ; then
    OS_VERSION=`/usr/share/umx-utils/b64encode.py $FILE`
fi
URL="${URL}&"
URL="${URL}osversion=$OS_VERSION"

TMP_DIR=/tmp/install-license-$$
mkdir -p $TMP_DIR

SRC_FILE=$TMP_DIR/1Box.info

echo "Attempting to get license file from $URL"

###################################
#
# Get license file
#
###################################

function get_license_file() {

    C=`which curl`
    W=`which wget`

    RET=1
# we are no longer installing "get-url.py" because the required libraries are rarely installed
#if [ -x /usr/share/1box-utils/get-url.py ] ; then
#	/usr/share/1box-utils/get-url.py "$URL" >  $SRC_FILE 2> /dev/null
#	RET=$?
#fi

# check to see is we have an Internet connection
    INTERNET=yes # test does not always work - disable for now
    INET=`LANG=C /sbin/ifconfig | grep -c "inet addr"`
    if [ $INET -gt 1 ] ; then
        INTERNET=yes
    fi
    
    if [ $INTERNET == yes ] ; then
		[ -n "$DEBUG" ] && echo "attempting to get license"
        if [ $RET != 0 ] ; then
            if [ -n "$W" ] && [ -x $W ] ; then 
				[ -n "$DEBUG" ] && echo "using wget to install license"
                wget -q -T 15 -t 1 -O $SRC_FILE "$URL"
            elif [ -n "$C" ] && [ -x $C ] ; then 
				[ -n "$DEBUG" ] && echo "using curl to install license"
                curl --max-time 15 "$URL" > $SRC_FILE 2> /dev/null
            elif [ -n "$W" ] && [ -x $W ] ; then 
				[ -n "$DEBUG" ] && echo "using mystery second wget to install license"
                wget -T 15 -t 1 -O $SRC_FILE "$URL"
            else
                if [ -d /proc/$PROGRESS_DLG_PID ] ; then
					[ -n "$DEBUG" ] && echo "killing progres dialog"
                    $KILL_PROGRESS
                fi
				[ -n "$DEBUG" ] && echo "displaying error about how we couln't find a facility to download the license"
                MSG=`gettext "Could not use python urllib, find 'wget', or 'curl' to download license file. Go to http://keys.userful.com/ to get your license."`
                exec $ERR_DLG "$MSG"
            fi
        fi
    fi
}

# to make the command line version look a little nicer
if [ -z "$DISPLAY" ] ; then
    echo
fi

#################################
#
# verify license file
#
###############################
[ -n "$DEBUG" ] && echo "verifying license"

SIZE=0
MAX_TRIES=15
TRIES=0

while [ "$SIZE" -ne 256 ]  && [ $TRIES -lt $MAX_TRIES ] ; do

    get_license_file

    if [ -f "$SRC_FILE" ] ; then
        SIZE=`stat -c "%s" $SRC_FILE`
    fi

    TRIES=$(( $TRIES + 1 ))

    if [ "$SIZE" == 0 ] ; then
        sleep $(( $RANDOM % 8 ))
    fi

done

if [ -d /proc/$PROGRESS_DLG_PID ] ; then
	$KILL_PROGRESS
fi

if [ "$SIZE" != 256 ] ; then
	[ -n "$DEBUG" ] && echo "license is bad..."

	# license is bad: it's actually an error message - show the message (or do something else)
    if [ -n "$ACTIVATION_CODE" ] ; then
        echo "${0##*/}: $ACTIVATION_DLG -a $ACTIVATION_CODE -t `cat $SRC_FILE`"
        exec $ACTIVATION_DLG -a "$ACTIVATION_CODE" -t "`cat $SRC_FILE`" 
    else
        if [ -z "$DISPLAY" ] ; then
            echo "We did not receive a license file from the server."
            echo "Please go to http://keys.userful.com/ to register. After registering, run this script again."
            echo "The text returned from the server was: "
        fi
        exec $ACTIVATION_DLG -t "`cat $SRC_FILE`" 
    fi
else
	:
	# license is good
fi

##########################################
#
# license is good: install license file
#
#########################################
[ -n "$DEBUG" ] && echo "license file is good, installing..."

install_file $SRC_FILE

exit 0
