#!/bin/bash
#
#  dm-xconfig-module.sh
# Functions to be used during X config process
###############################################
COMMENTS_CHAR="#"

############################################################
# getSectionDevice
#   Create temporal file for Device Section of xorg.conf.new
#
#   Parameters:
#       $TempFileName 
############################################################
function getSectionDevice()
{
    TempFileName=$1
    [[ $xconfigfile ]] || xconfigfile=/root/xorg.conf.new

    SectionCount=0
    SectionFlag=0
    while read CONFLINE; do

        # Remove any redundent spaces
        CONFLINE=`echo $CONFLINE`

        if [[ $SectionCount -eq 1 ]]; then
            if [[ "$CONFLINE" == "EndSection" ]]; then
                echo "$bus_id^$vendor_name^$board_name^$driver" >> $TempFileName
                SectionCount=0

            # Remove comments
            ## RE in BASH doesn't work on some distros
            #if [[ ! "$CONFLINE" =~ "^[[:space:]]*#.*" ]]; then
            else
			    #if ! echo $CONFLINE | grep -q '^[[:space:]]*#.*'; then # Do nothing for comments
			    if [[ "${CONFLINE:0:1}" != "$COMMENTS_CHAR" ]]; then # Do nothing for comments
				    # Valid line
                    entry_name=`echo $CONFLINE | awk -F'"' '{print $1}' | xargs echo`
                    case $entry_name in
                        Driver)
                            driver=`echo $CONFLINE | awk -F'"' '{print $2}' | xargs echo`
                            ;;
                        VendorName)
                            vendor_name=`echo $CONFLINE | awk -F'"' '{print $2}' | xargs echo`
                            ;;
                        BoardName)
                            board_name=`echo $CONFLINE | awk -F'"' '{print $2}' | xargs echo`
                            ;;
                        BusID)
                            bus_id=`echo $CONFLINE | awk -F'"' '{print $2}' | xargs echo`
						    bus_id=${bus_id#PCI:} # strip PCI
                            ;;
                        *)
                            :
                            ;;
                    esac
                fi
            fi
        elif [[ "$CONFLINE" == 'Section "Device"' ]]; then # SectionCount == 0
            # Found 'Section "Device"'
            SectionCount=1
            SectionFlag=1
            continue

        elif [[ "${CONFLINE:0:7}" == "Section" && $SectionFlag -eq 1 ]]; then
            # Found other Section
            #echo "Not a same section: $CONFLINE" >> /var/log/userful-start.log
            break
        fi
    done < $xconfigfile
}

############################################################
# genTempFile
#   Create temporal file for each Section of xorg.conf.new
#
#   Required Variable:
#       MXVER : The version of Multi-X used.
#
#   Parameter:
#       $SectionName : Section name of X config file
#       $TempFileName : The name of temporal section file 
############################################################
function genTempFile()
{
    SectionName=$1
    TempFileName=$2
    [[ $xconfigfile ]] || xconfigfile=/root/xorg.conf.new

    SectionCount=0
    SectionFlag=0

    while read CONFLINE; do

        # Remove any redundent spaces
        CONFLINE=`echo $CONFLINE`

        if [[ $SectionCount -eq 1 ]]; then
            if [[ "$CONFLINE" == "EndSection" ]]; then
                echo -e "$CONFLINE" >> $TempFileName
                SectionCount=$(( SectionCount + 1 ))
                continue
            else
                OPTION_NAME=$(echo $CONFLINE | awk -F'"' '{print $1}')

                if [ "$OPTION_NAME" != "${OPTION_NAME//VendorName}" ] || [ "$OPTION_NAME" != "${OPTION_NAME//ModelName}" ]; then
                    OPTION_STR=$(echo $CONFLINE | awk -F'"' '{print $2}')
                    if [ x"$OPTION_STR" == x ]; then
                        OPTION_STR="Unknown"
                    fi
                    echo -e "\t$OPTION_NAME\t\"$OPTION_STR\"" >> $TempFileName

                else
                    echo -e "\t$CONFLINE" >> $TempFileName
                fi
            fi

        elif [[ "$CONFLINE" == 'Section "'$SectionName'"' ]]; then # SectionCount == 0 or greater than 1
                if [[ $SectionCount -eq 0 ]]; then
                    echo -e "$CONFLINE" > $TempFileName
                else
                    echo -e "$CONFLINE" >> $TempFileName
                fi

                SectionCount=1
                SectionFlag=1
                continue

        elif [[ "${CONFLINE:0:7}" == "Section" && $SectionFlag -eq 1 ]]; then
            # Found another Section
            #echo "Not a same section: $CONFLINE" >> /var/log/userful-start.log
            break
        fi

    done < $xconfigfile
}

#########################################################################
# parse_xorg_conf_new
#    Parse xorg.conf.new file and create a temporal file for each section.
#
#   Required Variable:
#       FILES_SECTION_FILE
#       MODULE_SECTION_FILE
#       MONITOR_SECTION_FILE
#       LIB : (Default - lib)
#       MXVER : The version of Multi-X used. (Default - MX11R6)
# 		xconfigtype : current config type  <-1box|-check|-one|-all>
#       
#   Parameters:
#       $xconfigfile : X config file to parse
#########################################################################
function parse_xorg_conf_new()
{
    xconfigfile=$1

    ###############
    # Files section
    ###############
    genTempFile Files $FILES_SECTION_FILE

    SECTION_STRING="Section \"Files\""

    if [[ "$xconfigtype" == "-1box" ]]; then

        # Module path from Multi-X
		if grep -q -m 1 "MX11R7" $FILES_SECTION_FILE; then
			MODULEPATH="ModulePath\t\"\/opt\/MX11R7\/lib\/xorg\/modules\""
		else
			if [[ "$MXVER" == "MX11R6" ]]; then
				MODULEPATH="ModulePath\t\"\/opt\/MX11R6\/$LIB\/modules\""
			else
				MODULEPATH="ModulePath\t\"\/opt\/MX11R7\/lib\/xorg\/modules\""
			fi
		fi

        # Module path from updates
		if [ -d /usr/$LIB/xorg/modules/updates ]; then
			MODULEPATH="$MODULEPATH\n\tModulePath\t\"\/usr\/$LIB\/xorg\/modules\/updates\""
		elif [ -d /usr/X11R6/$LIB/modules/updates ]; then
			MODULEPATH="$MODULEPATH\n\tModulePath\t\"\/usr\/X11R6\/$LIB\/modules\/updates\""
        fi

        # Module path from normal X
		if [ -d /usr/$LIB/xorg/modules ]; then
			MODULEPATH="$MODULEPATH\n\tModulePath\t\"\/usr\/$LIB\/xorg\/modules\""
		elif [ -d /usr/X11R6/$LIB/modules ]; then
			MODULEPATH="$MODULEPATH\n\tModulePath\t\"\/usr\/X11R6\/$LIB\/modules\""
		fi

		sed -i -e "/ModulePath/d" $FILES_SECTION_FILE
		sed -i -e "s/$SECTION_STRING/$SECTION_STRING\n\t$MODULEPATH/" $FILES_SECTION_FILE

    else
        # -one -all -check, These options create the configuration file for normal X server
		
		if [ -d /usr/$LIB/xorg/modules ]; then
			MODULEPATH="ModulePath\t\"\/usr\/$LIB\/xorg\/modules\""
		elif [ -d /usr/X11R6/$LIB/modules ]; then
			MODULEPATH="ModulePath\t\"\/usr\/X11R6\/$LIB\/modules\""
		fi

		if [ -d /usr/$LIB/xorg/modules/updates ]; then
			MODULEPATH="$MODULEPATH\n\tModulePath\t\"\/usr\/$LIB\/xorg\/modules\/updates\""
		elif [ -d /usr/X11R6/$LIB/modules/updates ]; then
			MODULEPATH="$MODULEPATH\n\tModulePath\t\"\/usr\/X11R6\/$LIB\/modules\/updates\""
		fi

		sed -i -e "/ModulePath/d" $FILES_SECTION_FILE
		sed -i -e "s/$SECTION_STRING/$SECTION_STRING\n\t$MODULEPATH/" $FILES_SECTION_FILE
    fi

    #################
    # Module section
    #################
    genTempFile Module $MODULE_SECTION_FILE

    sed -i -e "/\"vnc\"/d" $MODULE_SECTION_FILE
    sed -i -e "/\"dri\"/d" $MODULE_SECTION_FILE
    sed -i -e "/\"glx\"/d" $MODULE_SECTION_FILE

    ##
    ## TODO:
    ##  - record and xtrap module cause problem in SLED10 with the new Multi-X
    sed -i -e "/\"record\"/d" $MODULE_SECTION_FILE
    sed -i -e "/\"xtrap\"/d" $MODULE_SECTION_FILE

    SECTION_STRING="Section \"Module\""
    DISABLE_GLX="Disable\t\"glx\""
    sed -i -e "s/$SECTION_STRING/$SECTION_STRING\n\t$DISABLE_GLX/" $MODULE_SECTION_FILE

    DISABLE_DRI="Disable\t\"dri\""
    sed -i -e "s/$SECTION_STRING/$SECTION_STRING\n\t$DISABLE_DRI/" $MODULE_SECTION_FILE

    XFree86_DGA="SubSection \"extmod\"\n\t\tOption \"omit XFree86-DGA\"\n\tEndSubSection"
    sed -i -e "s/$SECTION_STRING/$SECTION_STRING\n\t$XFree86_DGA/" $MODULE_SECTION_FILE

    ##################
    # Monitor section
    ##################
    genTempFile Monitor $MONITOR_SECTION_FILE

    sed -i -e "/\"DPMS\"/d" $MONITOR_SECTION_FILE

    ##################
    # Device section
    ##################
    getSectionDevice $DEVICE_SECTION_FILE
}

################################################
# genHeader
#   Generate the header of a temporal xorg.conf
#
#   Required Variables:
#       TEMP_XORG_CONF
#   Parameters:
################################################
function genHeader()
{
    echo '###################################' > $TEMP_XORG_CONF
    if [[ "$xconfigtype" == "-1box" ]]; then
        echo '# userful.Mxorg.conf' >> $TEMP_XORG_CONF
    elif [[ "$xconfigtype" == "-all" ]]; then
        echo '# userful.xorg.conf.all' >> $TEMP_XORG_CONF
    elif [[ "$xconfigtype" == "-one" ]]; then
        echo '# userful.xorg.conf.one' >> $TEMP_XORG_CONF
    elif [[ "$xconfigtype" == "-check" ]]; then
        echo '# userful.xorg.conf.check' >> $TEMP_XORG_CONF
    fi
    echo '#    Created by Userful' >> $TEMP_XORG_CONF
    echo "###################################" >> $TEMP_XORG_CONF
}

#########################################
# genSectionCommon
#   Generate section based on parameters
#
#   Required Variables:
#       TEMP_XORG_CONF
#   Parameter:
#       $sectionname
#       $sectiontempfile
########################################
function genSectionCommon()
{
    sectionname=$1
    sectiontempfile=$2

    echo -e "\n########################" >> $TEMP_XORG_CONF
    echo "# $sectionname Section" >> $TEMP_XORG_CONF
    echo "########################" >> $TEMP_XORG_CONF

    if [[ -f $sectiontempfile ]]; then
        cat $sectiontempfile >> $TEMP_XORG_CONF
    fi
}

##########################################
# genFilesSection
# 	- Generate Files section
##########################################   
function genFilesSection()
{
	if [[ -f /usr/X11R6/$LIB/X11/rgb.txt ]]; then
		RGBPATH=/usr/X11R6/$LIB/X11/rgb
	elif [[ -f /usr/share/X11/rgb.txt ]]; then
		RGBPATH=/usr/share/X11/rgb
	fi
	echo '
#################
# Files Section
#################' >>  $TEMP_XORG_CONF
	echo 'Section "Files"
	RgbPath 	"'$RGBPATH'"' >>  $TEMP_XORG_CONF

	if [[ "$xconfigtype" == "-1box" ]]; then
		# add Modulepath for Multi-X config file
		if [[ "$MXVER" == "MX11R6" ]]; then
			echo '	ModulePath	"/opt/MX11R6/'$LIB'/modules"' >> $TEMP_XORG_CONF
		else
			echo '	ModulePath	"/opt/MX11R7/lib/xorg/modules"' >> $TEMP_XORG_CONF
		fi

		# add module path for nVidia driver. SuSE Linux uses different path for nvidia driver
		if [[ -f /usr/$LIB/xorg/modules/updates/drivers/nvidia_drv.o || \
			  -f /usr/$LIB/xorg/modules/updates/drivers/nvidia_drv.so ]]; then
			echo '	ModulePath	"/usr/'$LIB'/xorg/modules/updates"' >> $TEMP_XORG_CONF
		elif [[ -f /usr/X11R6/$LIB/modules/updates/drivers/nvidia_drv.o || \
			  -f /usr/X11R6/$LIB/modules/updates/drivers/nvidia_drv.so ]]; then
			echo '	ModulePath	"/usr/X11R6/'$LIB'/modules/updates"' >> $TEMP_XORG_CONF
		fi
	fi

	if [[ -d /usr/$LIB/xorg/modules ]]; then
		echo '	ModulePath	"/usr/'$LIB'/xorg/modules"
	FontPath	"/opt/MX11R7/share/X11/fonts/misc/"
	FontPath	"/opt/MX11R7/share/X11/fonts/TTF/"
	FontPath	"/opt/MX11R7/share/X11/fonts/OTF/"
	FontPath	"/opt/MX11R7/share/X11/fonts/Type1/"
	FontPath	"/opt/MX11R7/share/X11/fonts/75dpi/"
	FontPath	"/opt/MX11R7/share/X11/fonts/100dpi/"
EndSection' >> $TEMP_XORG_CONF

	elif [[ -d /usr/X11R6/$LIB/modules ]]; then
		echo '	ModulePath	"/usr/X11R6/'$LIB'/modules"
	FontPath	"/usr/X11R6/lib/X11/fonts/misc/"
	FontPath	"/usr/X11R6/lib/X11/fonts/TTF/"
	FontPath	"/usr/X11R6/lib/X11/fonts/OTF/"
	FontPath	"/usr/X11R6/lib/X11/fonts/Type1/"
	FontPath	"/usr/X11R6/lib/X11/fonts/75dpi/"
	FontPath	"/usr/X11R6/lib/X11/fonts/100dpi/"
EndSection' >> $TEMP_XORG_CONF
	fi
}

#########################################
# genServerflagsSection
#   Generate ServerFlags section
#
#   Required Variables:
#       TEMP_XORG_CONF
#   Parameter:
########################################
function genServerflagsSection()
{
    if [[ "$xconfigtype" == "-1box" ]]; then
        echo '
######################
# ServerFlags Section
######################
Section "ServerFlags"
	Option  "Xinerama"	"off"
   	Option  "Clone"		"off"
   	Option  "RandR"		"off"
	Option	"DontVTSwitch"	"on"
	#Option	"DontZap"	"on"
	Option	"DontZoom"	"on"
	Option  "DisableVidModeExtension"   "on"
	Option  "IgnoreABI"   "on"
	Option  "AutoAddDevices"   "false"
EndSection' >>  $TEMP_XORG_CONF

    elif [[ "$xconfigtype" == "-one" ]]; then
        echo '
######################
# ServerFlags Section
######################
Section "ServerFlags"
	Option  "IgnoreABI"   "on"
	Option  "AutoAddDevices"   "false"
EndSection' >>  $TEMP_XORG_CONF
    fi
}

##############################
# genModuleSection
# 	- Generate Module section
##############################
function genModuleSection()
{
	echo '
#################
# Module Section
#################
Section "Module"
	Load "dbe"
	SubSection "extmod"
            Option "omit XFree86-DGA"
    	EndSubSection
	#Load "record"
	Load "freetype"
	Load "type1"
    Disable "glx"
    Disable "dri"
EndSection' >>  $TEMP_XORG_CONF
}

#########################################
# Resolutions
#   Returns Modelines for specified Monitor 
#
#   Required Variables:
#   Parameter: 
#      $BUSID 
#      $SCREENID
#      $RESOLUTIONS - /etc/X11/monitors.conf
########################################
genResolutions() 
{
	local MODE=$1 # not used
	local BUSID=$2
	local SCREENID=$3
	local RESOLUTIONS=$4
	local Modes=`cat $RESOLUTIONS | grep "$BUSID\^$SCREENID" | cut -d"^" -f5`
	Modes=${Modes//"'Modeline"/        Modeline}
	Modes=${Modes//"'"/^}
	echo $Modes
}

#########################################
# genMonitorSection
#   Generate Monitor section
#
#   Required Variables:
#       TEMP_XORG_CONF
#   Parameter:
#
# TODO:
#   - Find Monitor information. (read-edid)
#   - Find Modeline for monitor
########################################
function genMonitorSection()
{
	local TEMP_XORG_CONF=$1
	local MVGAINFO=$2
	local RESOLUTIONS=$3
    echo '
######################
# Monitor Section
######################' >> $TEMP_XORG_CONF

    # Check the Probed Monitor first
    # Can be a conflict with umx-probe-monitor.sh
    if [ -s /tmp/Monitor.Probed ]; then
        cat /tmp/Monitor.Probed >>  $TEMP_XORG_CONF
    fi

#    if [[ -s $MONITOR_SECTION_FILE ]]; then
#        cat $MONITOR_SECTION_FILE >>  $TEMP_XORG_CONF
#    fi

    if [[ "$xconfigtype" == "-1box" ]]; then 
        if [[ "$SYS_RESOLUTION" == "1" ]]; then
            H_SYNC="28-78"
		    V_REFRESH="43-75"
        elif [[ "$SYS_RESOLUTION" == "2" ]]; then
            H_SYNC="28-78"
		    V_REFRESH="43-75"
        elif [[ "$SYS_RESOLUTION" == "3" ]]; then
            H_SYNC="28-78"
		    V_REFRESH="43-75"
        elif [[ "$SYS_RESOLUTION" == "4" ]]; then
            H_SYNC="31-82"
		    V_REFRESH="56-75"
        else
            H_SYNC="28-78"
		    V_REFRESH="43-75"
        fi

    else # "-one" || "-all" || "-check"
        H_SYNC="28-78"
		V_REFRESH="43-75"
    fi

	log "Looking for monitors resolutions in %s" "$RESOLUTIONS"
	if [ -e $RESOLUTIONS ]; then 
        # we are relaying on the semantic fact that the order of 
        # BUSID is the same in Devices and /etc/X11/monitors.conf
        # if some monitors were not detected durring installation 
        # and user uses monitors from different vendor or model 
        # we can have a problem
		local Screen=0

		while read VIDEOCARD; do
			log "Looking configured monitors for %s videocard." "$VIDEOCARD"
			local BUSID=`echo "$VIDEOCARD" | awk -F'^' '{print $1}'`
			local HEADNO=`echo "$VIDEOCARD" | awk -F'^' '{print $5}'`
			log "Found %i monitors on %s videocard." "$HEADNO" "$BUSID"
			local i
			for (( i=0; i<=$HEADNO-1; i++ )); do
				log "Configuring monitor(%i) on %s(%s) videocard." "$i" "$BOARDNAME" "$BUSID"
				IFS='^' && Modelines=( $(genResolutions $1 "$BUSID" "$i" "$RESOLUTIONS") )
				echo '
Section "Monitor"
	Identifier "Monitor'$Screen'" # '$BOARDNAME'('$BUSID') Screen '$Screen'
	VendorName "Monitor Vendor"
	ModelName "Monitor Model"
	HorizSync '$H_SYNC'
	VertRefresh '$V_REFRESH'
'${Modelines[*]}'
EndSection' >> $TEMP_XORG_CONF
				Screen=$((Screen+1))
			done
			log "%i monitors were succesfully configured for %s(%s) videocard." "$HEADNO" "$BOARDNAME" "$BUSID"
		done < $MVGAINFO
	else
		echo 'Section "Monitor"
	Identifier  "Monitor"
	VendorName  "Monitor Vendor"
	ModelName   "General Monitor"
	HorizSync '$H_SYNC'
	VertRefresh '$V_REFRESH'
EndSection' >> $TEMP_XORG_CONF
	fi
}

########################################
# mga_multihead_config
#
#   Required Variables:
#       xconfigtype, CURR_CARD_TYPE, SYS_TYPE, STATION_TYPE[]
########################################
function mga_multihead_config()
{
	# if single-view, laptop or not userful.Moxrg.conf configuration
	if [[ "$CURR_CARD_TYPE" == "S" || "$xconfigtype" != "-1box" || "$SYS_TYPE" == "1" ]]; then
		echo 'Section "Device"
	Identifier "Videocard'$MXDEVNO'"
	Driver "'$DRIVER'"
	VendorName "'$VENDORNAME'"
	BoardName "'$BOARDNAME'"
	BusID "PCI:'$BUSID'"
	Screen 0
	Option "MGASDRAM" "on"
	Option "SWCursor" "on"
	Option "HWCursor" "off"
EndSection' >>  $TEMP_XORG_CONF
		echo 'Section "Device"
	Identifier "Videocard'$((MXDEVNO+1))'"
	Driver "'$DRIVER'"
	VendorName "'$VENDORNAME'"
	BoardName "'$BOARDNAME'"
	BusID "PCI:'$BUSID'"
	Screen 1
	Option "MGASDRAM" "on"
	Option "SWCursor" "on"
	Option "HWCursor" "off"
EndSection' >>  $TEMP_XORG_CONF
   		MXDEVNO=$((MXDEVNO+2))

	else # dualview configuration
		echo 'Section "Device"
	Identifier "Videocard'$MXDEVNO'"
	Driver "'$DRIVER'"
	VendorName "'$VENDORNAME'"
	BoardName "'$BOARDNAME'"
	BusID "PCI:'$BUSID'"
	Screen 0
	Option "MergedFB"
	Option "Monitor2Position" "RightOf"
	Option "Monitor2HSync" "'$H_SYNC'"
	Option "Monitor2VRefresh" "'$V_REFRESH'"
	Option "MGASDRAM" "on"
	Option "SWCursor" "on"
	Option "HWCursor" "off"' >> $TEMP_XORG_CONF

		if [ "$SYS_RESOLUTION" == "1" ]; then
			echo '	Option "MetaModes" "640x480-640x480"' >>  $TEMP_XORG_CONF
		elif [ "$SYS_RESOLUTION" == "2" ]; then
			echo '	Option "MetaModes" "800x600-800x600 640x480-640x480"' >>  $TEMP_XORG_CONF
		elif [ "$SYS_RESOLUTION" == "3" ]; then
			echo '	Option "MetaModes" "1024x768-1024x768 800x600-800x600 640x480-640x480"' >> $TEMP_XORG_CONF
		elif [ "$SYS_RESOLUTION" == "4" ]; then
			echo '	Option "MetaModes" "1280x1024-1280x1024 1024x768-1024x768 800x600-800x600 640x480-640x480"' >> $TEMP_XORG_CONF
		elif [ "$SYS_RESOLUTION" == "5" ]; then
            PROBED_RES=$(grep -m1 "ModeLine" /tmp/Monitor.Probed | awk -- '{print $2}' | tr -d '"')
			echo '	Option "MetaModes" "'${PROBED_RES}'-'${PROBED_RES}'"' >>  $TEMP_XORG_CONF
		fi
		echo 'EndSection' >> $TEMP_XORG_CONF

		STATION_TYPE[$MXDEVNO]=D # Set the Vitual in Screen section
		SYS_DUALVIEW=$((SYS_DUALVIEW-1))
   		MXDEVNO=$((MXDEVNO+1))
	fi
}

#########################################
# genScreenSection
#   Generate Screen section
#
#   Required Variables:
#       TEMP_XORG_CONF, SXDEVNO, MXDEVNO, xconfigtype, SYS_RESOLUTION
########################################
function genScreenSection()
{
	[[ $SXDEVNO ]] || SXDEVNO=1
	[[ $MXDEVNO ]] || MXDEVNO=1

	if [[ "$xconfigtype" == "-one" ]]; then
		NUM_OF_HEAD=$SXDEVNO
	else
		NUM_OF_HEAD=$MXDEVNO
	fi

	if [[ "$xconfigtype" == "-one" || "$xconfigtype" == "-all" || "$xconfigtype" == "-check" ]]; then
		MODES="Modes \"640x480\""
	else # "-1box"
		if [[ "$SYS_RESOLUTION" == "1" ]]; then
			MODES="Modes \"640x480\""
			VIRTUAL="Virtual 1280 480"
		elif [[ "$SYS_RESOLUTION" == "2" ]]; then
			MODES="Modes \"800x600\" \"640x480\""
			VIRTUAL="Virtual 1600 600"
		elif [ "$SYS_RESOLUTION" == "3" ]; then
			MODES="Modes \"1024x768\" \"800x600\" \"640x480\""
			VIRTUAL="Virtual 2048 768"
		elif [ "$SYS_RESOLUTION" == "4" ]; then
			MODES="Modes \"1280x1024\" \"1024x768\" \"800x600\" \"640x480\""
			VIRTUAL="Virtual 2560 1024"
		elif [ "$SYS_RESOLUTION" == "5" ]; then
            PROBED_RES=$(grep -m1 "ModeLine" /tmp/Monitor.Probed | awk -- '{print $2}' | tr -d '"')
			MODES="Modes \"$PROBED_RES\""
            VIRTUAL="Virtual $(( ${PROBED_RES%x*[0-9]} * 2)) ${PROBED_RES#[0-9]*x}"
		else
			# If SYS_RESOLUTION doesn't have vaild value.
			MODES="Modes \"800x600\" \"640x480\""
			VIRTUAL="Virtual 1600 600"
		fi
	fi

    echo '
######################
# Screen Section
######################' >> $TEMP_XORG_CONF
    echo $NUM_OF_HEAD >&2

	for (( i=0; i<$NUM_OF_HEAD; i++ )) ; do
		echo 'Section "Screen"
	Identifier "Screen'$i'"
	Device "Videocard'$i'"' >> $TEMP_XORG_CONF
	if [ -e /etc/X11/monitors.conf ]; then 
		echo '  Monitor "Monitor'$i'' >> $TEMP_XORG_CONF
		echo '  Option  "ExactModeTimingsDVI"   "TRUE"' >> $TEMP_XORG_CONF
	else
		# Use generic Monitor on every screen for now.
		if [ "$SYS_RESOLUTION" == "5" ]; then
		    echo "	Monitor \"probed\"" >> $TEMP_XORG_CONF
        else
		    echo "	Monitor \"Monitor\"" >> $TEMP_XORG_CONF
        fi
	fi

		# handle dual-view and single-view
		if [[ "${STATION_TYPE[i]}" == S ]]; then
			# SingleView - no Virtual option
			echo '	DefaultDepth 16
	SubSection "Display"
		Depth 8
		'$MODES'
	EndSubSection
	SubSection "Display"
		Depth 15
		'$MODES'
	EndSubSection
	SubSection "Display"
		Depth 16
		'$MODES'
	EndSubSection
	SubSection "Display"
		Depth 24
		'$MODES'
	EndSubSection
EndSection' >> $TEMP_XORG_CONF
			echo >> $TEMP_XORG_CONF
		else
			# DualView - With Virtual option
			echo '	DefaultDepth 16
	SubSection "Display"
		Depth 8
		'$MODES'
		'$VIRTUAL'
	EndSubSection
	SubSection "Display"
		Depth 15
		'$MODES'
		'$VIRTUAL'
	EndSubSection
	SubSection "Display"
		Depth 16
		'$MODES'
		'$VIRTUAL'
	EndSubSection
	SubSection "Display"
		Depth 24
		'$MODES'
		'$VIRTUAL'
	EndSubSection
EndSection' >> $TEMP_XORG_CONF
			echo >> $TEMP_XORG_CONF 
		fi
	done
}

########################################
# radeon_multihead_config
#
#   Required Variables:
#       xconfigtype, CURR_CARD_TYPE, SYS_TYPE, STATION_TYPE[]
#
#  TODO:
#   - Added AccelMethod EXA. Need to test
########################################
function radeon_multihead_config()
{
	# if single-view, laptop or not userful.Moxrg.conf configuration
	if [[ "$CURR_CARD_TYPE" == "S" || "$xconfigtype" != "-1box" || "$SYS_TYPE" == "1" ]]; then

		echo 'Section "Device"
	Identifier "Videocard'$MXDEVNO'"
	Driver "radeon"
	VendorName "'$VENDORNAME'"
	BoardName "'$BOARDNAME'"
	BusID "PCI:'$BUSID'"
	Screen 0' >> $TEMP_XORG_CONF

		if [ "$SYS_TYPE" == "1" ]; then # Laptop
			echo '	Option "MonitorLayout" "LVDS, CRT"' >> $TEMP_XORG_CONF
		else
			echo '	Option "MonitorLayout" "CRT, CRT"' >> $TEMP_XORG_CONF
		fi
		echo '	Option "DisplayPriority" "BIOS"' >> $TEMP_XORG_CONF
		echo '	Option "SWCursor" "on"
	Option "HWCursor" "off"
	Option "ColorTiling" "off"
        Option "AccelMethod" "EXA"
EndSection' >> $TEMP_XORG_CONF

   		echo 'Section "Device"
	Identifier "Videocard'$((MXDEVNO+1))'"
	Driver "radeon"
	VendorName "'$VENDORNAME'"
	BoardName "'$BOARDNAME'"
	BusID "PCI:'$BUSID'"
	Screen 1' >> $TEMP_XORG_CONF

		if [ "$SYS_TYPE" == "1" ]; then
			echo '	Option "MonitorLayout" "LVDS, CRT"' >> $TEMP_XORG_CONF
		else
			echo '	Option "MonitorLayout" "CRT, CRT"' >> $TEMP_XORG_CONF
		fi
		echo '	Option "DisplayPriority" "BIOS"' >> $TEMP_XORG_CONF
		echo '	Option "SWCursor" "on"
	Option "HWCursor" "off"
	Option "ColorTiling" "off"
        Option "AccelMethod" "EXA"
EndSection' >> $TEMP_XORG_CONF
		MXDEVNO=$((MXDEVNO+2))

	else # dualview configuration
		echo 'Section "Device"
	Identifier "Videocard'$MXDEVNO'"
	Driver "radeon"
	VendorName "'$VENDORNAME'"
	BoardName "'$BOARDNAME'"
	BusID "PCI:'$BUSID'"
	Option "CRT2ForceOn" "true"
	Option "ForceCRT2Type" "VGA"
	Option "ForceCRT1" "true"
	Option "MergedFB" "true"
	Option "CRT2Position" "RightOf"
	Option "SWCursor" "on"
	Option "HWCursor" "off"
	Option "ColorTiling" "off"
EndSection' >> $TEMP_XORG_CONF
		STATION_TYPE[$MXDEVNO]=D # Set the Virtual in screen section
		SYS_DUALVIEW=$((SYS_DUALVIEW-1))
    	MXDEVNO=$((MXDEVNO+1))
	fi
}

########################################
# nvidia_multihead_config
#
#   Required Variables:
#       xconfigtype, CURR_CARD_TYPE, SYS_TYPE, STATION_TYPE[]
########################################
function nvidia_multihead_config()
{
	# if single-view, laptop or not userful.Moxrg.conf configuration
	if [[ "$CURR_CARD_TYPE" == "S" || "$xconfigtype" != "-1box" || "$SYS_TYPE" == "1" ]]; then

		echo 'Section "Device"
	Identifier "Videocard'$MXDEVNO'"
	Driver "nvidia"
	VendorName "'$VENDORNAME'"
	BoardName "'$BOARDNAME'"
	BusID "PCI:'$BUSID'"
	Screen 0' >> $TEMP_XORG_CONF
		if [ "$SYS_TYPE" == "1" ]; then
			echo '	Option "ConnectedMonitor" "DFP, CRT"' >> $TEMP_XORG_CONF
		else
			echo '	Option "ConnectedMonitor" "CRT, CRT"' >> $TEMP_XORG_CONF
		fi
		echo '	Option "SWCursor" "on"
	Option "HWCursor" "off"
EndSection' >> $TEMP_XORG_CONF
		echo >>  $TEMP_XORG_CONF

   		echo 'Section "Device"
	Identifier "Videocard'$((MXDEVNO+1))'"
	Driver "nvidia"
	VendorName "'$VENDORNAME'"
	BoardName "'$BOARDNAME'"
	BusID "PCI:'$BUSID'"
	Screen 1' >> $TEMP_XORG_CONF
		if [ "$SYS_TYPE" == "1" ]; then
			echo '	Option "ConnectedMonitor" "DFP, CRT"' >>  $TEMP_XORG_CONF
		else
			echo '	Option "ConnectedMonitor" "CRT, CRT"' >> $TEMP_XORG_CONF
		fi
		echo '	Option "SWCursor" "on"
	Option "HWCursor" "off"
EndSection' >> $TEMP_XORG_CONF
		echo >> $TEMP_XORG_CONF
		MXDEVNO=$((MXDEVNO+2))

	else # dualview configuration
		echo 'Section "Device"
	Identifier "Videocard'$MXDEVNO'"
	Driver "nvidia"
	VendorName "'$VENDORNAME'"
	BoardName "'$BOARDNAME'"
	BusID "PCI:'$BUSID'"
	Option "TwinView" "on"
	Option "TwinViewOrientation" "RightOf"
	Option "ConnectedMonitor" "CRT,CRT"' >> $TEMP_XORG_CONF
		if [ "$SYS_RESOLUTION" == "1" ]; then
			echo '	Option "MetaModes" "640x480,640x480"' >> $TEMP_XORG_CONF
		elif [ "$SYS_RESOLUTION" == "2" ]; then
			echo '	Option "MetaModes" "800x600,800x600; 640x480,640x480"' >> $TEMP_XORG_CONF
		elif [ "$SYS_RESOLUTION" == "3" ]; then
			echo '	Option "MetaModes" "1024x768,1024x768; 800x600,800x600; 640x480,640x480"' >> $TEMP_XORG_CONF
		elif [ "$SYS_RESOLUTION" == "4" ]; then
			echo '	Option "MetaModes" "1280x1024,1280x1024; 1024x768,1024x768; 800x600,800x600; 640x480,640x480"' >> $TEMP_XORG_CONF
		elif [ "$SYS_RESOLUTION" == "5" ]; then
            PROBED_RES=$(grep -m1 "ModeLine" /tmp/Monitor.Probed | awk -- '{print $2}' | tr -d '"')
			echo '	Option "MetaModes" "'${PROBED_RES}','${PROBED_RES}'"' >>  $TEMP_XORG_CONF
		fi
		echo '	Option "NoTwinViewXineramaInfo" "true"  # Dont change this option, otherwise X will crash.
	Option "SWCursor" "on"
	Option "HWCursor" "off"
EndSection' >> $TEMP_XORG_CONF
		STATION_TYPE[$MXDEVNO]=S # Do not set the Vitual in Screen section
		SYS_DUALVIEW=$((SYS_DUALVIEW-1))
    	MXDEVNO=$((MXDEVNO+1))
	fi
}

############################
# configSingleCard
#   Create Device section for the first video card.
#
#   Required Variables:
#   	DRIVER, SXDEVNO, VENDORNAME, BOARDNAME, BUSID
#
#############################
function configSingleCard()
{
	echo 'Section "Device"
	Identifier "Videocard'$SXDEVNO'"
	Driver "'$DRIVER'"
	VendorName "'$VENDORNAME'"
	BoardName "'$BOARDNAME'"
	BusID "PCI:'$BUSID'"
	Option "ColorTiling" "off"
EndSection' >> $TEMP_XORG_CONF

	SXDEVNO=$((SXDEVNO+1))
}

############################
# configMultiCard
#   Create Device section for Multiple video cards.
#
#   Required Variables:
#   	HEADNO, DRIVER, MXDEVNO, VENDORNAME, BOARDNAME, BUSID, xconfigtype
#
#############################
function configMultiCard()
{
	if [[ $HEADNO -eq 1 ]]; then # Single head configuration
		case "$DRIVER" in
			"savage")
				echo 'Section "Device"
	Identifier "Videocard'$MXDEVNO'"
	Driver "'$DRIVER'"
	VendorName "'$VENDORNAME'"
	BoardName "'$BOARDNAME'"
	BusID "PCI:'$BUSID'"
	#Option "shadowFB" "on"
	Option "SWCursor" "on"
	Option "HWCursor" "off"
EndSection' >>  $TEMP_XORG_CONF
				MXDEVNO=$((MXDEVNO+1))
				;;

			"i810"|"intel")
				echo 'Section "Device"
	Identifier "Videocard'$MXDEVNO'"
	Driver "'$DRIVER'"
	VendorName "'$VENDORNAME'"
	BoardName "'$BOARDNAME'"
	BusID "PCI:'$BUSID'"
	Option "SWCursor" "on"
	Option "HWCursor" "off"
        Option "NoAccel" "true"
EndSection' >>  $TEMP_XORG_CONF
				MXDEVNO=$((MXDEVNO+1))
				;;

			*)
				echo 'Section "Device"
	Identifier "Videocard'$MXDEVNO'"
	Driver "'$DRIVER'"
	VendorName "'$VENDORNAME'"
	BoardName "'$BOARDNAME'"
	BusID "PCI:'$BUSID'"
	Option "SWCursor" "on"
	Option "HWCursor" "off"
	Option "ColorTiling" "off"
EndSection' >>  $TEMP_XORG_CONF 
				MXDEVNO=$((MXDEVNO+1))
				;;
		esac

	else # dual head card
		case "$DRIVER" in
			"mga")
				mga_multihead_config 
				;;
			"radeon"|"ati")
				DISABLEXVIDMODE=1
				radeon_multihead_config
				;;
			"nv"|"nvidia")
				nvidia_multihead_config
				;;
			*)
				echo 'Section "Device"
	Identifier "Videocard'$MXDEVNO'"
	Driver "'$DRIVER'"
	VendorName "'$VENDORNAME'"
	BoardName "'$BOARDNAME'"
	BusID "PCI:'$BUSID'"
	Screen 0
	Option "SWCursor" "on"
	Option "HWCursor" "off"
EndSection' >>  $TEMP_XORG_CONF
    			echo 'Section "Device"
	Identifier "Videocard'$((MXDEVNO+1))'"
	Driver "'$DRIVER'"
	VendorName "'$VENDORNAME'"
	BoardName "'$BOARDNAME'"
	BusID "PCI:'$BUSID'"
	Screen 1
	Option "SWCursor" "on"
	Option "HWCursor" "off"
EndSection' >>  $TEMP_XORG_CONF
    			MXDEVNO=$((MXDEVNO+2))
				;;
		esac
	fi
}

##########################
# fetch
# 	- Wrapper around db_get
##########################
function fetch()
{
	FETCH_VAL=""
	db_get $DB_PATH$1
	if [ $? -eq 0 -a -n "$RET" ]; then
		FETCH_VAL="$RET"
	fi

	if [ -z "$FETCH_VAL" ]; then
		case "$1" in
			rules)
				FETCH_VAL="xorg"
				;;
			layout)
				FETCH_VAL="us"
				;;
			model)
				FETCH_VAL="pc105"
				;;
			variant)
				FETCH_VAL="NULL"
				;;
			options)
				FETCH_VAL="NULL"
				;;
			*)
				bomb "Invalid argument for db_get."
				;;
		esac
	fi
}

##########################
# genKbdSection
# 	- Generate InputDevice section for keyboard
#
##########################
function genKbdSection()
{
    # Check keyboard sysconfig file
	if [[ -f /etc/sysconfig/keyboard ]]; then
		source /etc/sysconfig/keyboard
		KEYTABLE=`echo $KEYTABLE | awk -F'.' '{printf $1}'`
		XKBSTRING=`cat /opt/userful/bin/XKB-KEYTABLE | grep ^$KEYTABLE\^`
		XKBLAYOUT=`echo $XKBSTRING | awk -F'^' '{printf $2}'`
		XKBMODEL=`echo $XKBSTRING | awk -F'^' '{printf $3}'`
		XKBVARIANT=`echo $XKBSTRING | awk -F'^' '{printf $4}'`
		XKBOPTIONS=`echo $XKBSTRING | awk -F'^' '{printf $5}'`

    # Check console-setup
	elif [[ -f /etc/default/console-setup ]]; then

 	        # XKBRULES
	        XKBRULES=$(grep "XKBRULES" /etc/default/console-setup | awk -- '{print $1}' | tr -d 'XKBRULES=""')

                # XKBLAYOUT
	        XKBLAYOUT=$(grep "XKBLAYOUT" /etc/default/console-setup | awk -- '{print $1}'| tr -d 'XKBLAYOUT=""')

                # XKBMODEL
	        XKBMODEL=$(grep "XKBMODEL" /etc/default/console-setup | awk -- '{print $1}' |tr -d 'XKBMODEL=""')

                # XKBVARIANT
	        XKBVARIANT=$(grep "XKBVARIANT" /etc/X11/xorg.conf | awk -- '{print $1}' | tr -d 'XKBVARIANT=""')    

                # XKBOPTIONS
    	        XKBOPTIONS=$(grep "XKBOPTIONS" /etc/X11/xorg.conf | awk -- '{print $1}' | tr -d 'XKBOPTIONS=""')    


    # Check xorg.conf
	elif [[ -f /etc/X11/xorg.conf ]]; then

		# XKBRULES
		XKBRULES=$(grep "XkbRules" /etc/X11/xorg.conf | awk -- '{print $3}' | tr -d '"')

		# XKBLAYOUT
		XKBLAYOUT=$(grep "XkbLayout" /etc/X11/xorg.conf | awk -- '{print $3}' | tr -d '"')

		# XKBMODEL
		XKBMODEL=$(grep "XkbModel" /etc/X11/xorg.conf | awk -- '{print $3}' | tr -d '"')

		# XKBVARIANT
		XKBVARIANT=$(grep "XkbVariant" /etc/X11/xorg.conf | awk -- '{print $3}' | tr -d '"')

		# XKBOPTIONS
		XKBOPTIONS=$(grep "XkbOptions" /etc/X11/xorg.conf | awk -- '{print $3}' | tr -d '"')


    fi

    if [[ -z "$XKBRULES" ]]; then
	    if [[ -f /usr/X11R6/bin/XFree86 ]]; then
		    XKBRULES="xfree86"
	    else
		    XKBRULES="xorg"
	    fi
    fi

    [[ -z "$XKBLAYOUT" ]] && XKBLAYOUT="us"
    [[ -z "$XKBMODEL" ]] && XKBMODEL="pc105"
    [[ -z "$XKBVARIANT" ]] && XKBVARIANT="NULL"
    [[ -z "$XKBOPTIONS" ]] && XKBOPTIONS="NULL"

	echo '######################
# InputDevice Section
######################' >>  $TEMP_XORG_CONF
	if [[ "$xconfigtype" == "-1box" ]]; then
		# For keyboard
		for (( i=0; i<$NUM_OF_HEAD; i++ )); do
			echo 'Section "InputDevice"
	Identifier "Keyboard'$i'"
	Driver "usbkbd"
	Option "Protocol" "usb_event"
	Option "Device" "/etc/X11/Xinput/keyboard'$i'"
	Option "Screen" "'$i'"
	Option "XkbRules" "'$XKBRULES'"
	Option "XkbLayout" "'$XKBLAYOUT'"
	Option "XkbModel" "'$XKBMODEL'"' >>  $TEMP_XORG_CONF

			if [[ "$XKBVARIANT" != "NULL" ]]; then
				echo '	Option "XkbVariant" "'$XKBVARIANT'"' >> $TEMP_XORG_CONF
			fi
			if [[ "$XKBOPTIONS" != "NULL" ]]; then
				echo '	Option "XkbOptions" "'$XKBOPTIONS'"' >> $TEMP_XORG_CONF
			fi
			
			echo 'EndSection' >> $TEMP_XORG_CONF
		done
		echo >> $TEMP_XORG_CONF

		# For Barcode scanner
		for (( i=0; i<$NUM_OF_HEAD; i++ )); do
			echo 'Section "InputDevice"
	Identifier "Barcode'$i'"
	Driver "usbkbd"
	Option "Protocol" "usb_event"
	Option "Device" "/etc/X11/Xinput/barcode'$i'"
	Option "Screen" "'$i'"
	Option "XkbRules" "'$XKBRULES'"
	Option "XkbLayout" "'$XKBLAYOUT'"
	Option "XkbModel" "'$XKBMODEL'"' >>  $TEMP_XORG_CONF

			if [[ "$XKBVARIANT" != "NULL" ]]; then
				echo '	Option "XkbVariant" "'$XKBVARIANT'"' >> $TEMP_XORG_CONF
			fi
			if [[ "$XKBOPTIONS" != "NULL" ]]; then
				echo '	Option "XkbOptions" "'$XKBOPTIONS'"' >> $TEMP_XORG_CONF
			fi
			
			echo 'EndSection' >> $TEMP_XORG_CONF
		done
		echo >> $TEMP_XORG_CONF

		# Margnetic reader
		for (( i=0; i<$NUM_OF_HEAD; i++ )); do
			echo 'Section "InputDevice"
	Identifier "Magreader'$i'"
	Driver "usbkbd"
	Option "Protocol" "usb_event"
	Option "Device" "/etc/X11/Xinput/magreader'$i'"
	Option "Screen" "'$i'"
	Option "XkbRules" "'$XKBRULES'"
	Option "XkbLayout" "'$XKBLAYOUT'"
	Option "XkbModel" "'$XKBMODEL'"' >> $TEMP_XORG_CONF

			if [[ "$XKBVARIANT" != "NULL" ]]; then
				echo '	Option "XkbVariant" "'$XKBVARIANT'"' >> $TEMP_XORG_CONF
			fi
			if [[ "$XKBOPTIONS" != "NULL" ]]; then
				echo '	Option "XkbOptions" "'$XKBOPTIONS'"' >> $TEMP_XORG_CONF
			fi
			
			echo 'EndSection' >> $TEMP_XORG_CONF
		done
		echo >> $TEMP_XORG_CONF

	else
		echo 'Section "InputDevice"
	Identifier  "Keyboard0"
	Driver      "kbd"
	Option "XkbRules" "'$XKBRULES'"
	Option "XkbLayout" "'$XKBLAYOUT'"
	Option "XkbModel" "'$XKBMODEL'"' >> $TEMP_XORG_CONF

		if [[ "$XKBVARIANT" != "NULL" ]]; then
			echo '	Option "XkbVariant" "'$XKBVARIANT'"' >> $TEMP_XORG_CONF
		fi
		if [[ "$XKBOPTIONS" != "NULL" ]]; then
			echo '	Option "XkbOptions" "'$XKBOPTIONS'"' >> $TEMP_XORG_CONF
		fi
		
		echo 'EndSection' >> $TEMP_XORG_CONF
		echo >> $TEMP_XORG_CONF
	fi
}

##########################
# genMseSection
# 	- Generate InputDevice section for Mouse
#
##########################
function genMseSection()
{
	if [[ "$xconfigtype" == "-1box" ]]; then
		for (( i=0; i<$NUM_OF_HEAD; i++ )); do
			echo 'Section "InputDevice"
	Identifier  "Mouse'$i'"
	Driver      "mouse"
	Option      "Protocol" "IMPS/2"
	Option      "Device" "/etc/X11/Xinput/mouse'$i'"
	Option      "ZAxisMapping" "4 5"
	Option      "Emulate3Buttons" "no"
	Option      "Screen"  "'$i'"
EndSection' >> $TEMP_XORG_CONF
		done
		echo >> $TEMP_XORG_CONF

	else
		echo 'Section "InputDevice"
	Identifier  "Mouse0"
	Driver      "mouse"
	Option      "Device" "/dev/input/mice"
	Option      "Protocol" "ImPS/2"
	Option      "Emulate3Buttons" "off"
	Option      "ZAxisMapping" "4 5"
EndSection' >> $TEMP_XORG_CONF
		echo >> $TEMP_XORG_CONF
	fi
}

##########################
# genTouchSection
# 	- Generate InputDevice section for Touchscreen pointer
#
##########################
function genTouchSection()
{
	if [[ "$xconfigtype" == "-1box" ]]; then
		for (( i=0; i<$NUM_OF_HEAD; i++ )); do
        	echo 'Section "InputDevice"
	Identifier "Touch'$i'"
	Driver "evtouch"
	#Option "Calibrate" "1"
	Option "Device" "/etc/X11/Xinput/touch'$i'"
	Option "DeviceName" "Touch'$i'"
	Option "MinX" "67"
	Option "MinY" "96"
	Option "MaxX" "4000"
	Option "MaxY" "4003"
	Option "ReportingMode" "Raw"
	Option "SendCoreEvents" "On"
	Option "DragTimer" "300"
	Option "MoveLimit" "500"
	Option "Screen"  "'$i'"
	#Option "SwapX"
	#Option "SwapY"
EndSection' >> $TEMP_XORG_CONF
		done
		echo >> $TEMP_XORG_CONF
	fi
}

#################################################
# ServerLayout_screen
# 	- Create Screen part in ServerLayout section
#################################################
function ServerLayout_screen()
{
	if [[ "$xconfigtype" == "-one" ]] ; then
		echo '	Screen	"Screen0"' >> $TEMP_XORG_CONF
	#elif [ "$xconfigtype" == "-all" ] ; then
	else
		for (( i=0; i<$NUM_OF_HEAD; i++ )); do
			echo '	Screen	"Screen'$i'"' >> $TEMP_XORG_CONF
		done
	fi
}

#################################################
# ServerLayout_kbd
# 	- Create Keyboard part in ServerLayout section
#################################################
function ServerLayout_kbd()
{
	if [[ "$xconfigtype" == "-one" || "$xconfigtype" == "-all" || "$xconfigtype" == "-check" ]] ; then
		echo '	InputDevice	"Keyboard0"	"CoreKeyboard"' >> $TEMP_XORG_CONF
	else # -1box
		echo '	InputDevice	"Keyboard0"	"CoreKeyboard"' >> $TEMP_XORG_CONF
		for (( i=1; i<$NUM_OF_HEAD; i++ )); do
			echo '	InputDevice	"Keyboard'$i'"	"AlwaysCore"' >> $TEMP_XORG_CONF
		done
		for (( i=0; i<$NUM_OF_HEAD; i++ )); do
			echo '	InputDevice	"Barcode'$i'"	"AlwaysCore"' >> $TEMP_XORG_CONF
		done
		for (( i=0; i<$NUM_OF_HEAD; i++ )); do
			echo '	InputDevice	"Magreader'$i'"	"AlwaysCore"' >> $TEMP_XORG_CONF
		done
	fi
}

#################################################
# ServerLayout_mse
# 	- Create Mouse part in ServerLayout section
#################################################
function ServerLayout_mse()
{
	if [[ "$xconfigtype" == "-one" || "$xconfigtype" == "-all" || "$xconfigtype" == "-check" ]]; then
		echo '	InputDevice	"Mouse0"	"CorePointer"' >> $TEMP_XORG_CONF
	else
		echo '	InputDevice	"Mouse0"	"CorePointer"' >> $TEMP_XORG_CONF
		for (( i=1; i<$NUM_OF_HEAD; i++ )); do
			echo '	InputDevice	"Mouse'$i'"	"AlwaysCore"' >> $TEMP_XORG_CONF
		done
	fi
}

#################################################
# ServerLayout_touch
# 	- Create TouchScreen part in ServerLayout section
#################################################
function ServerLayout_touch()
{
	if [[ "$xconfigtype" == "-1box" ]] ; then
		for (( i=0; i<$NUM_OF_HEAD; i++ )); do
			echo '	InputDevice "Touch'$i'"	"AlwaysCore"' >> $TEMP_XORG_CONF
		done
	fi
}

##########################
# genServerlayoutSection
# 	- Generate ServerLayout section for Mouse
#
##########################
function genServerlayoutSection()
{
	echo '#######################
# ServerLayout Section
#######################' >>  $TEMP_XORG_CONF
	echo 'Section "ServerLayout"' >> $TEMP_XORG_CONF
	if [[ "$xconfigtype" == "-one" ]] ; then
		echo '	Identifier     "userful.xorg.conf.one"' >> $TEMP_XORG_CONF
	elif [[ "$xconfigtype" == "-all" ]] ; then
		echo '	Identifier     "userful.xorg.conf.all"' >> $TEMP_XORG_CONF
	elif [[ "$xconfigtype" == "-check" ]] ; then
		echo '	Identifier     "userful.xorg.conf.check"' >> $TEMP_XORG_CONF
	elif [[ "$xconfigtype" == "-1box" ]] ; then
		echo '	Identifier     "userful.Mxorg.conf"' >> $TEMP_XORG_CONF
	fi

	ServerLayout_screen
	ServerLayout_kbd
	ServerLayout_mse
	if [[ -e /etc/usbinput ]]; then
		if grep "tch" /etc/usbinput > /dev/null 2>&1 ; then
			ServerLayout_touch
		fi
	fi
	echo 'EndSection' >> $TEMP_XORG_CONF
}

##############################################################
# getDeviceID
#  - Get Device ID (VendorID:ProductID) using BUSID.
#
# 	Required Variables:
# 		BUSID
##############################################################
function getDeviceID()
{
	#
	#  PCI ID (BUSNUM, DEVICENUM, FUNCTIONNUM)
	BUSNUM=`echo $BUSID | awk -- '{split($1,a,":"); print a[1]}'`
	DEVICENUM=`echo $BUSID | awk -- '{split($1,a,":"); print a[2]}'`
	FUNCTIONNUM=`echo $BUSID | awk -- '{split($1,a,":"); print a[3]}'`

	BUSNUM=`printf %x $BUSNUM`
	DEVICENUM=`printf %x $DEVICENUM`
	FUNCTIONNUM=`printf %x $FUNCTIONNUM`

	VGAID=`$LSPCI -s "$BUSNUM:$DEVICENUM.$FUNCTIONNUM" -n | awk '{print $3}'`

	if [ "$VGAID" == "0300:" ] || [ "$VGAID" == "0380:" ]; then
		DEVID=`$LSPCI -s "$BUSNUM:$DEVICENUM.$FUNCTIONNUM" -n | awk '{print $4}'`
	else
		DEVID=`$LSPCI -s "$BUSNUM:$DEVICENUM.$FUNCTIONNUM" -n | awk '{print $3}'`
	fi
}

##############################################################
# getCardInfo
#  - Find video card information based on current config type
#
# 	Required Variables:
# 		xconfigtype, VIDEOCARD, MXVER
##############################################################
function getCardInfo()
{
	if [[ "$xconfigtype" == "-1box" ]]; then
		# get information from /etc/mvgainfo
		BUSID=`echo $VIDEOCARD | awk -F'^' '{print $1}'`
		VENDORNAME=`echo $VIDEOCARD | awk -F'^' '{print $2}'`
		BOARDNAME=`echo $VIDEOCARD | awk -F'^' '{print $3}'`
		DRIVER=`echo $VIDEOCARD | awk -F'^' '{print $4}'`
		HEADNO=`echo $VIDEOCARD | awk -F'^' '{print $5}'`
		DEVID=`echo $VIDEOCARD | awk -F'^' '{print $6}'`

        ##
        ##   - Check VGAINFO to get Driver information
        if [ "$DEVID" != "" ]; then
		    if grep "$DEVID" /opt/userful/bin/VGAINFO > /dev/null 2>&1 ; then
                DRIVER_CONFIGURED=`grep "$DEVID" /opt/userful/bin/VGAINFO | awk -- '{print $4}'`

                if [[ "$DRIVER_CONFIGURED" != "NULL" && \
                    "$DRIVER_CONFIGURED" != "null" && \
                    "$DRIVER_CONFIGURED" != "" && \
                    "${DRIVER_CONFIGURED:0:1}" != "[" ]]; then

                    DRIVER=$DRIVER_CONFIGURED
                fi
		    fi
        fi

        ##
        ## TODO:
        ##  - If there is only one video card and it is Intel, use vesa driver.
        #if [ "$DRIVER" == "intel" -o "$DRIVER" == "i810" ]; then
        #    LINE_OF_MVGAINFO=`cat /etc/mvgainfo 2> /dev/null | wc -l`
        #    if [ $LINE_OF_MVGAINFO -eq 1 ]; then
        #        DRIVER=vesa
        #    fi
        #fi

        ##
        ## TODO:
        ##   -No intel driver in MX11R7. Need to find why
		#if [[ "$DRIVER" == "intel" ]]; then
		#	DRIVER=i810
		#fi	

        ##  nouveau driver is not supported yet.
		if [[ "$DRIVER" == "nouveau" ]]; then
            DRIVER=nv
        fi

        ## User vesa driver instead of vga
		if [[ "$DRIVER" == "vga" ]]; then
            DRIVER=vesa
        fi

	else
    	if [[ -s $DEVICE_SECTION_FILE ]]; then
			# get information from /tmp/DeviceSection.XXXXXX
			BUSID=`echo $VIDEOCARD | awk -F'^' '{print $1}'`
			VENDORNAME=`echo $VIDEOCARD | awk -F'^' '{print $2}'`
			BOARDNAME=`echo $VIDEOCARD | awk -F'^' '{print $3}'`
			DRIVER=`echo $VIDEOCARD | awk -F'^' '{print $4}'`
			
            ##
            ## TODO:
            ##    nouveau driver is not supported yet.
		    if [[ "$DRIVER" == "nouveau" ]]; then
                DRIVER=nv
            fi

			if [[ "$DRIVER" == "nv" && "$IS_NVIDIA_DRIVER" == "true" ]]; then
                DRIVER=nvidia
            fi

            ##
            ## TODO:
            ##   -No intel driver in MX11R7. Need to find why
		    #if [[ "$DRIVER" == "intel" ]]; then
			#	DRIVER=i810
			#fi	

		else
			# get information from lspci result
			# ex: 01:00.0 Class 0300: 1002:5159
			busid_len=`$LSPCI -n | grep -m 1 "0300: " | cut -d ' ' -f 1 | wc -m`
			if [[ $busid_len -lt 12 ]]; then
				bus_num=`echo $VIDEOCARD | awk -- '{split($1,a,"."); split(a[1],b,":"); print b[1]}'`
				device_num=`echo $VIDEOCARD | awk -- '{split($1,a,"."); split(a[1],b,":"); print b[2]}'`
			else
				bus_num=`echo $VIDEOCARD | awk -- '{split($1,a,"."); split(a[1],b,":"); print b[2]}'`
				device_num=`echo $VIDEOCARD | awk -- '{split($1,a,"."); split(a[1],b,":"); print b[3]}'`
			fi
			function_num=`echo $VIDEOCARD | awk -- '{split($1,a,"."); split(a[1],b,":"); print a[2]}'`

			bus_num=$(( 16#$bus_num ))
			device_num=$(( 16#$device_num ))
			function_num=$(( 16#$function_num ))

			# Find BUSID
			BUSID=$bus_num:$device_num:$function_num

			# Find VENDORNAME, BOARDNAME and DRIVER
			if echo $VIDEOCARD | tr "[A-Z]" "[a-z]" | grep class > /dev/null 2>&1 ; then
            	VENDORID=`echo $VIDEOCARD |awk '{split($4,a,":"); print a[1]}'`
            	VIDEOID=`echo $VIDEOCARD |awk '{split($4,a,":"); print a[2]}'`
			else
            	VENDORID=`echo $VIDEOCARD |awk '{split($3,a,":"); print a[1]}'`
            	VIDEOID=`echo $VIDEOCARD |awk '{split($3,a,":"); print a[2]}'`
			fi

			VIDEONAME=`grep "0x"$VENDORID /etc/userful/pcitable |
						grep "0x"$VIDEOID |
						awk -- '{if(substr($3,2,4) == "Card") {split($0,a,"\""); print a[2]}}' |
						awk -- '{split($0,b,":"); print b[2]}'`
   			VENDORNAME=`grep "0x"$VENDORID /etc/userful/pcitable |
						grep "0x"$VIDEOID |
						awk -- '{if(substr($3,2,4) == "Card") {split($0,a,"\""); print a[4]}}' |
						awk -- '{split($0,b,"|"); print b[1]}'`
   			BOARDNAME=$VIDEONAME

			echo $VIDEONAME |sed -e 's/(/\\(/g' | sed -e 's/)/\\)/g' | sed -e 's/\//\\\//g' > /tmp/tmp-video
			VIDEO=`cat /tmp/tmp-video`

			# Find the driver name from /etc/userful/Cards, given VIDEO card name
   			found=0
   			while [ $found == 0 ]; do
   				test=`awk 'BEGIN { i = 0 }
					   		/NAME '"$VIDEO"'$/ { i=1 }
					   		/DRIVER/ && i==1 { print "1 "$0; next }
				   	   		/SEE/ && i==1 { print "0 "$0; next }
				   	   		i==1 && NF==0 { i=0; }' /etc/userful/Cards`

   				if [ "${test:0:1}" == "0" ]; then
   	    			VIDEONAME=${test:6:60}
					echo $VIDEONAME |sed -e 's/(/\\(/g' | sed -e 's/)/\\)/g'| sed -e 's/\//\\\//g' > /tmp/tmp-video
   	    			VIDEO=`cat /tmp/tmp-video`
   				else
   	    			DRIVER=${test:9:60}
   	    			found=1
   				fi
   			done

            # TODO:
            #   - Need to update default driver
			# use default driver for unknow ATI and nVidia cards.
			if [[ "$DRIVER" == "" ]]; then
				if [[ "$VENDORID" == "1002" ]]; then # ATI card
					VENDORNAME="ATI"
					BOARDNAME="unknown ATI card"
					DRIVER="radeon"
				elif [[ "$VENDORID" == "10de" ]]; then # nVidia card
					VENDORNAME="nVidia"
					BOARDNAME="unknown nVidia card"
					DRIVER="nv"
				elif [[ "$VENDORID" == "102b" ]]; then # Matrox card
					VENDORNAME="Matrox"
					BOARDNAME="unknown Matrox card"
					DRIVER="mga"
				fi
			fi
		fi

		# get DEVID using BUSID
		getDeviceID
	fi
}

###
# Check whitelist and blacklist
###
function checkWBLists()
{
	local DEVID=$1
	local VIDEO_WHITELIST=$2
	local VIDEO_BLACKLIST=$3
	if [[ -s "$VIDEO_WHITELIST" ]]; then
		local IS_WHITELIST=`grep "^[[:space:]]*VIDEO_WHITELIST=" $VIDEO_WHITELIST | xargs echo | awk '{print $1}' | awk -F'=' '{print $2}'`
	fi
	if [[ -s "$VIDEO_BLACKLIST" ]]; then
		local IS_BLACKLIST=`grep "^[[:space:]]*VIDEO_BLACKLIST=" $VIDEO_BLACKLIST | xargs echo | awk '{print $1}' | awk -F'=' '{print $2}'`
	fi
	if [[ "$IS_WHITELIST" == "on" ]]; then
		if ! grep "^[[:space:]]*$DEVID" "$VIDEO_WHITELIST" > /dev/null 2>&1; then
			# Current $DEVID is not on video-whitelist
			# Skip current video card.
			log "VIDEO_WHITELIST=on, %s is not on the whitelist. Skip." "$DEVID"
			echo skip_whitelist
		fi
		
	elif [[ "$IS_BLACKLIST" == "on" ]]; then
		if grep "^[[:space:]]*$DEVID" "$VIDEO_BLACKLIST" > /dev/null 2>&1; then
		    # Current $DEVID is on video-blacklist
			# Skip current video card.
			log "VIDEO_WHITELIST=on, %s is not on the blacklist. Skip." "$DEVID"
			echo skip_blacklist
		fi
	fi
}

##
## Check if the current device is VGA. Some video cards provide "Display device" instead of VGA 
## under certain circumstances.
function isVGA() {
	local BUSID=$1
	local LSPCI=$2
	local bus_num=$(echo $BUSID | awk -F':' '{print $1}')
	local device_num=$(echo $BUSID | awk -F':' '{print $2}')
	local function_num=$(echo $BUSID | awk -F':' '{print $3}')

	local bus_num=$(printf %x $bus_num)
	local device_num=$(printf %x $device_num)
	local function_num=$(printf %x $function_num)
#   echo $bus_num:$device_num.$function_num
	if ! $LSPCI -s $bus_num:$device_num.$function_num -n | grep "0300: " > /dev/null 2>&1 && \
       ! $LSPCI -s $bus_num:$device_num.$function_num -n | grep "0380: " > /dev/null 2>&1 ; then
		echo false
	else
		echo true	  
	fi
}
# The function requires variables
# $logfile $OSNAME $OSVER $BINDIR $LSPCI $IS_NVIDIA_DRIVER
function checkDevices() 
{
	local TEMP_XORG_CONF=$1
	local VIDEOCARDS=$2
	local MVGAINFO=$3
	local VGAINFO=$4
	local is_processing=/tmp/dm-config-processing
	local VIDEO_WHITELIST=/etc/userful/video-whitelist
	local VIDEO_BLACKLIST=/etc/userful/video-blacklist
	rm -f $MVGAINFO
	local base_tmp_conf=$(mktemp /tmp/basetmpconf.XXXXXXXX)
 	cp -f $TEMP_XORG_CONF $base_tmp_conf

	while read VIDEOCARD; do
		log "Testing %s." "$VIDEOCARD"
		getCardInfo # $BUSID $VENDORNAME $BOARDNAME $DRIVER $DEVID
#		echo ${LINENO} $BUSID $VENDORNAME $BOARDNAME -- $DRIVER -- $DEVID -- $HEADNO -- >&2
        ##
        ## Check if the current device is VGA. 
        ## Some video cards provide "Display device" instead of VGA 
        ## under certain circumstances.
		is_vga=$(isVGA "$BUSID" "$LSPCI")
		if [ "false" == "$is_vga" ]; then
			continue
		fi		

		###
		# Check whitelist and blacklist
		###
		is_skipped=$(checkWBLists "$DEVID" "$VIDEO_WHITELIST" "$VIDEO_BLACKLIST")
		if [ ! -z "$is_skipped" ]; then
			continue
		fi
        VENDOR_ID=`echo $DEVID | awk -F':' '{print $1}'`

		if [[ "$DRIVER" == "" ]]; then
			log "Couldn't find VGA driver for device %s" "$BOARDNAME"
			HEADNO=1
			DRIVER=vesa
		elif grep "^\<$DEVID\>" "$VGAINFO" > /dev/null; then
			HEADNO=`grep "$DEVID" "$VGAINFO" | awk -- '{print $2}'`
			DRIVER=`grep "$DEVID" "$VGAINFO" | awk -- '{print $4}'`
			if [ -z "${BOARDNAME//*known*}" ]; then
				BOARDNAME='['`grep "$DEVID" "$VGAINFO" | cut -d"[" -f2`
			fi
			log "Found %s(%s) in %s." "$BOARDNAME" "$DEVID" "$VGAINFO"
		elif [[ "$DRIVER" == "nv" && "$IS_NVIDIA_DRIVER" == "false" ]]; then
			log "nvidia driver is not available."
			HEADNO=1
			
			# System has been freezed during "-check" processing.
			# In Novell 10, multi-head configuration doesn't work with normal X
		elif [[ -f "$is_processing" ]] ||
			[[ "$OSNAME" == "novell" && "$OSVER" == "10" ]]; then
			log "%s %s doesn't support auto-configuration for your video card: %s." \
				"$OSNAME" "$OSVER" "$BOARDNAME"
			log "\tIf you have dual head cards, please add card information into %s." \
				"$VGAINFO"			
			HEADNO=1
			
        # TODO:
        #   Need better way to check dual-head card
        #   Consider every intel card as one head for now.
        elif [[ "$VENDOR_ID" == "8086" ]]; then
			log "Found Intel graphics card. Use it as a single head card by default."
			HEADNO=1
            
		else
			#touch flag file
			touch $is_processing
			
 			cp -f $base_tmp_conf $TEMP_XORG_CONF 
			MXDEVNO=0
			HEADNO=2
			
			configMultiCard
			genScreenSection
			genKbdSection
			genMseSection
			genServerlayoutSection
			
			# create temporal checking config file
			mv -f  $TEMP_XORG_CONF /etc/X11/userful.xorg.conf.check
			log "Test VGA card: %s." "$BOARDNAME"
				
			XPATH=`which X`

			$BINDIR/mh-check $XPATH 2> /dev/null

			if [[ "$?" == "0" ]]; then
				pkill -SIGTERM X 2> /dev/null
				sleep 5
				# make sure X is not running
				while [[ -f /tmp/.X0-lock ]] && [[ $KILLX -lt 6 ]]; do
					sleep 2
					KILLX=$((KILLX+1))
				done
				if [[ -f /tmp/.X0-lock ]]; then  # if X exist
					pkill -9 X 2> /dev/null
					touch /tmp/mx-config-error
				fi

			else
				HEADNO=1
				pkill -SIGTERM X 2> /dev/null
				sleep 5
			fi
			
			#delete flag file 
			rm -f $is_processing
		fi
		if [[ "$DRIVER" == "nv" && "$IS_NVIDIA_DRIVER" == "true" ]]; then
            DRIVER=nvidia
        fi
		echo "$BUSID^$VENDORNAME^$BOARDNAME^$DRIVER^$HEADNO^$DEVID" >> $MVGAINFO		
	done < $VIDEOCARDS
	rm -f $base_tmp_conf
}

###################################################
# genDeviceSection
#   Generate Device section
#
#   Required Variables:
#       TEMP_XORG_CONF, MXVER, IS_NVIDIA_DRIVER, SYS_DUALVIEW
#   Parameter:
#
##################################################
function genDeviceSection()
{
	tmp_video_list=$(mktemp /tmp/videolist.XXXXXXXX)
	local VIDEO_WHITELIST=/etc/userful/video-whitelist
	local VIDEO_BLACKLIST=/etc/userful/video-blacklist
    echo '
######################
# Device Section
######################' >> $TEMP_XORG_CONF

	if [[ "$xconfigtype" == "-1box" ]]; then
		# use /etc/mvgainfo as the input
		if [[ -f /etc/mvgainfo ]]; then
			cp -f /etc/mvgainfo $tmp_video_list
		fi

	else
    	if [[ -s $DEVICE_SECTION_FILE ]]; then	    
			# use /tmp/DeviceSection.XXXXXX as the input
			cp -f $DEVICE_SECTION_FILE $tmp_video_list
		else
			# xorg.conf.new is not available
			$LSPCI -n | grep "0300: " > $tmp_video_list
		fi

		if [[ "$xconfigtype" == "-check" ]]; then
			BINDIR=/opt/userful/bin
            #will create /etc/mvgainfo file
			checkDevices "$TEMP_XORG_CONF" "$tmp_video_list" "/etc/mvgainfo" "/opt/userful/bin/VGAINFO"
			rm -f $tmp_video_list
			return 0
		fi
	fi

	while read VIDEOCARD; do
		#
		# get video card info from xorg.conf.new
		# get video card info from lspci and VGA-DB
		# Get video card information
		getCardInfo
		#echo "$BUSID $VENDORNAME $BOARDNAME $DRIVER $DEVID $xconfigtype"

        ##
        ## Check if the current device is VGA. 
        ## Some video cards provide "Display device" instead of VGA 
        ## under certain circumstances.
		is_vga=$(isVGA "$BUSID" "$LSPCI")
		if [ "false" == "$is_vga" ]; then
			continue
		fi		

		###
		# Check whitelist and blacklist
		###
		is_skipped=$(checkWBLists "$DEVID" "$VIDEO_WHITELIST" "$VIDEO_BLACKLIST")
		if [ ! -z "$is_skipped" ]; then
			continue
		fi

		if [[ "$xconfigtype" == "-1box" ]] ;then
			if [[ "$DRIVER" == "nv" && "$IS_NVIDIA_DRIVER" == "false" ]]; then
				# "nv" driver doesn't support dualview and dual-head. 
				# "nvidia driver is required.
				log "nvidia driver is not available."
				HEADNO=1

			elif [[ $SYS_DUALVIEW -gt 0 ]]; then
				# Configure dualview if current card supports dualview
				if grep "$DEVID" /opt/userful/bin/VGAINFO > /dev/null 2>&1 ; then
					CURR_CARD_TYPE=`grep "$DEVID" /opt/userful/bin/VGAINFO | awk -- '{print $3}'`
				fi

			else
			    CURR_CARD_TYPE=S # set to singleview card
			fi

			# Configure for Multiple video card
			configMultiCard

		elif [[ "$xconfigtype" == "-one" ]] ;then
			#if [[ "$DRIVER" == "" ]]; then
			#	echo "`date +"%b %d %T"` $ProgramName: Couldn't find video driver for device $BOARDNAME" >> $logfile
			#	DRIVER=vesa
			#fi

			# TODO:
			#  - For now, use vesa driver
			if [[ "$DRIVER" != "nvidia" ]] && [[ "$DRIVER" != "nv" ]] && \
               [[ "$DRIVER" != "radeon" ]] && [[ "$DRIVER" != "ati" ]] ; then
			    DRIVER=vesa
            fi
			configSingleCard

			# Only configure the first card, so break.
			break


		elif [[ "$xconfigtype" == "-all" ]] ;then
			if [[ "$DRIVER" == "" ]]; then
				log "Couldn't find video driver for device %s" "$BOARDNAME"
				DRIVER=vesa
			fi

			HEADNO=1
			configMultiCard
		fi
	done < $tmp_video_list
	
	rm -f $tmp_video_list $base_tmp_conf
}

######################
# reconfXConfig
#   - Reconfigure X config file based on xconfigtype
#
#####################################################
function reconfXConfig()
{
	if [[ "$xconfigtype" == "-one" ]] ; then
		cp -f  $TEMP_XORG_CONF /etc/X11/userful.xorg.conf.one
	elif [ "$xconfigtype" == "-all" ] ; then
		cp -f  $TEMP_XORG_CONF /etc/X11/userful.xorg.conf.all
	elif [ "$xconfigtype" == "-1box" ] ; then
		echo $NUM_OF_HEAD > /etc/X11/stations
		cp -f  $TEMP_XORG_CONF /etc/X11/userful.Mxorg.conf
		if [ $DISABLEXVIDMODE -eq 0 ]; then
			sed -i -e "/DisableVidModeExtension/d" /etc/X11/userful.Mxorg.conf
		fi
	fi
}
