/*
Copyright (C) 2002-2003 Victor Luchits

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "cg_local.h"

// cg_view.c -- player rendering positioning


//===================================================================

//================
//CG_ThirdPerson_CameraUpdate
//================
static void CG_ThirdPerson_CameraUpdate( void )
{
	float	dist, f, r;
	vec3_t	dest, stop;
	vec3_t	chase_dest;
	trace_t	trace;
	vec3_t	mins = { -4, -4, -4 };
	vec3_t	maxs = { 4, 4, 4 };

	// calc exact destination
	VectorCopy( cg.refdef.vieworg, chase_dest );
	r = DEG2RAD( cg_thirdPersonAngle->value );
	f = -cos( r );
	r = -sin( r );
	VectorMA( chase_dest, cg_thirdPersonRange->value * f, cg.v_forward, chase_dest );
	VectorMA( chase_dest, cg_thirdPersonRange->value * r, cg.v_right, chase_dest );
	chase_dest[2] += 8;

	// find the spot the player is looking at
	VectorMA( cg.refdef.vieworg, 512, cg.v_forward, dest );
	CG_Trace( &trace, cg.refdef.vieworg, mins, maxs, dest, cg.chasedNum + 1, MASK_SOLID );

	// calculate pitch to look at the same spot from camera
	VectorSubtract( trace.endpos, cg.refdef.vieworg, stop );
	dist = sqrt( stop[0] * stop[0] + stop[1] * stop[1] );
	if( dist < 1 )
		dist = 1;
	cg.refdef.viewangles[PITCH] = RAD2DEG( -atan2(stop[2], dist) );
	cg.refdef.viewangles[YAW] -= cg_thirdPersonAngle->value;
	AngleVectors( cg.refdef.viewangles, cg.v_forward, cg.v_right, cg.v_up );

	// move towards destination
	CG_Trace( &trace, cg.refdef.vieworg, mins, maxs, chase_dest, cg.chasedNum + 1, MASK_SOLID );

	if( trace.fraction != 1.0 ) {
		VectorCopy( trace.endpos, stop );
		stop[2] += ( 1.0 - trace.fraction ) * 32;
		CG_Trace( &trace, cg.refdef.vieworg, mins, maxs, stop, cg.chasedNum + 1, MASK_SOLID );
		VectorCopy( trace.endpos, chase_dest );
	}

	VectorCopy( chase_dest, cg.refdef.vieworg );
}

//================
//CG_CalcViewBob
//================
static void CG_CalcViewBob( void )
{
	float bobMove, bobTime;

	if( cg.thirdPerson )
		return;

	//
	// calculate speed and cycle to be used for
	// all cyclic walking effects
	//
	cg.xyspeed = sqrt( cg.predictedVelocity[0]*cg.predictedVelocity[0] + cg.predictedVelocity[1]*cg.predictedVelocity[1] );
	
	bobMove = 0;
	if( cg_gunbob->integer == 1 )
	{
		if( cg.xyspeed < 5 )
			cg.oldBobTime = 0;			// start at beginning of cycle again
		else if( cg.player.viewContents & MASK_WATER )
			bobMove = cg.frameTime * 0.75;
		else if( cg.frame.playerState.pmove.pm_flags & PMF_DUCKED )
			bobMove = cg.frameTime * 1.5;
		else if( cg.player.isOnGround )
			bobMove = cg.frameTime * 2.5;
	}

	bobTime = (cg.oldBobTime += bobMove);

	cg.bobCycle = (int)bobTime;
	cg.bobFracSin = fabs(sin(bobTime*M_PI));
}

//============================================================================

//==================
//CG_SkyPortal
//==================
int CG_SkyPortal( void )
{
	float fov;
	vec3_t origin;

	if( cgs.configStrings[CS_SKYBOXORG][0] == '\0' )
		return 0;

	if( sscanf( cgs.configStrings[CS_SKYBOXORG], "%f %f %f %f", &origin[0], &origin[1], &origin[2], &fov ) == 4 ) {
		cg.refdef.skyportal.fov = fov;
		VectorCopy( origin, cg.refdef.skyportal.origin );
		return RDF_SKYPORTALINVIEW;
	}

	return 0;
}

//============================================================================

//==================
//CG_RenderFlags
//==================
static int CG_RenderFlags( void )
{
	int rdflags, contents;

	rdflags = 0;

	contents = CG_PointContents( cg.refdef.vieworg );
	if( contents & MASK_WATER )
		rdflags |= RDF_UNDERWATER;
	else
		rdflags &= ~RDF_UNDERWATER;

	if( cg.oldAreabits )
		rdflags |= RDF_OLDAREABITS;

	if( cg.portalInView )
		rdflags |= RDF_PORTALINVIEW;

	rdflags |= RDF_BLOOM;
	rdflags |= CG_SkyPortal ();

	return rdflags;
}

//============================================================================


//======================================================================
//					ChaseHack (In Eyes Chasecam)
//======================================================================

cg_chasecam_t	chaseCam;

void CG_ChasePrev( void )
{
	if( !chaseCam.mode < 0 || chaseCam.mode >= CAM_MODES )
		return;

	if( cg.frame.multipov )
	{
		int i = 0, nextpov = -1, thispov;

		for( i = 0; i < cg.frame.numplayers; i++ )
		{
			thispov = cg.frame.playerStates[i].POVnum;

			if( thispov == cg.chasedNum_latched + 1 )
				continue;

			if( nextpov == -1 )
				nextpov = thispov;
			else if( nextpov > cg.chasedNum_latched + 1 && (thispov < cg.chasedNum_latched + 1 || thispov > nextpov) )
				nextpov = thispov;
			else if( nextpov < cg.chasedNum_latched + 1 && thispov < cg.chasedNum_latched + 1 && thispov > nextpov )
				nextpov = thispov;
		}
		if( nextpov != -1 )
			cg.chasedNum_latched = nextpov - 1;
	}
	else
	{
		if( cgs.demoPlaying )
			return;

		trap_Cmd_ExecuteText( EXEC_NOW, "chaseprev" );
	}
}

void CG_ChaseNext( void )
{
	if( chaseCam.mode < 0 || chaseCam.mode >= CAM_MODES )
		return;

	if( cg.frame.multipov )
	{
		int i = 0, nextpov = -1, thispov;

		for( i = 0; i < cg.frame.numplayers; i++ )
		{
			thispov = cg.frame.playerStates[i].POVnum;

			if( thispov == cg.chasedNum_latched + 1 )
				continue;

			if( nextpov == -1 )
				nextpov = thispov;
			else if( nextpov < cg.chasedNum_latched + 1 && (thispov > cg.chasedNum_latched + 1 || thispov < nextpov) )
				nextpov = thispov;
			else if( nextpov > cg.chasedNum_latched + 1 && thispov > cg.chasedNum_latched + 1 && thispov < nextpov )
				nextpov = thispov;
		}
		if( nextpov != -1 )
			cg.chasedNum_latched = nextpov - 1;
	}
	else
	{
		if( cgs.demoPlaying )
			return;

		trap_Cmd_ExecuteText( EXEC_NOW, "chasenext" );
	}
}

//===============
//CG_PlayerPOV - IN-EYES chasecam
//===============
static void CG_PlayerPOV( player_state_t *ps )
{
	usercmd_t	cmd;

	if( cgs.demoPlaying || cg.frame.multipov ) {
#ifdef DEMOCAM
		if( !CG_Democam_OverrideChasedNum() )
#endif
		cg.chasedNum = ps->POVnum - 1;

		trap_NET_GetUserCmd( trap_NET_GetCurrentUserCmdNum() - 1, &cmd );
		if( cg.time > chaseCam.cmd_mode_delay ) {
			if( (cmd.buttons & BUTTON_ATTACK) )
			{
				chaseCam.mode = (chaseCam.mode != CAM_THIRDPERSON);
				chaseCam.cmd_mode_delay = cg.time + 200;
			}

			if( cg.frame.multipov ) {
				if( cmd.upmove > 0 ) {
					CG_ChaseNext();
					chaseCam.cmd_mode_delay = cg.time + 200;
				}

				if( cmd.upmove < 0 ) {
					CG_ChasePrev();
					chaseCam.cmd_mode_delay = cg.time + 200;
				}
			}
		}
	}
	else if( ps->pmove.pm_type == PM_CHASECAM ) { 
		cg.chasedNum = ps->POVnum - 1;//minus one for parallel with cgs.playerNum

		trap_NET_GetUserCmd( trap_NET_GetCurrentUserCmdNum() - 1, &cmd );
		if( cg.frame.playerState.stats[STAT_REALTEAM] == TEAM_SPECTATOR ) {
			if( (cmd.buttons & BUTTON_ATTACK) && cg.time > chaseCam.cmd_mode_delay )
			{
				chaseCam.mode++;
				if( chaseCam.mode >= CAM_MODES ) { // if exceedes the cycle, start free fly
					trap_Cmd_ExecuteText( EXEC_NOW, "camswitch" );
					chaseCam.mode = 0; // smallest, to start the new cycle
				}
				chaseCam.cmd_mode_delay = cg.time + 200;
			}
		}

		if( cg.frame.multipov ) {
			if( cmd.upmove > 100 )
				CG_ChaseNext();

			if( cmd.upmove < -100 )
				CG_ChasePrev();
		}
	} 
	else if( ps->pmove.pm_type == PM_SPECTATOR ) {
		cg.chasedNum = cgs.playerNum;
		trap_NET_GetUserCmd( trap_NET_GetCurrentUserCmdNum() - 1, &cmd );
		if( cg.frame.playerState.stats[STAT_REALTEAM] == TEAM_SPECTATOR ) {
			if( (cmd.buttons & BUTTON_ATTACK) && cg.time > chaseCam.cmd_mode_delay )
			{
				trap_Cmd_ExecuteText( EXEC_NOW, "camswitch" );
				chaseCam.cmd_mode_delay = cg.time + 200;
			}
		}
	}
	else {
		cg.chasedNum = cgs.playerNum;
		chaseCam.mode = CAM_INEYES;
	}

	// set up third-person
	if( chaseCam.mode == CAM_THIRDPERSON && (ps->pmove.pm_type == PM_CHASECAM || cgs.demoPlaying) )
		cg.thirdPerson = qtrue;
	else if( ps->pmove.pm_type == PM_SPECTATOR || ps->pmove.pm_type == PM_GIB || ps->pmove.pm_type == PM_FREEZE )
		cg.thirdPerson = qfalse;
	else
		cg.thirdPerson = ( cg_thirdPerson->integer != 0 );

	//determine if we have to draw the view weapon
	if( ps->pmove.pm_type == PM_SPECTATOR || cg_entities[cg.chasedNum+1].serverFrame != cg.frame.serverFrame
		|| cg_entities[cg.chasedNum+1].current.weapon == 0 )
	{
		vweap.active = qfalse;
	}
	else if( cg.frame.match.state >= MATCH_STATE_POSTMATCH )
		vweap.active = qfalse;
	else
		vweap.active = (cg.thirdPerson == qfalse);
}

//================
//CG_CathegorizePlayerStatePosition
//================
static void CG_CathegorizePlayerStatePosition( void )
{
	vec3_t			lerpedorigin;
	vec3_t			point;
	trace_t			trace;
	int				i;
	centity_t		*cent;
	player_state_t	*ps, *ops;

	//JALFIXME: just rewrite this one.

	ps = cg.player.curps;
	ops = cg.player.oldps;

	//using the entity
	cent = &cg_entities[cg.chasedNum+1]; // player in POV

	if( CG_PredictionActive() && !cg.thirdPerson )
	{
		for( i = 0; i < 3; i++ )
			lerpedorigin[i] = cg.predictedOrigin[i] - (1.0f - cg.lerpfrac) * cg.predictionError[i];
	}
	else
	{
		// using packet entity (for chased players)
		//for( i = 0; i < 3; i++ )
		//	lerpedorigin[i] = cent->prev.origin[i] + cg.lerpfrac * (cent->current.origin[i] - cent->prev.origin[i]);
		VectorCopy( cg.player.origin, lerpedorigin );
	}

	//in water check
	cg.player.viewContents = CG_PointContents( lerpedorigin );

	point[0] = lerpedorigin[0];
	point[1] = lerpedorigin[1];
	point[2] = lerpedorigin[2] - ((float)STEPSIZE * 1.4);

	//trace
	CG_Trace( &trace, lerpedorigin, playerbox_crouch_mins, playerbox_crouch_maxs, point, cent->current.number,
		MASK_PLAYERSOLID );
	if( trace.plane.normal[2] < 0.7 && !trace.startsolid )
		cg.player.isOnGround = qfalse;
	else
		cg.player.isOnGround = qtrue;
}

//================
//CG_LerpPlayerState
//================
static void CG_LerpPlayerState( void )
{
	int			i;
	player_state_t	*ps, *ops;
	
	// find the previous frame to interpolate from
	ps = &cg.frame.playerState;
	ops = &cg.oldFrame.playerState;

	// see if the player entity was teleported this frame
	if( abs(ops->pmove.origin[0] - ps->pmove.origin[0]) > 256
		|| abs(ops->pmove.origin[1] - ps->pmove.origin[1]) > 256
		|| abs(ops->pmove.origin[2] - ps->pmove.origin[2]) > 256 )
		ops = ps;		// don't interpolate

	// interpolate some values into cg_clientstate_t so
	// they are easier to use
	for( i = 0; i < 3; i++ ) {
		cg.player.origin[i] = ops->pmove.origin[i] +
			cg.lerpfrac * ( ps->pmove.origin[i] - ops->pmove.origin[i] );

		cg.player.viewangles[i] = LerpAngle( ops->viewangles[i], ps->viewangles[i], cg.lerpfrac );
	}

	cg.player.fov = ops->fov + cg.lerpfrac * (ps->fov - ops->fov);
	VectorSet( cg.player.viewoffset, 0.0f, 0.0f, ops->viewheight + cg.lerpfrac * (ps->viewheight - ops->viewheight) );

	cg.player.oldps = ops;
	cg.player.curps = ps;

	CG_PlayerPOV( ps ); // set up before cathegorize position

	// find out some information of common use later
	CG_CathegorizePlayerStatePosition();
	CG_CalcViewBob();
}

//============================================================================


//==============
//CG_AddLocalSounds
//==============
static void CG_AddLocalSounds( void )
{
	static int flagNextBipTimer = 100;
	static int lastBipTime;
	static qboolean postmatchsound_set = qfalse;

	// add sounds from announcer
	CG_ReleaseAnnouncerEvents();

	// if in postmatch, play postmatch song
	if( cg.frame.match.state >= MATCH_STATE_POSTMATCH ) {
		if( !postmatchsound_set ) {
			trap_S_StopBackgroundTrack();
			trap_S_StartBackgroundTrack( va( S_MUSIC_POSTMATCH_1_to_7, (int)brandom(1,7) ), NULL );
			postmatchsound_set = qtrue;
		}
	} else {
		if( postmatchsound_set ) {
			trap_S_StopBackgroundTrack();
			postmatchsound_set = qfalse;
		}
		// ctf flag sounds
		if( cg.frame.playerState.stats[STAT_GAMETYPE] == GAMETYPE_CTF ) {
			centity_t *cent = &cg_entities[cg.chasedNum+1];
			if( cg.frame.playerState.stats[STAT_RACE_TIME] == STAT_NOTSET ||
				!(cent->current.effects & EF_ENEMY_FLAG) ) // ignore if not a flag carrier
			{
				lastBipTime = STAT_NOTSET;
			}
			else { // the timer is up
				flagNextBipTimer -= cg.frameTime * 1000;
				if( flagNextBipTimer <= 0 ) {
					int curBipTime;

					curBipTime = cg.frame.playerState.stats[STAT_RACE_TIME];

					flagNextBipTimer = 1000;

					if( lastBipTime == STAT_NOTSET || lastBipTime > curBipTime + 1 ) {// counting down
						trap_S_StartGlobalSound( CG_MediaSfx( cgs.media.sfxTimerBipBip ), CHAN_AUTO, 0.5f );
						flagNextBipTimer = 1000;
					}
					else { // counting up
						trap_S_StartGlobalSound( CG_MediaSfx( cgs.media.sfxTimerPloink ), CHAN_AUTO, 0.5f );
						flagNextBipTimer = 2000;
					}

					lastBipTime = curBipTime;
				}
			}
		}
	}
}

//==============
//CG_AddKickAngles - wsw
//==============
void CG_AddKickAngles( vec3_t viewangles )
{
	float	time;
	float	uptime;
	float	delta;
	int		i;

	for( i = 0; i < MAX_ANGLES_KICKS; i++ ) 
	{
		if( cg.time > cg.kickangles[i].timestamp + cg.kickangles[i].kicktime )
			continue;

		time = (float)((cg.kickangles[i].timestamp + cg.kickangles[i].kicktime) - cg.time);
		uptime = ((float)cg.kickangles[i].kicktime) * 0.5f;
		delta = 1.0f - (abs(time - uptime) / uptime);
		//CG_Printf("Kick Delta:%f\n", delta );
		if( delta > 1.0f )
			delta = 1.0f;
		if( delta <= 0.0f )
			continue;

		viewangles[PITCH] += cg.kickangles[i].v_pitch * delta;
		viewangles[ROLL] += cg.kickangles[i].v_roll * delta;
	}
}

//===============
//CG_ViewAddStepOffset
//===============
static void CG_ViewAddStepOffset( void ) 
{
	int		timeDelta;

	// smooth out stair climbing
	timeDelta = cg.realTime - cg.predictedStepTime;
	if ( timeDelta < PREDICTED_STEP_TIME ) {
		cg.refdef.vieworg[2] -= cg.predictedStep * (PREDICTED_STEP_TIME - timeDelta) / PREDICTED_STEP_TIME;
	}
}

//===============
//CG_SetSensitivityScale
//Scale sensitivity for different view effects
//===============
float CG_SetSensitivityScale( const float sens )
{
	float sensScale = 1.0f;

	if( sens && (cg.frame.playerState.stats[STAT_LAYOUTS] & STAT_LAYOUT_ZOOM) ) {
		if( cg_zoomSens->value ) {
			return cg_zoomSens->value/sens;
		} else if( cg_fov->value ) {
			return (cg.player.fov / cg_fov->value); // FIXME: cg_fov might not be the actual fov?
		}
	}

	return sensScale;
}

//===============
//CG_CalcViewValues
//Sets refdef from player view point
//===============
static void CG_CalcViewValues( void )
{
	int			i;

#ifdef DEMOCAM
	if( CG_DemoCam() )
		return;
#endif
	// calculate the origin
	if( CG_PredictionActive() && !cg.thirdPerson )
	{
		// use predicted values
		for( i = 0; i < 3; i++ ) {
			cg.refdef.vieworg[i] =
				cg.predictedOrigin[i] + cg.player.viewoffset[i] - (1.0f - cg.lerpfrac) * cg.predictionError[i];
		}

		// smooth out stair climbing
		CG_ViewAddStepOffset();

	} else {
		// just use interpolated values
		VectorAdd( cg.player.origin, cg.player.viewoffset, cg.refdef.vieworg );
	}

	// if not running a demo or on a locked frame, add the local angle movement
	if( (cg.frame.playerState.pmove.pm_type < PM_DEAD) && !cgs.demoPlaying ) {
		// use predicted values
		for( i = 0; i < 3; i++ )
			cg.refdef.viewangles[i] = cg.predictedAngles[i];
	} else {
		// just use interpolated values
		VectorCopy( cg.player.viewangles, cg.refdef.viewangles );
	}

	if( cg_damage_kick->integer )
		CG_AddKickAngles( cg.refdef.viewangles );

	AngleVectors( cg.refdef.viewangles, cg.v_forward, cg.v_right, cg.v_up );

	// interpolate field of view
	if( !(cg.frame.playerState.stats[STAT_LAYOUTS] & STAT_LAYOUT_ZOOM) && !cg.thirdPerson
		&& ((cgs.demoPlaying && !cg_demo_truePOV->integer) || (!cgs.demoPlaying && cg.frame.multipov)) ) {
		cg.refdef.fov_x = (cg_fov->integer < 1 || cg_fov->integer > 160) ? 90 : cg_fov->integer;
	} else {
		cg.refdef.fov_x = cg.player.fov;
	}

	// used for outline LODs computations
	cg.view_fracDistFOV = tan( cg.refdef.fov_x * (M_PI/180) * 0.5f );
	cg.view_fracWeapFOV = ( 1.0f / cg.view_fracDistFOV ) * tan( cg_gun_fov->integer * (M_PI/180) * 0.5f );
}

/*
==================
CG_RenderView

==================
*/
#define	WAVE_AMPLITUDE	0.015	// [0..1]
#define	WAVE_FREQUENCY	0.6		// [0..1]
#ifdef ZEROTHREEAPI
unsigned int CG_RenderView( float frameTime, float realFrameTime, int realTime, unsigned int serverTime, float stereo_separation )
#else
void CG_RenderView( float frameTime, float realFrameTime, int realTime, unsigned int serverTime, float stereo_separation )
#endif
{
	// update time
	cg.realTime = realTime;
	cg.frameTime = frameTime;
	cg.realFrameTime = realFrameTime;
	cg.frameCount++;
	cg.time = serverTime;

	if( !cgs.precacheDone || !cg.frame.valid ) {
		CG_DrawLoading();
#ifdef ZEROTHREEAPI
		return cg.time;
#else
		return;
#endif
	}

	if( cg.oldFrame.serverTime == cg.frame.serverTime ) {
		cg.lerpfrac = 1.0f;
	} else {
		cg.lerpfrac = (double)(cg.time - cg.oldFrame.serverTime) / (double)(cg.frame.serverTime - cg.oldFrame.serverTime);
	}

	if( cg_showClamp->integer ) {
		if( cg.lerpfrac > 1.0f )
			CG_Printf( "high clamp %f\n", cg.lerpfrac );
		else if( cg.lerpfrac < 0.0f )
			CG_Printf( "low clamp  %f\n", cg.lerpfrac );
	}

	clamp( cg.lerpfrac, 0.0f, 1.0f );

	if( !cgs.configStrings[CS_MODELS+1][0] ) {
		trap_R_DrawStretchPic( 0, 0, cgs.vidWidth, cgs.vidHeight, 0, 0, 1, 1, colorBlack, cgs.shaderWhite );
#ifdef ZEROTHREEAPI
		return cg.time;
#else
		return;
#endif
	}

	// predict all unacknowledged movements
	CG_PredictMovement();
#ifdef PREDICTSHOOTING
	CG_PredictWeapon();
#endif

	CG_LerpPlayerState();	// interpolated player state info
	CG_LerpEntities();		// interpolate packet entities positions

	CG_CalcVrect(); // find sizes of the 3d drawing screen
	CG_TileClear(); // clear any dirty part of the background

	// run lightstyles
	CG_RunLightStyles();

	trap_R_ClearScene();

	// finds the refdef, loads v_forward, etc.
	CG_CalcViewValues();
	if( cg.thirdPerson )
		CG_ThirdPerson_CameraUpdate();

	// build a refresh entity list
	CG_AddEntities();

	CG_AddLightStyles();
#ifdef _DEBUG
	CG_AddTest();
#endif

	// offset vieworg appropriately if we're doing stereo separation
	if( stereo_separation != 0 )
		VectorMA( cg.refdef.vieworg, stereo_separation, cg.v_right, cg.refdef.vieworg );

	// never let it sit exactly on a node line, because a water plane can
	// dissapear when viewed with the eye exactly on it.
	// the server protocol only specifies to 1/16 pixel, so add 1/16 in each axis
	cg.refdef.vieworg[0] += 1.0/PM_VECTOR_SNAP;
	cg.refdef.vieworg[1] += 1.0/PM_VECTOR_SNAP;
	cg.refdef.vieworg[2] += 1.0/PM_VECTOR_SNAP;

	cg.refdef.x = scr_vrect.x;
	cg.refdef.y = scr_vrect.y;
	cg.refdef.width = scr_vrect.width;
	cg.refdef.height = scr_vrect.height;
	cg.refdef.fov_y = CalcFov( cg.refdef.fov_x, cg.refdef.width, cg.refdef.height );

	cg.refdef.time = cg.time * 0.001;
	cg.refdef.areabits = cg.frame.areabits;

	cg.refdef.rdflags = CG_RenderFlags();

	// warp if underwater
	if( cg.refdef.rdflags & RDF_UNDERWATER ) {
		float phase = cg.refdef.time * WAVE_FREQUENCY * M_TWOPI;
		float v = WAVE_AMPLITUDE * (sin( phase ) - 1.0) + 1;
		cg.refdef.fov_x *= v;
		cg.refdef.fov_y *= v;
	}

	CG_AddLocalSounds();
	CG_SetSceneTeamColors(); // update the team colors in the renderer
	
	trap_R_RenderScene( &cg.refdef );

	cg.oldAreabits = qtrue;

	// update audio
	// Medar: TODO: velocity
	trap_S_Update( cg.refdef.vieworg, vec3_origin, cg.v_forward, cg.v_right, cg.v_up );

	CG_Draw2D();

	CG_ResetTemporaryBoneposesCache(); // skelmod : reset for next frame
#ifdef ZEROTHREEAPI
	return cg.time;
#endif
}
