/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

#ifdef linux
/* Must come before any kernel header file --hpreg */
#   include "driver-config.h"

#   include <linux/string.h>
#endif

#include "vmware.h"
#include "machine.h"
#include "hostif.h"
#include "cpuid.h"

static CPUOem oem;
static uint32 features;
static uint32 version;

/*
 *-----------------------------------------------------------------------------
 *
 * CPUIDExtendedSupported  --
 *
 *     Determine whether processor supports extended CPUID (0x8000xxxx)
 *     and how many of them.
 *
 * Results:
 *     0         if extended CPUID is not supported
 *     otherwise maximum extended CPUID supported (bit 31 set)
 *
 * Side effects:
 *     None.
 *
 *-----------------------------------------------------------------------------
 */

static uint32
CPUIDExtendedSupported(void)
{
   uint32 regs[4];

   HostIF_GetCPUID(0x80000000, regs);
   if ((regs[0] & 0x80000000) != 0x80000000) {
      return 0;
   }
   return regs[0];
}


void
CPUID_Init(void)
{
   uint32 regs[4]; /* eax[0], ebx[1], ecx[2], edx[3] */
   uint32 *ptr;
   char name[16];

   HostIF_GetCPUID(1, regs);
   version = regs[0];
   features = regs[3];

   HostIF_GetCPUID(0, regs);
   ptr = (uint32 *)name;
   ptr[0] = regs[1];
   ptr[1] = regs[3];
   ptr[2] = regs[2];
   ptr[3] = 0;

   if (strcmp(name, "GenuineIntel") == 0) {
      oem = CPUID_OEM_INTEL;
   } else if (strcmp(name, "AuthenticAMD") == 0) {
      oem = CPUID_OEM_AMD;
   } else if (strcmp(name, "CyrixInstead") == 0) {
      oem = CPUID_OEM_CYRIX;
   } else {
      Warning("VMMON CPUID: Unrecognized CPU\n");
      oem = CPUID_OEM_OTHER;
   }
}

CPUOem
CPUID_GetOEM(void)
{
   return oem;
}

uint32
CPUID_GetFeatures(void)
{
   return features;
}

uint32
CPUID_GetVersion(void)
{
   return version;
}

/*
 *-----------------------------------------------------------------------------
 *
 * SyscallSupported  --
 *
 *     Determine whether processor supports syscall opcode and MSRs.
 *
 * Results:
 *     FALSE     if processor does not support syscall
 *     TRUE      if processor supports syscall
 *
 * Side effects:
 *     It determines value only on first call, caching it for future.
 *
 *-----------------------------------------------------------------------------
 */

Bool
CPUID_SyscallSupported(void)
{
   static int supported = -1;

   if (UNLIKELY(supported == -1)) {
      if (CPUIDExtendedSupported() >= 0x80000001) {
         uint32 regs[4];
         HostIF_GetCPUID(0x80000001, regs);
         supported = (regs[3] & (1 << 11)) != 0;
      } else {
         supported = 0;
      }
   }
   return (supported != 0) ? TRUE : FALSE;
}

Bool
CPUID_LongModeSupported(void)
{
   static int supported = -1;

   if (UNLIKELY(supported == -1)) {
      if (CPUIDExtendedSupported() >= 0x80000001) {
         uint32 regs[4];
         HostIF_GetCPUID(0x80000001, regs);
         supported = (regs[3] & (1 << 29)) != 0;
      } else {
         supported = 0;
      }
   }
   return (supported != 0) ? TRUE : FALSE;
}
