/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package xnap.gui;

import xnap.*;
import xnap.net.ITransferContainer;
import xnap.util.*;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.text.*;
import javax.swing.*;
import javax.swing.border.*;

import org.apache.log4j.Logger;

public class Dialogs
{
    //--- Constant(s) ---

    public static final int COPY_FILES = 0;
    public static final int DELETE_FILES = 1;
    public static final int MOVE_FILES = 2;

    //--- Data Field(s) ---

    protected static Preferences prefs = Preferences.getInstance();
    
    private static Logger logger = Logger.getLogger(Dialogs.class);

    //--- Constructor(s) ---

    public static boolean copy(File[] files, File target)
    {
	boolean success = true;

	files = Dialogs.showCopyDialog(null, files, target);
	if (files != null) {
	    for (int i = 0; i < files.length; i++) {
		try {
		    FileHelper.copy(files[i], new File
			(target, files[i].getName()));
		}
		catch (IOException e) {
		    success = false;
		    JOptionPane.showMessageDialog
			(null, 
			 "Could not copy file: " + e.getLocalizedMessage(),
			 XNap.tr("Error"), JOptionPane.ERROR_MESSAGE);
		}
	    }
	}
	else {
	    success = false;
	}
	
	return success;
    }

    public static void error(Component parent, String message)
    {
	JOptionPane.showMessageDialog
	    (parent, message, XNap.tr("Error"), JOptionPane.ERROR_MESSAGE);
    }

    public static boolean move(File[] files, File target)
    {
	boolean success = true;

	files = Dialogs.showMoveDialog(null, files, target);
	if (files != null) {
	    for (int i = 0; i < files.length; i++) {
		try {
		    FileHelper.moveUnique(files[i], target.getAbsolutePath());
		}
		catch (IOException e) {
		    success = false;
		    JOptionPane.showMessageDialog
			(null, 
			 "Could not move file: " + e.getLocalizedMessage(),
			 XNap.tr("Error"), JOptionPane.ERROR_MESSAGE);
		}
	    }
	}
	else {
	    success = false;
	}
	
	return success;
    }

    /**
     * Returns true, if user presses okay button.
     */
    public static boolean showCloseDialog(Component c) 
    {
	JCheckBox jcb 
	    = new JCheckBox(XNap.tr("Always exit without prompting me."));

        Object[] message = new Object[] {
            XNap.tr("This will abort any downloads and uploads."),
	    XNap.tr("Do you really want to exit XNap?"), 
	    jcb
        };

        int response = JOptionPane.showConfirmDialog
            (c, message, XNap.tr("Exit XNap"), JOptionPane.OK_CANCEL_OPTION );

	prefs.setShowCloseDialog(!jcb.isSelected());
	prefs.write();

	return (response == JOptionPane.OK_OPTION);
    }	

    public static File[] showCopyDialog(Component c, File files[], File dir)
    {
	if (!prefs.getShowCopyDialog())
	    return files;
	
	String question = MessageFormat.format(XNap.tr("Do you really want to copy these files to \"{0}\"?"), 
					       new Object[] { dir });

	return showFilesActionDialog(c, files, question, COPY_FILES,
				     XNap.tr("Copy"));
    }

    public static File[] showDeleteFilesDialog(Component c, File files[])
    {
	if (!prefs.getShowDeleteFilesDialog())
	    return files;

	return showFilesActionDialog(c, files, XNap.tr("Do you really want to delete these files?"),
				     DELETE_FILES,
				     XNap.tr("Delete"));
    }

    public static ITransferContainer[]
	showDeleteDownloadsDialog(Component c, ITransferContainer[] transfers)
    {
	if (!prefs.getShowDeleteDownloadsDialog()) {
	    return  transfers;
	}

	JList jList = new JList(transfers);
	jList.setCellRenderer(new TransferListCellRenderer());
	jList.setVisibleRowCount(4);
	JScrollPane jsp = new JScrollPane();
	jsp.setViewportView(jList);

	// select all transfers
	jList.setSelectionInterval(0, transfers.length - 1);

	JCheckBox jcb = new JCheckBox(XNap.tr("Do not ask me again."));

	Object[] message = new Object[] {
	    jsp, jcb, XNap.tr("Do you really want to delete these files?")
	};
	
	int response = JOptionPane.showConfirmDialog
	    (c, message, 
	     MessageFormat.format(XNap.tr("{0} Files"), 
				  new Object[] { XNap.tr("Delete") }),
	     JOptionPane.YES_NO_OPTION);
	
	prefs.setShowDeleteDownloadsDialog(!jcb.isSelected());
	
	if (response == JOptionPane.YES_OPTION) {
	    Object[] rows = jList.getSelectedValues();
	    ITransferContainer[] selected = 
		new ITransferContainer[rows.length];
	    System.arraycopy(rows, 0, selected, 0, selected.length);
	    return selected;
	}
	else {
	    return null;
	}
    }

    protected static File[] showFilesActionDialog(Component c, File files[],
						  String question, int action,
						  String capitalizedAction)
    {
	JList jList = new JList(files);
	jList.setCellRenderer(new FileListCellRenderer());
	jList.setVisibleRowCount(4);
	JScrollPane jsp = new JScrollPane();
	jsp.setViewportView(jList);

	// select all files
	jList.setSelectionInterval(0, files.length - 1);

	JCheckBox jcb = new JCheckBox(XNap.tr("Do not ask me again."));

	Object[] message = new Object[] {
	    jsp, jcb, question
	};
	
	int response = JOptionPane.showConfirmDialog
            (c, message, 
	     MessageFormat.format(XNap.tr("{0} Files"),
				  new Object[] { capitalizedAction }),
	     JOptionPane.YES_NO_OPTION);
	
	setShowDialog(action, !jcb.isSelected());
	
	if (response == JOptionPane.YES_OPTION) {
	    Object rows[] = jList.getSelectedValues();
	    File selected[] = new File[rows.length];
	    System.arraycopy(rows, 0, selected, 0, selected.length);
	    return selected;
	}
	else {
	    return null;
	}
    }

    public static void showFullPathSharing(Component c) 
    {
	JCheckBox jcb = new JCheckBox
	    (XNap.tr("Share Full Path"), prefs.getShareFullPath());

        Object[] message = new Object[] {
            XNap.tr("A new feature has been added in this XNap release, which shares your files \n with their full path and not only with a meaningless number as directory entry. \nThis makes browsing much nicer from other clients and therefore it makes \na lot of sense to leave this enabled. \nPlease change this setting only if you really know what you are doing!"),
	    jcb
        };

        JOptionPane.showMessageDialog
            (c, message, XNap.tr("New Version"), JOptionPane.PLAIN_MESSAGE);

	prefs.setShareFullPath(jcb.isSelected());
    }	

    public static File[] showMoveDialog(Component c, File files[], File dir)
    {
	if (!prefs.getShowMoveDialog())
	    return files;
	
	String question = MessageFormat.format(XNap.tr("Do you really want to move these files to \"{0}\"?"), 
					       new Object[] { dir });
	
	return showFilesActionDialog(c, files, question, MOVE_FILES,
				     XNap.tr("Move"));
    }

    public static void showNotification(Component c, String dialogName, 
					String title, String message) 
    {
	if (!prefs.getNotShowDialog(dialogName)) {
	    NotificationDialog d 
		= new NotificationDialog(dialogName, title, message);
	    d.show(c);
	}
    }	

    public static void showPortRange(Component c) 
    {
	String text = XNap.tr("XNap now supports port ranges. You can set a port range in the OpenNap preferences panel like '6600-6700'. XNap will pick a random port that is available in that range. Also multiple distinct ports or ranges can be seperated with a semikolon like '6699;7788'.");

	MultiLineLabel mllText = new MultiLineLabel(text);
	mllText.setBorder(new TitledBorder(XNap.tr("Port Range", 1)));
	mllText.setColumns(40);

        Object[] message = new Object[] {
	    mllText
        };

        JOptionPane.showMessageDialog
            (c, message, XNap.tr("New Version"), JOptionPane.PLAIN_MESSAGE);
    }	

    public static String showRenameDialog(Component c, String filename) 
    {

	JTextField jtf = new JTextField(filename);

        Object[] message = new Object[] {
	    XNap.tr("Filename"),
	    jtf
        };

        int response = JOptionPane.showConfirmDialog
            (c, message, XNap.tr("Rename File"), JOptionPane.OK_CANCEL_OPTION);

	return (response == JOptionPane.OK_OPTION) ? jtf.getText() : null;
    }	

    public static void showRestartNotification(Component c)
    {
	JOptionPane.showMessageDialog
	    (c, XNap.tr("You need to restart XNap to activate your changes."),
	     XNap.tr("Preferences"), JOptionPane.INFORMATION_MESSAGE);
    }

    private static void setShowDialog(int i, boolean show)
    {
	switch(i) {
	case COPY_FILES:
	    prefs.setShowCopyDialog(show);
	    return;
	case DELETE_FILES:
	    prefs.setShowDeleteFilesDialog(show);
	    return;
	case MOVE_FILES:
	    prefs.setShowMoveDialog(show);
	    return;
	}
	    
    }

    public static class NotificationDialog extends DefaultDialog
    {
    //--- Data field(s) ---

	private static Preferences prefs = Preferences.getInstance();
	private JCheckBox jcbDoNotShowAgain;
	private String dialogName;

	//--- Constructor(s) ---
	    
	private NotificationDialog(String dialogName, String title, 
				   String message) 
	{
	    super(BUTTON_OKAY);

	    this.dialogName = dialogName;

	    // text panel
	    MultiLineLabel mllText = new MultiLineLabel(message);
	    mllText.setBorder(new TitledBorder(" " + title + " "));
	    mllText.setColumns(40);

	    // checkbox
	    jcbDoNotShowAgain = new JCheckBox
		(XNap.tr("Do not show again"), prefs.getNotShowDialog(dialogName));

	    // content
	    setTitle(XNap.tr("Notification"));
	    
	    getMainPanel().setBorder(new EmptyBorder(5, 5, 5, 5));
	    getMainPanel().setLayout(new GridBagLayout());
	    GridBagHelper.add(getMainPanel(), mllText);
	    GridBagHelper.add(getMainPanel(), jcbDoNotShowAgain);
	    pack();
	}
	
	// --- Methods ---

	public void apply()
	{
	    prefs.setNotShowDialog(dialogName, jcbDoNotShowAgain.isSelected());
	}

    }

}
