/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui;

import xnap.XNap;
import xnap.gui.event.PopupListener;
import xnap.util.*;

import java.awt.*;
import java.awt.event.*;
import java.beans.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import org.apache.log4j.Logger;

public class EditableComboBox extends JComboBox 
    implements ActionListener, PropertyChangeListener {
    
    //--- Constant(s) ---

    //--- Data field(s) ---

    protected static Logger logger = Logger.getLogger(EditableComboBox.class);

    protected boolean autoComplete;
    //protected boolean markedChars = false;
    protected int historySize;
    protected boolean itemsOnly;
    protected JTextField jtf;

    //--- Constructor(s) ---

    public EditableComboBox(Action enterAction, int historySize)
    {
	this.historySize = historySize;

	setAutoComplete(Preferences.getInstance().getAutoComplete());
	jtf = (JTextField)getEditor().getEditorComponent();
	jtf.addFocusListener(new FocusHandler());
	jtf.addKeyListener(new KeyHandler());
	
	setEditable(true);

	if (enterAction != null) {
	    KeyStroke k = KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0);
	    jtf.getInputMap().put(k, enterAction);
	    jtf.getActionMap().put(enterAction, enterAction);
	}
	
	Preferences.getInstance().addPropertyChangeListener
	    ("autoComplete", this);

	// popup menu
	JPopupMenu pmenu = new JPopupMenu();
	pmenu.add(new JMenuItem(new ClearHistoryAction()));
	jtf.addMouseListener(new PopupListener(pmenu));
    }

    public EditableComboBox(Action enterAction)
    {
	this(enterAction, 30);
    }

    public EditableComboBox()
    {
	this(null);
    }

    //--- Method(s) ---
    
    /**
     * Adds item if not already in list.
     */
    public void addDistinctItemAtTop(Object item)
    {
	logger.debug("i;: " + item);
	for (int i = getItemCount() - 1; i >= 0; i--) {
	    if (item.toString().equals(getItemAt(i).toString())) {
		removeItemAt(i);
	    }
	}

	insertItemAt(item, 0);
	setSelectedIndex(0);
    }

    public boolean getAutoComplete() 
    {
	return autoComplete;
    }

    public void setAutoComplete(boolean newValue) 
    {
	autoComplete = newValue;
    }

    public Dimension getMinimumSize()
    {
	return jtf.getMinimumSize();
    }

    public Dimension getPreferredSize()
    {
	return jtf.getPreferredSize();
    }

    public Dimension getMaximumSize()
    {
	return jtf.getMaximumSize();
    }
    
    public void setHistorySize(int newValue)
    {
	historySize = newValue;
    }

    public boolean isItemSelected() 
    {
	return isItem(jtf.getText());
    }
    
    public boolean getItemsOnly() 
    {
	return itemsOnly;
    }

    public void setItemsOnly(boolean newValue) 
    {
	this.itemsOnly = newValue;
    }

    public JTextField getJTextField()
    {
	return jtf;
    }

    /** 
     * Returns the selected object if autoComplete is true or editable is
     * false.
     * Otherwise you only get a string.
     * @return the selected object (if autoComplete is true or editable is 
     * false).
     */
    public Object getSelectedItem() 
    {
	ComboBoxModel boxModel = getModel();
	Object item = boxModel.getSelectedItem();

	if (item != null) {
	    if (item instanceof String) {
		Object o = getItem((String)item);
		if (o != null) {
		    return o;
		} 
		else {
		    return jtf.getText();
		}
	    } 
	    else {
		return item;
	    }
	} 
	else {
	    return jtf.getText();
	}
    }

    public String getText()
    {
	return jtf.getText();
    }
    
    public void setText(String newValue)
    {
	jtf.setText(newValue);
    }

    public void propertyChange(PropertyChangeEvent e)
    {
	setAutoComplete(Preferences.getInstance().getAutoComplete());
    }

    public void readBinaryHistoryFile(File file, Class type)
    {
    }

    public void readBinaryHistoryFile(File file)
    {
	logger.debug("reading binary history: " + file);
	ArrayList al = new ArrayList();
	
	try {
	    FileHelper.readBinary(file, al);
	    for (Iterator i = al.iterator(); i.hasNext();) {
		addItem(i.next());
	    }
	}
	catch (IOException e) {
	}
    }

    public void readHistoryFile(File file)
    {
	logger.debug("reading history: " + file);
	BufferedReader in = null;
	try {
	    in = new BufferedReader(new FileReader(file));
		
	    String line;
	    for (int i = 0;  i < historySize && (line = in.readLine()) != null;
		 i++)
		addItem(line);

	    in.close();
	}
	catch(IOException e) {
	}
	finally {
	    try {
		if (in != null) {
		    in.close();
		}
	    }
	    catch (IOException e) {
	    }
	}
    }

    public void writeBinaryHistoryFile(File file)
    {
	logger.debug("writing binary history: " + file);
	int count = Math.min(getItemCount(), historySize);
	ArrayList al = new ArrayList(count);
	for (int i = 0; i < count; i++) {
	    al.add(getItemAt(i));
	}
	try {
	    FileHelper.writeBinary(file, al);
	}
	catch (IOException e) {
	}
    }

    public void writeHistoryFile(File file)
    {
	logger.debug("writing history: " + file);
	BufferedWriter out = null;
	try {
	    out = new BufferedWriter(new FileWriter(file));
		
	    int count = getItemCount();
	    for (int i = 0; i < historySize && i < count; i++) {
		out.write(getItemAt(i).toString());
		out.newLine();
	    }
	}
	catch(IOException e) {
	}
	finally {
	    try {
		if (out != null) {
		    out.close();
		}
	    }
	    catch (IOException e) {
	    }
	}

    }

    protected boolean isItem(String name) 
    {
	return getItem(name) != null;
    }

    /** 
     * Returns the first object which corresponds to name.
     *@param name The name of the object to return.
     *@return the object whose name is name.
     */
    private Object getItem(String name) 
    {
	ComboBoxModel boxModel = getModel();
	for (int i = 0; i < boxModel.getSize(); i++) {
	    if (boxModel.getElementAt(i).equals(name)) {
		return boxModel.getElementAt(i);
	    }
	}

	return null;
    }

    private boolean selectFirstMatchingItem(String string) 
    {
	ComboBoxModel boxModel = getModel();
	int matchingItemIndex = -1;

	string = string.toLowerCase();
	for (int i = 0; i < boxModel.getSize(); i++) {
	    if (boxModel.getElementAt(i).toString().toLowerCase()
		.startsWith(string)) {
		matchingItemIndex = i;
		break;
	    }
	}

	//logger.debug(string + ":" + 1);

	if (matchingItemIndex != -1) {
	    /* Each selection is being cached, therefore the string in the
               editor won't be updated when selecting the same item twice.  */
	    setSelectedItem(null);
	    setSelectedIndex(matchingItemIndex);
	    return true;
	} 
	else {
	    return false;
	}
    }


    /**
     * Make sure a valid item is selected when focus is lost.
     */
    public class FocusHandler extends FocusAdapter {

	public void focusGained(FocusEvent e) 
	{
//  	    jtf.setCaretPosition(jtf.getText().length());
//  	    jtf.moveCaretPosition(0);
	}

	public void focusLost(FocusEvent e) 
	{
	    int caretPos = jtf.getCaretPosition();
	    if (getItemsOnly() && !isItemSelected()) {
		EditableComboBox.this.requestFocus();
		selectFirstMatchingItem(jtf.getText());
		jtf.setCaretPosition(jtf.getText().length());
		jtf.moveCaretPosition(caretPos);
	    }
	}

    }

    public class KeyHandler extends KeyAdapter
    {
	
	public void keyTyped(KeyEvent e)  
	{
	    if (!autoComplete) {
		return;
	    }

	    //logger.debug("key typed: " + e);
	    
	    String currentInput;
	    if (e.getKeyChar() == e.VK_ENTER) {
		hidePopup();
		jtf.setCaretPosition(jtf.getText().length());
		return;
	    }
	    else if (e.getKeyChar() == e.VK_BACK_SPACE) {
		hidePopup();
		return;
	    } 
	    else {
		currentInput = jtf.getText().substring
		    (0, jtf.getSelectionStart()) + e.getKeyChar();
	    }

	    // select item
	    if (selectFirstMatchingItem(currentInput)) {
		jtf.setCaretPosition(jtf.getText().length());		
		jtf.moveCaretPosition(currentInput.length());
		e.consume();
		showPopup();
	    } 
	    else {
		if (getItemsOnly()) {
		    e.consume();
		}
		else {
		    hidePopup();
		}
	    }
	} 
    }

    private class ClearHistoryAction extends AbstractAction
    {
	public ClearHistoryAction()
	{
	    putValue(Action.NAME, XNap.tr("Clear history"));
	    putValue(Action.SMALL_ICON, XNapFrame.getIcon("eraser.png"));
	}
	
	public void actionPerformed(ActionEvent event)
	{
	    removeAllItems();
	}
    }
}
