/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui;

import xnap.util.*;

import java.awt.*;
import java.awt.event.*;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Vector;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;

public class ToggableIconPane extends JPanel implements ChangeListener
{

    //--- Constant(s) ---

    public static final int BLINK_INTERVAL = 500;

    //--- Data field(s) ---

    /**
     * ChangeListeners.
     */
    Vector cls = new Vector();
    IconSplitPane ipane = null;
    int dividerSize;
    JTabbedPane tpane = null;
    boolean useTabbedPane;
    /**
     * Maps descriptions to components.
     */
    Hashtable components = new Hashtable();
    Timer timer;
    /**
     * Always set to non null value to make life easy.
     */
    Blinker blinker = new Blinker();
    
    //--- Constructor(s) ---

    public ToggableIconPane(boolean useTabbedPane, int dividerSize)
    {
	this.dividerSize = dividerSize;

	setLayout(new BorderLayout());
	setUseTabbedPane(useTabbedPane);

	addChangeListener(this);
    }

    public ToggableIconPane(boolean useTabbedPane)
    {
	this(useTabbedPane, -1);
    }


    //--- Method(s) ---

    public void insertTab(String description, Icon icon, JComponent c, 
			  String tooltip, int index) 
    {
	components.put(description, c);

	if (useTabbedPane)
	    tpane.insertTab(description, icon, c, tooltip, index);
	else
	    ipane.insertTab(description, icon, c, index);

	blinker.index++;
    }
    
    public void addTab(String description, Icon icon, JComponent c) 
    {
	components.put(description, c);

	if (useTabbedPane)
	    tpane.addTab(description, icon, c);
	else
	    ipane.addTab(description, icon, c);

	blinker.index++;
    }

    public Component[] getTabs()
    {
	if (useTabbedPane)
	    return tpane.getComponents();
	else 
	    return ipane.getTabs();
    }

    public void setDividerSize(int newValue)
    {
	dividerSize = newValue;
    }

    public void remove(Component c)
    {
	Object key = null;
	for (Enumeration e = components.keys(); e.hasMoreElements();) {
	    Object o = e.nextElement();
	    if (components.get(o) == c) {
		key = o;
	    }
	}
	if (key != null) {
	    components.remove(key);
	}

	if (useTabbedPane)
	    tpane.remove(c);
	else
	    ipane.remove(c);

	blinker.index--;
    }

    public void addChangeListener(ChangeListener l) 
    {
	cls.add(l);

	if (useTabbedPane)
	    tpane.addChangeListener(l);
	else
	    ipane.addChangeListener(l);
    }

    public void removeChangeListener(ChangeListener l) {
	cls.remove(l);

	if (useTabbedPane)
	    tpane.removeChangeListener(l);
	else
	    ipane.removeChangeListener(l);
    }

    public Icon getIconAt(int i)
    {
	if (useTabbedPane)
	    return tpane.getIconAt(i);
	else
	    return ipane.getIconAt(i);

    }

    public void setIconAt(int i, Icon icon) 
    {
	if (useTabbedPane)
	    tpane.setIconAt(i, icon);
	else
	    ipane.setIconAt(i, icon);
    }	

    public void blink(int i, Icon blinkIcon)
    {
	unblink();

	blinker = new Blinker(i, blinkIcon);
	timer = new Timer(BLINK_INTERVAL, blinker);
	timer.start();
    }

    public void unblink()
    {
	if (timer != null) {
	    timer.stop();
	    setIconAt(blinker.index, blinker.savedIcon);
	    timer = null;
	}
    }

    public int getSelectedIndex()
    {
	if (useTabbedPane)
	    return tpane.getSelectedIndex();
	else
	    return ipane.getSelectedIndex();
    }

    public Component getSelectedComponent()
    {
	if (useTabbedPane)
	    return tpane.getSelectedComponent();
	else
	    return ipane.getSelectedComponent();
    }

    public int getTabCount()
    {
	if (useTabbedPane)
	    return tpane.getTabCount();
	else
	    return ipane.getTabCount();
    }

    public void setSelectedComponent(String description)
    {
	setSelectedComponent((Component)components.get(description));
    }

    public void setSelectedComponent(Component c)
    {
	if (useTabbedPane)
	    tpane.setSelectedComponent(c);
	else
	    ipane.setSelectedComponent(c);
    }

    public void setUseTabbedPane(boolean newValue)
    {
	useTabbedPane = newValue;

	if (useTabbedPane && tpane == null) {
	    tpane = new JTabbedPane();
	    if (ipane != null) {
		super.remove(ipane);
		Component c = ipane.getSelectedComponent();

		for (Iterator i = cls.iterator(); i.hasNext();) {
		    ChangeListener l = (ChangeListener)i.next();
		    ipane.removeChangeListener(l);
		    tpane.addChangeListener(l);
		}

		while(ipane.getTabCount() > 0) {
		    String title = ipane.getTitleAt(0);
		    Icon icon = ipane.getIconAt(0);
		    Component component = ipane.getTabAt(0);
		    ipane.removeTabAt(0);
		    tpane.addTab(title, icon, component);
		}

		ipane = null;
		tpane.setSelectedComponent(c);
	    }
	    super.add(tpane, "Center");
	}
	else if (!useTabbedPane && ipane == null) {
	    ipane = new IconSplitPane();
	    if (dividerSize != -1) {
		ipane.setDividerSize(dividerSize);
	    }
	    if (tpane != null) {
		super.remove(tpane);
		Component c = tpane.getSelectedComponent();

		for (Iterator i = cls.iterator(); i.hasNext();) {
		    ChangeListener l = (ChangeListener)i.next();
		    tpane.removeChangeListener(l);
		    ipane.addChangeListener(l);
		}

		while(tpane.getTabCount() > 0) {
		    String title = tpane.getTitleAt(0);
		    Icon icon = tpane.getIconAt(0);
		    Component component = tpane.getComponentAt(0);
		    tpane.removeTabAt(0);
		    ipane.addTab(title, icon, component);
		}
		
		tpane = null;
		ipane.setSelectedComponent(c);
	    }
	    super.add(ipane, "Center");
	}

    }

    public void stateChanged(ChangeEvent e)
    {
	if (timer != null && getSelectedIndex() == blinker.index) {
	    unblink();
	}
    }

    protected class Blinker implements ActionListener 
    {
	public Icon blinkIcon; 
	public Icon savedIcon;
	public int index = 0;

	public Blinker(int index, Icon blinkIcon)
	{
	    this.index = index;
	    this.blinkIcon = blinkIcon;
	    savedIcon = getIconAt(index);
	}

	public Blinker()
	{
	}

	public void actionPerformed(ActionEvent e)
	{
	    Icon icon = (getIconAt(index) == savedIcon) ? blinkIcon : savedIcon;
	    setIconAt(index, icon);
	}
    }
}
