/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui;

import xnap.XNap;
import xnap.cmdl.Console;
import xnap.io.*;
import xnap.gui.action.MenuAction;
import xnap.gui.action.OpenFileWithMenu;
import xnap.gui.event.*;
import xnap.gui.menu.UserMenu;
import xnap.gui.table.*;
import xnap.net.*;
import xnap.util.*;
import xnap.util.launcher.LauncherSupport;

import java.io.*;
import java.awt.*;
import java.beans.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.TitledBorder;
import javax.swing.event.*;
import javax.swing.table.*;
import java.awt.event.*;
import java.lang.Math;


public class TransferSubPanel extends AbstractPanel 
    implements FileCollector, UserSupport
{
    
    //--- Data field(s) ---

    private boolean tabbed;
    private int dividerLoc = -1;

    private JTabbedPane jtp = null;
    private JSplitPane jsp = null;

    private JTable jtaDownloads;
    private TransferTableModel ttmDownloads;
    private JScrollPane jspDl;

    private JTable jtaUploads;
    private TransferTableModel ttmUploads;
    private JScrollPane jspUl;

    private AbortAction abortAction = new AbortAction();
    private ClearAction clearAction = new ClearAction();
    private DeleteAction deleteAction = new DeleteAction();
    private EditFilterAction acEditFilter = new EditFilterAction();
    private ResumeAction resumeAction = new ResumeAction();
    private OpenFileAction acOpenFile = new OpenFileAction(this);
    private ResumeSlashOpenAction acRSOAction = new ResumeSlashOpenAction();
    private MenuAction acMenu = new MenuAction(new UserMenu(this));

    //--- Constructor(s) ---

    public TransferSubPanel(boolean filtered, boolean tabbed) 
    {
        initialize(filtered);

	setTabbed(tabbed);
    }

    //--- Method(s) ---

    private void initialize(boolean filtered) 
    {
	// download popup
	JPopupMenu jmDownloads = new JPopupMenu();
	jmDownloads.add(resumeAction);
	jmDownloads.add(abortAction);
	jmDownloads.add(acEditFilter);
	jmDownloads.add(deleteAction);
	jmDownloads.addSeparator();
	jmDownloads.add(acOpenFile);
	jmDownloads.add(new OpenFileWithMenu(this));;
	jmDownloads.addSeparator();
	jmDownloads.add(new UserMenu(this));
	if (!filtered) {
	    jmDownloads.addSeparator();
	    jmDownloads.add(clearAction);
	}


	// download table
	if (filtered) {
	    ttmDownloads = new FilteredTransferTableModel
		(DownloadQueue.getInstance(), "download");
	}
	else {
	    ttmDownloads = new TransferTableModel
		(DownloadQueue.getInstance(), "download",
		 XNap.tr("Download Table"));
	}
	jtaDownloads = ttmDownloads.createJTable();
	jtaDownloads.setShowGrid(false);
	jtaDownloads.addFocusListener(new FocusSelectionListener());
	MouseListener mlDownloads = new PopupListener(jmDownloads);
	jtaDownloads.addMouseListener(mlDownloads);
	jtaDownloads.addMouseListener
	    (new DoubleClickListener(acRSOAction, jtaDownloads));
	jtaDownloads.getInputMap().put
	    (KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0), acRSOAction);
	jtaDownloads.getActionMap().put(acRSOAction, acRSOAction);

	jtaDownloads.getInputMap().put
	    (KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, 0), deleteAction);
	jtaDownloads.getActionMap().put(deleteAction, deleteAction);

	jtaDownloads.getInputMap().put
	    (KeyStroke.getKeyStroke(KeyEvent.VK_F2, 0), acEditFilter);
	jtaDownloads.getActionMap().put(acEditFilter, acEditFilter);

	jspDl = new JScrollPane();	
	jspDl.setViewportView(jtaDownloads);

	// upload popup
	JPopupMenu jmUploads = new JPopupMenu();
	jmUploads.add(new JMenuItem(new StartAction()));
	jmUploads.add(new JMenuItem(abortAction));
	if (!filtered) {
	    jmUploads.addSeparator();
	    jmUploads.add(new JMenuItem(clearAction));
	}
	jmUploads.addSeparator();
	jmUploads.add(new JMenuItem(acOpenFile));
	jmUploads.add(new OpenFileWithMenu(this));
	jmUploads.addSeparator();
	jmUploads.add(new UserMenu(this));

	// upload table	    
	if (filtered) {
	    ttmUploads = new FilteredTransferTableModel(UploadQueue.getInstance(),
							"upload");
	}
	else {
	    ttmUploads = new TransferTableModel
		(UploadQueue.getInstance(), "upload", XNap.tr("Upload Table"));
	}
	jtaUploads = ttmUploads.createJTable();
	jtaUploads.setShowGrid(false);

	jtaUploads.addFocusListener(new FocusSelectionListener());
	MouseListener mlUploads = new PopupListener(jmUploads);
	jtaUploads.addMouseListener(mlUploads);
	jtaUploads.addMouseListener
	    (new DoubleClickListener(acOpenFile, jtaUploads));
	jtaUploads.getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER,
							    0),
				     acOpenFile);
	jtaUploads.getActionMap().put(acOpenFile, acOpenFile);

	jspUl = new JScrollPane();	
	jspUl.setViewportView(jtaUploads);

	// content
	setLayout(new BorderLayout());
    }
    
    public AbstractAction[] getActions()
    {
		return (new AbstractAction[] { resumeAction, abortAction, null,
				       deleteAction, null, acMenu, null,
				       clearAction});
    }

    public int getDividerLocation()
    {
	return (tabbed) ? dividerLoc : jsp.getDividerLocation();
    }

    public void setDividerLocation(int newValue)
    {
	dividerLoc = newValue;

	if (tabbed) {
	    return;
	}

	if (newValue == -1) {
	    jsp.setDividerLocation(0.5);
	}
	else {
	    jsp.setDividerLocation(newValue);
	}
    }

    public void setTabbed(boolean newValue)
    {
	tabbed = newValue;

	if (tabbed) {
	    // tabbed pane
	    jtp = new JTabbedPane();
	    jtp.addTab(XNap.tr("Downloads"), jspDl);
	    jtp.addTab(XNap.tr("Uploads"), jspUl);
	    if (jsp != null) {
		dividerLoc = jsp.getDividerLocation();
		remove(jsp);
		jsp = null;
	    }
	    add(jtp, "Center");
	}
	else {
	    // split pane
	    jsp = new JSplitPane(JSplitPane.VERTICAL_SPLIT);

	    JPanel jp = new JPanel(new BorderLayout());
	    jp.setBorder(new TitledBorder(XNap.tr("Downloads", 1)));
	    jp.add(jspDl, "Center");
	    jsp.add(jp, JSplitPane.TOP);

	    jp = new JPanel(new BorderLayout());
	    jp.setBorder(new TitledBorder(XNap.tr("Uploads", 1)));
	    jp.add(jspUl, "Center");
	    jsp.add(jp, JSplitPane.BOTTOM);

	    jsp.setOneTouchExpandable(true);
	    jsp.setResizeWeight(1);
	    jsp.setDividerLocation(dividerLoc);
	    if (jtp != null) {
		remove(jtp);
		jtp = null;
	    }
	    add(jsp, "Center");
	}
    }

    /**
     * Return the active table.
     */
    protected JTable getSelectedTable() {
	if (tabbed) {
	    if (jtp.getSelectedComponent() == jspDl) {
		return jtaDownloads;
	    }
	    else if (jtp.getSelectedComponent() == jspUl) {
		return jtaUploads;
	    }
	}
	else {
	    if (jtaDownloads.getSelectedRowCount() > 0) {
		return jtaDownloads;
	    }
	    else if (jtaUploads.getSelectedRowCount() > 0) {
		return jtaUploads;
	    }
	}

	return null;
    }

    protected TransferQueue getSelectedQueue()
    {
	JTable jta = getSelectedTable();
	if (jta == null) {
	    return null;
	}
	TransferTableModel model = (TransferTableModel)jta.getModel();
	return (TransferQueue)model.getQueue();
    }

    protected ITransferContainer[] getSelectedTransfers()
    {
	JTable jta = getSelectedTable();
	if (jta == null) {
	    return new ITransferContainer[0];
	}
	
	TransferTableModel model = (TransferTableModel)jta.getModel();

	int[] rows = jta.getSelectedRows();
	ITransferContainer[] transfers = new ITransferContainer[rows.length];
	for (int i = 0; i < rows.length; i++) {
	    transfers[i] = model.get(rows[i]);
	}

	return transfers;
    }

    /**
     * Implements FileCollector.
     */
    public File[] getFiles()
    {
	ITransferContainer[] transfers = getSelectedTransfers();

	File[] files = new File[transfers.length];
	for (int i = 0; i < transfers.length; i++) {
	    files[i] = transfers[i].getFile();
	}

	return files;
    }

    public JMenu getDownloadTableMenu()
    {
	return ttmDownloads.createJMenu();
    }

    public JMenu getUploadTableMenu()
    {
	return ttmUploads.createJMenu();
    }

    /**
     * Implements UserSupport.
     */
    public IUser[] getUsers()
    {
	ITransferContainer[] transfers = getSelectedTransfers();
	if (transfers.length == 0) {
	    return null;
	}
	
	LinkedList l = new LinkedList();
	for (int i = 0; i < transfers.length; i++) {
	    IUser user = transfers[i].getUser();

	    if (user == null) {
		continue;
	    }
	    else if (user instanceof UserContainer) {
		l.addAll(Arrays.asList(((UserContainer)user).getUsers()));
	    }
	    else {
		l.add(user);
	    }
	}
	IUser[] users = new IUser[l.size()];
	System.arraycopy(l.toArray(), 0, users, 0, users.length);
	return users;
    }

    /**
     * Aborts all selected transfers.
     */
    private class AbortAction extends AbstractAction
    {
        public AbortAction()
	{
            putValue(Action.NAME, XNap.tr("Stop"));
            putValue(Action.SHORT_DESCRIPTION,
		     XNap.tr("Stop the selected transfer(s)"));
	    putValue(Action.SMALL_ICON, XNapFrame.getIcon("stop.png"));
            putValue(Action.MNEMONIC_KEY, new Integer('A'));
        }
	
        public void actionPerformed(ActionEvent event) 
	{
	    ITransferContainer[] transfers = getSelectedTransfers();
	    TransferQueue queue = getSelectedQueue();

	    if (transfers.length == 0) {
		setStatus(XNap.tr("Please select something first"));
	    }
	    else {
		for (int i = 0; i < transfers.length; i++) {
		    if (!queue.abort(transfers[i])) {
			setStatus(XNap.tr("Transfer not running"));
		    }
		}
	    }
        }
    }

    /**
     * Clears all finished transfers.
     */
    private class ClearAction extends AbstractAction {

        public ClearAction() 
	{
            putValue(Action.NAME, XNap.tr("Clear finished"));
            putValue(Action.SHORT_DESCRIPTION,
		     XNap.tr("Clear all finished transfers"));
	    putValue(Action.SMALL_ICON, XNapFrame.getIcon("edittrash.png"));
        }

        public void actionPerformed(ActionEvent event) 
	{
	    TransferQueue queue = getSelectedQueue();
	    if (queue == null) {
		//setStatus("Please select a table first.");
		DownloadQueue.getInstance().clearFinished();
		UploadQueue.getInstance().clearFinished();
	    }
	    else {
		queue.clearFinished();
	    }
        }

    }

    /**
     *
     */
    private class DeleteAction extends AbstractAction
    {
	public DeleteAction()
	{
	    putValue(Action.NAME, XNap.tr("Delete"));
            putValue(Action.SHORT_DESCRIPTION,
		     XNap.tr("Delete incomplete file"));
	    putValue(Action.SMALL_ICON, XNapFrame.getIcon("delete.png"));
            putValue(Action.MNEMONIC_KEY, new Integer('D'));
	}
	    
        public void actionPerformed(ActionEvent event) 
	{
	    if (getSelectedTable() != jtaDownloads) {
		setStatus(XNap.tr("You can only delete downloads."));
		return;
	    }
	    ITransferContainer[] transfers = getSelectedTransfers();
	    DownloadQueue dlQueue = (DownloadQueue)getSelectedQueue();

	    if (transfers.length == 0) {
		setStatus(XNap.tr("Please select something first"));
	    }
	    else {
		transfers = 
		    Dialogs.showDeleteDownloadsDialog(TransferSubPanel.this, 
						      transfers);
		if (transfers != null) {
		    for (int i = 0; i < transfers.length; i++) {
			dlQueue.delete((IDownloadContainer)transfers[i]);
		    }
		}
	    }	    
        }
    }

    private class ResumeAction extends AbstractAction
    {
	public ResumeAction()
	{
	    putValue(Action.NAME, XNap.tr("Resume"));
            putValue(Action.SHORT_DESCRIPTION, XNap.tr("Resume Download"));
	    putValue(Action.SMALL_ICON, XNapFrame.getIcon("reload3.png"));
            putValue(Action.MNEMONIC_KEY, new Integer('R'));
	}
	    
        public void actionPerformed(ActionEvent event) 
	{
	    if (getSelectedTable() != jtaDownloads) {
		setStatus(XNap.tr("You can only resume downloads."));
		return;
	    }
	    ITransferContainer[] transfers = getSelectedTransfers();
	    DownloadQueue dlQueue = (DownloadQueue)getSelectedQueue();

	    if (transfers.length == 0) {
		setStatus(XNap.tr("Please select something first"));
	    }
	    else {
		for (int i = 0; i < transfers.length; i++) {
		    IDownloadContainer d = (IDownloadContainer)transfers[i];
		    if (!dlQueue.resume(d)) {
			d.skip();
			setStatus(XNap.tr("Download already in progress or finished"));
		    }
		}
	    }	    
        }
    }

    private class FocusSelectionListener extends FocusAdapter
    {
	public void focusGained(FocusEvent e)
	{
	    if (e.getSource() == jtaDownloads) {
		jtaUploads.getSelectionModel().clearSelection();
	    }
	    else if (e.getSource() == jtaUploads) {
		jtaDownloads.getSelectionModel().clearSelection();
	    }
	}
    }

    private class EditFilterAction extends AbstractAction
    {
	public EditFilterAction()
	{
	    putValue(Action.NAME, XNap.tr("Edit search filter"));
            putValue(Action.SHORT_DESCRIPTION, XNap.tr("Edit search filter"));
	    putValue(Action.SMALL_ICON, XNapFrame.getIcon("edit.png"));
            putValue(Action.MNEMONIC_KEY, new Integer('E'));
	}
	    
        public void actionPerformed(ActionEvent event) 
	{
	    ITransferContainer[] transfers = getSelectedTransfers();

	    if (transfers.length == 0) {
		setStatus(XNap.tr("Please select something first"));
	    }
	    else if (transfers[0] instanceof AutoDownload) {
		AutoDownload ad = (AutoDownload)transfers[0];
		AutoDownloadEditorDialog.showDialog(TransferSubPanel.this, ad);
	    }
	    else {
		setStatus(XNap.tr("This search filter can not be edited"));
	    }
	}
    }

    private class StartAction extends AbstractAction
    {
	public StartAction()
	{
	    putValue(Action.NAME, XNap.tr("Start"));
            putValue(Action.SHORT_DESCRIPTION, XNap.tr("Start Upload"));
	    putValue(Action.SMALL_ICON, XNapFrame.getIcon("misc.png"));
            putValue(Action.MNEMONIC_KEY, new Integer('S'));
	}
	    
        public void actionPerformed(ActionEvent event) 
	{
	    if (getSelectedTable() != jtaUploads) {
		setStatus(XNap.tr("You can only start downloads"));
		return;
	    }
	    ITransferContainer[] transfers = getSelectedTransfers();
	    UploadQueue ulQueue = (UploadQueue)getSelectedQueue();

	    if (transfers.length == 0) {
		setStatus(XNap.tr("Please select something first"));
	    }
	    else {
		for (int i = 0; i < transfers.length; i++) {
		    IUploadContainer u = (IUploadContainer)transfers[i];
		    ulQueue.startUpload(u);
		}
	    }	    
        }
    }

    /**
     * Resumes a transfer if it isn't started yet and opens it otherwise.
     */
    private class ResumeSlashOpenAction extends AbstractAction
    {

	public ResumeSlashOpenAction()
	{
	}
	
	public void actionPerformed(ActionEvent event)
	{
	    if (getSelectedTable() != jtaDownloads) {
		return;
	    }

	    ITransferContainer[] transfers = getSelectedTransfers();
	    DownloadQueue dlQueue = (DownloadQueue)getSelectedQueue();

	    if (transfers.length == 0) {
		setStatus(XNap.tr("Please select something first"));
	    }
	    else {
		for (int i = 0; i < transfers.length; i++) {
		    IDownloadContainer d = (IDownloadContainer)transfers[i];
		    if (d.isResumable()) {
			if (!dlQueue.resume(d)) {
			    setStatus(XNap.tr("Download already in progress or finished"));
			}
		    }
		    else {
			try {
			    LauncherSupport.open(d.getFile());
			}
			catch (IOException e) {
			    Console.getInstance().println(e.getMessage());
			}
		    }
		}
	    }	    
	    
	}
    }
}
