/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui;

import xnap.XNap;
import xnap.user.UserManager;
import xnap.net.IUser;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;

public class UserEditorPanel extends AbstractSaveablePanel {
    
    //--- Data field(s) ---

    private IUser user;

    private JTextField jtName;
    private JTextField jtComment;
    private JComboBox jcbCategory;

    private JCheckBox jcbIgnoreChat;

    private JRadioButton jrbUploadDefault;
    private JRadioButton jrbUploadNever;
    private JRadioButton jrbUploadAlways;
    private ButtonGroup bgUpload;

    private JRadioButton jrbDownloadDefault;
    private JRadioButton jrbDownloadNever;
    private ButtonGroup bgDownload;

    //--- Constructor(s) ---
    
    public UserEditorPanel(IUser user)
    {
	initialize();

	setUser(user);
    }

    public UserEditorPanel()
    {
	initialize();
    }

    // --- Methods ---

    private void initialize()
    {
	setLayout(new GridBagLayout());

	GridBagHelper.addLabel(this, XNap.tr("Name"));
	jtName = new ValidatedTextField("", 20, ValidatedTextField.EMAIL);
	jtName.setEditable(false);
	GridBagHelper.add(this, jtName);

	GridBagHelper.addLabel(this, XNap.tr("Comment"));
	jtComment = new JTextField("", 20);
	GridBagHelper.add(this, jtComment);

	GridBagHelper.addLabel(this, XNap.tr("Category"));
	jcbCategory = new JComboBox(UserManager.CATEGORIES);
	GridBagHelper.add(this, jcbCategory);

	GridBagHelper.addLabel(this, "");
	jcbIgnoreChat = new JCheckBox(XNap.tr("Ignore Chat"));
	GridBagHelper.add(this, jcbIgnoreChat);

	// uploads
	JPanel jpUpload = new JPanel(new GridBagLayout());
	jpUpload.setBorder
	    (new TitledBorder(XNap.tr("Allow Uploads To This User", 1)));
	GridBagHelper.add(this, jpUpload);
	
	jrbUploadDefault = new JRadioButton(XNap.tr("Default"));
	jrbUploadDefault.setActionCommand(IUser.TRANSFER_DEFAULT + "");
	GridBagHelper.add(jpUpload, jrbUploadDefault);

	jrbUploadNever = new JRadioButton(XNap.tr("Never"));
	jrbUploadNever.setActionCommand(IUser.TRANSFER_NEVER + "");
	GridBagHelper.add(jpUpload, jrbUploadNever);

	jrbUploadAlways = new JRadioButton(XNap.tr("Always"));
	jrbUploadAlways.setActionCommand(IUser.TRANSFER_UNLIMITED + "");
	GridBagHelper.add(jpUpload, jrbUploadAlways);

	bgUpload = new ButtonGroup();
	bgUpload.add(jrbUploadDefault);
	bgUpload.add(jrbUploadNever);
	bgUpload.add(jrbUploadAlways);

	// downloads
	JPanel jpDownload = new JPanel(new GridBagLayout());
	jpDownload.setBorder
	    (new TitledBorder(XNap.tr("Allow Downloads From This User", 1)));
	GridBagHelper.add(this, jpDownload);

	jrbDownloadDefault = new JRadioButton(XNap.tr("Default"));
	jrbDownloadDefault.setActionCommand(IUser.TRANSFER_DEFAULT + "");
	GridBagHelper.add(jpDownload, jrbDownloadDefault);

	jrbDownloadNever = new JRadioButton(XNap.tr("Never"));
	jrbDownloadNever.setActionCommand(IUser.TRANSFER_NEVER + "");
	GridBagHelper.add(jpDownload, jrbDownloadNever);

	bgDownload = new ButtonGroup();
	bgDownload.add(jrbDownloadDefault);
	bgDownload.add(jrbDownloadNever);
    }

    public void apply()
    {
	user.setComment(jtComment.getText());
	user.setCategory(jcbCategory.getSelectedItem().toString());

	user.setChatIgnored(jcbIgnoreChat.isSelected());

	user.setMaxUploads(getValue(bgUpload));

	user.setMaxDownloads(getValue(bgDownload));

	user.setTemporary(false);
    }

    public IUser getUser()
    {
	return user;
    }

    public boolean hasChanged()
    {
	boolean dirty = false;

	dirty |= !(jtComment.getText().equals(user.getComment()));
	dirty |= !(jcbCategory.getSelectedItem().toString().equals
		   (user.getCategory()));
	dirty |= !(jcbIgnoreChat.isSelected() == user.isChatIgnored());

	dirty |= !(user.getMaxUploads() == getValue(bgUpload));

	dirty |= !(user.getMaxDownloads() == getValue(bgDownload));

	return dirty;
    }

    public void setUser(IUser user)
    {
	this.user = user;

	jtName.setText(user.getName());
	jtComment.setText(user.getComment());
	jcbCategory.setSelectedItem(user.getCategory());

	jcbIgnoreChat.setSelected(user.isChatIgnored());

	setValue(bgDownload, user.getMaxDownloads());

	setValue(bgUpload, user.getMaxUploads());
    }

    private int getValue(ButtonGroup bg)
    {
	return Integer.parseInt(bg.getSelection().getActionCommand());
    }

    private void setValue(ButtonGroup bg, int value)
    {
	for (Enumeration e = bg.getElements(); e.hasMoreElements();) {
	    AbstractButton b = (AbstractButton)e.nextElement();
	    if (b.getActionCommand().equals(value + "")) {
		b.setSelected(true);
		return;
	    }
	}

    }

}
